<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_Updater {

    protected string $plugin_file;
    protected string $plugin_basename;
    protected string $api_base_url = 'https://aegisify.com/wp-json/chue-license/v1';
    protected string $product_slug = 'aegiswaf';

    public function __construct() {
        $this->plugin_file = defined( 'AEGISWAF_PATH' ) ? AEGISWAF_PATH . 'aegiswaf.php' : __FILE__;
        $this->plugin_basename = plugin_basename( $this->plugin_file );
    }

    public function init() : void {
        // Allow override.
        $this->api_base_url = (string) apply_filters( 'aegiswaf_clm_api_base', $this->api_base_url );

        /**
         * Updates are intentionally disabled for WordPress.org hosted builds.
         * Licensing remains handled via AegisWAF_License_Manager.
         */
    }

    public function check_updates( $transient ) {
        if ( ! is_object( $transient ) ) { return $transient; }

        $current = defined( 'AEGISWAF_VERSION' ) ? AEGISWAF_VERSION : '0.0.0';

        $info = $this->remote_update_info( 'update' );
        if ( ! is_array( $info ) || empty( $info['version'] ) || empty( $info['download_url'] ) ) {
            return $transient;
        }

        if ( version_compare( (string) $info['version'], (string) $current, '<=' ) ) {
            return $transient;
        }

        $item = new stdClass();
        $item->slug = $this->product_slug;
        $item->plugin = $this->plugin_basename;
        $item->new_version = (string) $info['version'];
        $item->url = (string) ( $info['homepage'] ?? '' );
        $item->package = (string) $info['download_url'];
        if ( ! empty( $info['requires'] ) ) { $item->requires = (string) $info['requires']; }
        if ( ! empty( $info['requires_php'] ) ) { $item->requires_php = (string) $info['requires_php']; }
        if ( ! empty( $info['tested'] ) ) { $item->tested = (string) $info['tested']; }

        $transient->response[ $this->plugin_basename ] = $item;
        return $transient;
    }

    public function plugins_api( $result, $action, $args ) {
        if ( 'plugin_information' !== $action ) { return $result; }
        if ( empty( $args->slug ) || $args->slug !== $this->product_slug ) { return $result; }

        $info = $this->remote_update_info( 'info' );
        if ( ! is_array( $info ) || empty( $info['version'] ) ) {
            return $result;
        }

        $obj = new stdClass();
        $obj->name = (string) ( $info['name'] ?? 'AegisWAF – Web Application Firewall' );
        $obj->slug = $this->product_slug;
        $obj->version = (string) $info['version'];
        $obj->author = (string) ( $info['author'] ?? 'AI Technology Avenue' );
        $obj->homepage = (string) ( $info['homepage'] ?? '' );
        $obj->requires = (string) ( $info['requires'] ?? '6.8' );
        $obj->requires_php = (string) ( $info['requires_php'] ?? '8.2' );
        $obj->tested = (string) ( $info['tested'] ?? '' );
        $obj->last_updated = (string) ( $info['last_updated'] ?? '' );
        $obj->download_link = (string) ( $info['download_url'] ?? '' );
        $obj->sections = is_array( $info['sections'] ?? null ) ? $info['sections'] : [ 'description' => 'AegisWAF update details not provided.' ];

        return $obj;
    }

    protected function remote_update_info( string $endpoint ) : ?array {
        $lm = new AegisWAF_License_Manager();
        $lic = $lm->get_license_data();

        $body = [
            'site_url'    => home_url(),
            'plugin_slug' => $this->product_slug,
            'version'     => defined( 'AEGISWAF_VERSION' ) ? AEGISWAF_VERSION : '',
            'license_key' => (string) ( $lic['license_key'] ?? '' ),
            'email'       => (string) ( $lic['license_email'] ?? '' ),
        ];

        $url = trailingslashit( $this->api_base_url ) . $this->product_slug . '/' . $endpoint;
        $resp = wp_remote_post( $url, [ 'timeout' => 15, 'body' => $body ] );
        if ( is_wp_error( $resp ) ) { return null; }

        $code = (int) wp_remote_retrieve_response_code( $resp );
        $raw  = (string) wp_remote_retrieve_body( $resp );
        if ( $code !== 200 || $raw === '' ) { return null; }

        $data = json_decode( $raw, true );
        return is_array( $data ) ? $data : null;
    }
}
