<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_GeoASN_Provider {

    public static function resolve( string $ip ) : ?array {
        $intel = AegisWAF_Intel::get();
        $provider = (string) ( $intel['geo_provider'] ?? 'none' );

        if ( $provider === 'cloudflare' ) {
            return self::from_cloudflare_headers( $ip );
        }

        if ( $provider === 'maxmind_ext' ) {
            return self::from_maxmind_extension( $ip );
        }

        return null;
    }

    public static function register_filter() : void {
        add_filter( 'aegiswaf_geoasn_provider', function() {
            return function( $ip ) {
                return AegisWAF_GeoASN_Provider::resolve( (string) $ip );
            };
        } );
    }

    protected static function from_cloudflare_headers( string $ip ) : ?array {

        $cc = '';
        if ( isset( $_SERVER['HTTP_CF_IPCOUNTRY'] ) ) {
			$cc = strtoupper( sanitize_text_field( wp_unslash( (string) $_SERVER['HTTP_CF_IPCOUNTRY'] ) ) );
			$cc = preg_replace( '/[^A-Z]/', '', $cc );
			$cc = substr( $cc, 0, 2 );
        }
        if ( $cc === '' ) { return null; }
        return [
            'country' => $cc,
            'asn'     => '', // not available via standard CF headers
            'source'  => 'cloudflare_headers',
        ];
    }

    protected static function from_maxmind_extension( string $ip ) : ?array {
        if ( ! function_exists( 'maxminddb_open' ) ) {
            update_option( 'aegiswaf_mmdb_notice', 'missing_ext', false );
            return null;
        }

        $intel = AegisWAF_Intel::get();
        $path = (string) ( $intel['maxmind_mmdb_path'] ?? '' );
        $path = trim( $path );
        if ( $path === '' || ! file_exists( $path ) ) { return null; }

        $db = @maxminddb_open( $path );
        if ( ! $db ) { return null; }

        $res = @maxminddb_lookup_string( $db, $ip );
        @maxminddb_close( $db );
        if ( ! is_array( $res ) ) { return null; }

        $cc = '';
        if ( isset( $res['country']['iso_code'] ) ) {
            $cc = strtoupper( (string) $res['country']['iso_code'] );
        } elseif ( isset( $res['registered_country']['iso_code'] ) ) {
            $cc = strtoupper( (string) $res['registered_country']['iso_code'] );
        }

        $asn = '';
        if ( isset( $res['autonomous_system_number'] ) ) {
            $asn = 'AS' . (string) $res['autonomous_system_number'];
        } elseif ( isset( $res['traits']['autonomous_system_number'] ) ) {
            $asn = 'AS' . (string) $res['traits']['autonomous_system_number'];
        }

        if ( $cc === '' && $asn === '' ) { return null; }

        return [
            'country' => $cc,
            'asn'     => strtoupper( $asn ),
            'source'  => 'maxmind_mmdb_ext',
        ];
    }
}
