<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_Engine {

    public static function init() : void {

        add_action( 'template_redirect', [ __CLASS__, 'inspect_front_requests' ], 1 );
        add_filter( 'rest_pre_dispatch', [ __CLASS__, 'rest_enforce_policies' ], 10, 3 );
    }

    public static function inspect_front_requests() : void {
        if ( is_admin() ) { return; }
        if ( wp_doing_ajax() ) { return; }
        if ( defined( 'WP_CLI' ) && WP_CLI ) { return; }

        $settings = AegisWAF_Storage::get_settings();
		
        if ( ! class_exists( 'AegisWAF_Logger' ) || ! method_exists( 'AegisWAF_Logger', 'log' ) ) {
            error_log( '[AegisWAF] Logger missing or invalid (inspect_front_requests) — enforcement will be limited.' );
        }
        if ( ! class_exists( 'AegisWAF_Normalizer' ) || ! method_exists( 'AegisWAF_Normalizer', 'normalize_front' ) ) {
            if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
                AegisWAF_Logger::log( '(unknown)', 'GET', 'engine', 'missing_dependency', [
                    'missing' => 'AegisWAF_Normalizer::normalize_front',
                    'scope'   => 'front',
                ] );
            }
            error_log( '[AegisWAF] Missing dependency: AegisWAF_Normalizer::normalize_front — allowing request.' );
            return;
        }

        $payload = AegisWAF_Normalizer::normalize_front();

        $path = (string) ( $payload['path'] ?? '' );
        if ( $path === '' ) { return; }

        if ( isset( $_SERVER['REQUEST_URI'] ) ) {
            $path = (string) wp_parse_url( (string) $_SERVER['REQUEST_URI'], PHP_URL_PATH );
        }
        if ( $path === '' ) { return; }

        $method = isset( $_SERVER['REQUEST_METHOD'] ) ? (string) $_SERVER['REQUEST_METHOD'] : 'GET';
        $ua     = (string) AegisWAF_Utils::user_agent();
        $ip     = (string) AegisWAF_Utils::client_ip();
		
		if ( class_exists( 'AegisWAF_Overrides' ) && AegisWAF_Overrides::is_allowed( (string) $ip, (string) $path, (string) $method ) ) {
			if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
				AegisWAF_Logger::log( (string) $path, (string) $method, 'override', 'allow', [
					'ip' => (string) $ip,
					'ua' => (string) $ua,
					'reason' => 'allow override matched (logs triage)',
				] );
			}
			return;
		}
		
		if ( class_exists( 'AegisWAF_Logger' ) ) {
			AegisWAF_Logger::log(
				$path,
				$method,
				'engine',
				'request_seen',
				[ 'ip' => $ip, 'ua' => $ua, 'scope' => 'front' ]
			);
		}

		$ddos = null;

		$ddos_settings = AegisWAF_DDoS_Storage::get();

		if ( ! class_exists( 'AegisWAF_DDoS_Engine' ) ) {
			if ( ! empty( $ddos_settings['enabled'] ) ) {
				if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
					AegisWAF_Logger::log( $path, $method, 'ddos_shield', 'engine_missing', [
						'missing' => 'AegisWAF_DDoS_Engine',
						'ip'      => $ip,
						'ua'      => $ua,
					] );
				} else {
					error_log( '[AegisWAF] DDoS enabled but AegisWAF_DDoS_Engine missing' );
				}
			}
		} else {
			$ddos = AegisWAF_DDoS_Engine::check_front( $ip, $path, $method, $ua );
		}

		if ( is_array( $ddos ) ) {
			AegisWAF_Logger::log( $path, $method, 'ddos_shield', (string) ( $ddos['action'] ?? 'blocked' ), $ddos );

			$act = (string) ( $ddos['action'] ?? '' );
			if ( $act === 'challenge' ) {
				AegisWAF_Challenge::render_page( esc_html__( 'Verifying your browser…', 'aegiswaf' ) );
			}
			if ( $act === 'rate_limit' ) {
				wp_die(
					esc_html__( 'AegisWAF: Too many requests.', 'aegiswaf' ),
					esc_html__( 'Rate limited', 'aegiswaf' ),
					[ 'response' => 429 ]
				);
			}

			wp_die(
				esc_html__( 'AegisWAF: Access denied.', 'aegiswaf' ),
				esc_html__( 'Blocked', 'aegiswaf' ),
				[ 'response' => 403 ]
			);
		}

        if ( $ip !== '' && AegisWAF_Intel::is_ip_blocked( $ip ) ) {
            AegisWAF_Logger::log( $path, $method, 'intel_ip_block', 'blocked', [ 'ip' => $ip ] );
            wp_die( esc_html__( 'AegisWAF: Access denied.', 'aegiswaf' ), esc_html__( 'Blocked', 'aegiswaf' ), [ 'response' => 403 ] );
        }

        if ( $ip !== '' && AegisWAF_Intel::geo_asn_blocked( $ip ) ) {
            AegisWAF_Logger::log( $path, $method, 'intel_geo_asn_block', 'blocked', [ 'ip' => $ip ] );
            wp_die( esc_html__( 'AegisWAF: Access denied.', 'aegiswaf' ), esc_html__( 'Blocked', 'aegiswaf' ), [ 'response' => 403 ] );
        }

        $is_pro = AegisWAF_Features::is_pro();
        $settings = AegisWAF_Storage::get_settings();
        $mr = is_array( $settings['managed_rules'] ?? null ) ? $settings['managed_rules'] : [];
        $base_cats = is_array( $mr['categories'] ?? null ) ? $mr['categories'] : [];
        $base_thr  = is_array( $mr['thresholds'] ?? null ) ? $mr['thresholds'] : [];
        $policy = AegisWAF_Endpoint_Policy::match_policy( $path );
        $cats = AegisWAF_Endpoint_Policy::effective_categories( $base_cats, $policy );
        $thr  = AegisWAF_Endpoint_Policy::effective_thresholds( $base_thr, $policy );
        $action = AegisWAF_Endpoint_Policy::effective_action( $is_pro ? (string) ( $mr['mode_pro'] ?? 'block' ) : (string) ( $mr['mode_free'] ?? 'log' ), $policy, $is_pro );
        if ( ! empty( $policy ) ) {
            AegisWAF_Logger::log( $path, $method, 'endpoint_policy', 'applied', [
                'ip'            => $ip,
                'ua'            => $ua,
                'path_pattern'  => (string) ( $policy['path_pattern'] ?? '' ),
                'effective_action' => $action,
                'effective_categories' => $cats,
                'effective_thresholds' => $thr,
            ] );
        }
		$ctx = [ 'categories' => $cats, 'thresholds' => $thr, 'action' => $action ];

        $mhit = AegisWAF_Managed_Rules::evaluate( $payload, $is_pro, $ctx );
        if ( is_array( $mhit ) ) {
            AegisWAF_Logger::log( $path, $method, 'managed_rule', (string) $mhit['action'], $mhit + [ 'ip' => $ip, 'ua' => $ua ] );
            $act = (string) $mhit['action'];
            if ( $act === 'rate_limit' ) {
                wp_die( esc_html__( 'AegisWAF: Too many requests.', 'aegiswaf' ), esc_html__( 'Rate limited', 'aegiswaf' ), [ 'response' => 429 ] );
            }
            if ( $act === 'challenge' ) {
                if ( AegisWAF_Challenge::enabled() ) {
                    if ( ! AegisWAF_Challenge::verify( $ip ) ) {
                        AegisWAF_Challenge::render_page( 'Challenge required to continue.' );
                    }
                }
                wp_die( esc_html__( 'AegisWAF: Challenge required.', 'aegiswaf' ), esc_html__( 'Challenge', 'aegiswaf' ), [ 'response' => 403 ] );
            }
            if ( $act === 'block' ) {
                wp_die( esc_html__( 'AegisWAF: Request blocked.', 'aegiswaf' ), esc_html__( 'Blocked', 'aegiswaf' ), [ 'response' => 403 ] );
            }
        }

        $hhit = AegisWAF_Heuristics::evaluate( $payload, $ctx );
        if ( is_array( $hhit ) ) {
            AegisWAF_Logger::log( $path, $method, 'heuristic', (string) ( $hhit['action'] ?? 'log' ), $hhit + [ 'ip' => $ip, 'ua' => $ua ] );
            $hact = (string) ( $hhit['action'] ?? 'log' );
            if ( $hact === 'rate_limit' ) {
                wp_die( esc_html__( 'AegisWAF: Too many requests.', 'aegiswaf' ), esc_html__( 'Rate limited', 'aegiswaf' ), [ 'response' => 429 ] );
            }
            if ( $hact === 'challenge' && AegisWAF_Challenge::enabled() && ! AegisWAF_Challenge::verify( $ip ) ) {
                AegisWAF_Challenge::render_page( 'Challenge required to continue.' );
            }
            if ( $hact === 'block' ) {
                wp_die( esc_html__( 'AegisWAF: Request blocked.', 'aegiswaf' ), esc_html__( 'Blocked', 'aegiswaf' ), [ 'response' => 403 ] );
            }
        }

        $hhit = AegisWAF_Heuristics::evaluate( $payload, $ctx );
        if ( is_array( $hhit ) ) {
            AegisWAF_Logger::log( $path, $method, 'heuristic', (string) ( $hhit['action'] ?? 'log' ), $hhit + [ 'ip' => $ip, 'ua' => $ua ] );
            $hact = (string) ( $hhit['action'] ?? 'log' );
            if ( $hact === 'rate_limit' ) {
                wp_die( esc_html__( 'AegisWAF: Too many requests.', 'aegiswaf' ), esc_html__( 'Rate limited', 'aegiswaf' ), [ 'response' => 429 ] );
            }
            if ( $hact === 'challenge' ) {
                if ( AegisWAF_Challenge::enabled() && ! AegisWAF_Challenge::verify( $ip ) ) {
                    AegisWAF_Challenge::render_page( 'Challenge required to continue.' );
                }
                return; // OK in void
            }
            if ( $hact === 'block' ) {
                wp_die( esc_html__( 'AegisWAF: Request blocked.', 'aegiswaf' ), esc_html__( 'Blocked', 'aegiswaf' ), [ 'response' => 403 ] );
            }
        }

        $hit = AegisWAF_Rules::evaluate( $path, $method, (string) ( $payload['norm'] ?? '' ) );
        if ( is_array( $hit ) ) {
        $hit = AegisWAF_Managed_Rules::evaluate( $payload, $is_pro, $ctx );
        if ( is_array( $hit ) ) {

            $act = (string) ( $hit['action'] ?? 'log' );

            AegisWAF_Logger::log( $path, $method, 'managed_rules', $act, $hit + [
                'ip'   => $ip,
                'ua'   => $ua,
                'scope'=> 'front',
            ] );

            if ( $act === 'rate_limit' ) {
                wp_die(
                    esc_html__( 'AegisWAF: Too many requests.', 'aegiswaf' ),
                    esc_html__( 'Rate limited', 'aegiswaf' ),
                    [ 'response' => 429 ]
                );
            }

            if ( $act === 'challenge' ) {
                if ( ! class_exists( 'AegisWAF_Challenge' ) ) {
                    AegisWAF_Logger::log( $path, $method, 'managed_rules', 'challenge_unavailable', [
                        'reason' => 'AegisWAF_Challenge class missing',
                        'ip'     => $ip,
                        'ua'     => $ua,
                    ] );
                    wp_die(
                        esc_html__( 'AegisWAF: Access denied.', 'aegiswaf' ),
                        esc_html__( 'Blocked', 'aegiswaf' ),
                        [ 'response' => 403 ]
                    );
                }

                if ( AegisWAF_Challenge::enabled() && ! AegisWAF_Challenge::verify( $ip ) ) {
                    AegisWAF_Logger::log( $path, $method, 'challenge', 'issued', [
                        'ip'   => $ip,
                        'ua'   => $ua,
                        'from' => 'managed_rules',
                    ] );
                    AegisWAF_Challenge::render_page( 'Challenge required to continue.' );
                }

                AegisWAF_Logger::log( $path, $method, 'challenge', 'passed_or_disabled', [
                    'ip'   => $ip,
                    'ua'   => $ua,
                    'from' => 'managed_rules',
                ] );
                return;
            }

            if ( $act === 'block' ) {
                wp_die(
                    esc_html__( 'AegisWAF: Access denied.', 'aegiswaf' ),
                    esc_html__( 'Blocked', 'aegiswaf' ),
                    [ 'response' => 403 ]
                );
            }
        }

        $score = AegisWAF_Intel::behavioral_score( 'front', $path );
        if ( $score > 0 ) {
            AegisWAF_Behavioral_Timeline::incr();
            $thr = (int) ( ( AegisWAF_Intel::get()['behavioral_threshold'] ?? 80 ) );
            if ( $score >= $thr ) {
                AegisWAF_Logger::log( $path, $method, 'intel_behavioral', 'rate_limited', [
                    'score'     => $score,
                    'threshold' => $thr,
                    'ip'        => $ip,
                    'ua'        => $ua,
                ] );

                wp_die(
                    esc_html__( 'AegisWAF: Too many requests.', 'aegiswaf' ),
                    esc_html__( 'Rate limited', 'aegiswaf' ),
                    [ 'response' => 429 ]
                );
            }
        }

        $bot = is_array( $settings['bot'] ?? null ) ? $settings['bot'] : [];
        if ( ! empty( $bot['enabled'] ) ) {
            if ( ! empty( $bot['block_bad_ua'] ) ) {
                $bad = (string) ( $bot['bad_ua_list'] ?? '' );
                foreach ( preg_split( "/\r\n|\r|\n/", $bad ) as $needle ) {
                    $needle = trim( (string) $needle );
                    if ( $needle === '' ) { continue; }
                                        if ( stripos( $ua, $needle ) !== false ) {
                        AegisWAF_Logger::log( $path, $method, 'bot_bad_ua', 'blocked', [
                            'ua'    => $ua,
                            'match' => $needle,
                            'ip'    => $ip,
                        ] );

                        wp_die(
                            esc_html__( 'AegisWAF: Bot blocked.', 'aegiswaf' ),
                            esc_html__( 'Blocked', 'aegiswaf' ),
                            [ 'response' => 403 ]
                        );
                    }
                }
            }
        }

		$default_limit = (int) ( $bot['default_limit_per_min'] ?? 0 );
		if ( $default_limit <= 0 ) {
			$default_limit = 180;
			AegisWAF_Logger::log( $path, $method, 'bot_control', 'config_missing', [
				'ip' => $ip,
				'ua' => $ua,
				'missing' => 'bot.default_limit_per_min (<=0); defaulting to 180',
			] );
		}

		$matched_pattern = '*';
		$limit = $default_limit;

		$rows = is_array( $bot['per_path_limits'] ?? null ) ? $bot['per_path_limits'] : [];
		if ( ! empty( $rows ) ) {
			$best_len = -1;
			foreach ( $rows as $row ) {
				if ( ! is_array( $row ) ) { continue; }
				$p = (string) ( $row['pattern'] ?? '' );
				$l = (int) ( $row['limit_per_min'] ?? 0 );
				if ( $p === '' || $l <= 0 ) { continue; }

				$rx = '#^' . str_replace( '\*', '.*', preg_quote( $p, '#' ) ) . '$#i';
				if ( preg_match( $rx, (string) $path ) ) {
					$plen = strlen( $p );
					if ( $plen > $best_len ) {
						$best_len = $plen;
						$matched_pattern = $p;
						$limit = max( 1, min( 10000, $l ) );
					}
				}
			}
		}

		$key = 'aegiswaf_bot_' . md5( $ip . '|' . $matched_pattern . '|front' );
		$data = get_transient( $key );
		if ( ! is_array( $data ) ) {
			$data = [ 'window_start' => 0, 'count' => 0 ];
		}

		$now = time();
		$window_start = (int) ( floor( $now / 60 ) * 60 );

		if ( (int) ( $data['window_start'] ?? 0 ) !== $window_start ) {
			$data['window_start'] = $window_start;
			$data['count'] = 0;
		}

		$data['count'] = (int) ( $data['count'] ?? 0 ) + 1;

		set_transient( $key, $data, 120 );

		if ( (int) $data['count'] === 1 ) {
			AegisWAF_Logger::log( $path, $method, 'bot_control', 'allow', [
				'ip' => $ip,
				'ua' => $ua,
				'matched_pattern' => $matched_pattern,
				'limit_per_min' => $limit,
				'window_start' => $window_start,
				'scope' => 'front',
			] );
		}

		if ( (int) $data['count'] > $limit ) {
			AegisWAF_Logger::log( $path, $method, 'bot_control', 'rate_limit', [
				'ip' => $ip,
				'ua' => $ua,
				'matched_pattern' => $matched_pattern,
				'limit_per_min' => $limit,
				'count' => (int) $data['count'],
				'window_start' => $window_start,
				'scope' => 'front',
			] );

			wp_die(
				esc_html__( 'AegisWAF: Too many requests.', 'aegiswaf' ),
				esc_html__( 'Rate limited', 'aegiswaf' ),
				[ 'response' => 429 ]
			);
		}
    }
	}

    public static function rest_enforce_policies( $result, $server, $request ) {
        if ( defined( 'WP_CLI' ) && WP_CLI ) { return $result; }

        $settings = AegisWAF_Storage::get_settings();
		if ( ! class_exists( 'AegisWAF_Logger' ) || ! method_exists( 'AegisWAF_Logger', 'log' ) ) {
            error_log( '[AegisWAF] Logger missing or invalid (rest_enforce_policies) — enforcement will be limited.' );
        }

        $route = '';
        if ( is_object( $request ) && method_exists( $request, 'get_route' ) ) {
            $route = (string) $request->get_route();
        }
        if ( $route === '' && isset( $_SERVER['REQUEST_URI'] ) ) {
            $route = (string) wp_parse_url( (string) $_SERVER['REQUEST_URI'], PHP_URL_PATH );
        }
        if ( $route === '' ) { return $result; }

        $path = $route;

        $method = (string) ( is_object( $request ) && method_exists( $request, 'get_method' ) ? $request->get_method() : ( $_SERVER['REQUEST_METHOD'] ?? 'GET' ) );
        $ua     = (string) AegisWAF_Utils::user_agent();
        $ip     = (string) AegisWAF_Utils::client_ip();
		
		if ( class_exists( 'AegisWAF_Overrides' ) && AegisWAF_Overrides::is_allowed( (string) $ip, (string) $route, (string) $method ) ) {
			if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
				AegisWAF_Logger::log( (string) $route, (string) $method, 'override', 'allow', [
					'ip' => (string) $ip,
					'ua' => (string) $ua,
					'reason' => 'allow override matched (logs triage)',
					'scope' => 'rest',
				] );
			}
			return $result;
		}

		if ( class_exists( 'AegisWAF_DDoS_Engine' ) ) {
			$ddos = AegisWAF_DDoS_Engine::check_rest( (string) $ip, (string) $route, (string) $method, (string) $ua );
			if ( is_array( $ddos ) ) {
				if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
					AegisWAF_Logger::log( (string) $route, (string) $method, 'ddos_shield', (string) ( $ddos['action'] ?? 'blocked' ), $ddos );
				} else {
					error_log( '[AegisWAF] DDoS REST enforcement triggered but logger missing.' );
				}

				$act = (string) ( $ddos['action'] ?? '' );
				if ( $act === 'rate_limit' ) {
					return new WP_Error( 'aegiswaf_ddos_rate_limited', 'AegisWAF: Too many requests.', [ 'status' => 429 ] );
				}
				if ( $act === 'challenge' ) {
					return new WP_Error( 'aegiswaf_ddos_challenge', 'AegisWAF: Challenge required.', [ 'status' => 403 ] );
				}
				return new WP_Error( 'aegiswaf_ddos_blocked', 'AegisWAF: Access denied.', [ 'status' => 403 ] );
			}
		} else {
			if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
				AegisWAF_Logger::log( (string) $route, (string) $method, 'ddos_shield', 'engine_missing_rest', [
					'missing' => 'AegisWAF_DDoS_Engine',
					'ip' => $ip,
				] );
			}
		}

		$bot = is_array( $settings['bot'] ?? null ) ? $settings['bot'] : [];
		if ( ! empty( $bot['enabled'] ) ) {

			$path_for_bot = '';
			if ( isset( $_SERVER['REQUEST_URI'] ) ) {
				$path_for_bot = (string) wp_parse_url( (string) $_SERVER['REQUEST_URI'], PHP_URL_PATH );
			}
			if ( $path_for_bot === '' ) {
				$path_for_bot = (string) $path;
			}

			if ( ! empty( $bot['block_bad_ua'] ) ) {
				$bad = (string) ( $bot['bad_ua_list'] ?? '' );

				if ( trim( $bad ) === '' ) {
					AegisWAF_Logger::log( $path_for_bot, $method, 'bot_control', 'config_missing', [
						'ip'      => $ip,
						'ua'      => $ua,
						'missing' => 'bot.bad_ua_list is empty while bot.block_bad_ua is ON',
						'scope'   => 'rest',
					] );
				} else {
					foreach ( preg_split( "/\r\n|\r|\n/", $bad ) as $needle ) {
						$needle = trim( (string) $needle );
						if ( $needle === '' ) { continue; }

						if ( stripos( $ua, $needle ) !== false ) {
							AegisWAF_Logger::log( $path_for_bot, $method, 'bot_bad_ua', 'blocked', [
								'ua'    => $ua,
								'match' => $needle,
								'ip'    => $ip,
								'scope' => 'rest',
							] );

							return new WP_Error(
								'aegiswaf_bot_bad_ua',
								'AegisWAF: Request blocked.',
								[ 'status' => 403 ]
							);
						}
					}
				}
			}

			$default_limit = (int) ( $bot['default_limit_per_min'] ?? 0 );
			if ( $default_limit <= 0 ) {
				$default_limit = 180;
				AegisWAF_Logger::log( $path_for_bot, $method, 'bot_control', 'config_missing', [
					'ip' => $ip,
					'ua' => $ua,
					'missing' => 'bot.default_limit_per_min (<=0); defaulting to 180',
					'scope' => 'rest',
				] );
			}

			$matched_pattern = '*';
			$limit = $default_limit;

			$rows = is_array( $bot['per_path_limits'] ?? null ) ? $bot['per_path_limits'] : [];
			if ( ! empty( $rows ) ) {
				$best_len = -1;
				foreach ( $rows as $row ) {
					if ( ! is_array( $row ) ) { continue; }
					$p = (string) ( $row['pattern'] ?? '' );
					$l = (int) ( $row['limit_per_min'] ?? 0 );
					if ( $p === '' || $l <= 0 ) { continue; }

					$rx = '#^' . str_replace( '\*', '.*', preg_quote( $p, '#' ) ) . '$#i';
					if ( preg_match( $rx, (string) $path_for_bot ) ) {
						$plen = strlen( $p );
						if ( $plen > $best_len ) {
							$best_len = $plen;
							$matched_pattern = $p;
							$limit = max( 1, min( 10000, $l ) );
						}
					}
				}
			}

			$key = 'aegiswaf_bot_' . md5( $ip . '|' . $matched_pattern . '|rest' );
			$data = get_transient( $key );
			if ( ! is_array( $data ) ) {
				$data = [ 'window_start' => 0, 'count' => 0 ];
			}

			$now = time();
			$window_start = (int) ( floor( $now / 60 ) * 60 );

			if ( (int) ( $data['window_start'] ?? 0 ) !== $window_start ) {
				$data['window_start'] = $window_start;
				$data['count'] = 0;
			}

			$data['count'] = (int) ( $data['count'] ?? 0 ) + 1;
			set_transient( $key, $data, 120 );

			if ( (int) $data['count'] === 1 ) {
				AegisWAF_Logger::log( $path_for_bot, $method, 'bot_control', 'allow', [
					'ip' => $ip,
					'ua' => $ua,
					'matched_pattern' => $matched_pattern,
					'limit_per_min' => $limit,
					'window_start' => $window_start,
					'scope' => 'rest',
				] );
			}

			if ( (int) $data['count'] > $limit ) {
				AegisWAF_Logger::log( $path_for_bot, $method, 'bot_control', 'rate_limit', [
					'ip' => $ip,
					'ua' => $ua,
					'matched_pattern' => $matched_pattern,
					'limit_per_min' => $limit,
					'count' => (int) $data['count'],
					'window_start' => $window_start,
					'scope' => 'rest',
				] );

				return new WP_Error( 'aegiswaf_bot_rate_limited', 'AegisWAF: Too many requests.', [ 'status' => 429 ] );
			}
		}

        $wp = is_array( $settings['wp_protect'] ?? null ) ? $settings['wp_protect'] : [];

        if ( empty( $wp ) ) {
            AegisWAF_Logger::log( $route, $method, 'api_shield', 'config_missing', [
                'ip' => $ip,
                'ua' => $ua,
                'missing' => 'settings.wp_protect',
            ] );
            error_log( '[AegisWAF] API Shield: wp_protect settings missing; REST enforcement skipped.' );
        }

        $api_shield_on = ( ! empty( $wp['enabled'] ) && ! empty( $wp['rest_protect'] ) );

        if ( $api_shield_on ) {

            AegisWAF_Logger::log( $route, $method, 'api_shield', 'request_seen', [
                'ip' => $ip,
                'ua' => $ua,
            ] );

            if ( ! empty( $wp['rest_only_unauth'] ) && is_user_logged_in() ) {
                AegisWAF_Logger::log( $route, $method, 'api_shield', 'allow', [
                    'ip' => $ip,
                    'ua' => $ua,
                    'reason' => 'rest_only_unauth enabled; user logged in',
                ] );
                return $result;
            }

            $allow_raw = (string) ( $wp['rest_allowlist'] ?? '' );
            if ( $allow_raw !== '' ) {
                $allow_lines = preg_split( "/\r\n|\r|\n/", $allow_raw );
                foreach ( $allow_lines as $al ) {
                    $al = trim( (string) $al );
                    if ( $al === '' ) { continue; }
                    if ( strpos( $route, $al ) === 0 ) {
                        AegisWAF_Logger::log( $route, $method, 'api_shield', 'allow', [
                            'ip' => $ip,
                            'ua' => $ua,
                            'reason' => 'route allowlisted',
                            'match' => $al,
                        ] );
                        return $result;
                    }
                }
            }

            if ( ! empty( $wp['rest_block_user_enum'] ) ) {
                if ( strpos( $route, '/wp/v2/users' ) === 0 && strtoupper( $method ) === 'GET' && ! is_user_logged_in() ) {
                    AegisWAF_Logger::log( $route, $method, 'api_shield', 'block', [
                        'ip' => $ip,
                        'ua' => $ua,
                        'reason' => 'rest_block_user_enum',
                    ] );
                    return new WP_Error( 'aegiswaf_users_blocked', 'AegisWAF: User enumeration blocked.', [ 'status' => 403 ] );
                }
            }

            if ( ! empty( $wp['rest_require_write_auth'] ) ) {
                $m = strtoupper( (string) $method );
                if ( in_array( $m, [ 'POST', 'PUT', 'PATCH', 'DELETE' ], true ) && ! is_user_logged_in() ) {
                    AegisWAF_Logger::log( $route, $method, 'api_shield', 'block', [
                        'ip' => $ip,
                        'ua' => $ua,
                        'reason' => 'rest_require_write_auth',
                    ] );
                    return new WP_Error( 'aegiswaf_write_auth_required', 'AegisWAF: Authentication required.', [ 'status' => 403 ] );
                }
            }

            if ( ! empty( $wp['rest_require_json'] ) ) {
                $m = strtoupper( (string) $method );
                if ( in_array( $m, [ 'POST', 'PUT', 'PATCH' ], true ) ) {
                    $ct = '';
                    if ( is_object( $request ) && method_exists( $request, 'get_header' ) ) {
                        $ct = (string) $request->get_header( 'content-type' );
                    } elseif ( isset( $_SERVER['CONTENT_TYPE'] ) ) {
                        $ct = (string) $_SERVER['CONTENT_TYPE'];
                    }
                    if ( stripos( $ct, 'application/json' ) === false ) {
                        AegisWAF_Logger::log( $route, $method, 'api_shield', 'block', [
                            'ip' => $ip,
                            'ua' => $ua,
                            'reason' => 'rest_require_json',
                            'content_type' => $ct,
                        ] );
                        return new WP_Error( 'aegiswaf_json_required', 'AegisWAF: JSON required.', [ 'status' => 415 ] );
                    }
                }
            }

            $max_bytes = (int) ( $wp['rest_max_body_bytes'] ?? 0 );
            if ( $max_bytes > 0 ) {
                $len = 0;
                if ( isset( $_SERVER['CONTENT_LENGTH'] ) ) {
                    $len = (int) $_SERVER['CONTENT_LENGTH'];
                }
                if ( $len > $max_bytes ) {
                    AegisWAF_Logger::log( $route, $method, 'api_shield', 'block', [
                        'ip' => $ip,
                        'ua' => $ua,
                        'reason' => 'rest_max_body_bytes',
                        'content_length' => $len,
                        'max_bytes' => $max_bytes,
                    ] );
                    return new WP_Error( 'aegiswaf_body_too_large', 'AegisWAF: Request body too large.', [ 'status' => 413 ] );
                }
            }

            $cors_raw = (string) ( $wp['rest_cors_allowlist'] ?? '' );
            if ( $cors_raw !== '' ) {
                $origin = '';
                if ( is_object( $request ) && method_exists( $request, 'get_header' ) ) {
                    $origin = (string) $request->get_header( 'origin' );
                } elseif ( isset( $_SERVER['HTTP_ORIGIN'] ) ) {
                    $origin = (string) $_SERVER['HTTP_ORIGIN'];
                }
                if ( $origin !== '' ) {
                    $allowed = false;
                    $cors_lines = preg_split( "/\r\n|\r|\n/", $cors_raw );
                    foreach ( $cors_lines as $o ) {
                        $o = trim( (string) $o );
                        if ( $o === '' ) { continue; }
                        if ( strcasecmp( $o, $origin ) === 0 ) { $allowed = true; break; }
                    }
                    if ( ! $allowed ) {
                        AegisWAF_Logger::log( $route, $method, 'api_shield', 'block', [
                            'ip' => $ip,
                            'ua' => $ua,
                            'reason' => 'rest_cors_allowlist',
                            'origin' => $origin,
                        ] );
                        return new WP_Error( 'aegiswaf_origin_blocked', 'AegisWAF: Origin not allowed.', [ 'status' => 403 ] );
                    }
                }
            }

            if ( ! empty( $wp['rest_api_key_enabled'] ) ) {
                $hdr = strtolower( (string) ( $wp['rest_api_key_header'] ?? 'x-aegis-key' ) );
                $expected = (string) ( $wp['rest_api_key_value'] ?? '' );

                if ( $expected === '' ) {
                    AegisWAF_Logger::log( $route, $method, 'api_shield', 'config_missing', [
                        'ip' => $ip,
                        'ua' => $ua,
                        'missing' => 'rest_api_key_value',
                    ] );
                    error_log( '[AegisWAF] API Shield: API key enabled but rest_api_key_value is empty.' );
                } else {
                    $got = '';
                    if ( is_object( $request ) && method_exists( $request, 'get_header' ) ) {
                        $got = (string) $request->get_header( $hdr );
                    } else {
                        // Fallback: try server-style header name
                        $server_key = 'HTTP_' . strtoupper( str_replace( '-', '_', $hdr ) );
                        if ( isset( $_SERVER[ $server_key ] ) ) {
                            $got = (string) $_SERVER[ $server_key ];
                        }
                    }

                    if ( $got === '' || ! hash_equals( $expected, $got ) ) {
                        AegisWAF_Logger::log( $route, $method, 'api_shield', 'block', [
                            'ip' => $ip,
                            'ua' => $ua,
                            'reason' => 'rest_api_key_enabled',
                            'header' => $hdr,
                            'present' => ( $got !== '' ),
                        ] );
                        return new WP_Error( 'aegiswaf_api_key_required', 'AegisWAF: API key required.', [ 'status' => 403 ] );
                    }

                    AegisWAF_Logger::log( $route, $method, 'api_shield', 'allow', [
                        'ip' => $ip,
                        'ua' => $ua,
                        'reason' => 'api_key_verified',
                        'header' => $hdr,
                    ] );
                }
            }

			$window = (int) ( $wp['window_seconds_rest'] ?? 60 );
			if ( $window < 15 ) { $window = 15; }

			list( $challenge_at, $rate_limit_at, $block_at, $rt_meta ) = self::rest_effective_thresholds(
				is_array( $wp ) ? $wp : [],
				(string) $route,
				(string) $method
			);

			if ( ! empty( $rt_meta['matched'] ) ) {
				AegisWAF_Logger::log( $route, $method, 'api_shield', 'rest_route_override_matched', [
					'ip' => $ip,
					'ua' => $ua,
					'route' => $route,
					'method' => $method,
					'pattern' => (string) ( $rt_meta['pattern'] ?? '' ),
					'category' => (string) ( $rt_meta['category'] ?? '' ),
					'profile' => (string) ( $rt_meta['profile'] ?? '' ),
					'thresholds_applied' => (array) ( $rt_meta['thresholds_applied'] ?? [] ),
					'source' => (string) ( $rt_meta['source'] ?? 'base' ),
				] );
			}

			if ( $challenge_at > 0 || $rate_limit_at > 0 || $block_at > 0 ) {

				$key = 'aegiswaf_rest_rl_' . md5( $ip . '|' . $route . '|' . $method );
				$state = get_transient( $key );
                if ( ! is_array( $state ) ) {
                    $state = [ 'count' => 0 ];
                }
                $state['count'] = (int) ( $state['count'] ?? 0 ) + 1;
                set_transient( $key, $state, $window );

                $count = (int) $state['count'];

                AegisWAF_Logger::log( $route, $method, 'api_shield', 'rate_window', [
                    'ip' => $ip,
                    'ua' => $ua,
                    'count' => $count,
                    'window_seconds' => $window,
                    'challenge_at' => $challenge_at,
                    'rate_limit_at' => $rate_limit_at,
                    'block_at' => $block_at,
					'pattern'  => (string) ( $rt_meta['pattern'] ?? '' ),
					'category' => (string) ( $rt_meta['category'] ?? '' ),
					'profile'  => (string) ( $rt_meta['profile'] ?? '' ),
					'source'   => (string) ( $rt_meta['source'] ?? 'base' ),
                ] );

                if ( $block_at > 0 && $count >= $block_at ) {
                    AegisWAF_Logger::log( $route, $method, 'api_shield', 'block', [
                        'ip' => $ip,
                        'ua' => $ua,
                        'reason' => 'threshold_block_at',
                        'count' => $count,
                        'block_at' => $block_at,
                    ] );
                    return new WP_Error( 'aegiswaf_rest_blocked', 'AegisWAF: Access denied.', [ 'status' => 403 ] );
                }

                if ( $rate_limit_at > 0 && $count >= $rate_limit_at ) {
                    AegisWAF_Logger::log( $route, $method, 'api_shield', 'rate_limit', [
                        'ip' => $ip,
                        'ua' => $ua,
                        'reason' => 'threshold_rate_limit_at',
                        'count' => $count,
                        'rate_limit_at' => $rate_limit_at,
                    ] );
                    return new WP_Error( 'aegiswaf_rest_rate_limited', 'AegisWAF: Too many requests.', [ 'status' => 429 ] );
                }

                if ( $challenge_at > 0 && $count >= $challenge_at ) {
                    AegisWAF_Logger::log( $route, $method, 'api_shield', 'challenge', [
                        'ip' => $ip,
                        'ua' => $ua,
                        'reason' => 'threshold_challenge_at',
                        'count' => $count,
                        'challenge_at' => $challenge_at,
                        'note' => 'REST cannot render challenge UI; returning 403',
                    ] );
                    return new WP_Error( 'aegiswaf_rest_challenge', 'AegisWAF: Challenge required.', [ 'status' => 403 ] );
                }
            } else {
                AegisWAF_Logger::log( $route, $method, 'api_shield', 'warning', [
                    'ip' => $ip,
                    'ua' => $ua,
                    'reason' => 'thresholds_rest all zero; rate-limit/challenge/block will never trigger',
                ] );
            }

            AegisWAF_Logger::log( $route, $method, 'api_shield', 'allow', [
                'ip' => $ip,
                'ua' => $ua,
                'reason' => 'passed_all_checks',
            ] );
        }

        if ( $ip !== '' && AegisWAF_Intel::is_ip_blocked( $ip ) ) {
            AegisWAF_Logger::log( $route, $method, 'intel_ip_block', 'blocked', [ 'ip' => $ip ] );
            return new WP_Error( 'aegiswaf_denied', 'AegisWAF: Access denied.', [ 'status' => 403 ] );
        }

        if ( $ip !== '' && AegisWAF_Intel::geo_asn_blocked( $ip ) ) {
            AegisWAF_Logger::log( $route, $method, 'intel_geo_asn_block', 'blocked', [ 'ip' => $ip ] );
            return new WP_Error( 'aegiswaf_denied', 'AegisWAF: Access denied.', [ 'status' => 403 ] );
        }

        $payload = AegisWAF_Normalizer::normalize_rest( $request );

        $is_pro = AegisWAF_Features::is_pro();
        $settings = AegisWAF_Storage::get_settings();
        $mr = is_array( $settings['managed_rules'] ?? null ) ? $settings['managed_rules'] : [];
        $base_cats = is_array( $mr['categories'] ?? null ) ? $mr['categories'] : [];
        $base_thr  = is_array( $mr['thresholds'] ?? null ) ? $mr['thresholds'] : [];
        $policy = AegisWAF_Endpoint_Policy::match_policy( $route );
        $cats = AegisWAF_Endpoint_Policy::effective_categories( $base_cats, $policy );
        $thr  = AegisWAF_Endpoint_Policy::effective_thresholds( $base_thr, $policy );
        $action = AegisWAF_Endpoint_Policy::effective_action( $is_pro ? (string) ( $mr['mode_pro'] ?? 'block' ) : (string) ( $mr['mode_free'] ?? 'log' ), $policy, $is_pro );
        $ctx = [ 'categories' => $cats, 'thresholds' => $thr, 'action' => $action ];

        $mhit = AegisWAF_Managed_Rules::evaluate( $payload, $is_pro, $ctx );
        if ( is_array( $mhit ) ) {
            AegisWAF_Logger::log( $route, $method, 'managed_rule', (string) $mhit['action'], $mhit + [ 'ip' => $ip, 'ua' => $ua ] );
            $act = (string) $mhit['action'];
            if ( $act === 'rate_limit' ) {
                return new WP_Error( 'aegiswaf_rate_limited', 'AegisWAF: Too many requests.', [ 'status' => 429 ] );
            }
            if ( $act === 'challenge' ) {
                return new WP_Error( 'aegiswaf_challenge', 'AegisWAF: Challenge required.', [ 'status' => 403 ] );
            }
            if ( $act === 'block' ) {
                return new WP_Error( 'aegiswaf_rule_block', 'AegisWAF: Request blocked.', [ 'status' => 403 ] );
            }
        }

        $hhit = AegisWAF_Heuristics::evaluate( $payload, $ctx );
        if ( is_array( $hhit ) ) {
            AegisWAF_Logger::log( $route, $method, 'heuristic', (string) ( $hhit['action'] ?? 'log' ), $hhit + [ 'ip' => $ip, 'ua' => $ua ] );
            $hact = (string) ( $hhit['action'] ?? 'log' );
            if ( $hact === 'rate_limit' ) {
                return new WP_Error( 'aegiswaf_rate_limited', 'AegisWAF: Too many requests.', [ 'status' => 429 ] );
            }
            if ( $hact === 'challenge' ) {

                return new WP_Error( 'aegiswaf_challenge', 'AegisWAF: Challenge required.', [ 'status' => 403 ] );
            }
            if ( $hact === 'block' ) {
                return new WP_Error( 'aegiswaf_rule_block', 'AegisWAF: Request blocked.', [ 'status' => 403 ] );
            }
        }

        $hit = AegisWAF_Rules::evaluate( $route, $method, (string) ( $payload['norm'] ?? '' ) );
        if ( is_array( $hit ) ) {
        $hit = AegisWAF_Managed_Rules::evaluate( $payload, $is_pro, $ctx );
        if ( is_array( $hit ) ) {

            $act = (string) ( $hit['action'] ?? 'log' );

            AegisWAF_Logger::log( $path, $method, 'managed_rules', $act, $hit + [
                'ip'   => $ip,
                'ua'   => $ua,
                'scope'=> 'front',
            ] );

            if ( $act === 'rate_limit' ) {
                wp_die(
                    esc_html__( 'AegisWAF: Too many requests.', 'aegiswaf' ),
                    esc_html__( 'Rate limited', 'aegiswaf' ),
                    [ 'response' => 429 ]
                );
            }

            if ( $act === 'challenge' ) {
                if ( ! class_exists( 'AegisWAF_Challenge' ) ) {
                    AegisWAF_Logger::log( $path, $method, 'managed_rules', 'challenge_unavailable', [
                        'reason' => 'AegisWAF_Challenge class missing',
                        'ip'     => $ip,
                        'ua'     => $ua,
                    ] );
                    wp_die(
                        esc_html__( 'AegisWAF: Access denied.', 'aegiswaf' ),
                        esc_html__( 'Blocked', 'aegiswaf' ),
                        [ 'response' => 403 ]
                    );
                }

                if ( AegisWAF_Challenge::enabled() && ! AegisWAF_Challenge::verify( $ip ) ) {
                    AegisWAF_Logger::log( $path, $method, 'challenge', 'issued', [
                        'ip'   => $ip,
                        'ua'   => $ua,
                        'from' => 'managed_rules',
                    ] );
                    AegisWAF_Challenge::render_page( 'Challenge required to continue.' );
                }

                AegisWAF_Logger::log( $path, $method, 'challenge', 'passed_or_disabled', [
                    'ip'   => $ip,
                    'ua'   => $ua,
                    'from' => 'managed_rules',
                ] );
                return;
            }

            if ( $act === 'block' ) {
                wp_die(
                    esc_html__( 'AegisWAF: Access denied.', 'aegiswaf' ),
                    esc_html__( 'Blocked', 'aegiswaf' ),
                    [ 'response' => 403 ]
                );
            }
        }

        $score = AegisWAF_Intel::behavioral_score( 'rest', $route );
        if ( $score > 0 ) {
            AegisWAF_Behavioral_Timeline::incr();
            $thr = (int) ( ( AegisWAF_Intel::get()['behavioral_threshold'] ?? 80 ) );
            if ( $score >= $thr ) {
                AegisWAF_Logger::log( $route, $method, 'intel_behavioral', 'blocked', [ 'score' => $score, 'threshold' => $thr, 'ip' => $ip ] );
                return new WP_Error( 'aegiswaf_rate_limited', 'AegisWAF: Too many requests.', [ 'status' => 429 ] );
            }
        }

        return $result;
    }
	}
	private static function rest_effective_thresholds( array $wp, string $route, string $method ) : array {
		$thr = is_array( $wp['thresholds_rest'] ?? null ) ? $wp['thresholds_rest'] : [];
		$base = [
			'challenge_at'  => (int) ( $thr['challenge_at'] ?? 0 ),
			'rate_limit_at' => (int) ( $thr['rate_limit_at'] ?? 0 ),
			'block_at'      => (int) ( $thr['block_at'] ?? 0 ),
		];

		$meta = [
			'matched' => false,
			'pattern' => '',
			'category' => '',
			'profile' => '',
			'source' => 'base',
			'thresholds_applied' => [
				'challenge_at'  => $base['challenge_at'],
				'rate_limit_at' => $base['rate_limit_at'],
				'block_at'      => $base['block_at'],
			],
		];

		$raw = (string) ( $wp['rest_route_overrides'] ?? '' );
		if ( trim( $raw ) === '' ) {
			if ( ! empty( $wp['rest_route_auto_profiles'] ) ) {
				$ap = self::rest_auto_profile_for_route( $route, $method );
				$meta['category'] = (string) ( $ap['category'] ?? '' );
				$meta['profile']  = (string) ( $ap['profile'] ?? '' );
				$meta['source']   = 'auto';
			}
			return [ $base['challenge_at'], $base['rate_limit_at'], $base['block_at'], $meta ];
		}

		$parsed = self::parse_rest_route_overrides( $raw, $route, $method );
		$rules  = (array) ( $parsed['rules'] ?? [] );

		$m = strtoupper( trim( $method ) );
		if ( $m === '' ) { $m = 'GET'; }

		foreach ( $rules as $r ) {
			$rx = (string) ( $r['regex'] ?? '' );
			if ( $rx === '' ) { continue; }

			$ok = @preg_match( $rx, $route );
			if ( $ok !== 1 ) { continue; }

			// Matched a pattern
			$meta['matched'] = true;
			$meta['pattern'] = (string) ( $r['pattern'] ?? '' );
			$meta['category'] = (string) ( $r['category'] ?? '' );
			$meta['profile']  = (string) ( $r['profile'] ?? '' );

			$methods = is_array( $r['methods'] ?? null ) ? $r['methods'] : [];
			$picked  = null;

			if ( isset( $methods[ $m ] ) && is_array( $methods[ $m ] ) ) {
				$picked = $methods[ $m ];
			} elseif ( isset( $methods['ANY'] ) && is_array( $methods['ANY'] ) ) {
				$picked = $methods['ANY'];
			}

			if ( is_array( $picked ) ) {
				$meta['source'] = 'override';
				$meta['thresholds_applied'] = [
					'challenge_at'  => (int) ( $picked['challenge_at'] ?? $base['challenge_at'] ),
					'rate_limit_at' => (int) ( $picked['rate_limit_at'] ?? $base['rate_limit_at'] ),
					'block_at'      => (int) ( $picked['block_at'] ?? $base['block_at'] ),
				];
			} else {
				// Matched for tagging, but no thresholds specified for this method
				$meta['source'] = 'tag_only';
			}

			// Auto-profiles if enabled and category/profile empty
			if ( ! empty( $wp['rest_route_auto_profiles'] ) && ( $meta['category'] === '' || $meta['profile'] === '' ) ) {
				$ap = self::rest_auto_profile_for_route( $route, $method );
				if ( $meta['category'] === '' ) { $meta['category'] = (string) ( $ap['category'] ?? '' ); }
				if ( $meta['profile']  === '' ) { $meta['profile']  = (string) ( $ap['profile'] ?? '' ); }
				if ( $meta['source'] === 'base' ) { $meta['source'] = 'auto'; }
			}

			$c = (int) ( $meta['thresholds_applied']['challenge_at'] ?? $base['challenge_at'] );
			$rL= (int) ( $meta['thresholds_applied']['rate_limit_at'] ?? $base['rate_limit_at'] );
			$b = (int) ( $meta['thresholds_applied']['block_at'] ?? $base['block_at'] );

			return [ $c, $rL, $b, $meta ];
		}
		if ( ! empty( $wp['rest_route_auto_profiles'] ) ) {
			$ap = self::rest_auto_profile_for_route( $route, $method );
			$meta['category'] = (string) ( $ap['category'] ?? '' );
			$meta['profile']  = (string) ( $ap['profile'] ?? '' );
			$meta['source']   = 'auto';
		}

		return [ $base['challenge_at'], $base['rate_limit_at'], $base['block_at'], $meta ];
	}

	private static function parse_rest_route_overrides( string $raw, string $route, string $method ) : array {
		static $cache = [];

		$cache_key = md5( $raw );
		if ( isset( $cache[ $cache_key ] ) ) {
			return $cache[ $cache_key ];
		}

		$rules  = [];
		$errors = [];

		$lines = preg_split( "/\r\n|\n|\r/", $raw );
		if ( ! is_array( $lines ) ) { $lines = []; }

		foreach ( $lines as $idx => $line ) {
			$orig = (string) $line;
			$line = trim( $orig );

			if ( $line === '' ) { continue; }
			if ( strpos( $line, '#' ) === 0 || strpos( $line, '//' ) === 0 ) { continue; }

			$parts = array_map( 'trim', explode( '|', $line ) );
			$parts = array_values( array_filter( $parts, static function( $v ) { return $v !== ''; } ) );

			if ( empty( $parts[0] ) ) {
				$errors[] = [ 'line' => $orig, 'reason' => 'missing_pattern', 'index' => $idx + 1 ];
				continue;
			}

			$pattern  = (string) $parts[0];
			$category = '';
			$profile  = '';
			$methods  = [];

			for ( $i = 1; $i < count( $parts ); $i++ ) {
				$p = (string) $parts[$i];

				if ( preg_match( '/^(ANY|GET|POST|PUT|PATCH|DELETE|HEAD|OPTIONS)\s*[:=]\s*(.+)$/i', $p, $mm ) ) {
					$m = strtoupper( $mm[1] );
					$nums = trim( (string) $mm[2] );

					$nums = str_replace( '/', ',', $nums );
					$arr  = array_map( 'trim', explode( ',', $nums ) );

					$c  = isset( $arr[0] ) ? (int) $arr[0] : 0;
					$rl = isset( $arr[1] ) ? (int) $arr[1] : 0;
					$b  = isset( $arr[2] ) ? (int) $arr[2] : 0;

					$methods[ $m ] = [
						'challenge_at'  => $c,
						'rate_limit_at' => $rl,
						'block_at'      => $b,
					];
					continue;
				}

				if ( $category === '' ) { $category = $p; continue; }
				if ( $profile  === '' ) { $profile  = $p; continue; }
			}

			$rx = preg_quote( $pattern, '#' );
			$rx = str_replace( '\*', '.*', $rx );
			$rx = '#^' . $rx . '$#';

			if ( @preg_match( $rx, $route ) === false ) {
				$errors[] = [ 'line' => $orig, 'reason' => 'invalid_regex', 'index' => $idx + 1, 'pattern' => $pattern ];
				continue;
			}

			$rules[] = [
				'pattern'  => $pattern,
				'regex'    => $rx,
				'category' => $category,
				'profile'  => $profile,
				'methods'  => $methods,
			];
		}

		if ( ! empty( $errors ) && class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
			foreach ( $errors as $e ) {
				$sig = md5( (string) ( $e['index'] ?? '' ) . '|' . (string) ( $e['reason'] ?? '' ) . '|' . (string) ( $e['line'] ?? '' ) );
				$tkey = 'aegiswaf_rest_override_err_' . $sig;

				if ( get_transient( $tkey ) ) { continue; }
				set_transient( $tkey, 1, HOUR_IN_SECONDS );

				AegisWAF_Logger::log( $route, $method, 'api_shield', 'rest_route_override_parse_error', [
					'route' => $route,
					'method' => $method,
					'line_no' => (int) ( $e['index'] ?? 0 ),
					'reason' => (string) ( $e['reason'] ?? '' ),
					'line' => (string) ( $e['line'] ?? '' ),
					'pattern' => (string) ( $e['pattern'] ?? '' ),
				] );
			}
		}

		$cache[ $cache_key ] = [ 'rules' => $rules, 'errors' => $errors ];
		return $cache[ $cache_key ];
	}

	private static function rest_auto_profile_for_route( string $route, string $method ) : array {
		$r = (string) $route;

		if ( strpos( $r, '/wp/v2/users' ) === 0 ) {
			return [ 'category' => 'user_enum', 'profile' => 'sensitive' ];
		}
		if ( strpos( $r, '/jwt-auth' ) === 0 || strpos( $r, '/oauth' ) !== false ) {
			return [ 'category' => 'auth', 'profile' => 'auth' ];
		}
		if ( strpos( $r, '/wc/' ) === 0 || strpos( $r, '/wc-' ) !== false ) {
			return [ 'category' => 'ecommerce', 'profile' => ( strtoupper( $method ) === 'GET' ? 'read_heavy' : 'write_heavy' ) ];
		}
		if ( strpos( $r, '/wp/v2/posts' ) === 0 || strpos( $r, '/wp/v2/pages' ) === 0 ) {
			return [ 'category' => 'content', 'profile' => ( strtoupper( $method ) === 'GET' ? 'read_heavy' : 'write_heavy' ) ];
		}

		return [ 'category' => 'rest', 'profile' => 'default' ];
	}

}
