<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_Intel {

    public static function default_intel() : array {
        return [
            'ip_blocklist' => '',     // PRO, newline separated (IP or CIDR)
            'country_block' => '',    // PRO, CSV list (US,CA,...)
            'asn_block' => '',        // PRO, newline separated (e.g., AS13335)
            'behavioral_enabled' => true,
            'behavioral_threshold' => 80,
        ];
    }

    public static function get() : array {
        $cur = AegisWAF_Storage::get_intel();
        if ( ! is_array( $cur ) ) { $cur = []; }
        return wp_parse_args( $cur, self::default_intel() );
    }

    public static function is_ip_blocked( string $ip ) : bool {
        if ( ! AegisWAF_Features::is_pro() ) { return false; }

        $cfg = self::get();
        $raw = (string) ( $cfg['ip_blocklist'] ?? '' );
        $lines = preg_split( '/\r\n|\r|\n/', $raw ) ?: [];
        foreach ( $lines as $l ) {
            $l = trim( $l );
            if ( $l === '' ) { continue; }
            if ( self::cidr_match( $ip, $l ) ) {
                return true;
            }
        }
        return false;
    }

    public static function behavioral_score( string $scope, string $path ) : int {
        if ( ! AegisWAF_Features::is_pro() ) { return 0; }

        $ip = AegisWAF_Utils::client_ip();
        if ( $ip === '' ) { return 0; }

        $bucket = gmdate( 'YmdHi' );
        $key = 'aegiswaf_bhv_' . md5( $scope . '|' . $ip . '|' . $bucket );
        $count = (int) get_transient( $key );
        $count++;
        set_transient( $key, $count, 70 );

        $score = 0;
        if ( $count > 30 ) { $score += 30; }
        if ( $count > 80 ) { $score += 40; }

        $sens = [ '/wp-login.php', '/xmlrpc.php', '/wp-admin', '/wp-json', '/?author=' ];
        foreach ( $sens as $s ) {
            if ( stripos( $path, $s ) !== false ) {
                $score += 25;
                break;
            }
        }

        // Cap
        if ( $score > 100 ) { $score = 100; }
        return $score;
    }

    public static function has_geo_asn_provider() : bool {
        $provider = apply_filters( 'aegiswaf_geoasn_provider', null );
        return is_callable( $provider );
    }

    public static function lookup_geo_asn( string $ip ) : ?array {
        $provider = apply_filters( 'aegiswaf_geoasn_provider', null );
        if ( ! is_callable( $provider ) ) { return null; }
        $res = call_user_func( $provider, $ip );
        return is_array( $res ) ? $res : null;
    }

    public static function geo_asn_blocked( string $ip ) : bool {
        if ( ! AegisWAF_Features::is_pro() ) { return false; }

        $cfg = self::get();
        $res = self::lookup_geo_asn( $ip );
        if ( ! $res ) { return false; }

        $cc = strtoupper( (string) ( $res['country'] ?? '' ) );
        $asn = strtoupper( (string) ( $res['asn'] ?? '' ) );

        $cc_list = array_filter( array_map( 'trim', explode( ',', (string) ( $cfg['country_block'] ?? '' ) ) ) );
        $cc_list = array_map( 'strtoupper', $cc_list );

        if ( $cc !== '' && in_array( $cc, $cc_list, true ) ) {
            return true;
        }

        $asn_lines = preg_split( '/\r\n|\r|\n/', (string) ( $cfg['asn_block'] ?? '' ) ) ?: [];
        foreach ( $asn_lines as $l ) {
            $l = strtoupper( trim( $l ) );
            if ( $l === '' ) { continue; }
            if ( $asn !== '' && $asn === $l ) { return true; }
        }

        return false;
    }

    private static function cidr_match( string $ip, string $cidr ) : bool {
        $cidr = trim( $cidr );
        if ( $cidr === '' ) { return false; }
        if ( strpos( $cidr, '/' ) === false ) {
            return $ip === $cidr;
        }
        if ( filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 ) === false ) { return false; }
        list( $subnet, $mask ) = explode( '/', $cidr, 2 );
        $mask = (int) $mask;
        if ( filter_var( $subnet, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 ) === false ) { return false; }
        if ( $mask < 0 || $mask > 32 ) { return false; }
        $ip_long = ip2long( $ip );
        $sub_long = ip2long( $subnet );
        $mask_long = -1 << ( 32 - $mask );
        $subnet_long = $sub_long & $mask_long;
        return ( $ip_long & $mask_long ) === $subnet_long;
    }
}
