<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_Overrides {

    private const OPT = 'aegiswaf_allow_overrides';

    /**
     * Store allow overrides as:
     * [
     *   "<sha1sig>" => [ 'ip'=>..., 'route'=>..., 'method'=>..., 'expires'=>timestamp, 'created'=>timestamp ]
     * ]
     */
    public static function get_all() : array {
        $raw = get_option( self::OPT, [] );
        return is_array( $raw ) ? $raw : [];
    }

    public static function purge_expired() : void {
        $all = self::get_all();
        $now = time();

        $changed = false;
        foreach ( $all as $sig => $row ) {
            $exp = (int) ( $row['expires'] ?? 0 );
            if ( $exp > 0 && $exp < $now ) {
                unset( $all[ $sig ] );
                $changed = true;
            }
        }

        if ( $changed ) {
            update_option( self::OPT, $all, false );
        }
    }

    private static function sig( string $ip, string $route, string $method ) : string {
        return sha1( strtolower( trim( $ip ) ) . '|' . strtoupper( trim( $method ) ) . '|' . trim( $route ) );
    }

    public static function add_allow( string $ip, string $route, string $method, int $ttl_seconds = 86400 ) : bool {
        $ip = trim( $ip );
        $route = trim( $route );
        $method = strtoupper( trim( $method ) );

        if ( $ip === '' || $route === '' || $method === '' ) {
            return false;
        }

        self::purge_expired();

        $all = self::get_all();
        $sig = self::sig( $ip, $route, $method );

        $now = time();
        $all[ $sig ] = [
            'ip'      => $ip,
            'route'   => $route,
            'method'  => $method,
            'created' => $now,
            'expires' => $now + max( 300, (int) $ttl_seconds ), // minimum 5 minutes
        ];

        return (bool) update_option( self::OPT, $all, false );
    }

    public static function remove_allow( string $ip, string $route, string $method ) : bool {
        self::purge_expired();

        $all = self::get_all();
        $sig = self::sig( $ip, $route, $method );

        if ( isset( $all[ $sig ] ) ) {
            unset( $all[ $sig ] );
            return (bool) update_option( self::OPT, $all, false );
        }

        return true;
    }

    public static function is_allowed( string $ip, string $route, string $method ) : bool {
        self::purge_expired();

        $sig = self::sig( $ip, $route, $method );
        $all = self::get_all();

        if ( empty( $all[ $sig ] ) ) {
            return false;
        }

        $exp = (int) ( $all[ $sig ]['expires'] ?? 0 );
        return ( $exp <= 0 ) ? false : ( $exp >= time() );
    }
}
