<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_Rules {

    public static function normalize_list( string $raw ) : array {
        $lines = preg_split( '/\r\n|\r|\n/', (string) $raw ) ?: [];
        $out = [];
        foreach ( $lines as $l ) {
            $l = trim( (string) $l );
            if ( $l !== '' ) { $out[] = $l; }
        }
        return $out;
    }

    public static function validate_regex_list( array $regexes ) : array {
        $valid = [];
        foreach ( $regexes as $r ) {
            $r = trim( (string) $r );
            if ( $r === '' ) { continue; }

            if ( $r[0] !== '#' ) {
                $r = '#' . str_replace('#', '\#', $r) . '#i';
            }
            set_error_handler( function() {} );
            $ok = @preg_match( $r, '' ) !== false;
            restore_error_handler();
            if ( $ok ) { $valid[] = $r; }
        }
        return $valid;
    }

    public static function evaluate( ?string $path, string $method, $haystack ) : ?array {
        if ( empty( $path ) ) { return null; }
        $haystack = (string) $haystack;

        $settings = AegisWAF_Storage::get_settings();
        if ( empty( $settings['rules']['enabled'] ) ) { return null; }

        $rules = AegisWAF_Storage::get_waf_rules();
        if ( empty( $rules ) || ! is_array( $rules ) ) { return null; }

        $is_pro = AegisWAF_Features::is_pro();
        $hl = strtolower( $haystack );

        if ( ! $is_pro && count( $rules ) > 5 ) {
            $rules = array_slice( $rules, 0, 5 );
        }

        foreach ( $rules as $rule ) {
            if ( empty( $rule['enabled'] ) ) { continue; }

            $pattern = isset( $rule['path_pattern'] ) ? (string) $rule['path_pattern'] : '';
            if ( $pattern === '' ) { continue; }

            $rm = isset( $rule['method'] ) ? strtoupper( (string) $rule['method'] ) : 'ANY';
            if ( $rm !== 'ANY' && $rm !== strtoupper( $method ) ) { continue; }

            if ( ! self::path_match( $path, $pattern ) ) { continue; }

            $contains = isset( $rule['contains'] ) ? (string) $rule['contains'] : '';
            $needles = self::normalize_list( $contains );
            $needles_ok = true;
            foreach ( $needles as $n ) {
                $n = strtolower( $n );
                if ( $n !== '' && strpos( $hl, $n ) === false ) {
                    $needles_ok = false;
                    break;
                }
            }
            if ( ! $needles_ok ) { continue; }

            $regex_raw = isset( $rule['regex'] ) ? (string) $rule['regex'] : '';
            $regexes = self::normalize_list( $regex_raw );
            $regexes = self::validate_regex_list( $regexes );

            if ( ! empty( $regexes ) && ! $is_pro ) {
                continue;
            }

            $regex_ok = true;
            foreach ( $regexes as $rx ) {
                if ( @preg_match( $rx, $haystack ) !== 1 ) {
                    $regex_ok = false;
                    break;
                }
            }
            if ( ! $regex_ok ) { continue; }

            $action = isset( $rule['action'] ) ? (string) $rule['action'] : 'log';
            $action = AegisWAF_Utils::sanitize_action( $action );
            if ( ! in_array( $action, [ 'log', 'block', 'allow' ], true ) ) {
                $action = 'log';
            }

            return [
                'rule_id' => $rule['id'] ?? '',
                'name' => $rule['name'] ?? 'Custom Rule',
                'action' => $action,
                'path_pattern' => $pattern,
            ];
        }

        return null;
    }

    private static function path_match( string $path, string $pattern ) : bool {

        $path    = (string) $path;
        $pattern = (string) $pattern;

        if ( $path === '' || $pattern === '' ) {
            return false;
        }

        if ( $path[0] !== '/' ) {
            $path = '/' . $path;
        }
        if ( $pattern[0] !== '/' && $pattern[0] !== '*' ) {
            $pattern = '/' . $pattern;
        }

        if ( strpos( $pattern, '*' ) === false && strpos( $pattern, '?' ) === false ) {
            return strcasecmp( $path, $pattern ) === 0;
        }

        $quoted = preg_quote( $pattern, '~' );

        $quoted = str_replace( '\*', '.*', $quoted );
        $quoted = str_replace( '\?', '[^/]', $quoted );

        $regex = '~^' . $quoted . '$~i';

        return (bool) preg_match( $regex, $path );
    }

}
