<?php
/**
 * Plugin Name: AegisWAF – Web Application Firewall
 * Plugin URI: https://aegisify.com/aegiswaf
 * Description: WordPress-native WAF with policy persistence, Web Application Firewall, API Shield, Bot Control, DDOS Layer7, Logs and Attack Story.
 * Version: 1.8.7
 * Requires at least: 6.8
 * Requires PHP: 8.2
 * Author:      Aegisify | AegisWAF | AegisShield | AegisSEO
 * Author URI:  https://aegisify.com/
 * Text Domain: aegiswaf
 * License: GPLv2 or later
 * Text Domain: aegiswaf
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

define( 'AEGISWAF_VERSION', '1.8.7' );
define( 'AEGISWAF_PATH', plugin_dir_path( __FILE__ ) );
define( 'AEGISWAF_URL', plugin_dir_url( __FILE__ ) );
define( 'AEGISWAF_SLUG', 'aegiswaf' );
define( 'AEGISWAF_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

define( 'AEGISWAF_CLM_BASE', 'https://aegisify.com' );
define( 'AEGISWAF_CLM_FREE_REG_ENDPOINT', AEGISWAF_CLM_BASE . '/wp-json/chue-license/v1/aegiswaf/free-register' );
define( 'AEGISWAF_CLM_UPDATE_ENDPOINT',    AEGISWAF_CLM_BASE . '/wp-json/chue-license/v1/aegiswaf/update-check' );

require_once AEGISWAF_PATH . 'includes/class-aegiswaf-features.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-utils.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-storage.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-overrides.php';
require_once AEGISWAF_PATH . 'includes/ddos/class-aegiswaf-ddos-storage.php';
require_once AEGISWAF_PATH . 'includes/ddos/class-aegiswaf-ddos-rules.php';
require_once AEGISWAF_PATH . 'includes/ddos/class-aegiswaf-ddos-engine.php';
require_once AEGISWAF_PATH . 'includes/ddos/class-aegiswaf-ddos-admin.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-logger.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-engine.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-normalizer.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-managed-rules.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-heuristics.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-endpoint-policy.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-challenge.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-wp-protect.php';
require_once AEGISWAF_PATH . 'includes/licensing/class-aegiswaf-license-manager.php';
require_once AEGISWAF_PATH . 'includes/licensing/class-aegiswaf-updater.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-rules.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-intel.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-behavioral-timeline.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-rule-sets.php';
require_once AEGISWAF_PATH . 'includes/providers/class-aegiswaf-geoasn-provider.php';
require_once AEGISWAF_PATH . 'includes/admin/class-aegiswaf-admin.php';

function aegiswaf_update_log( string $msg, array $ctx = array() ) : void {
    if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
        error_log( '[AegisWAF Update] ' . $msg . ( $ctx ? ' | ' . wp_json_encode( $ctx ) : '' ) );
    }
}

function aegiswaf_clm_register_free_install() : void {

    if ( get_transient( 'aegiswaf_clm_free_reg_sent' ) ) {
        return;
    }

    $site_url = home_url();
    $domain   = wp_parse_url( $site_url, PHP_URL_HOST );

    $payload = array(
        'product'        => AEGISWAF_SLUG,
        'email'          => (string) get_option( 'admin_email' ),
        'site_url'       => (string) $site_url,
        'domain'         => (string) $domain,
        'plugin_version' => (string) AEGISWAF_VERSION,
        'wp_version'     => (string) get_bloginfo( 'version' ),
        'php_version'    => (string) PHP_VERSION,
    );

    $args = array(
        'timeout' => 15,
        'headers' => array( 'Content-Type' => 'application/json' ),
        'body'    => wp_json_encode( $payload ),
    );

    $resp = wp_remote_post( AEGISWAF_CLM_FREE_REG_ENDPOINT, $args );

    if ( is_wp_error( $resp ) ) {
        aegiswaf_update_log( 'CLM free-register failed (WP_Error).', array( 'error' => $resp->get_error_message() ) );
        return;
    }

    $code = (int) wp_remote_retrieve_response_code( $resp );
    $body = (string) wp_remote_retrieve_body( $resp );

    // Even if CLM rejects, we still backoff for a day to prevent log spam.
    set_transient( 'aegiswaf_clm_free_reg_sent', 1, DAY_IN_SECONDS );

    aegiswaf_update_log( 'CLM free-register response.', array(
        'http_code' => $code,
        'body'      => $body,
    ) );
}

function aegiswaf_clm_direct_update_check( string $current_version ) : array {

    $site_url = home_url();
    $domain   = wp_parse_url( $site_url, PHP_URL_HOST );

    $license_key = (string) get_option( 'aegiswaf_license_key', '' );

    $payload = array(
        'slug'            => AEGISWAF_SLUG,
        'product'         => AEGISWAF_SLUG,
        'version'         => $current_version,
        'domain'          => (string) $domain,
        'site_url'        => (string) $site_url,
        'license_key'     => $license_key,
        'wp_version'      => (string) get_bloginfo( 'version' ),
        'php_version'     => (string) PHP_VERSION,
        'plugin_basename' => (string) AEGISWAF_PLUGIN_BASENAME,
    );

    $args = array(
        'timeout' => 20,
        'headers' => array( 'Content-Type' => 'application/json' ),
        'body'    => wp_json_encode( $payload ),
    );

    $resp = wp_remote_post( AEGISWAF_CLM_UPDATE_ENDPOINT, $args );

    if ( is_wp_error( $resp ) ) {
        return array(
            'success' => false,
            'status'  => 'wp_error',
            'message' => $resp->get_error_message(),
        );
    }

    $code = (int) wp_remote_retrieve_response_code( $resp );
    $body = (string) wp_remote_retrieve_body( $resp );

    if ( $code < 200 || $code >= 300 ) {
        return array(
            'success' => false,
            'status'  => 'http_' . $code,
            'message' => 'CLM HTTP ' . $code,
            'raw'     => $body,
        );
    }

    $data = json_decode( $body, true );
    if ( ! is_array( $data ) ) {
        return array(
            'success' => false,
            'status'  => 'bad_json',
            'message' => 'CLM returned invalid JSON',
            'raw'     => $body,
        );
    }

    if ( empty( $data['version'] ) && ! empty( $data['latest_version'] ) ) {
        $data['version'] = (string) $data['latest_version'];
    }

    return $data;
}

add_filter( 'pre_set_site_transient_update_plugins', function( $transient ) {

    aegiswaf_update_log( 'Hook fired: pre_set_site_transient_update_plugins', array(
        'is_object'     => is_object( $transient ),
        'checked_count' => ( is_object( $transient ) && ! empty( $transient->checked ) && is_array( $transient->checked ) )
            ? count( $transient->checked )
            : 0,
    ) );

    if ( ! is_object( $transient ) || empty( $transient->checked ) || ! is_array( $transient->checked ) ) {
        aegiswaf_update_log( 'Transient missing or empty checked list; skipping.' );
        return $transient;
    }

    if ( empty( $transient->checked[ AEGISWAF_PLUGIN_BASENAME ] ) ) {
        return $transient;
    }

    set_site_transient( $last_check_key, time(), DAY_IN_SECONDS );

    $current_version = (string) $transient->checked[ AEGISWAF_PLUGIN_BASENAME ];
    $data            = aegiswaf_clm_direct_update_check( $current_version );

    aegiswaf_update_log( 'CLM response received', array(
        'type' => gettype( $data ),
        'data' => $data,
    ) );

    if ( empty( $data['success'] ) ) {
        aegiswaf_update_log( 'ERROR: CLM success=false.', array( 'data' => $data ) );
        return $transient;
    }

    $remote_version = ! empty( $data['version'] ) ? (string) $data['version'] : '';
	
	if ( $remote_version !== '' && $current_version !== '' && ! version_compare( $remote_version, $current_version, '>' ) ) {
		aegiswaf_update_log( 'Remote version is not newer than installed; ignoring update flag.', array(
			'remote_version'  => $remote_version,
			'current_version' => $current_version,
		) );
		return $transient;
	}

    if ( empty( $data['update_available'] ) ) {
        aegiswaf_update_log( 'CLM reports no update available.', array(
            'remote_version' => $remote_version,
        ) );
        return $transient;
    }

    $package = ! empty( $data['download_url'] ) ? (string) $data['download_url'] : '';
    if ( $package === '' ) {
        aegiswaf_update_log( 'CLM says update available but download_url missing.', array( 'data' => $data ) );
        return $transient;
    }

    // Create update object
    $update              = new stdClass();
    $update->slug        = AEGISWAF_SLUG;
    $update->plugin      = AEGISWAF_PLUGIN_BASENAME;
    $update->new_version = $remote_version;
    $update->package     = $package;
    $update->url         = 'https://aegisify.com/aegiswaf';

    $transient->response[ AEGISWAF_PLUGIN_BASENAME ] = $update;
	set_site_transient( $cache_key, $update, DAY_IN_SECONDS );

    aegiswaf_update_log( 'Update injected into WP transient.', array(
        'new_version' => $remote_version,
        'package'     => $package,
    ) );

    return $transient;

}, 25, 1 );

register_activation_hook( __FILE__, [ 'AegisWAF_Logger', 'activate' ] );

add_action( 'plugins_loaded', function() {

    aegiswaf_clm_register_free_install();

    ( new AegisWAF_License_Manager() )->init();
    ( new AegisWAF_Updater() )->init();
    AegisWAF_GeoASN_Provider::register_filter();
    require_once AEGISWAF_PATH . 'includes/admin/class-aegiswaf-notices.php';
    AegisWAF_Admin_Notices::init();
    AegisWAF_Engine::init();
    AegisWAF_WP_Protect::init();
} );

add_action( 'init', function() {
    load_plugin_textdomain( 'aegiswaf', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
} );
