<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

if ( ! function_exists( 'aegiswaf_waf_rules_get_visual_data' ) ) {
	function aegiswaf_waf_rules_get_visual_data() : array {
		global $wpdb;

		if ( ! class_exists( 'AegisWAF_Logger' ) || ! method_exists( 'AegisWAF_Logger', 'table_name' ) ) {
			return [
				'trend_labels' => [],
				'trend_counts' => [],
				'cat_labels'   => [],
				'cat_counts'   => [],
				'route_labels' => [],
				'route_counts' => [],
				'action_labels'=> [],
				'action_counts'=> [],
				'totals'       => [ 'last_24h' => 0, 'last_7d' => 0 ],
				'since_gmt'    => gmdate( 'Y-m-d H:i:s', time() - ( 7 * DAY_IN_SECONDS ) ),
			];
		}

		$table = AegisWAF_Logger::table_name();

		$since_7d_gmt  = gmdate( 'Y-m-d H:i:s', time() - ( 7 * DAY_IN_SECONDS ) );
		$since_24h_gmt = gmdate( 'Y-m-d H:i:s', time() - DAY_IN_SECONDS );

		$total_7d = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$table}
				 WHERE event_time >= %s
				   AND category IN ('managed_rule','heuristic')",
				$since_7d_gmt
			)
		);
		$total_24h = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$table}
				 WHERE event_time >= %s
				   AND category IN ('managed_rule','heuristic')",
				$since_24h_gmt
			)
		);

		$trend_rows = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT DATE(event_time) AS d, COUNT(*) AS c
				   FROM {$table}
				  WHERE event_time >= %s
				    AND category IN ('managed_rule','heuristic')
				  GROUP BY DATE(event_time)
				  ORDER BY d ASC",
				$since_7d_gmt
			),
			ARRAY_A
		);

		$trend_labels = [];
		$trend_counts = [];
		foreach ( (array) $trend_rows as $r ) {
			$trend_labels[] = (string) ( $r['d'] ?? '' );
			$trend_counts[] = (int) ( $r['c'] ?? 0 );
		}

		$cat_rows = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT category, details
				   FROM {$table}
				  WHERE event_time >= %s
				    AND category IN ('managed_rule','heuristic')
				  ORDER BY id DESC
				  LIMIT 2000",
				$since_7d_gmt
			),
			ARRAY_A
		);

		$cat_counts_map = [];
		foreach ( (array) $cat_rows as $r ) {
			$type = (string) ( $r['category'] ?? '' );
			$details = [];
			if ( ! empty( $r['details'] ) ) {
				$decoded = json_decode( (string) $r['details'], true );
				if ( is_array( $decoded ) ) { $details = $decoded; }
			}

			if ( $type === 'managed_rule' ) {
				$k = (string) ( $details['category'] ?? 'unknown' );
			} else {
				$k = 'heuristic';
			}

			$k = sanitize_key( $k );
			if ( $k === '' ) { $k = 'unknown'; }
			$cat_counts_map[ $k ] = ( $cat_counts_map[ $k ] ?? 0 ) + 1;
		}

		arsort( $cat_counts_map );
		$top = array_slice( $cat_counts_map, 0, 8, true );
		$other = array_slice( $cat_counts_map, 8, null, true );

		$cat_labels = array_keys( $top );
		$cat_counts = array_values( $top );
		$other_sum  = array_sum( $other );
		if ( $other_sum > 0 ) {
			$cat_labels[] = 'other';
			$cat_counts[] = (int) $other_sum;
		}

		$route_rows = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT route, COUNT(*) AS c
				   FROM {$table}
				  WHERE event_time >= %s
				    AND category IN ('managed_rule','heuristic')
				    AND action_taken IN ('block','challenge','rate_limit')
				  GROUP BY route
				  ORDER BY c DESC
				  LIMIT 8",
				$since_7d_gmt
			),
			ARRAY_A
		);

		$route_labels = [];
		$route_counts = [];
		foreach ( (array) $route_rows as $r ) {
			$route_labels[] = (string) ( $r['route'] ?? '' );
			$route_counts[] = (int) ( $r['c'] ?? 0 );
		}

		$action_rows = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT action_taken, COUNT(*) AS c
				   FROM {$table}
				  WHERE event_time >= %s
				    AND category IN ('managed_rule','heuristic')
				  GROUP BY action_taken
				  ORDER BY c DESC",
				$since_7d_gmt
			),
			ARRAY_A
		);

		$action_labels = [];
		$action_counts = [];
		foreach ( (array) $action_rows as $r ) {
			$action_labels[] = (string) ( $r['action_taken'] ?? '' );
			$action_counts[] = (int) ( $r['c'] ?? 0 );
		}

		return [
			'trend_labels' => $trend_labels,
			'trend_counts' => $trend_counts,
			'cat_labels'   => $cat_labels,
			'cat_counts'   => $cat_counts,
			'route_labels' => $route_labels,
			'route_counts' => $route_counts,
			'action_labels'=> $action_labels,
			'action_counts'=> $action_counts,
			'totals'       => [ 'last_24h' => $total_24h, 'last_7d' => $total_7d ],
			'since_gmt'    => $since_7d_gmt,
		];
	}
}

add_action( 'admin_enqueue_scripts', function() {
	if ( ! is_admin() ) { return; }
	if ( empty( $_GET['page'] ) || $_GET['page'] !== 'aegiswaf' ) { return; }

	$tab = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'overview';
	if ( $tab !== 'waf_rules' ) { return; }

	$base_url = plugin_dir_url( dirname( __FILE__, 3 ) ); // plugin root

	wp_enqueue_script(
		'aegiswaf-chartjs',
		$base_url . 'assets/js/chart.umd.min.js',
		[],
		'1.6.9',
		true
	);

	$data = aegiswaf_waf_rules_get_visual_data();
	$json = wp_json_encode( $data );

	$inline = "
	(function(){
		if (typeof Chart === 'undefined') { return; }
		var data = {$json};
		function el(id){ return document.getElementById(id); }

		// Trend (Area/Line)
		var c1 = el('aegiswaf_waf_chart_trend');
		if (c1) {
			new Chart(c1.getContext('2d'), {
				type: 'line',
				data: {
					labels: data.trend_labels || [],
					datasets: [{
						label: 'WAF violations (last 7 days)',
						data: data.trend_counts || [],
						fill: true,
						tension: 0.35,
						pointRadius: 2
					}]
				},
				options: {
					responsive: true,
					maintainAspectRatio: false,
					plugins: { legend: { display: true } },
					scales: { y: { beginAtZero: true } }
				}
			});
		}

		// Threat categories (Donut)
		var c2 = el('aegiswaf_waf_chart_categories');
		if (c2) {
			new Chart(c2.getContext('2d'), {
				type: 'doughnut',
				data: {
					labels: data.cat_labels || [],
					datasets: [{ data: data.cat_counts || [] }]
				},
				options: {
					responsive: true,
					maintainAspectRatio: false,
					plugins: { legend: { position: 'bottom' } }
				}
			});
		}

		// Top routes (Bar)
		var c3 = el('aegiswaf_waf_chart_routes');
		if (c3) {
			new Chart(c3.getContext('2d'), {
				type: 'bar',
				data: {
					labels: data.route_labels || [],
					datasets: [{
						label: 'Top attacked routes (block/challenge/rate-limit)',
						data: data.route_counts || []
					}]
				},
				options: {
					responsive: true,
					maintainAspectRatio: false,
					plugins: { legend: { display: true } },
					scales: { y: { beginAtZero: true } }
				}
			});
		}

		// Actions (Donut)
		var c4 = el('aegiswaf_waf_chart_actions');
		if (c4) {
			new Chart(c4.getContext('2d'), {
				type: 'doughnut',
				data: {
					labels: data.action_labels || [],
					datasets: [{ data: data.action_counts || [] }]
				},
				options: {
					responsive: true,
					maintainAspectRatio: false,
					plugins: { legend: { position: 'bottom' } }
				}
			});
		}

		// Totals text
		var t24 = el('aegiswaf_waf_total_24h');
		var t7d = el('aegiswaf_waf_total_7d');
		if (t24) { t24.textContent = String((data.totals && data.totals.last_24h) ? data.totals.last_24h : 0); }
		if (t7d) { t7d.textContent = String((data.totals && data.totals.last_7d) ? data.totals.last_7d : 0); }
	})();";

	wp_add_inline_script( 'aegiswaf-chartjs', $inline, 'after' );
}, 21 );

class AegisWAF_Page_WAF_Rules {

    public function render() : void {
        if ( ! current_user_can( 'manage_options' ) ) { return; }

        $is_pro = AegisWAF_Features::is_pro();

        echo '<style>
        .aegiswaf-subhead, .aegiswaf-sectionhead{
            display:flex;
            align-items:flex-start;
            gap:10px;
            margin:16px 0 10px;
        }
        .aegiswaf-subhead .dashicons, .aegiswaf-sectionhead .dashicons{
            font-size:18px;
            line-height:18px;
            margin-top:2px;
            color:#2271b1;
        }
        .aegiswaf-subhead h3, .aegiswaf-sectionhead h3{
            margin:0;
            padding:0;
            line-height:1.2;
        }
        .aegiswaf-subhead p, .aegiswaf-sectionhead p{
            margin:3px 0 0;
            color:#646970;
        }
		.aegiswaf-fields-grid{
			display:grid;
			grid-template-columns: repeat(auto-fit, minmax(160px, 1fr));
			gap:12px 16px;
			margin:10px 0 20px;
		}
		.aegiswaf-field label{
			display:flex;
			justify-content:space-between;
			align-items:center;
			font-weight:500;
		}
		.aegiswaf-field input[type="number"]{
			width:70px;
		}
		/* WAF Visual Intelligence (4-up charts) */
		.aegiswaf-wafviz-grid{
			display:grid;
			grid-template-columns: repeat(4, minmax(220px, 1fr));
			gap:12px;
			margin-top:12px;
		}
		@media (max-width: 1300px){
			.aegiswaf-wafviz-grid{ grid-template-columns: repeat(2, minmax(240px, 1fr)); }
		}
		@media (max-width: 782px){
			.aegiswaf-wafviz-grid{ grid-template-columns: 1fr; }
		}
		.aegiswaf-wafviz-panel{
			border:1px solid #e5e7eb;
			border-radius:10px;
			padding:12px;
			background:#fff;
		}
		.aegiswaf-wafviz-title{
			font-weight:700;
			margin:0 0 4px;
		}
		.aegiswaf-wafviz-sub{
			color:#646970;
			font-size:12px;
			margin:0 0 10px;
		}
		.aegiswaf-wafviz-canvas{
			height:220px; /* fixed panel height for consistent 4-up layout */
			position:relative;
		}
        </style>';

        if ( isset( $_POST['aegiswaf_save_managed_rules'] ) ) {
            check_admin_referer( 'aegiswaf_save_managed_rules' );
            $settings = AegisWAF_Storage::get_settings();
            $mr = is_array( $settings['managed_rules'] ?? null ) ? $settings['managed_rules'] : [];
            $mr['enabled'] = ! empty( $_POST['mr_enabled'] );
            $mr['mode_pro'] = isset($_POST['mr_mode_pro']) ? sanitize_text_field( wp_unslash($_POST['mr_mode_pro']) ) : 'block';
            $mr['sensitivity'] = isset($_POST['mr_sensitivity']) ? sanitize_text_field( wp_unslash($_POST['mr_sensitivity']) ) : 'balanced';
            $mr['inspect_headers'] = ! empty($_POST['mr_headers']);
            $mr['inspect_cookies'] = ! empty($_POST['mr_cookies']);
            $mr['inspect_body'] = ! empty($_POST['mr_body']);
            $mr['max_body_bytes'] = isset($_POST['mr_max_body']) ? max(1024, (int) $_POST['mr_max_body']) : 65536;
            // thresholds
            $mr['thresholds'] = is_array( $mr['thresholds'] ?? null ) ? $mr['thresholds'] : [];
            $tcats = [ 'sqli','xss','path_traversal','rce','cmd_injection','ssrf','file_upload' ];
            foreach ( $tcats as $t ) {
                if ( isset( $_POST['mr_thr_' . $t ] ) ) { $mr['thresholds'][ $t ] = max( 1, (int) $_POST['mr_thr_' . $t ] ); }
            }

            $mr['heuristics'] = is_array( $mr['heuristics'] ?? null ) ? $mr['heuristics'] : [];
            $mr['heuristics']['enabled'] = ! empty( $_POST['mr_heur_enabled'] );
            if ( isset( $_POST['mr_heur_score'] ) ) { $mr['heuristics']['score_threshold'] = max( 10, (int) $_POST['mr_heur_score'] ); }
            if ( isset( $_POST['mr_heur_pct'] ) ) { $mr['heuristics']['pct_threshold'] = max( 1, (int) $_POST['mr_heur_pct'] ); }
            if ( isset( $_POST['mr_heur_meta'] ) ) { $mr['heuristics']['meta_threshold'] = max( 1, (int) $_POST['mr_heur_meta'] ); }
            if ( isset( $_POST['mr_heur_tok'] ) ) { $mr['heuristics']['token_len_threshold'] = max( 100, (int) $_POST['mr_heur_tok'] ); }

            $mr['challenge'] = is_array( $mr['challenge'] ?? null ) ? $mr['challenge'] : [];

            $mr['categories'] = is_array( $mr['categories'] ?? null ) ? $mr['categories'] : [];
            $mr['challenge']['enabled'] = ! empty( $_POST['mr_chal_enabled'] );
            if ( isset( $_POST['mr_chal_ttl'] ) ) { $mr['challenge']['ttl_seconds'] = max( 300, min( 86400, (int) $_POST['mr_chal_ttl'] ) ); }
            $cats = [ 'sqli','xss','rce','path_traversal','file_upload','cmd_injection','ssrf' ];
            foreach ( $cats as $c ) {
                $mr['categories'][$c] = ! empty( $_POST['mr_cat_' . $c ] );
            }
            $settings['managed_rules'] = $mr;
            AegisWAF_Storage::update_settings( $settings );
            echo '<div class="notice notice-success"><p>Managed rules updated.</p></div>';
			if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
				AegisWAF_Logger::log( 'admin', 'POST', 'settings', 'managed_rules_updated', [
					'enabled' => ! empty( $mr['enabled'] ),
					'inspect_headers' => ! empty( $mr['inspect_headers'] ),
					'inspect_cookies' => ! empty( $mr['inspect_cookies'] ),
					'inspect_body' => ! empty( $mr['inspect_body'] ),
					'max_body_bytes' => (int) ( $mr['max_body_bytes'] ?? 0 ),
					'thresholds' => $mr['thresholds'] ?? [],
					'heuristics' => $mr['heuristics'] ?? [],
					'challenge' => $mr['challenge'] ?? [],
				] );
			} else {
				error_log( '[AegisWAF] managed_rules_updated: logger missing' );
			}
        }

        if ( isset( $_POST['aegiswaf_save_endpoint_policies'] ) ) {
            check_admin_referer( 'aegiswaf_save_endpoint_policies' );
            $settings = AegisWAF_Storage::get_settings();
            $pol = [];
            if ( ! empty( $_POST['ep_path'] ) && is_array( $_POST['ep_path'] ) ) {
                $count = count( $_POST['ep_path'] );
                for ( $i = 0; $i < $count; $i++ ) {
                    $path = sanitize_text_field( wp_unslash( $_POST['ep_path'][ $i ] ) );
                    if ( $path === '' ) { continue; }
                    $row = [
                        'path_pattern' => $path,
                        'mode_pro' => sanitize_text_field( wp_unslash( $_POST['ep_mode_pro'][ $i ] ?? 'block' ) ),
                        'mode_free' => 'log',
                        'categories' => [],
                        'thresholds' => [],
                    ];
                    $cats = [ 'sqli','xss','rce','path_traversal','file_upload','cmd_injection','ssrf' ];
                    foreach ( $cats as $c ) {
                        $row['categories'][ $c ] = ! empty( $_POST['ep_cat_' . $c ][ $i ] );
                        $row['thresholds'][ $c ] = isset( $_POST['ep_thr_' . $c ][ $i ] ) ? max( 0, (int) $_POST['ep_thr_' . $c ][ $i ] ) : 0;
                    }
                    $pol[] = $row;
                }
            }
            $settings['endpoint_policies'] = $pol;
            AegisWAF_Storage::update_settings( $settings );
            echo '<div class="notice notice-success"><p>Endpoint policies updated.</p></div>';
			if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
				AegisWAF_Logger::log( 'admin', 'POST', 'settings', 'endpoint_policies_updated', [
					'count' => is_array( $pol ) ? count( $pol ) : 0,
				] );
			} else {
				error_log( '[AegisWAF] endpoint_policies_updated: logger missing' );
			}
        }

        $rules = AegisWAF_Storage::get_waf_rules();

        if ( isset( $_GET['apply_ruleset'] ) ) {
            check_admin_referer( 'aegiswaf_apply_ruleset' );

            if ( ! $is_pro ) {
                echo '<div class="notice notice-warning"><p>Rule Sets are PRO.</p></div>';
            } else {
                $key = sanitize_key( wp_unslash( $_GET['apply_ruleset'] ) );
                if ( $key === '' ) {
                    echo '<div class="notice notice-error"><p>Invalid ruleset key.</p></div>';
                    error_log( '[AegisWAF] apply_ruleset failed: empty key' );
                } elseif ( ! class_exists( 'AegisWAF_Rule_Sets' ) || ! method_exists( 'AegisWAF_Rule_Sets', 'presets' ) ) {
                    echo '<div class="notice notice-error"><p>Rule Sets engine not available.</p></div>';
                    error_log( '[AegisWAF] apply_ruleset failed: AegisWAF_Rule_Sets missing' );
                    AegisWAF_Logger::log( 'admin', 'GET', 'error', 'ruleset_missing', [ 'key' => $key ] );
				} else {
					if ( $key === 'wp_default' ) {

						$backup = get_option( 'aegiswaf_rules_backup', [] );

						if ( is_array( $backup ) && ! empty( $backup['rules'] ) && is_array( $backup['rules'] ) ) {

							AegisWAF_Storage::update_waf_rules( $backup['rules'] );
							$rules = $backup['rules'];

							$when = ! empty( $backup['saved_at'] ) ? gmdate( 'Y-m-d H:i:s', (int) $backup['saved_at'] ) . ' UTC' : 'previous snapshot';

							echo '<div class="notice notice-success"><p>Restored WP Default rules from ' . esc_html( $when ) . '.</p></div>';

							if ( class_exists( 'AegisWAF_Logger' ) ) {
								AegisWAF_Logger::log( 'admin', 'GET', 'info', 'ruleset_restore_wp_default', [
									'saved_at' => (int) ( $backup['saved_at'] ?? 0 ),
									'count'    => count( $backup['rules'] ),
								] );
							}

						} else {

							echo '<div class="notice notice-warning"><p>No WP Default snapshot found yet. Apply a preset first to create a revert point.</p></div>';

							if ( class_exists( 'AegisWAF_Logger' ) ) {
								AegisWAF_Logger::log( 'admin', 'GET', 'warning', 'ruleset_restore_missing', [] );
							}
						}

					} else {

						$presets = AegisWAF_Rule_Sets::presets();

						if ( empty( $presets[ $key ] ) || ! is_array( $presets[ $key ] ) ) {

							echo '<div class="notice notice-error"><p>Unknown ruleset.</p></div>';
							error_log( '[AegisWAF] apply_ruleset failed: unknown key ' . $key );

							if ( class_exists( 'AegisWAF_Logger' ) ) {
								AegisWAF_Logger::log( 'admin', 'GET', 'error', 'ruleset_unknown', [ 'key' => $key ] );
							}

						} else {

							$set       = $presets[ $key ];
							$add_rules = is_array( $set['rules'] ?? null ) ? $set['rules'] : [];
							$current   = is_array( $rules ) ? $rules : [];

							update_option( 'aegiswaf_rules_backup', [
								'saved_at' => time(),
								'rules'    => $current,
							], false );

							foreach ( $add_rules as $r ) {
								if ( ! is_array( $r ) ) { continue; }
								$current[] = [
									'id'           => 'r_' . wp_generate_password( 8, false, false ),
									'name'         => sanitize_text_field( (string) ( $r['name'] ?? 'Preset Rule' ) ),
									'enabled'      => ! empty( $r['enabled'] ),
									'path_pattern' => sanitize_text_field( (string) ( $r['path_pattern'] ?? '' ) ),
									'method'       => strtoupper( sanitize_text_field( (string) ( $r['method'] ?? 'ANY' ) ) ),
									'action'       => AegisWAF_Utils::sanitize_action( (string) ( $r['action'] ?? 'log' ) ),
									'contains'     => (string) ( $r['contains'] ?? '' ),
									'regex'        => (string) ( $r['regex'] ?? '' ),
								];
							}

							AegisWAF_Storage::update_waf_rules( $current );
							$rules = $current;

							echo '<div class="notice notice-success"><p>Applied ruleset: ' . esc_html( (string) ( $set['name'] ?? $key ) ) . '</p></div>';

							if ( class_exists( 'AegisWAF_Logger' ) ) {
								AegisWAF_Logger::log( 'admin', 'GET', 'info', 'ruleset_applied', [
									'key'  => $key,
									'name' => (string) ( $set['name'] ?? $key ),
								] );
							}
						}
					}
				}
            }
        }

        if ( isset( $_GET['aegiswaf_export_rules'] ) ) {
            if ( ! $is_pro ) {
                wp_die( esc_html__( 'AegisWAF: Export is PRO.', 'aegiswaf' ) );
            }
            $export = AegisWAF_Storage::get_waf_rules();
            nocache_headers();
            header( 'Content-Type: application/json; charset=utf-8' );
            header( 'Content-Disposition: attachment; filename="aegiswaf-waf-rules.json"' );
            echo wp_json_encode( $export );
            exit;
        }

        if ( isset( $_POST['aegiswaf_import_rules'] ) ) {
            check_admin_referer( 'aegiswaf_import_rules' );
            if ( ! $is_pro ) {
                echo '<div class="notice notice-warning"><p>Import is PRO.</p></div>';
            } else {
                $raw = isset( $_POST['rules_json'] ) ? (string) wp_unslash( $_POST['rules_json'] ) : '';
                $data = json_decode( $raw, true );
                if ( is_array( $data ) ) {

                    $norm = [];
                    foreach ( $data as $r ) {
                        if ( ! is_array( $r ) ) { continue; }
                        $path = isset( $r['path_pattern'] ) ? sanitize_text_field( (string) $r['path_pattern'] ) : '';
                        if ( $path === '' ) { continue; }
                        $norm[] = [
                            'id' => isset( $r['id'] ) ? sanitize_text_field( (string) $r['id'] ) : ('r_' . wp_generate_password( 8, false, false )),
                            'name' => isset( $r['name'] ) ? sanitize_text_field( (string) $r['name'] ) : 'Imported Rule',
                            'enabled' => ! empty( $r['enabled'] ),
                            'path_pattern' => $path,
                            'method' => isset( $r['method'] ) ? strtoupper( sanitize_text_field( (string) $r['method'] ) ) : 'ANY',
                            'action' => isset( $r['action'] ) ? AegisWAF_Utils::sanitize_action( (string) $r['action'] ) : 'log',
                            'contains' => isset( $r['contains'] ) ? (string) $r['contains'] : '',
                            'regex' => isset( $r['regex'] ) ? (string) $r['regex'] : '',
                        ];
                    }
                    AegisWAF_Storage::update_waf_rules( $norm );
                    $rules = $norm;
                    echo '<div class="notice notice-success"><p>Rules imported.</p></div>';
                } else {
                    echo '<div class="notice notice-error"><p>Invalid JSON.</p></div>';
                }
            }
        }

        if ( ! is_array( $rules ) ) { $rules = []; }

        if ( isset( $_POST['aegiswaf_save_rules'] ) ) {
            check_admin_referer( 'aegiswaf_save_rules' );

            $names   = isset( $_POST['rule_name'] ) && is_array( $_POST['rule_name'] ) ? $_POST['rule_name'] : [];
            $enabled = isset( $_POST['rule_enabled'] ) && is_array( $_POST['rule_enabled'] ) ? $_POST['rule_enabled'] : [];
            $paths   = isset( $_POST['rule_path'] ) && is_array( $_POST['rule_path'] ) ? $_POST['rule_path'] : [];
            $methods = isset( $_POST['rule_method'] ) && is_array( $_POST['rule_method'] ) ? $_POST['rule_method'] : [];
            $actions = isset( $_POST['rule_action'] ) && is_array( $_POST['rule_action'] ) ? $_POST['rule_action'] : [];
            $contains= isset( $_POST['rule_contains'] ) && is_array( $_POST['rule_contains'] ) ? $_POST['rule_contains'] : [];
            $regex   = isset( $_POST['rule_regex'] ) && is_array( $_POST['rule_regex'] ) ? $_POST['rule_regex'] : [];
            $ids     = isset( $_POST['rule_id'] ) && is_array( $_POST['rule_id'] ) ? $_POST['rule_id'] : [];

            $new = [];
            $count = max( count($names), count($paths), count($ids) );
            for ( $i=0; $i<$count; $i++ ) {
                $id = isset( $ids[$i] ) ? sanitize_text_field( wp_unslash( $ids[$i] ) ) : '';
                if ( $id === '' ) {
                    $id = 'r_' . wp_generate_password( 8, false, false );
                }
                $name = isset( $names[$i] ) ? sanitize_text_field( wp_unslash( $names[$i] ) ) : '';
                $path = isset( $paths[$i] ) ? sanitize_text_field( wp_unslash( $paths[$i] ) ) : '';
                if ( $path === '' ) { continue; }

                $method = isset( $methods[$i] ) ? strtoupper( sanitize_text_field( wp_unslash( $methods[$i] ) ) ) : 'ANY';
                $allowed_methods = [ 'ANY','GET','POST','PUT','PATCH','DELETE','HEAD','OPTIONS' ];
                if ( ! in_array( $method, $allowed_methods, true ) ) { $method = 'ANY'; }

                $action = isset( $actions[$i] ) ? sanitize_text_field( wp_unslash( $actions[$i] ) ) : 'log';
                $action = AegisWAF_Utils::sanitize_action( $action );
                if ( ! in_array( $action, [ 'log','block','allow' ], true ) ) { $action = 'log'; }

                $en = ! empty( $enabled[$i] );

                $c = isset( $contains[$i] ) ? (string) wp_unslash( $contains[$i] ) : '';
                $rx= isset( $regex[$i] ) ? (string) wp_unslash( $regex[$i] ) : '';

                if ( ! $is_pro ) {
                    $rx = '';
                }

                $new[] = [
                    'id' => $id,
                    'name' => $name !== '' ? $name : 'Rule ' . ( $i + 1 ),
                    'enabled' => $en,
                    'path_pattern' => $path,
                    'method' => $method,
                    'action' => $action,
                    'contains' => $c,
                    'regex' => $rx,
                ];
            }

            if ( ! $is_pro && count( $new ) > 5 ) {
                $new = array_slice( $new, 0, 5 );
            }

            AegisWAF_Storage::update_waf_rules( $new );
            $rules = $new;

            echo '<div class="notice notice-success"><p>WAF Rules saved.</p></div>';
			if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
				AegisWAF_Logger::log( 'admin', 'POST', 'settings', 'waf_rules_saved', [
					'count' => is_array( $rules ) ? count( $rules ) : 0,
					'is_pro' => (bool) $is_pro,
				] );
			} else {
				error_log( '[AegisWAF] waf_rules_saved: logger missing' );
			}
        }

        echo '<div class="wrap">';
        echo '<h1>Web Application Firewall Rules</h1>';
        AegisWAF_Pro_Notice::render();
        echo '<p class="description">The WAF Rules tab controls the core rule engine that inspects and evaluates incoming requests against predefined security rules. Administrators can enable or disable rule categories, apply default rule presets, and leverage heuristic scoring to detect malicious behavior. Pro unlocks advanced rule sensitivity tuning, custom rule overrides, and virtual patching capabilities, allowing sites to respond quickly to emerging vulnerabilities without waiting for application updates.</p>';

		echo '<div class="aegiswaf-card" style="margin-top:14px;">';
		echo '<div class="aegiswaf-sectionhead"><span class="dashicons dashicons-chart-area"></span><div>';
		echo '<h3>WAF Visual Intelligence</h3>';
		echo '<p class="description" style="margin:4px 0 0;">Instant visibility into WAF threat activity (managed rules + heuristics) without digging through logs.</p>';
		echo '<p style="margin:10px 0 0;font-weight:600;">Last 24h: <span id="aegiswaf_waf_total_24h">0</span> &nbsp; | &nbsp; Last 7d: <span id="aegiswaf_waf_total_7d">0</span></p>';
		echo '</div></div>';

		echo '<div class="aegiswaf-wafviz-grid">';
			echo '<div class="aegiswaf-wafviz-panel">';
			echo '<div class="aegiswaf-wafviz-title">Violations Trend</div>';
			echo '<div class="aegiswaf-wafviz-sub">Last 7 days (all WAF hits)</div>';
			echo '<div class="aegiswaf-wafviz-canvas"><canvas id="aegiswaf_waf_chart_trend"></canvas></div>';
			echo '</div>';

			echo '<div class="aegiswaf-wafviz-panel">';
			echo '<div class="aegiswaf-wafviz-title">Threat Categories</div>';
			echo '<div class="aegiswaf-wafviz-sub">SQLi / XSS / RCE / etc.</div>';
			echo '<div class="aegiswaf-wafviz-canvas"><canvas id="aegiswaf_waf_chart_categories"></canvas></div>';
			echo '</div>';

			echo '<div class="aegiswaf-wafviz-panel">';
			echo '<div class="aegiswaf-wafviz-title">Top Attacked Routes</div>';
			echo '<div class="aegiswaf-wafviz-sub">Only block/challenge/rate-limit</div>';
			echo '<div class="aegiswaf-wafviz-canvas"><canvas id="aegiswaf_waf_chart_routes"></canvas></div>';
			echo '</div>';

			echo '<div class="aegiswaf-wafviz-panel">';
			echo '<div class="aegiswaf-wafviz-title">Action Outcomes</div>';
			echo '<div class="aegiswaf-wafviz-sub">What AegisWAF is doing</div>';
			echo '<div class="aegiswaf-wafviz-canvas"><canvas id="aegiswaf_waf_chart_actions"></canvas></div>';
			echo '</div>';
		echo '</div>'; // grid

		echo '</div>'; // card

        $settings = AegisWAF_Storage::get_settings();
        $mr = is_array( $settings['managed_rules'] ?? null ) ? $settings['managed_rules'] : [];
        $thr = is_array( $mr['thresholds'] ?? null ) ? $mr['thresholds'] : [];
        $h = is_array( $mr['heuristics'] ?? null ) ? $mr['heuristics'] : [];
        $ch = is_array( $mr['challenge'] ?? null ) ? $mr['challenge'] : [];
		$health_errors = [];

		if ( ! class_exists( 'AegisWAF_Managed_Rules' ) || ! method_exists( 'AegisWAF_Managed_Rules', 'evaluate' ) ) {
			$health_errors[] = 'Managed Rules engine missing (AegisWAF_Managed_Rules::evaluate). Managed Rules + Category Thresholds cannot be enforced.';
		}

		if ( ! class_exists( 'AegisWAF_Heuristics' ) || ! method_exists( 'AegisWAF_Heuristics', 'evaluate' ) ) {
			$health_errors[] = 'Heuristics engine missing (AegisWAF_Heuristics::evaluate). Heuristic Layer cannot be enforced.';
		}

		if ( ! class_exists( 'AegisWAF_Endpoint_Policy' ) ) {
			$health_errors[] = 'Endpoint Policy engine missing (AegisWAF_Endpoint_Policy). Endpoint Policies cannot be enforced.';
		}

		if ( ! class_exists( 'AegisWAF_Challenge' ) ) {
			$health_errors[] = 'Challenge engine missing (AegisWAF_Challenge). Challenge Token cannot be enforced.';
		}

		$thr_configured = is_array( $thr ) && array_filter( $thr, static function( $v ){ return (int)$v > 1; } );
		$mr_enabled = ! empty( $mr['enabled'] );

		if ( $thr_configured && ! $mr_enabled ) {
			$health_errors[] = 'Category Thresholds are configured (>1) but Managed Rules is disabled — thresholds will not be applied.';
		}

		if ( ! empty( $health_errors ) ) {
			echo '<div class="notice notice-warning"><p><strong>AegisWAF WAF Rules — Configuration Warning</strong></p><ul>';
			foreach ( $health_errors as $msg ) {
				echo '<li>' . esc_html( $msg ) . '</li>';
			}
			echo '</ul><p>Check your Logs tab for details.</p></div>';

			if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
				AegisWAF_Logger::log( 'admin', 'GET', 'warning', 'waf_rules_health', [
					'errors' => $health_errors,
					'managed_rules_enabled' => $mr_enabled,
					'thresholds' => $thr,
					'heuristics_enabled' => ! empty( $h['enabled'] ),
					'challenge_enabled' => ! empty( $ch['enabled'] ),
				] );
			} else {
				error_log( '[AegisWAF] WAF Rules health warnings: ' . wp_json_encode( $health_errors ) );
			}
		}

        echo '<div class="aegiswaf-card" style="width:100%;max-width:100%;box-sizing:border-box;margin-top:12px;">';
		echo '<div class="aegiswaf-cardhead"><h3>Heuristic Rules (PRO) + False Positive Tuning</h3></div>';
        if ( $is_pro ) {
            echo '<form method="post">';
            wp_nonce_field( 'aegiswaf_save_managed_rules' );
            echo '<p><label><input type="checkbox" name="mr_enabled" value="1" ' . checked( ! empty( $mr['enabled'] ), true, false ) . '> Managed rules enabled</label></p>';
            echo '<p class="aegiswaf-inline"><label><input type="checkbox" name="mr_headers" value="1" ' . checked( ! empty( $mr['inspect_headers'] ), true, false ) . '> Inspect headers</label> ';
            echo '<label><input type="checkbox" name="mr_cookies" value="1" ' . checked( ! empty( $mr['inspect_cookies'] ), true, false ) . '> Inspect cookies</label> ';
            echo '<label><input type="checkbox" name="mr_body" value="1" ' . checked( ! empty( $mr['inspect_body'] ), true, false ) . '> Inspect body</label></p>';
            echo '<p><label>Max body bytes <input type="number" name="mr_max_body" min="1024" max="262144" value="' . esc_attr( (string) ( $mr['max_body_bytes'] ?? 65536 ) ) . '"></label></p>';

            echo '<div class="aegiswaf-subhead"><span class="dashicons dashicons-search"></span><h3>Heuristic Layer</h3></div><p class="description">Scores each request for suspicious patterns to catch evasive attacks with fewer false positives.</p>';
            echo '<p><label><input type="checkbox" name="mr_heur_enabled" value="1" ' . checked( ! empty( $h['enabled'] ), true, false ) . '> Enable heuristics</label></p>';
            echo '<p><label>Heuristic score threshold <input type="number" name="mr_heur_score" value="' . esc_attr( (string) ( $h['score_threshold'] ?? 40 ) ) . '" min="10" max="200"></label></p>';
            echo '<p><label>% encoding threshold <input type="number" name="mr_heur_pct" value="' . esc_attr( (string) ( $h['pct_threshold'] ?? 8 ) ) . '" min="1" max="50"></label> ';
            echo '<label>Meta char threshold <input type="number" name="mr_heur_meta" value="' . esc_attr( (string) ( $h['meta_threshold'] ?? 12 ) ) . '" min="1" max="200"></label></p>';
            echo '<p><label>Very long token length <input type="number" name="mr_heur_tok" value="' . esc_attr( (string) ( $h['token_len_threshold'] ?? 900 ) ) . '" min="100" max="5000"></label></p>';

			echo '<div class="aegiswaf-subhead">
					<span class="dashicons dashicons-filter"></span>
					<h3>Category Thresholds</h3>
				  </div>
				  <p class="description">
					Increase to reduce false positives. <strong>1 = immediate</strong>.</br>
					Thresholds: 	1, 2, 3+</br>
					Definitions: 	First hit triggers immediately, Needs 2 separate rule hits, Requires multiple indicators -> fewer false positives
				  </p>';

			$cats = [
				'sqli'=>'SQLi',
				'xss'=>'XSS',
				'path_traversal'=>'Path Traversal',
				'rce'=>'RCE',
				'cmd_injection'=>'Command Injection',
				'ssrf'=>'SSRF',
				'file_upload'=>'File Upload'
			];

			echo '<table class="widefat striped" style="max-width:900px">
					<thead>
						<tr>
							<th>Category</th>
							<th style="width:140px">Hits Required</th>
						</tr>
					</thead>
					<tbody>';

			foreach ( $cats as $k => $label ) {
				$v = isset( $thr[$k] ) ? (int) $thr[$k] : 1;

				echo '<tr>
						<td><strong>' . esc_html( $label ) . '</strong></td>
						<td>
							<input type="number"
								   name="mr_thr_' . esc_attr($k) . '"
								   value="' . esc_attr((string)$v) . '"
								   min="1"
								   max="10"
								   style="width:80px">
						</td>
					  </tr>';
			}

			echo '</tbody></table>';

            echo '<div class="aegiswaf-subhead"><span class="dashicons dashicons-lock"></span><h3>Challenge Token</h3></div><p class="description">Adds a lightweight browser challenge to slow down bots before allowing risky traffic.</p>';
            echo '<p><label><input type="checkbox" name="mr_chal_enabled" value="1" ' . checked( ! empty( $ch['enabled'] ), true, false ) . '> Enable JS challenge token</label></p>';
            echo '<p><label>Challenge TTL seconds <input type="number" name="mr_chal_ttl" value="' . esc_attr( (string) ( $ch['ttl_seconds'] ?? 1800 ) ) . '" min="300" max="86400"></label></p>';

            echo '<p><button class="button button-primary" name="aegiswaf_save_managed_rules" value="1">Save Tuning</button></p>';
            echo '</form>';
        } else {
            AegisWAF_Utils::pro_gate_box('Heuristics + Tuning (PRO)','Unlock heuristics, per-category thresholds, and challenge mode.');
        }
        echo '</div>';

        echo '<div class="aegiswaf-card" style="width:100%;max-width:100%;box-sizing:border-box;margin-top:12px;">';
		echo '<div class="aegiswaf-cardhead"><div class="aegiswaf-sectionhead"><span class="dashicons dashicons-admin-links"></span><div><h3>Endpoint Policies</h3></div>';
		echo '<p class="description">Override rule categories, thresholds, and action mode for specific paths like <code>/wp-json/*</code>.</p></div></div>';
        if ( $is_pro ) {
            $eps = is_array( $settings['endpoint_policies'] ?? null ) ? $settings['endpoint_policies'] : [];
            echo '<form method="post">';
            wp_nonce_field( 'aegiswaf_save_endpoint_policies' );
            echo '<p class="description">Override categories/thresholds per path. Use wildcards like <code>/wp-json/*</code>.</p>';
            echo '<table class="widefat striped"><thead><tr><th>Path Pattern</th><th>Mode (PRO)</th><th>SQLi</th><th>XSS</th><th>Traversal</th><th>RCE</th><th>CMD</th><th>SSRF</th><th>Upload</th></tr></thead><tbody>';

            $rows = max( 1, count( $eps ) );
            for ( $i=0; $i<$rows; $i++ ) {
                $row = $eps[$i] ?? [];
                $path = esc_attr( (string) ( $row['path_pattern'] ?? '' ) );
                $mode = (string) ( $row['mode_pro'] ?? 'block' );
                $catsv = is_array( $row['categories'] ?? null ) ? $row['categories'] : [];
                $thrv  = is_array( $row['thresholds'] ?? null ) ? $row['thresholds'] : [];
                echo '<tr>';
                echo '<td><input style="width:100%" name="ep_path[]" value="' . $path . '" placeholder="/wp-login.php"></td>';
                echo '<td><select name="ep_mode_pro[]">'
                    . '<option value="block" ' . selected($mode,'block',false) . '>Block</option>'
                    . '<option value="challenge" ' . selected($mode,'challenge',false) . '>Challenge</option>'
                    . '<option value="rate_limit" ' . selected($mode,'rate_limit',false) . '>Rate-limit</option>'
                    . '<option value="log" ' . selected($mode,'log',false) . '>Log</option>'
                    . '</select></td>';
                $cc = [ 'sqli','xss','path_traversal','rce','cmd_injection','ssrf','file_upload' ];
                foreach ( $cc as $c ) {
                    $chk = ! empty( $catsv[$c] );
                    $thr = isset( $thrv[$c] ) ? (int) $thrv[$c] : 0;
                    echo '<td><label><input type="checkbox" name="ep_cat_' . esc_attr($c) . '[]" value="1" ' . checked($chk,true,false) . '> <input type="number" name="ep_thr_' . esc_attr($c) . '[]" value="' . esc_attr((string)$thr) . '" min="0" max="10" style="width:60px" title="threshold"></label></td>';
                }
                echo '</tr>';
            }
            echo '</tbody></table>';
            echo '<p><button class="button button-primary" name="aegiswaf_save_endpoint_policies" value="1">Save Endpoint Policies</button></p>';
            echo '</form>';
        } else {
            AegisWAF_Utils::pro_gate_box('Endpoint Policies (PRO)','Enable per-path category controls and thresholds.');
        }
        echo '</div>';

		echo '<div class="aegiswaf-card" style="width:100%;max-width:100%;box-sizing:border-box;margin-top:12px;">';
		echo '<div class="aegiswaf-cardhead"><span class="dashicons dashicons-list-view"></span><h3>Rule Sets</h3></div>';
		echo '<p class="description">Apply prebuilt presets to instantly protect common WordPress entry points.</p>';
		if ( $is_pro ) {

			$sets = AegisWAF_Rule_Sets::presets();

			$sets = array_merge(
				[
					'wp_default' => [
						'name'  => 'WP Default (Revert)',
						'rules' => [],
					],
				],
				is_array( $sets ) ? $sets : []
			);

			$desc = [
				'wp_default'        => 'Revert WAF rules back to the state before the last preset was applied.',
				'wp_login_protect'  => 'Blocks abusive POST traffic to /wp-login.php to reduce brute-force and credential stuffing.',
				'xmlrpc_block'      => 'Blocks all access to /xmlrpc.php, removing a common remote attack surface.',
				'wpjson_throttle'   => 'Logs traffic to /wp-json/* so admins can review REST API usage and detect abuse patterns.',
			];

			echo '<div class="aegiswaf-rulesets">';

			foreach ( $sets as $k => $s ) {

				$name  = (string) ( $s['name'] ?? $k );
				$rules = is_array( $s['rules'] ?? null ) ? $s['rules'] : [];

				$url = wp_nonce_url(
					add_query_arg( [ 'apply_ruleset' => $k ] ),
					'aegiswaf_apply_ruleset'
				);

				echo '<div class="aegiswaf-card" style="margin:0 0 12px;">';
				echo '<div class="aegiswaf-cardhead"><span class="dashicons dashicons-list-view"></span><h2>' . esc_html( $name ) . '</h2></div>';

				if ( ! empty( $desc[ $k ] ) ) {
					echo '<p class="description">' . esc_html( $desc[ $k ] ) . '</p>';
				}

				if ( $k === 'wp_default' ) {
					echo '<ul style="margin:0 0 10px 18px;list-style:disc;">';
					echo '<li><strong>What it does:</strong> Restores your rules to the last saved snapshot.</li>';
					echo '<li><strong>When snapshot is created:</strong> Automatically when you apply any preset below.</li>';
					echo '<li><strong>If missing:</strong> You will see a warning, and nothing will change.</li>';
					echo '</ul>';
				} else {

					echo '<table class="widefat striped" style="margin-top:8px;">';
					echo '<thead><tr>';
					echo '<th style="width:18%;">Rule Name</th>';
					echo '<th style="width:18%;">Path Pattern</th>';
					echo '<th style="width:10%;">Method</th>';
					echo '<th style="width:10%;">Action</th>';
					echo '<th style="width:18%;">Contains</th>';
					echo '<th style="width:18%;">Regex</th>';
					echo '<th style="width:8%;">Enabled</th>';
					echo '</tr></thead><tbody>';

					foreach ( $rules as $r ) {
						if ( ! is_array( $r ) ) { continue; }

						$r_name  = (string) ( $r['name'] ?? '' );
						$path    = (string) ( $r['path_pattern'] ?? '' );
						$method  = (string) ( $r['method'] ?? 'ANY' );
						$action  = (string) ( $r['action'] ?? 'log' );
						$contains= (string) ( $r['contains'] ?? '' );
						$regex   = (string) ( $r['regex'] ?? '' );
						$enabled = ! empty( $r['enabled'] ) ? 'Yes' : 'No';

						echo '<tr>';
						echo '<td>' . esc_html( $r_name ) . '</td>';
						echo '<td><code>' . esc_html( $path ) . '</code></td>';
						echo '<td>' . esc_html( strtoupper( $method ) ) . '</td>';
						echo '<td><strong>' . esc_html( $action ) . '</strong></td>';
						echo '<td>' . ( $contains !== '' ? '<code>' . esc_html( $contains ) . '</code>' : '<span class="description">—</span>' ) . '</td>';
						echo '<td>' . ( $regex !== '' ? '<code>' . esc_html( $regex ) . '</code>' : '<span class="description">—</span>' ) . '</td>';
						echo '<td>' . esc_html( $enabled ) . '</td>';
						echo '</tr>';
					}

					echo '</tbody></table>';
				}

				echo '<p style="margin:12px 0 0;">';
				echo '<a class="button button-primary" href="' . esc_url( $url ) . '">Apply</a>';
				echo '</p>';

				echo '</div>'; // card
			}

			echo '</div>'; // rulesets wrapper

		} else {
			AegisWAF_Utils::pro_gate_box('Rule Sets (PRO)','Apply prebuilt protection presets.');
		}
		echo '</div>';
		echo '<div class="aegiswaf-card" style="width:100%;max-width:100%;box-sizing:border-box;margin-top:12px;">';
		echo '<div class="aegiswaf-cardhead"><span class="dashicons dashicons-migrate"></span><div><h3>Import / Export</h3></div>';
		echo '<p class="description">Move rules between sites by exporting JSON and importing it on another install.</p></div></div>';
        if ( $is_pro ) {
            $url = add_query_arg( [ 'page' => 'aegiswaf', 'tab' => 'waf_rules', 'aegiswaf_export_rules' => 1 ], admin_url( 'admin.php' ) );
            echo '<p><a class="button" href="' . esc_url( $url ) . '">Export Rules (JSON)</a></p>';
            echo '<form method="post">';
            wp_nonce_field( 'aegiswaf_import_rules' );
            echo '<p><textarea name="rules_json" rows="6" style="width:100%;" placeholder="Paste rules JSON here..."></textarea></p>';
            echo '<p><button class="button" name="aegiswaf_import_rules" value="1">Import Rules</button></p>';
            echo '</form>';
        } else {
            AegisWAF_Utils::pro_gate_box( 'Import / Export (PRO)', 'Move rules between sites. Export JSON + Import JSON.' );
        }
        echo '</div>';


        if ( ! $is_pro ) {
            AegisWAF_Utils::pro_gate_box(
                'Advanced Rules (PRO)',
                'PRO unlocks unlimited rules and regex matching for deep request inspection.'
            );
        }

        echo '<form method="post">';
        wp_nonce_field( 'aegiswaf_save_rules' );

        $rows = array_values( $rules );
        $max = $is_pro ? 10 : 5; // starter UI rows

        echo '<table class="widefat striped"><thead><tr>';
        echo '<th>On</th><th>Name</th><th>Path Pattern</th><th>Method</th><th>Action</th><th>Contains (all)</th><th>Regex (PRO)</th>';
        echo '</tr></thead><tbody>';

        for ( $i=0; $i<$max; $i++ ) {
            $r = $rows[$i] ?? [];
            $on = ! empty( $r['enabled'] );
            $name = (string) ( $r['name'] ?? '' );
            $path = (string) ( $r['path_pattern'] ?? '' );
            $method = (string) ( $r['method'] ?? 'ANY' );
            $action = (string) ( $r['action'] ?? 'log' );
            $contains = (string) ( $r['contains'] ?? '' );
            $regex = (string) ( $r['regex'] ?? '' );
            $id = (string) ( $r['id'] ?? '' );

            echo '<tr>';
            echo '<td><input type="checkbox" name="rule_enabled['.$i.']" value="1" '.checked($on,true,false).'></td>';
            echo '<td><input type="hidden" name="rule_id['.$i.']" value="'.esc_attr($id).'"><input type="text" name="rule_name['.$i.']" value="'.esc_attr($name).'" placeholder="Block login abuse" style="width:180px;"></td>';
            echo '<td><input type="text" name="rule_path['.$i.']" value="'.esc_attr($path).'" placeholder="/wp-login.php" style="width:220px;"></td>';
            echo '<td><select name="rule_method['.$i.']">';
            foreach ( [ 'ANY','GET','POST','PUT','PATCH','DELETE','HEAD','OPTIONS' ] as $m ) {
                echo '<option value="'.esc_attr($m).'" '.selected($method,$m,false).'>'.esc_html($m).'</option>';
            }
            echo '</select></td>';
            echo '<td><select name="rule_action['.$i.']">';
            foreach ( [ 'log'=>'Log', 'block'=>'Block', 'allow'=>'Allow' ] as $k=>$lab ) {
                echo '<option value="'.esc_attr($k).'" '.selected($action,$k,false).'>'.esc_html($lab).'</option>';
            }
            echo '</select></td>';
            echo '<td><textarea name="rule_contains['.$i.']" rows="2" style="width:220px;">'.esc_textarea($contains).'</textarea><div class="description">1 token per line</div></td>';

            if ( $is_pro ) {
                echo '<td><textarea name="rule_regex['.$i.']" rows="2" style="width:220px;">'.esc_textarea($regex).'</textarea><div class="description">1 regex per line</div></td>';
            } else {
                echo '<td><div class="aegiswaf-dim"><textarea disabled rows="2" style="width:220px;"></textarea><div class="description">PRO only</div></div><input type="hidden" name="rule_regex['.$i.']" value=""></td>';
            }

            echo '</tr>';
        }

        echo '</tbody></table>';

        submit_button( 'Save WAF Rules', 'primary', 'aegiswaf_save_rules' );
        echo '</form>';
    }
}
