<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_Managed_Rules {

    public static function evaluate( array $payload, bool $is_pro, array $ctx = [] ) : ?array {
        $settings = AegisWAF_Storage::get_settings();
        $mr = is_array( $settings['managed_rules'] ?? null ) ? $settings['managed_rules'] : [];

        if ( empty( $mr['enabled'] ) ) { return null; }
        if ( empty( $mr['categories'] ) || ! is_array( $mr['categories'] ) ) {
            if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
                AegisWAF_Logger::log( 'engine', 'SCAN', 'managed_rules', 'config_missing', [
                    'missing' => 'categories',
                ] );
            }
            error_log( '[AegisWAF] Managed Rules enabled but categories missing/invalid — nothing will match.' );
        }
        if ( empty( $mr['thresholds'] ) || ! is_array( $mr['thresholds'] ) ) {
            if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
                AegisWAF_Logger::log( 'engine', 'SCAN', 'managed_rules', 'config_missing', [
                    'missing' => 'thresholds',
                ] );
            }
            error_log( '[AegisWAF] Managed Rules enabled but thresholds missing/invalid — defaults will be used.' );
        }


        $categories = is_array( $mr['categories'] ?? null ) ? $mr['categories'] : [];
        $thresholds = is_array( $mr['thresholds'] ?? null ) ? $mr['thresholds'] : [];
        $mode = $is_pro ? (string) ( $mr['mode_pro'] ?? 'block' ) : (string) ( $mr['mode_free'] ?? 'log' );
        $sensitivity = (string) ( $mr['sensitivity'] ?? 'balanced' );

        if ( ! $is_pro ) {
            $mode = 'log';

            $categories = array_merge(
                [ 'sqli' => true, 'xss' => true, 'path_traversal' => true ],
                $categories
            );
            $categories['rce'] = false;
            $categories['cmd_injection'] = false;
            $categories['ssrf'] = false;
            $categories['file_upload'] = false;
        }

        $hay = (string) ( $payload['norm'] ?? '' );
        if ( $hay === '' ) { return null; }

        $packs = self::signature_packs( $sensitivity );

        foreach ( $packs as $cat => $rules ) {
            $enabled = $categories;
            if ( isset( $ctx['categories'] ) && is_array( $ctx['categories'] ) ) { $enabled = $ctx['categories']; }
            if ( empty( $enabled[ $cat ] ) ) { continue; }
            foreach ( $rules as $r ) {
                $id = (string) $r['id'];
                $re = (string) $r['re'];
                if ( @preg_match( $re, '' ) === false ) {
                    if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
                        AegisWAF_Logger::log( 'engine', 'SCAN', 'managed_rule', 'regex_invalid', [ 'category' => $cat, 'rule_id' => $id, 'regex' => $re ] );
                    }
                    error_log( '[AegisWAF] Invalid managed rule regex in ' . $cat . ' (' . $id . '): ' . $re );
                    continue;
                }
                if ( preg_match( $re, $hay ) ) {
                    $need = max( 1, (int) ( $ctx['thresholds'][ $cat ] ?? ( $thresholds[ $cat ] ?? 1 ) ) );
                    $got  = 1;
                    if ( $got < $need ) { continue; }
                    return [
                        'type' => 'managed',
                        'category' => $cat,
                        'rule_id' => $id,
                        'action' => self::sanitize_action( $mode ),
                        'message' => (string) ( $r['msg'] ?? 'Managed rule hit' ),
                    ];
                }
            }
        }

        return null;
    }

    protected static function sanitize_action( string $mode ) : string {
        $mode = strtolower( trim( $mode ) );
        $allowed = [ 'block', 'challenge', 'rate_limit', 'log' ];
        return in_array( $mode, $allowed, true ) ? $mode : 'log';
    }

    public static function signature_packs( string $sensitivity = 'balanced' ) : array {
        $strict = ( $sensitivity === 'strict' );

        return [
            'sqli' => [
                [ 'id' => 'sqli_01', 're' => '#\b(union\s+select|select\s+.*\s+from|insert\s+into|update\s+\w+\s+set|delete\s+from)\b#i', 'msg' => 'SQLi keyword pattern' ],
                [ 'id' => 'sqli_02', 're' => '#(\bor\b|\band\b)\s+\d+=\d+\b#i', 'msg' => 'SQLi tautology pattern' ],
                [ 'id' => 'sqli_03', 're' => '#\b(sleep\s*\(|benchmark\s*\()#i', 'msg' => 'Time-based SQLi function' ],
            ],
            'xss' => [
                [ 'id' => 'xss_01', 're' => '#<\s*script\b#i', 'msg' => 'XSS script tag' ],
                [ 'id' => 'xss_02', 're' => '#(onerror\s*=|onload\s*=|onclick\s*=)#i', 'msg' => 'XSS inline event handler' ],
                [ 'id' => 'xss_03', 're' => '#javascript\s*:#i', 'msg' => 'XSS javascript: scheme' ],
            ],
            'rce' => [
                [ 'id' => 'rce_01', 're' => '#\b(system\(|exec\(|shell_exec\(|passthru\()#i', 'msg' => 'RCE PHP exec functions' ],
                [ 'id' => 'rce_02', 're' => '#\b(powershell|cmd\.exe|/bin/sh|/bin/bash)\b#i', 'msg' => 'RCE shell invocation' ],
            ],
            'path_traversal' => [
                [ 'id' => 'pth_01', 're' => '#\.\./|\.\.\\#', 'msg' => 'Path traversal ../' ],
                [ 'id' => 'pth_02', 're' => '#(%2e%2e%2f|%2e%2e%5c)#i', 'msg' => 'Encoded traversal' ],
            ],
            'cmd_injection' => [
                [ 'id' => 'cmd_01', 're' => '#(;|\|\||&&|\|)\s*(curl|wget|nc|bash|sh)\b#i', 'msg' => 'Command injection chaining' ],
                [ 'id' => 'cmd_02', 're' => '#\$\(([^\)]{1,80})\)|`([^`]{1,80})`#', 'msg' => 'Command substitution' ],
            ],
            'ssrf' => [
                [ 'id' => 'ssrf_01', 're' => '#\b(http|https)://(127\.0\.0\.1|localhost|0\.0\.0\.0|169\.254\.)#i', 'msg' => 'SSRF localhost/meta' ],
                [ 'id' => 'ssrf_02', 're' => '#\b(file|gopher|dict|ftp)://#i', 'msg' => 'SSRF dangerous scheme' ],
            ],
            'file_upload' => [
                [ 'id' => 'up_01', 're' => '#\.(php\d*|phtml|phar|asp|aspx|jsp)\b#i', 'msg' => 'Executable upload extension' ],
                [ 'id' => 'up_02', 're' => '#content-type\s*:\s*application/x-php#i', 'msg' => 'PHP content-type' ],
            ],
        ];
    }
}
