<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_WP_Protect {

    public static function init() : void {
        add_action( 'login_init', [ __CLASS__, 'inspect_login_request' ], 1 );

        add_filter( 'authenticate', [ __CLASS__, 'authenticate_guard' ], 1, 3 );
        add_action( 'wp_login_failed', [ __CLASS__, 'on_login_failed' ], 10, 1 );
        add_action( 'wp_login', [ __CLASS__, 'on_login_success' ], 10, 2 );

        add_action( 'admin_init', [ __CLASS__, 'inspect_admin_surface' ], 1 );

        add_filter( 'rest_pre_dispatch', [ __CLASS__, 'rest_pre_dispatch' ], 1, 3 );

        add_action( 'init', [ __CLASS__, 'inspect_xmlrpc' ], 0 );
    }

    protected static function cfg() : array {
        $s = AegisWAF_Storage::get_settings();
        $wp = is_array( $s['wp_protect'] ?? null ) ? $s['wp_protect'] : [];
        return $wp;
    }

    protected static function ip() : string {
        return (string) AegisWAF_Utils::client_ip();
    }

    protected static function now() : int { return time(); }

    protected static function inc( string $key, int $window ) : int {
        $key = 'aegiswaf_' . preg_replace( '/[^a-z0-9_]/i', '_', $key );
        $cur = (int) get_transient( $key );
        $cur++;
        set_transient( $key, $cur, max( 30, $window ) );
        return $cur;
    }

    protected static function get( string $key ) : int {
        $key = 'aegiswaf_' . preg_replace( '/[^a-z0-9_]/i', '_', $key );
        return (int) get_transient( $key );
    }

    protected static function reset( string $key ) : void {
        $key = 'aegiswaf_' . preg_replace( '/[^a-z0-9_]/i', '_', $key );
        delete_transient( $key );
    }

    protected static function progressive_action( int $count, array $t ) : string {
        $challenge = (int) ( $t['challenge_at'] ?? 0 );
        $rate      = (int) ( $t['rate_limit_at'] ?? 0 );
        $block     = (int) ( $t['block_at'] ?? 0 );

        if ( $block > 0 && $count >= $block ) { return 'block'; }
        if ( $rate > 0 && $count >= $rate ) { return 'rate_limit'; }
        if ( $challenge > 0 && $count >= $challenge ) { return 'challenge'; }
        return 'log';
    }

    protected static function enforce( string $surface, string $path, string $method, string $action, array $meta = [] ) : void {
        $ip = self::ip();
        $ua = (string) AegisWAF_Utils::user_agent();

        AegisWAF_Logger::log( $path, $method, 'wp_protect', $action, array_merge( [
            'surface' => $surface,
            'ip' => $ip,
            'ua' => $ua,
        ], $meta ) );

        if ( $action === 'log' ) {
            return;
        }

        if ( $action === 'rate_limit' ) {
            if ( is_admin() || wp_doing_ajax() ) {
                wp_die( esc_html__( 'AegisWAF: Too many requests.', 'aegiswaf' ), esc_html__( 'Rate limited', 'aegiswaf' ), [ 'response' => 429 ] );
            }
            wp_die( esc_html__( 'AegisWAF: Too many requests.', 'aegiswaf' ), esc_html__( 'Rate limited', 'aegiswaf' ), [ 'response' => 429 ] );
        }

        if ( $action === 'challenge' ) {
            if ( class_exists( 'AegisWAF_Challenge' ) && AegisWAF_Challenge::enabled() ) {
                if ( ! AegisWAF_Challenge::verify( $ip ) ) {
                    AegisWAF_Challenge::render_page( 'Challenge required to continue.' );
                }
            }
            return;
        }

        wp_die( esc_html__( 'AegisWAF: Request blocked.', 'aegiswaf' ), esc_html__( 'Blocked', 'aegiswaf' ), [ 'response' => 403 ] );
    }

    public static function inspect_login_request() : void {
        $wp = self::cfg();
        if ( empty( $wp['enabled'] ) || empty( $wp['login_protect'] ) ) { return; }

        $path = '/wp-login.php';
        $method = isset( $_SERVER['REQUEST_METHOD'] ) ? (string) $_SERVER['REQUEST_METHOD'] : 'GET';

        $ip = self::ip();
        $window = (int) ( $wp['window_seconds'] ?? 300 );
        $t = is_array( $wp['thresholds_login'] ?? null ) ? $wp['thresholds_login'] : [];

        $count = self::inc( 'login_ip_' . md5( $ip ), $window );

        $action = self::progressive_action( $count, $t );
        self::enforce( 'wp-login', $path, $method, $action, [ 'count' => $count, 'window' => $window ] );
    }

    public static function authenticate_guard( $user, $username, $password ) {
        $wp = self::cfg();
        if ( empty( $wp['enabled'] ) || empty( $wp['bruteforce_enabled'] ) ) { return $user; }

        $ip = self::ip();
        $window = (int) ( $wp['window_seconds'] ?? 300 );
        $t_ip = is_array( $wp['thresholds_login_ip'] ?? null ) ? $wp['thresholds_login_ip'] : [];
        $t_user = is_array( $wp['thresholds_login_user'] ?? null ) ? $wp['thresholds_login_user'] : [];

        $u = (string) $username;
        $u_key = $u !== '' ? 'login_user_' . md5( strtolower( $u ) ) : '';

        $count_ip = self::get( 'login_fail_ip_' . md5( $ip ) );
        $count_u  = $u_key ? self::get( 'login_fail_' . $u_key ) : 0;

        $act_ip = self::progressive_action( $count_ip, $t_ip );
        $act_u  = $u_key ? self::progressive_action( $count_u, $t_user ) : 'log';

        $priority = [ 'log' => 0, 'challenge' => 1, 'rate_limit' => 2, 'block' => 3 ];
        $action = ( $priority[ $act_ip ] >= $priority[ $act_u ] ) ? $act_ip : $act_u;

        if ( $action === 'log' ) { return $user; }

        $path = '/wp-login.php';
        $method = 'POST';
        self::enforce( 'login_velocity', $path, $method, $action, [
            'count_ip' => $count_ip,
            'count_user' => $count_u,
            'window' => $window,
            'username' => $u !== '' ? $u : '(empty)',
        ] );

        return $user;
    }

    public static function on_login_failed( $username ) : void {
        $wp = self::cfg();
        if ( empty( $wp['enabled'] ) || empty( $wp['bruteforce_enabled'] ) ) { return; }

        $ip = self::ip();
        $window = (int) ( $wp['window_seconds'] ?? 300 );

        self::inc( 'login_fail_ip_' . md5( $ip ), $window );

        $u = (string) $username;
        if ( $u !== '' ) {
            self::inc( 'login_fail_login_user_' . md5( strtolower( $u ) ), $window );
        }

        if ( ! empty( $wp['distributed_enabled'] ) ) {
            $bucket = (int) floor( self::now() / max( 30, $window ) );
            $k = 'dist_login_' . $bucket;
            $raw = get_transient( 'aegiswaf_' . $k );
            $set = is_array( $raw ) ? $raw : [];
            $set[ $ip ] = 1;

            if ( count( $set ) > 500 ) {

                $set = array_slice( $set, -500, null, true );
            }
            set_transient( 'aegiswaf_' . $k, $set, max( 30, $window ) );
        }
    }

    public static function on_login_success( $user_login, $user ) : void {
        $wp = self::cfg();
        if ( empty( $wp['enabled'] ) || empty( $wp['bruteforce_enabled'] ) ) { return; }

        if ( ! empty( $wp['reset_user_on_success'] ) ) {
            $u = (string) $user_login;
            if ( $u !== '' ) {
                self::reset( 'login_fail_login_user_' . md5( strtolower( $u ) ) );
            }
        }
    }

    public static function inspect_admin_surface() : void {
        $wp = self::cfg();
        if ( empty( $wp['enabled'] ) ) { return; }

        $method = isset( $_SERVER['REQUEST_METHOD'] ) ? (string) $_SERVER['REQUEST_METHOD'] : 'GET';
        $path = isset( $_SERVER['REQUEST_URI'] ) ? (string) wp_parse_url( (string) $_SERVER['REQUEST_URI'], PHP_URL_PATH ) : '';

        if ( wp_doing_ajax() ) {
            if ( empty( $wp['ajax_protect'] ) ) { return; }

            $ip = self::ip();
            $window = (int) ( $wp['window_seconds'] ?? 60 );
            $t = is_array( $wp['thresholds_ajax'] ?? null ) ? $wp['thresholds_ajax'] : [];

            $count = self::inc( 'ajax_ip_' . md5( $ip ), $window );
            $action = self::progressive_action( $count, $t );
            self::enforce( 'admin-ajax', $path !== '' ? $path : '/wp-admin/admin-ajax.php', $method, $action, [ 'count' => $count, 'window' => $window ] );
            return;
        }

        if ( empty( $wp['admin_protect'] ) ) { return; }

        if ( is_user_logged_in() ) { return; }

        $ip = self::ip();
        $window = (int) ( $wp['window_seconds_admin'] ?? 300 );
        $t = is_array( $wp['thresholds_admin'] ?? null ) ? $wp['thresholds_admin'] : [];
        $count = self::inc( 'admin_ip_' . md5( $ip ), $window );
        $action = self::progressive_action( $count, $t );
        self::enforce( 'wp-admin', $path !== '' ? $path : '/wp-admin/', $method, $action, [ 'count' => $count, 'window' => $window ] );
    }

protected static function enforce_rest( string $surface, string $route, string $method, string $action, array $meta = [] ) {
    $ip = self::ip();
    $ua = (string) AegisWAF_Utils::user_agent();

    AegisWAF_Logger::log( $route, $method, 'wp_protect', $action, array_merge( [
        'surface' => $surface,
        'ip' => $ip,
        'ua' => $ua,
    ], $meta ) );

    if ( $action === 'log' ) {
        return null;
    }

    if ( $action === 'rate_limit' ) {
        return new WP_Error( 'aegiswaf_rate_limited', 'AegisWAF: Too many requests.', [ 'status' => 429 ] );
    }

    if ( $action === 'challenge' ) {
        if ( class_exists( 'AegisWAF_Challenge' ) && AegisWAF_Challenge::enabled() ) {
            if ( AegisWAF_Challenge::verify( $ip ) ) {
                return null;
            }
        }
        return new WP_Error( 'aegiswaf_challenge', 'AegisWAF: Challenge required.', [ 'status' => 403 ] );
    }

    return new WP_Error( 'aegiswaf_blocked', 'AegisWAF: Request blocked.', [ 'status' => 403 ] );
}

protected static function route_matches_any( string $route, string $list ) : bool {
    $lines = preg_split( '/\r\n|\r|\n/', (string) $list ) ?: [];
    foreach ( $lines as $l ) {
        $l = trim( (string) $l );
        if ( $l === '' ) { continue; }
        if ( strpos( $l, '*' ) !== false ) {
            $re = '#^' . str_replace( '\\*', '.*', preg_quote( $l, '#' ) ) . '$#i';
            if ( preg_match( $re, $route ) ) { return true; }
        } else {
            if ( strtolower( $l ) === strtolower( $route ) ) { return true; }
        }
    }
    return false;
}

public static function rest_pre_dispatch( $result, $server, $request ) {
    $wp = self::cfg();
    if ( empty( $wp['enabled'] ) || empty( $wp['rest_protect'] ) ) { return $result; }

    if ( ! ( $request instanceof WP_REST_Request ) ) { return $result; }

    $route  = (string) $request->get_route();
    if ( $route === '' ) { return $result; }

    if ( ! empty( $wp['rest_allowlist'] ) && self::route_matches_any( $route, (string) $wp['rest_allowlist'] ) ) {
        return $result;
    }

    if ( ! empty( $wp['rest_only_unauth'] ) && is_user_logged_in() ) {
        return $result;
    }

    if ( ! empty( $wp['rest_block_user_enum'] ) && ! is_user_logged_in() ) {
        if ( preg_match( '#^/wp/v2/users(?:/|$)#i', $route ) && in_array( strtoupper( $method ), [ 'GET', 'HEAD' ], true ) ) {
            return self::enforce_rest( 'rest_api', $route, $method, 'block', [ 'reason' => 'user_enumeration' ] );
        }
    }

    if ( ! empty( $wp['rest_require_write_auth'] ) && ! is_user_logged_in() ) {
        if ( ! in_array( strtoupper( $method ), [ 'GET', 'HEAD', 'OPTIONS' ], true ) ) {
            return self::enforce_rest( 'rest_api', $route, $method, 'block', [ 'reason' => 'unauthenticated_write' ] );
        }
    }

    if ( ! empty( $wp['rest_require_json'] ) ) {
        $body = (string) $request->get_body();
        if ( $body !== '' && ! in_array( strtoupper( $method ), [ 'GET', 'HEAD', 'OPTIONS' ], true ) ) {
            $ct = (string) $request->get_header( 'content-type' );
            if ( $ct === '' || stripos( $ct, 'application/json' ) === false ) {
                return self::enforce_rest( 'rest_api', $route, $method, 'block', [ 'reason' => 'content_type', 'content_type' => $ct ] );
            }
        }
    }

    $max_body = (int) ( $wp['rest_max_body_bytes'] ?? 0 );
    if ( $max_body > 0 ) {
        $body_len = strlen( (string) $request->get_body() );
        if ( $body_len > $max_body ) {
            return self::enforce_rest( 'rest_api', $route, $method, 'block', [ 'reason' => 'body_too_large', 'body_len' => $body_len, 'max' => $max_body ] );
        }
    }

    $cors_allow = (string) ( $wp['rest_cors_allowlist'] ?? '' );
    if ( $cors_allow !== '' ) {
        $origin = (string) $request->get_header( 'origin' );
        if ( $origin !== '' ) {
            $ok = self::origin_matches_any( $origin, $cors_allow );
            if ( ! $ok ) {
                return self::enforce_rest( 'rest_api', $route, $method, 'block', [ 'reason' => 'cors_origin', 'origin' => $origin ] );
            }
        }
    }

    if ( ! empty( $wp['rest_api_key_enabled'] ) && ! is_user_logged_in() ) {
        $hdr = (string) ( $wp['rest_api_key_header'] ?? 'x-aegis-key' );
        $expected = (string) ( $wp['rest_api_key_value'] ?? '' );
        if ( $expected !== '' ) {
            $got = (string) $request->get_header( $hdr );
            if ( $got === '' || ! hash_equals( $expected, $got ) ) {
                return self::enforce_rest( 'rest_api', $route, $method, 'block', [ 'reason' => 'api_key', 'header' => $hdr ] );
            }
        }
    }



    $method = (string) $request->get_method();
    $ip = self::ip();

    $window = (int) ( $wp['window_seconds_rest'] ?? 60 );
    $t = is_array( $wp['thresholds_rest'] ?? null ) ? $wp['thresholds_rest'] : [ 'challenge_at' => 40, 'rate_limit_at' => 80, 'block_at' => 140 ];

    $count_ip = self::inc( 'rest_ip_' . md5( $ip ), $window );
    $count_route = self::inc( 'rest_route_ip_' . md5( $route . '|' . $ip ), $window );

    $count = max( $count_ip, $count_route );

    $action = self::progressive_action( $count, $t );

    $err = self::enforce_rest( 'rest_api', $route, $method, $action, [
        'count' => $count,
        'count_ip' => $count_ip,
        'count_route' => $count_route,
        'window' => $window,
    ] );

    if ( is_wp_error( $err ) ) {
        return $err;
    }

    return $result;
}

    public static function inspect_xmlrpc() : void {
        $wp = self::cfg();
        if ( empty( $wp['enabled'] ) || empty( $wp['xmlrpc_enabled'] ) ) { return; }

        if ( ! defined( 'XMLRPC_REQUEST' ) || ! XMLRPC_REQUEST ) { return; }

        $mode = (string) ( $wp['xmlrpc_mode'] ?? 'restrict' ); // disable|restrict|rate_limit
        $ip = self::ip();
        $method = 'POST';
        $path = '/xmlrpc.php';

        if ( $mode === 'disable' ) {
            self::enforce( 'xmlrpc', $path, $method, 'block', [ 'mode' => 'disable' ] );
            return;
        }

        if ( $mode === 'restrict' ) {
            $allow = (string) ( $wp['xmlrpc_allowlist'] ?? '' );
            $lines = preg_split( '/
|
|
/', $allow ) ?: [];
            $ok = false;
            foreach ( $lines as $l ) {
                $l = trim( (string) $l );
                if ( $l === '' ) { continue; }
                if ( $l === $ip ) { $ok = true; break; }
            }
            if ( ! $ok ) {
                self::enforce( 'xmlrpc', $path, $method, 'block', [ 'mode' => 'restrict' ] );
            }
            return;
        }

        $window = (int) ( $wp['window_seconds_xmlrpc'] ?? 300 );
        $t = is_array( $wp['thresholds_xmlrpc'] ?? null ) ? $wp['thresholds_xmlrpc'] : [];
        $count = self::inc( 'xmlrpc_ip_' . md5( $ip ), $window );
        $action = self::progressive_action( $count, $t );
        self::enforce( 'xmlrpc', $path, $method, $action, [ 'count' => $count, 'window' => $window, 'mode' => 'rate_limit' ] );
    }
}
