<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

if ( ! function_exists( 'aegiswaf_ddos_get_visual_data' ) ) {
	function aegiswaf_ddos_get_visual_data() : array {
		global $wpdb;

		if ( ! class_exists( 'AegisWAF_Logger' ) || ! method_exists( 'AegisWAF_Logger', 'table_name' ) ) {
			return [
				'totals' => [ 'last_24h' => 0, 'last_7d' => 0 ],
				'hour_labels' => [], 'hour_challenge' => [], 'hour_rate' => [], 'hour_block' => [],
				'group_labels' => [], 'group_challenge' => [], 'group_rate' => [], 'group_block' => [],
				'action_labels' => [], 'action_counts' => [],
				'scatter_points' => [],
			];
		}

		$table = AegisWAF_Logger::table_name();
		$since_7d_gmt  = gmdate( 'Y-m-d H:i:s', time() - ( 7 * DAY_IN_SECONDS ) );
		$since_24h_gmt = gmdate( 'Y-m-d H:i:s', time() - DAY_IN_SECONDS );
		$total_7d = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$table}
				  WHERE event_time >= %s AND category = 'ddos_shield'",
				$since_7d_gmt
			)
		);
		$total_24h = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$table}
				  WHERE event_time >= %s AND category = 'ddos_shield'",
				$since_24h_gmt
			)
		);

		$hour_rows = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT DATE_FORMAT(event_time, '%%Y-%%m-%%d %%H:00') AS h,
				        SUM(CASE WHEN action_taken='challenge' THEN 1 ELSE 0 END) AS c_ch,
				        SUM(CASE WHEN action_taken='rate_limit' THEN 1 ELSE 0 END) AS c_rl,
				        SUM(CASE WHEN action_taken='block' THEN 1 ELSE 0 END) AS c_bl
				   FROM {$table}
				  WHERE event_time >= %s
				    AND category = 'ddos_shield'
				  GROUP BY h
				  ORDER BY h ASC",
				$since_24h_gmt
			),
			ARRAY_A
		);

		$hour_labels = [];
		$hour_ch = [];
		$hour_rl = [];
		$hour_bl = [];
		foreach ( (array) $hour_rows as $r ) {
			$hour_labels[] = (string) ( $r['h'] ?? '' );
			$hour_ch[] = (int) ( $r['c_ch'] ?? 0 );
			$hour_rl[] = (int) ( $r['c_rl'] ?? 0 );
			$hour_bl[] = (int) ( $r['c_bl'] ?? 0 );
		}
		$raw_rows = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT action_taken, details
				   FROM {$table}
				  WHERE event_time >= %s
				    AND category = 'ddos_shield'
				  ORDER BY id DESC
				  LIMIT 4000",
				$since_7d_gmt
			),
			ARRAY_A
		);

		$known_groups = [ 'global', 'wp_login', 'xmlrpc', 'wp_admin', 'rest', 'search', 'cache_bypass' ];
		$group_map = [];
		foreach ( $known_groups as $g ) {
			$group_map[ $g ] = [ 'challenge' => 0, 'rate_limit' => 0, 'block' => 0 ];
		}
		$group_map['other'] = [ 'challenge' => 0, 'rate_limit' => 0, 'block' => 0 ];

		foreach ( (array) $raw_rows as $r ) {
			$act = sanitize_key( (string) ( $r['action_taken'] ?? '' ) );
			if ( ! in_array( $act, [ 'challenge', 'rate_limit', 'block' ], true ) ) {
				continue;
			}

			$details = [];
			if ( ! empty( $r['details'] ) ) {
				$d = json_decode( (string) $r['details'], true );
				if ( is_array( $d ) ) { $details = $d; }
			}
			$g = sanitize_key( (string) ( $details['group'] ?? 'other' ) );
			if ( $g === '' ) { $g = 'other'; }
			if ( ! isset( $group_map[ $g ] ) ) { $g = 'other'; }

			$group_map[ $g ][ $act ]++;
		}

		$group_labels = [];
		$group_ch = [];
		$group_rl = [];
		$group_bl = [];
		foreach ( $group_map as $g => $counts ) {
			$total_g = (int)$counts['challenge'] + (int)$counts['rate_limit'] + (int)$counts['block'];
			if ( $total_g <= 0 ) { continue; }

			$group_labels[] = $g;
			$group_ch[] = (int) $counts['challenge'];
			$group_rl[] = (int) $counts['rate_limit'];
			$group_bl[] = (int) $counts['block'];
		}

		$action_rows = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT action_taken, COUNT(*) AS c
				   FROM {$table}
				  WHERE event_time >= %s
				    AND category = 'ddos_shield'
				  GROUP BY action_taken
				  ORDER BY c DESC",
				$since_7d_gmt
			),
			ARRAY_A
		);

		$action_labels = [];
		$action_counts = [];
		foreach ( (array) $action_rows as $r ) {
			$action_labels[] = (string) ( $r['action_taken'] ?? '' );
			$action_counts[] = (int) ( $r['c'] ?? 0 );
		}

		$scatter_rows = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT t.ip,
				        SUM(t.minute_c) AS total_hits,
				        MAX(t.minute_c) AS max_per_min
				   FROM (
				        SELECT ip,
				               DATE_FORMAT(event_time, '%%Y-%%m-%%d %%H:%%i') AS m,
				               COUNT(*) AS minute_c
				          FROM {$table}
				         WHERE event_time >= %s
				           AND category = 'ddos_shield'
				           AND ip IS NOT NULL AND ip <> ''
				         GROUP BY ip, m
				   ) t
				  GROUP BY t.ip
				  ORDER BY total_hits DESC
				  LIMIT 12",
				$since_24h_gmt
			),
			ARRAY_A
		);

		$scatter_points = [];
		foreach ( (array) $scatter_rows as $r ) {
			$scatter_points[] = [
				'label' => (string) ( $r['ip'] ?? '' ),
				'x' => (int) ( $r['total_hits'] ?? 0 ),
				'y' => (int) ( $r['max_per_min'] ?? 0 ),
			];
		}

		return [
			'totals' => [ 'last_24h' => $total_24h, 'last_7d' => $total_7d ],
			'hour_labels' => $hour_labels,
			'hour_challenge' => $hour_ch,
			'hour_rate' => $hour_rl,
			'hour_block' => $hour_bl,
			'group_labels' => $group_labels,
			'group_challenge' => $group_ch,
			'group_rate' => $group_rl,
			'group_block' => $group_bl,
			'action_labels' => $action_labels,
			'action_counts' => $action_counts,
			'scatter_points' => $scatter_points,
		];
	}
}

add_action( 'admin_enqueue_scripts', function() {
	if ( empty( $_GET['page'] ) || $_GET['page'] !== 'aegiswaf' ) { return; }

	$tab = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'overview';
	if ( $tab !== 'ddos' ) { return; }

	$base_url = plugin_dir_url( dirname( __FILE__, 3 ) ); // plugin root (…/aegiswaf/)

	wp_enqueue_script(
		'aegiswaf-chartjs',
		$base_url . 'assets/js/chart.umd.min.js',
		[],
		'1.6.9',
		true
	);

	$data = aegiswaf_ddos_get_visual_data();
	$json = wp_json_encode( $data );

	$inline = "
	(function(){
		if (typeof Chart === 'undefined') { return; }
		var data = {$json};
		function el(id){ return document.getElementById(id); }

		// Totals
		var t24 = el('aegiswaf_ddos_total_24h');
		var t7d = el('aegiswaf_ddos_total_7d');
		if (t24) { t24.textContent = String((data.totals && data.totals.last_24h) ? data.totals.last_24h : 0); }
		if (t7d) { t7d.textContent = String((data.totals && data.totals.last_7d) ? data.totals.last_7d : 0); }

		// 1) Line (hourly trend 24h)
		var c1 = el('aegiswaf_ddos_chart_hourly');
		if (c1) {
			new Chart(c1.getContext('2d'), {
				type: 'line',
				data: {
					labels: data.hour_labels || [],
					datasets: [
						{ label: 'Challenge', data: data.hour_challenge || [], tension: 0.3, pointRadius: 1 },
						{ label: 'Rate-limit', data: data.hour_rate || [], tension: 0.3, pointRadius: 1 },
						{ label: 'Block', data: data.hour_block || [], tension: 0.3, pointRadius: 1 }
					]
				},
				options: { responsive:true, maintainAspectRatio:false, scales:{ y:{ beginAtZero:true } } }
			});
		}

		// 2) Stacked bar (by group, 7d)
		var c2 = el('aegiswaf_ddos_chart_groups');
		if (c2) {
			new Chart(c2.getContext('2d'), {
				type: 'bar',
				data: {
					labels: data.group_labels || [],
					datasets: [
						{ label: 'Challenge', data: data.group_challenge || [], stack: 'ddos' },
						{ label: 'Rate-limit', data: data.group_rate || [], stack: 'ddos' },
						{ label: 'Block', data: data.group_block || [], stack: 'ddos' }
					]
				},
				options: {
					responsive:true,
					maintainAspectRatio:false,
					scales: { x: { stacked:true }, y: { stacked:true, beginAtZero:true } }
				}
			});
		}

		// 3) Doughnut (action distribution 7d)
		var c3 = el('aegiswaf_ddos_chart_actions');
		if (c3) {
			new Chart(c3.getContext('2d'), {
				type: 'doughnut',
				data: { labels: data.action_labels || [], datasets: [{ data: data.action_counts || [] }] },
				options: { responsive:true, maintainAspectRatio:false, plugins:{ legend:{ position:'bottom' } } }
			});
		}

		// 4) Scatter (burst offenders 24h)
		var c4 = el('aegiswaf_ddos_chart_scatter');
		if (c4) {
			var pts = (data.scatter_points || []).map(function(p){
				return { x: p.x, y: p.y, _label: p.label };
			});
			new Chart(c4.getContext('2d'), {
				type: 'scatter',
				data: { datasets: [{ label:'Burst offenders (24h)', data: pts }] },
				options: {
					responsive:true,
					maintainAspectRatio:false,
					plugins: {
						tooltip: { callbacks: { label: function(ctx){
							var p = ctx.raw || {};
							return (p._label || 'ip') + ' — total:' + p.x + ', max/min:' + p.y;
						}}}
					},
					scales: {
						x: { beginAtZero:true, title:{ display:true, text:'Total DDOS events (24h)' } },
						y: { beginAtZero:true, title:{ display:true, text:'Max events per minute (24h)' } }
					}
				}
			});
		}
	})();";

	wp_add_inline_script( 'aegiswaf-chartjs', $inline, 'after' );
}, 21 );

class AegisWAF_DDoS_Admin {

    public function render() : void {
        if ( ! current_user_can( 'manage_options' ) ) { return; }

        $is_pro = AegisWAF_Features::is_pro();
        $ddos = AegisWAF_DDoS_Storage::get();

        if ( isset( $_POST['aegiswaf_save_ddos_shield'] ) ) {
            check_admin_referer( 'aegiswaf_save_ddos_shield' );

            $ddos['enabled'] = ! empty( $_POST['ddos_enabled'] );
            $ddos['ignore_logged_in'] = ! empty( $_POST['ddos_ignore_logged_in'] );
            $ddos['cooldown_seconds'] = max( 30, (int) ( $_POST['ddos_cooldown_seconds'] ?? 300 ) );
            $allow = isset( $_POST['ddos_allowlist'] ) ? (string) wp_unslash( $_POST['ddos_allowlist'] ) : '';
            $allow_lines = AegisWAF_DDoS_Storage::allowlist_lines( $allow );
            $valid = [];
            foreach ( $allow_lines as $item ) {
                if ( filter_var( $item, FILTER_VALIDATE_IP ) ) {
                    $valid[] = $item;
                    continue;
                }
                if ( strpos( $item, '/' ) !== false ) {

                    [ $subnet, $bits ] = array_pad( explode( '/', $item, 2 ), 2, '' );
                    $subnet = trim( (string) $subnet );
                    $bits = trim( (string) $bits );
                    if ( filter_var( $subnet, FILTER_VALIDATE_IP ) && $bits !== '' && ctype_digit( $bits ) ) {
                        $valid[] = $subnet . '/' . $bits;
                    }
                }
            }
            $ddos['allowlist'] = implode( "\n", array_values( array_unique( $valid ) ) );

			if ( $is_pro ) {

				$groups = is_array( $ddos['groups'] ?? null ) ? $ddos['groups'] : [];
				$group_keys = [ 'global', 'wp_login', 'xmlrpc', 'wp_admin', 'admin_ajax', 'wp_cron', 'rest', 'search', 'cache_bypass' ];

				foreach ( $group_keys as $g ) {
					$groups[ $g ] = is_array( $groups[ $g ] ?? null ) ? $groups[ $g ] : [ 'challenge' => 0, 'rate_limit' => 0, 'block' => 0 ];

					$c = max( 1, (int) ( $_POST[ 'ddos_' . $g . '_challenge' ] ?? ( $groups[ $g ]['challenge'] ?? 1 ) ) );
					$r = max( $c, (int) ( $_POST[ 'ddos_' . $g . '_rate_limit' ] ?? ( $groups[ $g ]['rate_limit'] ?? $c ) ) );
					$b = max( $r, (int) ( $_POST[ 'ddos_' . $g . '_block' ] ?? ( $groups[ $g ]['block'] ?? $r ) ) );

					$groups[ $g ]['challenge'] = $c;
					$groups[ $g ]['rate_limit'] = $r;
					$groups[ $g ]['block'] = $b;
				}

				$ddos['groups'] = $groups;
				$ddos['enable_cache_bypass_detection'] = ! empty( $_POST['ddos_enable_cache_bypass_detection'] );
				$ddos['enable_search_detection'] = ! empty( $_POST['ddos_enable_search_detection'] );

			} else {

				$pro_attempt = false;

				if ( isset( $_POST['ddos_enable_cache_bypass_detection'] ) || isset( $_POST['ddos_enable_search_detection'] ) ) {
					$pro_attempt = true;
				} else {

					foreach ( [ 'global','wp_login','xmlrpc','wp_admin','admin_ajax','wp_cron','rest','search','cache_bypass' ] as $g ) {
						if ( isset( $_POST[ 'ddos_' . $g . '_challenge' ] ) || isset( $_POST[ 'ddos_' . $g . '_rate_limit' ] ) || isset( $_POST[ 'ddos_' . $g . '_block' ] ) ) {
							$pro_attempt = true;
							break;
						}
					}
				}

				if ( $pro_attempt ) {
					if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
						AegisWAF_Logger::log( 'admin', 'POST', 'ddos_shield', 'pro_fields_ignored', [
							'reason' => 'Free license',
						] );
					} else {
						error_log( '[AegisWAF] DDOS: PRO-only fields were posted but ignored (Free license).' );
					}
				}
			}

			if ( $mult < 0.10 ) { $mult = 0.10; }
			if ( $mult > 1.00 ) { $mult = 1.00; }

			$ddos['emergency_multiplier'] = $mult;
			$ok = false;

			if ( ! class_exists( 'AegisWAF_DDoS_Storage' ) || ! method_exists( 'AegisWAF_DDoS_Storage', 'update' ) ) {
				if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
					AegisWAF_Logger::log( 'admin', 'POST', 'ddos_shield', 'save_failed_missing_storage', [
						'missing' => 'AegisWAF_DDoS_Storage::update',
					] );
				} else {
					error_log( '[AegisWAF] DDoS save failed: missing AegisWAF_DDoS_Storage::update' );
				}
			} else {
				$ok = AegisWAF_DDoS_Storage::update( $ddos );
			}

			if ( ! $ok ) {
				if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
					AegisWAF_Logger::log( 'admin', 'POST', 'ddos_shield', 'save_failed', [
						'reason' => 'update_settings returned false',
					] );
				} else {
					error_log( '[AegisWAF] DDoS save failed: update_settings returned false' );
				}

				echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__( 'DDoS Shield settings failed to save. Check debug log.', 'aegiswaf' ) . '</p></div>';
			} else {
				echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'DDoS Shield settings saved.', 'aegiswaf' ) . '</p></div>';
			}

        }
        echo '<div class="wrap">';
        echo '<h1>DDOS Protection</h1>';
        AegisWAF_Pro_Notice::render();
        echo '<p class="description">The DDoS Protection tab defends against traffic floods and resource exhaustion attacks by monitoring request rates and detecting abnormal spikes. Free includes basic rate limiting to stop simple floods, while Pro adds adaptive thresholds, <b>Layer 7 detection</b>, and automated mitigation logic that responds dynamically as attack patterns change. All detected DDoS activity is logged for visibility and post-incident review.</p>';

		echo '<style>
		.aegiswaf-ddosviz-grid{
			display:grid;
			grid-template-columns: repeat(4, minmax(220px, 1fr));
			gap:12px;
			margin:12px 0 14px;
		}
		@media (max-width: 1300px){ .aegiswaf-ddosviz-grid{ grid-template-columns: repeat(2, minmax(240px, 1fr)); } }
		@media (max-width: 782px){ .aegiswaf-ddosviz-grid{ grid-template-columns: 1fr; } }

		.aegiswaf-ddosviz-panel{
			border:1px solid #e5e7eb;
			border-radius:10px;
			padding:12px;
			background:#fff;
		}
		.aegiswaf-ddosviz-title{ font-weight:700; margin:0 0 4px; }
		.aegiswaf-ddosviz-sub{ color:#646970; font-size:12px; margin:0 0 10px; }
		.aegiswaf-ddosviz-canvas{ height:220px; position:relative; }
		/* PRO dim/lock section (DDOS) */
		.aegiswaf-pro-dimwrap{
			position: relative;
			margin-top: 10px;
		}
		.aegiswaf-pro-dimcontent{
			opacity: 0.45;
			filter: grayscale(0.15);
			pointer-events: none;
		}
		.aegiswaf-pro-overlay{
			position: static;
			inset: auto;
			display: block;
			margin: 0 0 10px 0;
			pointer-events: auto;
		}
		.aegiswaf-pro-overlay .aegiswaf-pro-box{
			margin: 0;
			background: rgba(255,255,255,0.92);
			border: 1px solid #dcdcde;
			border-left: 4px solid #d63638;
			border-radius: 6px;
			padding: 12px 14px;
			max-width: none;
			width: 100%;
			box-shadow: 0 1px 2px rgba(0,0,0,0.06);
		}
		.aegiswaf-pro-overlay .aegiswaf-pro-box h4{
			margin: 0 0 6px 0;
			font-size: 14px;
		}
		.aegiswaf-pro-overlay .aegiswaf-pro-box p{
			margin: 0 0 10px 0;
			color: #50575e;
		}
		.aegiswaf-pro-overlay .aegiswaf-pro-box .button{
			margin-right: 8px;
		}
		</style>';

		echo '<div class="aegiswaf-card" style="margin-top:12px;">';
		echo '<h2 style="margin:0 0 8px;">DDoS Visual Intelligence</h2>';
		echo '<p class="description" style="margin:0 0 10px;">DDOS-only insights from logged DDoS Shield enforcement. Built for quick incident triage.</p>';
		echo '<p style="margin:0 0 10px;font-weight:600;">Last 24h: <span id="aegiswaf_ddos_total_24h">0</span> &nbsp; | &nbsp; Last 7d: <span id="aegiswaf_ddos_total_7d">0</span></p>';

		echo '<div class="aegiswaf-ddosviz-grid">';

			echo '<div class="aegiswaf-ddosviz-panel">';
				echo '<div class="aegiswaf-ddosviz-title">Enforcement Trend</div>';
				echo '<div class="aegiswaf-ddosviz-sub">Line (24h): challenge / rate-limit / block</div>';
				echo '<div class="aegiswaf-ddosviz-canvas"><canvas id="aegiswaf_ddos_chart_hourly"></canvas></div>';
			echo '</div>';

			echo '<div class="aegiswaf-ddosviz-panel">';
				echo '<div class="aegiswaf-ddosviz-title">Hot Surfaces</div>';
				echo '<div class="aegiswaf-ddosviz-sub">Stacked bar (7d): actions by group</div>';
				echo '<div class="aegiswaf-ddosviz-canvas"><canvas id="aegiswaf_ddos_chart_groups"></canvas></div>';
			echo '</div>';

			echo '<div class="aegiswaf-ddosviz-panel">';
				echo '<div class="aegiswaf-ddosviz-title">Action Distribution</div>';
				echo '<div class="aegiswaf-ddosviz-sub">Doughnut (7d): what the shield is doing</div>';
				echo '<div class="aegiswaf-ddosviz-canvas"><canvas id="aegiswaf_ddos_chart_actions"></canvas></div>';
			echo '</div>';

			echo '<div class="aegiswaf-ddosviz-panel">';
				echo '<div class="aegiswaf-ddosviz-title">Burst Offenders</div>';
				echo '<div class="aegiswaf-ddosviz-sub">Scatter (24h): total hits vs max/minute per IP</div>';
				echo '<div class="aegiswaf-ddosviz-canvas"><canvas id="aegiswaf_ddos_chart_scatter"></canvas></div>';
			echo '</div>';

		echo '</div>'; // grid
		echo '</div>'; // card

        echo '<h2>' . esc_html__( 'DDoS Shield', 'aegiswaf' ) . '</h2>';
        echo '<p class="description">' . esc_html__( 'Application-layer (HTTP) surge protection. Uses lightweight counters and progressive mitigation (challenge → rate limit → block).', 'aegiswaf' ) . '</p>';

        echo '<form method="post">';
        wp_nonce_field( 'aegiswaf_save_ddos_shield' );

        echo '<table class="form-table"><tbody>';

        echo '<tr><th scope="row">' . esc_html__( 'Enable DDoS Shield', 'aegiswaf' ) . '</th><td>';
        echo '<label><input type="checkbox" name="ddos_enabled" value="1" ' . checked( ! empty( $ddos['enabled'] ), true, false ) . '> ' . esc_html__( 'On', 'aegiswaf' ) . '</label>';
        echo '</td></tr>';

        echo '<tr><th scope="row">' . esc_html__( 'Ignore logged-in users', 'aegiswaf' ) . '</th><td>';
        echo '<label><input type="checkbox" name="ddos_ignore_logged_in" value="1" ' . checked( ! empty( $ddos['ignore_logged_in'] ), true, false ) . '> ' . esc_html__( 'Do not apply DDoS checks to authenticated users', 'aegiswaf' ) . '</label>';
        echo '</td></tr>';

        echo '<tr><th scope="row">' . esc_html__( 'Cooldown (seconds)', 'aegiswaf' ) . '</th><td>';
        echo '<input type="number" min="30" step="10" name="ddos_cooldown_seconds" value="' . esc_attr( (string) ( $ddos['cooldown_seconds'] ?? 300 ) ) . '" style="width:120px;">';
        echo '<p class="description">' . esc_html__( 'When an IP triggers enforcement, keep it in a short cooldown where repeated requests are rate-limited.', 'aegiswaf' ) . '</p>';
        echo '</td></tr>';
		
		echo '<tr><th scope="row">' . esc_html__( 'Emergency Mode', 'aegiswaf' ) . '</th><td>';
		echo '<label><input type="checkbox" name="ddos_emergency_mode" value="1"' . checked( ! empty( $ddos['emergency_mode'] ), true, false ) . '> ';
		echo esc_html__( 'Tighten thresholds immediately during an active attack', 'aegiswaf' ) . '</label>';
		echo '<br><label style="display:inline-block;margin-top:6px;">' . esc_html__( 'Emergency multiplier', 'aegiswaf' ) . ' ';
		echo '<input type="number" min="0.10" max="1.00" step="0.05" name="ddos_emergency_multiplier" value="' . esc_attr( (string) ( $ddos['emergency_multiplier'] ?? 0.50 ) ) . '" style="width:120px;"></label>';
		echo '<p class="description">' . esc_html__( 'Example: 0.50 cuts Challenge/Rate/Block thresholds in half. Use 1.00 for normal behavior.', 'aegiswaf' ) . '</p>';
		echo '</td></tr>';

        echo '<tr><th scope="row">' . esc_html__( 'Allowlist (IP / CIDR)', 'aegiswaf' ) . '</th><td>';
        echo '<textarea name="ddos_allowlist" rows="6" style="width:420px;">' . esc_textarea( (string) ( $ddos['allowlist'] ?? '' ) ) . '</textarea>';
        echo '<p class="description">' . esc_html__( 'One per line. Example: 192.0.2.10 or 203.0.113.0/24', 'aegiswaf' ) . '</p>';
        echo '</td></tr>';

        echo '</tbody></table>';
		$pro_disabled = $is_pro ? '' : ' disabled="disabled"';

		if ( ! $is_pro ) {
			echo '<div class="aegiswaf-pro-dimwrap">';
			echo '<div class="aegiswaf-pro-overlay"><div class="aegiswaf-pro-box">';
			echo '<h4>' . esc_html__( 'Upgrade to PRO to unlock advanced DDoS controls', 'aegiswaf' ) . '</h4>';
			echo '<p>' . esc_html__( 'PRO features: per-group thresholds, advanced detection toggles, and higher fidelity enforcement tuning.', 'aegiswaf' ) . '</p>';
			echo '<a class="button button-primary" href="' . esc_url( admin_url( 'admin.php?page=aegiswaf&tab=license' ) ) . '">' . esc_html__( 'Upgrade to PRO', 'aegiswaf' ) . '</a>';
			echo '<a class="button" href="' . esc_url( admin_url( 'admin.php?page=aegiswaf&tab=license' ) ) . '">' . esc_html__( 'View Plans', 'aegiswaf' ) . '</a>';
			echo '</div></div>';
			echo '<div class="aegiswaf-pro-dimcontent">';
		}

        echo '<h3>' . esc_html__( 'Thresholds per 60 seconds (per IP)', 'aegiswaf' ) . '</h3>';
        echo '<p class="description">' . esc_html__( 'Progressive enforcement uses these thresholds in order: Challenge → Rate Limit → Block.', 'aegiswaf' ) . '</p>';

        $groups = is_array( $ddos['groups'] ?? null ) ? $ddos['groups'] : [];
        $rows = [
		'global' => esc_html__( 'Global (all traffic)', 'aegiswaf' ),
		'wp_login' => esc_html__( 'wp-login.php', 'aegiswaf' ),
		'xmlrpc' => esc_html__( 'xmlrpc.php', 'aegiswaf' ),
		'wp_admin' => esc_html__( 'wp-admin/', 'aegiswaf' ),
		'admin_ajax' => esc_html__( 'admin-ajax.php', 'aegiswaf' ),
		'wp_cron' => esc_html__( 'wp-cron.php', 'aegiswaf' ),
		'rest' => esc_html__( 'REST API (/wp-json/)', 'aegiswaf' ),
		'search' => esc_html__( 'Search (?s=)', 'aegiswaf' ),
		'cache_bypass' => esc_html__( 'Cache-bypass query floods', 'aegiswaf' ),
        ];

        echo '<table class="widefat striped" style="max-width:900px;"><thead><tr>';
        echo '<th>' . esc_html__( 'Group', 'aegiswaf' ) . '</th>';
        echo '<th>' . esc_html__( 'Challenge', 'aegiswaf' ) . '</th>';
        echo '<th>' . esc_html__( 'Rate Limit (429)', 'aegiswaf' ) . '</th>';
        echo '<th>' . esc_html__( 'Block (403)', 'aegiswaf' ) . '</th>';
        echo '</tr></thead><tbody>';

        foreach ( $rows as $k => $label ) {
            $g = is_array( $groups[ $k ] ?? null ) ? $groups[ $k ] : [ 'challenge' => 1, 'rate_limit' => 1, 'block' => 1 ];
            $c = (int) ( $g['challenge'] ?? 1 );
            $r = (int) ( $g['rate_limit'] ?? $c );
            $b = (int) ( $g['block'] ?? $r );

            echo '<tr>';
            echo '<td><strong>' . $label . '</strong></td>';
			echo '<td><input type="number" min="1" name="ddos_' . esc_attr( $k ) . '_challenge" value="' . esc_attr( (string) $c ) . '" style="width:120px;"' . $pro_disabled . '></td>';
			echo '<td><input type="number" min="1" name="ddos_' . esc_attr( $k ) . '_rate_limit" value="' . esc_attr( (string) $r ) . '" style="width:140px;"' . $pro_disabled . '></td>';
			echo '<td><input type="number" min="1" name="ddos_' . esc_attr( $k ) . '_block" value="' . esc_attr( (string) $b ) . '" style="width:120px;"' . $pro_disabled . '></td>';
            echo '</tr>';
        }

        echo '</tbody></table>';

        echo '<h3>' . esc_html__( 'Detection toggles', 'aegiswaf' ) . '</h3>';
        echo '<table class="form-table"><tbody>';

        echo '<tr><th scope="row">' . esc_html__( 'Cache-bypass detection', 'aegiswaf' ) . '</th><td>';
		echo '<label><input type="checkbox" name="ddos_enable_cache_bypass_detection" value="1"' . checked( ! empty( $ddos['enable_cache_bypass_detection'] ), true, false ) . $pro_disabled . '> ' . esc_html__( 'Detect and limit cache-bypass patterns', 'aegiswaf' ) . '</label>';
        echo '</td></tr>';

        echo '<tr><th scope="row">' . esc_html__( 'Search detection', 'aegiswaf' ) . '</th><td>';
        echo '<label><input type="checkbox" name="ddos_enable_search_detection" value="1"' . checked( ! empty( $ddos['enable_search_detection'] ), true, false ) . $pro_disabled . '> ' . esc_html__( 'Treat search requests as a higher-cost group', 'aegiswaf' ) . '</label>';
        echo '</td></tr>';

        echo '</tbody></table>';
		if ( ! $is_pro ) {
			echo '</div></div>';
		}

        submit_button( esc_html__( 'Save DDoS Shield', 'aegiswaf' ), 'primary', 'aegiswaf_save_ddos_shield' );

        echo '</form>';

        if ( ! $is_pro ) {
            echo '<p class="description" style="margin-top:12px;">' . esc_html__( 'Tip: Pairing with an edge provider (CDN/WAF) helps mitigate network-layer floods. AegisWAF focuses on application-layer (HTTP) surges.', 'aegiswaf' ) . '</p>';
        }
    }
}
