<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_DDoS_Engine {

    public static function check_front( string $ip, string $path, string $method, string $ua ) : ?array {
        $ddos = AegisWAF_DDoS_Storage::get();

        if ( empty( $ddos['enabled'] ) ) { return null; }

        if ( ! empty( $ddos['ignore_logged_in'] ) && is_user_logged_in() ) { return null; }

        if ( AegisWAF_DDoS_Storage::ip_allowlisted( $ip, (string) ( $ddos['allowlist'] ?? '' ) ) ) { return null; }

        $group = AegisWAF_DDoS_Rules::group_for_front( $path );

        if ( $group === 'cache_bypass' && empty( $ddos['enable_cache_bypass_detection'] ) ) { $group = 'global'; }
        if ( $group === 'search' && empty( $ddos['enable_search_detection'] ) ) { $group = 'global'; }

        return self::evaluate_and_decide( $ip, $group, $path, $method, $ua, $ddos );
    }

    public static function check_rest( string $ip, string $route, string $method, string $ua ) : ?array {
        $ddos = AegisWAF_DDoS_Storage::get();

        if ( empty( $ddos['enabled'] ) ) { return null; }

        if ( ! empty( $ddos['ignore_logged_in'] ) && is_user_logged_in() ) { return null; }

        if ( AegisWAF_DDoS_Storage::ip_allowlisted( $ip, (string) ( $ddos['allowlist'] ?? '' ) ) ) { return null; }

        $group = AegisWAF_DDoS_Rules::group_for_rest( $route );

        if ( $group === 'cache_bypass' && empty( $ddos['enable_cache_bypass_detection'] ) ) { $group = 'rest'; }

        return self::evaluate_and_decide( $ip, $group, $route, $method, $ua, $ddos );
    }

    private static function evaluate_and_decide( string $ip, string $group, string $path_or_route, string $method, string $ua, array $ddos ) : ?array {
        $thr = AegisWAF_DDoS_Rules::thresholds_for( $group, $ddos );
		if ( ! empty( $ddos['emergency_mode'] ) ) {
			$mult = isset( $ddos['emergency_multiplier'] ) ? (float) $ddos['emergency_multiplier'] : 0.50;
			if ( $mult < 0.10 ) { $mult = 0.10; }
			if ( $mult > 1.00 ) { $mult = 1.00; }

			$mins = is_array( $ddos['emergency_minimums'] ?? null ) ? $ddos['emergency_minimums'] : [];
			$min_c = max( 1, (int) ( $mins['challenge'] ?? 3 ) );
			$min_r = max( $min_c, (int) ( $mins['rate_limit'] ?? 5 ) );
			$min_b = max( $min_r, (int) ( $mins['block'] ?? 8 ) );

			$thr['challenge']   = max( $min_c, (int) floor( (int) $thr['challenge']   * $mult ) );
			$thr['rate_limit']  = max( $min_r, (int) floor( (int) $thr['rate_limit']  * $mult ) );
			$thr['block']       = max( $min_b, (int) floor( (int) $thr['block']       * $mult ) );
		}

        $cooldown_key = self::cooldown_key( $ip, $group );
        $cd_until = (int) get_transient( $cooldown_key );
        if ( $cd_until > time() ) {
            return [
                'action' => 'rate_limit',
                'group' => $group,
                'count' => -1,
                'limit' => $thr['rate_limit'],
                'window' => 60,
                'ip' => $ip,
                'reason' => 'cooldown_active',
                'cooldown_until' => $cd_until,
            ];
        }

        $count = self::incr_counter( $ip, $group );

        $action = null;
        if ( $count >= (int) $thr['block'] ) {
            $action = 'block';
        } elseif ( $count >= (int) $thr['rate_limit'] ) {
            $action = 'rate_limit';
        } elseif ( $count >= (int) $thr['challenge'] ) {
            $action = 'challenge';
        }

		if ( ! $action ) {
			$seen_key = 'aegiswaf_ddos_allowlog_' . md5( $ip . '|' . $group );
			if ( false === get_transient( $seen_key ) ) {
				set_transient( $seen_key, 1, 60 );

				if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
					AegisWAF_Logger::log( (string) $path_or_route, (string) $method, 'ddos_shield', 'allowed', [
						'group' => $group,
						'count' => $count,
						'ip'    => $ip,
						'ua'    => $ua,
						'reason'=> 'below_threshold',
					] );
				}
			}

			return null;
		}

        $cooldown_seconds = max( 30, (int) ( $ddos['cooldown_seconds'] ?? 300 ) );
        $set_cd = ( $action === 'rate_limit' || $action === 'block' || $count >= ( (int) $thr['challenge'] * 2 ) );
        if ( $set_cd ) {
            set_transient( $cooldown_key, time() + $cooldown_seconds, $cooldown_seconds );
        }

        return [
            'action' => $action,
            'group' => $group,
            'count' => $count,
            'limit' => (int) $thr[ $action === 'block' ? 'block' : ( $action === 'rate_limit' ? 'rate_limit' : 'challenge' ) ],
            'window' => 60,
            'ip' => $ip,
            'reason' => 'threshold_exceeded',
            'method' => $method,
            'ua' => $ua,
            'path' => $path_or_route,
        ];
    }

    private static function incr_counter( string $ip, string $group ) : int {
        $key = self::counter_key( $ip, $group );

        $now = time();
        $data = get_transient( $key );
        if ( ! is_array( $data ) ) {
            $data = [ 'c' => 0, 'start' => $now ];
        }

        $start = (int) ( $data['start'] ?? $now );
        $c = (int) ( $data['c'] ?? 0 );

        if ( ( $now - $start ) >= 60 ) {
            $start = $now;
            $c = 0;
        }

        $c++;
        $data = [ 'c' => $c, 'start' => $start ];

        $ok = set_transient( $key, $data, 120 );

		if ( ! $ok ) {
			$lf_key = 'aegiswaf_ddos_ctr_fail_' . md5( $ip . '|' . $group );
			if ( false === get_transient( $lf_key ) ) {
				set_transient( $lf_key, 1, 300 ); // log at most once per 5 minutes per ip+group

				if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
					AegisWAF_Logger::log( '/', 'GET', 'ddos_shield', 'counter_store_failed', [
						'ip'    => $ip,
						'group' => $group,
						'reason'=> 'set_transient returned false',
					] );
				} else {
					error_log( '[AegisWAF] DDoS counter store failed: set_transient returned false for group=' . $group );
				}
			}
		}

        return $c;
    }

    private static function counter_key( string $ip, string $group ) : string {
        return 'aegiswaf_ddos_' . md5( $ip . '|' . $group );
    }

    private static function cooldown_key( string $ip, string $group ) : string {
        return 'aegiswaf_ddos_cd_' . md5( $ip . '|' . $group );
    }
}
