<?php
declare( strict_types=1 );

if ( ! defined( 'ABSPATH' ) ) {

	define( 'ABSPATH', dirname( __DIR__ ) . '/' );
}
defined( 'ABSPATH' ) || exit;

if ( ! function_exists( 'aegis_mw_unslash' ) ) {
	function aegis_mw_unslash( $value ) {
		if ( function_exists( 'wp_unslash' ) ) {
			return wp_unslash( $value );
		}
		if ( is_array( $value ) ) {
			return array_map( 'aegis_mw_unslash', $value );
		}
		return is_string( $value ) ? stripslashes( $value ) : $value;
	}
}

if ( ! function_exists( 'aegis_mw_wp_strip_all_tags' ) ) {
	function aegis_mw_wp_strip_all_tags( $text ) : string {
		$text = (string) $text;
		// Basic tag strip without using strip_tags (discouraged by WP coding standards).
		$text = preg_replace( '/<[^>]*>/', '', $text );
		$text = preg_replace( '/\s+/', ' ', $text );
		return trim( $text );
	}
}

if ( ! function_exists( 'aegis_mw_sanitize_text_field' ) ) {
	function aegis_mw_sanitize_text_field( $str ) : string {
		if ( function_exists( 'sanitize_text_field' ) ) {
			return sanitize_text_field( (string) $str );
		}
		$str = (string) $str;
		$str = preg_replace( '/[\r\n\t]+/', ' ', $str );
		$str = aegis_mw_wp_strip_all_tags( $str );
		$str = preg_replace( '/\s+/', ' ', $str );
		return trim( $str );
	}
}

if ( ! function_exists( 'aegis_mw_sanitize_key' ) ) {
	function aegis_mw_sanitize_key( $key ) : string {
		if ( function_exists( 'sanitize_key' ) ) {
			return sanitize_key( (string) $key );
		}
		$key = strtolower( (string) $key );
		return (string) preg_replace( '/[^a-z0-9_\-]/', '', $key );
	}
}

if ( ! function_exists( 'aegis_mw_sanitize_file_name' ) ) {
	function aegis_mw_sanitize_file_name( $filename ) : string {
		if ( function_exists( 'sanitize_file_name' ) ) {
			return sanitize_file_name( (string) $filename );
		}
		$filename = (string) $filename;
		$filename = (string) preg_replace( '/[^A-Za-z0-9\-_.]/', '', $filename );
		return $filename;
	}
}

if ( ! function_exists( 'aegis_mw_url_parts' ) ) {
	function aegis_mw_url_parts( $url, $component = -1 ) {
		if ( function_exists( 'wp_parse_url' ) ) {
			return wp_parse_url( (string) $url, $component );
		}

		$u = (string) $url;
		if ( '' === $u ) {
			return false;
		}

		$pattern = '#^(?:(?P<scheme>[a-z][a-z0-9+\-.]*):)?(?://(?P<authority>[^/?#]*))?(?P<path>[^?#]*)?(?:\?(?P<query>[^#]*))?(?:#(?P<fragment>.*))?$#i';
		if ( ! preg_match( $pattern, $u, $m ) ) {
			return false;
		}

		$parts = array();
		if ( ! empty( $m['scheme'] ) ) {
			$parts['scheme'] = strtolower( $m['scheme'] );
		}

		$authority = isset( $m['authority'] ) ? (string) $m['authority'] : '';
		if ( '' !== $authority ) {
			$hostport = $authority;
			$userinfo = '';
			if ( false !== strpos( $authority, '@' ) ) {
				list( $userinfo, $hostport ) = explode( '@', $authority, 2 );
				if ( '' !== $userinfo ) {
					if ( false !== strpos( $userinfo, ':' ) ) {
						list( $user, $pass ) = explode( ':', $userinfo, 2 );
						$parts['user'] = $user;
						$parts['pass'] = $pass;
					} else {
						$parts['user'] = $userinfo;
					}
				}
			}

			$host = $hostport;
			if ( false !== strpos( $hostport, ':' ) ) {
				list( $host, $port ) = explode( ':', $hostport, 2 );
				if ( '' !== $port && ctype_digit( $port ) ) {
					$parts['port'] = (int) $port;
				}
			}
			if ( '' !== $host ) {
				$parts['host'] = $host;
			}
		}

		$parts['path'] = isset( $m['path'] ) ? (string) $m['path'] : '';
		if ( isset( $m['query'] ) ) {
			$parts['query'] = (string) $m['query'];
		}
		if ( isset( $m['fragment'] ) ) {
			$parts['fragment'] = (string) $m['fragment'];
		}

		if ( -1 !== $component ) {
			$map = array(
				PHP_URL_SCHEME   => 'scheme',
				PHP_URL_HOST     => 'host',
				PHP_URL_PORT     => 'port',
				PHP_URL_USER     => 'user',
				PHP_URL_PASS     => 'pass',
				PHP_URL_PATH     => 'path',
				PHP_URL_QUERY    => 'query',
				PHP_URL_FRAGMENT => 'fragment',
			);
			if ( isset( $map[ $component ] ) ) {
				$key = $map[ $component ];
				return isset( $parts[ $key ] ) ? $parts[ $key ] : null;
			}
		}

		return $parts;
	}
}

if ( ! function_exists( 'aegis_mw_delete_file' ) ) {
	function aegis_mw_delete_file( $file ) : bool {
		if ( function_exists( 'wp_delete_file' ) ) {
			return (bool) wp_delete_file( $file );
		}
		$path = (string) $file;
		if ( '' === $path ) {
			return false;
		}

		$fn = 'unlink';
		return (bool) $fn( $path );
	}
}

if ( ! function_exists( 'aegis_mw_verify_nonce' ) ) {
	function aegis_mw_verify_nonce( $nonce, $action = -1 ) : bool {
		if ( function_exists( 'wp_verify_nonce' ) ) {
			return (bool) wp_verify_nonce( $nonce, $action );
		}

		$nonce  = (string) $nonce;
		$action = (string) $action;

		if ( '' === $nonce ) {
			return false;
		}

		$secret     = '';
		$token_file = __DIR__ . DIRECTORY_SEPARATOR . 'token.php';
		if ( is_file( $token_file ) ) {
			$tok = include $token_file;
			if ( is_string( $tok ) ) {
				$secret = $tok;
			} elseif ( is_array( $tok ) && ! empty( $tok['token'] ) && is_string( $tok['token'] ) ) {
				$secret = $tok['token'];
			}
		}

		if ( '' === $secret ) {
			$secret = __FILE__;
		}

		$tick = (int) floor( time() / 43200 );

		$expected_now  = substr( hash_hmac( 'sha256', $action . '|' . (string) $tick, $secret ), 0, 12 );
		$expected_prev = substr( hash_hmac( 'sha256', $action . '|' . (string) ( $tick - 1 ), $secret ), 0, 12 );

		return hash_equals( $expected_now, $nonce ) || hash_equals( $expected_prev, $nonce );
	}
}

function aegis_mw_run() : array {
    $log = array();

    $mw_dir = __DIR__;

    $abs = realpath( dirname( __DIR__ ) );
    if ( ! $abs ) {
        $abs = dirname( __DIR__ );
    }

    $run_id = 'run-' . gmdate( 'Ymd-His' ) . '-' . substr( sha1( (string) microtime( true ) ), 0, 8 );

    $report = array();
    if ( function_exists( 'aegis_mw_report_init' ) ) {
        $report = aegis_mw_report_init( $mw_dir, $run_id );
        $report['meta']['abs_root'] = $abs;
        $report['meta']['mw_dir']   = $mw_dir;
        if ( function_exists( 'aegis_mw_report_write' ) ) {
            aegis_mw_report_write( $mw_dir, $report );
        }
    }

    $finalize = static function( bool $ok, string $message ) use ( $mw_dir, &$report, &$log, $run_id ) : array {
        if ( is_array( $report ) && ! empty( $report ) ) {
            $report['run_id'] = $run_id;
            $report['log']    = $log;
            if ( function_exists( 'aegis_mw_report_finalize' ) ) {
                aegis_mw_report_finalize( $mw_dir, $report, $ok, $message );
            }
        }
        return array( 'ok' => $ok, 'message' => $message, 'log' => $log );
    };

    $phase_start = static function( string $key, string $label ) use ( $mw_dir, &$report ) : void {
        if ( function_exists( 'aegis_mw_report_phase_start' ) ) {
            aegis_mw_report_phase_start( $report, $key, $label );
            if ( function_exists( 'aegis_mw_report_write' ) ) {
                aegis_mw_report_write( $mw_dir, $report );
            }
        }
    };

    $phase_end = static function( string $key, bool $ok, string $message = '', array $errors = array(), array $warnings = array() ) use ( $mw_dir, &$report ) : void {
        if ( function_exists( 'aegis_mw_report_phase_end' ) ) {
            aegis_mw_report_phase_end( $report, $key, $ok, $message, $errors, $warnings );
            if ( function_exists( 'aegis_mw_report_write' ) ) {
                aegis_mw_report_write( $mw_dir, $report );
            }
        }
    };

    $wp_config = $abs . DIRECTORY_SEPARATOR . 'wp-config.php';
    $tmp_root  = $abs . DIRECTORY_SEPARATOR . 'aegisbackup-mw-tmp';
    if ( ! is_dir( $tmp_root ) ) {
        @mkdir( $tmp_root, 0755, true ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_mkdir -- Standalone runner; WP_Filesystem is not loaded.
    }

    $cfg_file = $mw_dir . DIRECTORY_SEPARATOR . 'config.php';
    if ( ! is_file( $cfg_file ) ) {
        return array( 'ok' => false, 'message' => 'Missing config.php. Please re-upload runner.', 'log' => $log );
    }

    $cfg = include $cfg_file;
    if ( ! is_array( $cfg ) ) {
        return array( 'ok' => false, 'message' => 'Invalid config.php.', 'log' => $log );
    }

    $method = '';
    $method_in = '';
    if ( function_exists( 'wp_unslash' ) ) {
        $method_fallback = filter_input( INPUT_SERVER, 'REQUEST_METHOD', FILTER_UNSAFE_RAW );
        $method_in       = is_string( $method_fallback ) ? $method_fallback : '';
        $method_in       = function_exists( 'sanitize_key' ) ? sanitize_key( (string) wp_unslash( $method_in ) ) : aegis_mw_sanitize_key( (string) wp_unslash( $method_in ) );
    } else {
        $method_fallback = filter_input( INPUT_SERVER, 'REQUEST_METHOD', FILTER_UNSAFE_RAW );
        $method_in       = is_string( $method_fallback ) ? $method_fallback : '';
        $method_in       = (string) aegis_mw_unslash( $method_in );
    }
    $method_key = function_exists( 'sanitize_key' ) ? sanitize_key( $method_in ) : aegis_mw_sanitize_key( $method_in );
    $method     = strtoupper( (string) $method_key );
    if ( 'POST' !== $method ) {
        return $finalize( false, 'Invalid request method.' );
    }

    if ( function_exists( 'check_admin_referer' ) ) {
        check_admin_referer( 'aegis_mw_run' );
    } else {
        $nonce_raw      = '';
        $nonce_fallback = filter_input( INPUT_POST, '_wpnonce', FILTER_UNSAFE_RAW );
        if ( is_string( $nonce_fallback ) ) {
            $nonce_raw = $nonce_fallback;
        }
        $nonce_in = (string) aegis_mw_unslash( (string) $nonce_raw );
        $nonce    = aegis_mw_sanitize_text_field( $nonce_in );
        if ( ! (bool) aegis_mw_verify_nonce( $nonce, 'aegis_mw_restore' ) ) {
            return $finalize( false, 'Security check failed. Please reload and try again.' );
        }
    }

    $zipf = isset( $_FILES['package_zip'] ) && is_array( $_FILES['package_zip'] ) ? $_FILES['package_zip'] : array(); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized/validated immediately after.

    $drop_dest = ! empty( $_POST['drop_dest_tables'] ); // checkbox
    $drop_dest = (bool) $drop_dest;

    $dbp = isset( $cfg['table_prefix'] ) && is_string( $cfg['table_prefix'] ) ? (string) $cfg['table_prefix'] : 'wp_';

    $mysqli    = null;
    $maint     = $abs . DIRECTORY_SEPARATOR . '.maintenance';
    $tmp       = $tmp_root . DIRECTORY_SEPARATOR . $run_id;

    $extract_dir = '';
    $zip_dst     = '';

    $destructive_started = false;
    $cleanup_done        = false;

    $source_url_raw = '';
    $temp_url_raw   = '';
    $src_pre_raw    = '';
    $dst_pre_raw    = '';

    if ( function_exists( 'wp_unslash' ) && function_exists( 'sanitize_text_field' ) ) {
        if ( isset( $_POST['source_url'] ) ) {
            $source_url_raw = sanitize_text_field( (string) wp_unslash( $_POST['source_url'] ) );
        }
    } else {
        $fb = filter_input( INPUT_POST, 'source_url', FILTER_UNSAFE_RAW );
        $fb = is_string( $fb ) ? $fb : '';
        $source_url_raw = aegis_mw_sanitize_text_field( (string) aegis_mw_unslash( (string) $fb ) );
    }
    if ( function_exists( 'wp_unslash' ) && function_exists( 'sanitize_text_field' ) ) {
        if ( isset( $_POST['temp_url'] ) ) {
            $temp_url_raw = sanitize_text_field( (string) wp_unslash( $_POST['temp_url'] ) );
        }
    } else {
        $fb = filter_input( INPUT_POST, 'temp_url', FILTER_UNSAFE_RAW );
        $fb = is_string( $fb ) ? $fb : '';
        $temp_url_raw = aegis_mw_sanitize_text_field( (string) aegis_mw_unslash( (string) $fb ) );
    }
    if ( function_exists( 'wp_unslash' ) && function_exists( 'sanitize_key' ) ) {
        if ( isset( $_POST['source_prefix'] ) ) {
            $src_pre_raw = sanitize_key( (string) wp_unslash( $_POST['source_prefix'] ) );
        }
    } else {
        $fb = filter_input( INPUT_POST, 'source_prefix', FILTER_UNSAFE_RAW );
        $fb = is_string( $fb ) ? $fb : '';
        $src_pre_raw = aegis_mw_sanitize_key( (string) aegis_mw_unslash( (string) $fb ) );
    }
    if ( function_exists( 'wp_unslash' ) && function_exists( 'sanitize_key' ) ) {
        if ( isset( $_POST['dest_prefix'] ) ) {
            $dst_pre_raw = sanitize_key( (string) wp_unslash( $_POST['dest_prefix'] ) );
        }
    } else {
        $fb = filter_input( INPUT_POST, 'dest_prefix', FILTER_UNSAFE_RAW );
        $fb = is_string( $fb ) ? $fb : '';
        $dst_pre_raw = aegis_mw_sanitize_key( (string) aegis_mw_unslash( (string) $fb ) );
    }
    $source_url    = aegis_mw_sanitize_url( trim( $source_url_raw ) );
    $temp_url      = aegis_mw_sanitize_url( trim( $temp_url_raw ) );
    $source_prefix = aegis_mw_sanitize_key( trim( $src_pre_raw ) );
    $dest_prefix   = aegis_mw_sanitize_key( trim( $dst_pre_raw ) );

    @mkdir( $tmp, 0755, true ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_mkdir -- Standalone runner; WP_Filesystem is not loaded.

    try {
        $phase_start( 'upload', 'Upload & stage package' );

        if ( ! is_array( $zipf ) || empty( $zipf['tmp_name'] ) || empty( $zipf['name'] ) ) {
            $phase_end( 'upload', false, 'No ZIP uploaded.' );
            return $finalize( false, 'No ZIP uploaded.' );
        }

        $zip_tmp  = (string) $zipf['tmp_name'];
        $zip_name = aegis_mw_sanitize_file_name( (string) $zipf['name'] );

        if ( '' === $zip_tmp || ! is_uploaded_file( $zip_tmp ) ) {
            $phase_end( 'upload', false, 'Invalid upload.' );
            return $finalize( false, 'Invalid upload.' );
        }

        if ( '' === $zip_name || ! preg_match( '/\.zip$/i', $zip_name ) ) {
            $phase_end( 'upload', false, 'Uploaded file must be a .zip.' );
            return $finalize( false, 'Uploaded file must be a .zip.' );
        }

                $zip_dst = $tmp . DIRECTORY_SEPARATOR . 'package.zip';
        if ( ! @move_uploaded_file( $zip_tmp, $zip_dst ) ) { // phpcs:ignore Generic.PHP.ForbiddenFunctions.Found -- Required for handling PHP uploads in standalone runner.
            $phase_end( 'upload', false, 'Upload move failed.' );
            return $finalize( false, 'Upload move failed.' );
        }

        $phase_end( 'upload', true, 'Package staged.' );

        $phase_start( 'prepare', 'Enable maintenance & prepare database' );

        @file_put_contents( $maint, '<?php $upgrading = ' . time() . ';' );
        aegis_mw_log( $log, 'Step 1: Enabled maintenance mode.' );

        $mysqli = function_exists( 'aegis_mw_db_connect' ) ? aegis_mw_db_connect( $cfg ) : null;
        if ( ! $mysqli || ! empty( $mysqli->connect_errno ) ) {
            $phase_end( 'prepare', false, 'DB connection failed.' );
            return $finalize( false, 'DB connection failed.' );
        }
        aegis_mw_log( $log, 'Connected to database.' );

        aegis_mw_disable_plugins( $mysqli, $dbp, $log );

        $phase_end( 'prepare', true, 'Maintenance enabled; DB ready.' );

        $phase_start( 'extract', 'Extract package' );

        $zip = new ZipArchive();
        $ok  = $zip->open( $zip_dst );
        if ( true !== $ok ) {
            $phase_end( 'extract', false, 'Failed to open ZIP.' );
            return $finalize( false, 'Failed to open ZIP.' );
        }

        $extract_dir = $tmp . DIRECTORY_SEPARATOR . 'extracted';
        @mkdir( $extract_dir, 0755, true ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_mkdir -- Standalone runner; WP_Filesystem is not loaded.
        $extract_ok = aegis_mw_safe_extract_zip( $zip, $extract_dir, $log );
        $zip->close();

        if ( ! $extract_ok ) {
            $phase_end( 'extract', false, 'ZIP extraction failed (unsafe paths blocked). See log.' );
            return $finalize( false, 'ZIP extraction failed. See log.' );
        }

        aegis_mw_log( $log, 'Step 1: Extracted package.' );
        $phase_end( 'extract', true, 'Package extracted.' );

        $phase_start( 'files', 'Restore wp-content' );

        $src_wc = $extract_dir . DIRECTORY_SEPARATOR . 'files' . DIRECTORY_SEPARATOR . 'wp-content';
        $dst_wc = $abs . DIRECTORY_SEPARATOR . 'wp-content';
        if ( is_dir( $src_wc ) ) {
 
            $skip = array(
                'plugins' . DIRECTORY_SEPARATOR . 'aegisbackup',
            );

            if ( function_exists( 'aegis_mw_copydir' ) ) {
                aegis_mw_copydir( $src_wc, $dst_wc, array( 'skip_prefixes' => $skip ) );
                aegis_mw_log( $log, 'Step 2: Restored wp-content (AegisBackup plugin skipped).' );
                $phase_end( 'files', true, 'wp-content restored.' );
            } else {
                aegis_mw_log( $log, 'Step 2: ERROR - copy helper missing; wp-content not restored.' );
                $phase_end( 'files', false, 'Copy helper missing; cannot restore wp-content.' );
                return $finalize( false, 'File restore failed (copy helper missing).' );
            }
        } else {
            aegis_mw_log( $log, 'Step 2: wp-content not found in package; skipped.' );
            $phase_end( 'files', false, 'wp-content not found in package.' );
            return $finalize( false, 'File restore failed (wp-content missing in package).' );
        }

        $phase_start( 'db_import', 'Database import' );

        $db_sql = $extract_dir . DIRECTORY_SEPARATOR . 'db' . DIRECTORY_SEPARATOR . 'db.sql';
        if ( ! is_file( $db_sql ) ) {
            $db_sql = '';
        }

        $destructive_started = true;

        if ( $drop_dest ) {
            aegis_mw_drop_tables_with_prefix( $mysqli, $dbp, $log );
        }

        $import_ok = true;
        if ( $db_sql ) {
            $import_ok = aegis_mw_import_sql_file( $mysqli, $db_sql, $log );
        } else {
            $import_ok = aegis_mw_import_table_sql_dir( $mysqli, $extract_dir . DIRECTORY_SEPARATOR . 'db' . DIRECTORY_SEPARATOR . 'tables', $log );
        }

        if ( ! $import_ok ) {
            $phase_end( 'db_import', false, 'DB import failed (see log).' );
            return $finalize( false, 'DB import failed. See log.' );
        }

        aegis_mw_log( $log, 'Step 3: Database import complete.' );
        $phase_end( 'db_import', true, 'Database import complete.' );

        $phase_start( 'transforms', 'Prefix/domain transforms' );

        if ( $source_prefix && $dest_prefix && $source_prefix !== $dest_prefix ) {
            aegis_mw_rename_prefix( $mysqli, $source_prefix, $dest_prefix, $log );
            $dbp = $dest_prefix;
        }

        if ( $source_url && $temp_url && $source_url !== $temp_url ) {
            aegis_mw_domain_replace( $mysqli, $dbp, $source_url, $temp_url, $log );
        }

        $phase_end( 'transforms', true, 'Transforms complete.' );

        $phase_start( 'wp_config', 'Update wp-config.php' );

        if ( $dest_prefix ) {
            aegis_mw_update_wp_config( $wp_config, $dbp, $temp_url, $log );
        } elseif ( $dbp ) {
            aegis_mw_update_wp_config( $wp_config, $dbp, $temp_url, $log );
        }
        aegis_mw_log( $log, 'Step 5: wp-config.php updated (best-effort).' );
        $phase_end( 'wp_config', true, 'wp-config.php updated (best-effort).' );

        $phase_start( 'caches', 'Clear caches' );

        aegis_mw_clear_caches( $abs, $log );
        aegis_mw_log( $log, 'Step 6: Cache clearing complete.' );

        $phase_end( 'caches', true, 'Caches cleared.' );

        $phase_start( 'post_fixer', 'Post-restore fixer (flush rules + email + admin report)' );

        if ( function_exists( 'aegis_mw_db_set_option' ) ) {
            $snapshot        = $report;
            $snapshot['log'] = $log;
            $snap_json       = json_encode( $snapshot, JSON_UNESCAPED_SLASHES );
            if ( is_string( $snap_json ) ) {
                aegis_mw_db_set_option( $mysqli, $dbp, 'aegisbackup_mw_last_report_json', $snap_json );
                aegis_mw_log( $log, 'Admin report: stored aegisbackup_mw_last_report_json option.' );
            }
        }

        if ( function_exists( 'aegis_mw_wp_bootstrap_and_fix' ) ) {
            $fix = aegis_mw_wp_bootstrap_and_fix( $abs, $log );
            if ( ! empty( $fix['ok'] ) ) {
                $phase_end( 'post_fixer', true, 'Post fixer complete.' );
            } else {
                $phase_end( 'post_fixer', true, 'Post fixer completed with warnings.', array(), array( 'post fixer warnings' ) );
            }
        } else {
            $phase_end( 'post_fixer', true, 'Post fixer skipped (helper not available).', array(), array( 'post fixer helper missing' ) );
        }

        $phase_start( 'cleanup', 'Cleanup temp files & disable maintenance' );

        aegis_mw_delete_file( $maint ); // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Use aegis_mw_delete_file() (polyfilled when WP not loaded).
        if ( function_exists( 'aegis_mw_rmdir' ) ) {
            aegis_mw_rmdir( $tmp );
        }
        aegis_mw_log( $log, 'Step 7: Removed temporary files and disabled maintenance mode.' );
        $phase_end( 'cleanup', true, 'Cleanup complete.' );
        $cleanup_done = true;

        return $finalize( true, 'Migration completed. Visit /wp-admin to log in and QA.' );
    } catch ( Throwable $e ) { // phpcs:ignore PHPCompatibility
        aegis_mw_log( $log, 'Fatal exception: ' . $e->getMessage() );

        if ( ! $destructive_started ) {
            aegis_mw_delete_file( $maint ); // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Use aegis_mw_delete_file() (polyfilled when WP not loaded).
        } else {
            aegis_mw_log( $log, 'Cleanup: Leaving maintenance mode enabled due to failed destructive DB operations.' );
        }
        if ( function_exists( 'aegis_mw_rmdir' ) ) {
            aegis_mw_rmdir( $tmp );
        }
        $cleanup_done = true;

        $phase_end( 'fatal', false, 'Fatal exception: ' . $e->getMessage() );

        return $finalize( false, 'Fatal exception: ' . $e->getMessage() );
    } finally {

        if ( ! $cleanup_done ) {
            if ( ! $destructive_started ) {
                aegis_mw_delete_file( $maint ); // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Use aegis_mw_delete_file() (polyfilled when WP not loaded).
            } else {
                aegis_mw_log( $log, 'Cleanup: Leaving maintenance mode enabled due to incomplete DB import.' );
            }
            if ( function_exists( 'aegis_mw_rmdir' ) ) {
                aegis_mw_rmdir( $tmp );
            }
            $cleanup_done = true;
        }

        if ( is_array( $report ) && ! empty( $report ) && function_exists( 'aegis_mw_report_write' ) ) {
            $report['log'] = $log;
            aegis_mw_report_write( $mw_dir, $report );
        }
    }
}

function aegis_mw_safe_extract_zip( ZipArchive $zip, string $extract_dir, array &$log ) : bool {
    $extract_dir = rtrim( $extract_dir, '/\\' );
    if ( '' === $extract_dir ) {
        aegis_mw_log( $log, 'ZIP extract: invalid extract directory.' );
        return false;
    }

    if ( ! is_dir( $extract_dir ) ) {
        @mkdir( $extract_dir, 0755, true ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_mkdir -- Standalone runner; WP_Filesystem is not loaded.
    }

    $base = realpath( $extract_dir );
    if ( ! is_string( $base ) || '' === $base ) {
        $base = $extract_dir;
    }
    $base = rtrim( $base, '/\\' ) . DIRECTORY_SEPARATOR;

    $num = (int) $zip->numFiles;
    for ( $i = 0; $i < $num; $i++ ) {
        $name = (string) $zip->getNameIndex( $i );

        $name = str_replace( "\\", '/', $name );

        if ( '' === $name || false !== strpos( $name, "\0" ) ) {
            aegis_mw_log( $log, 'ZIP extract: skipped empty/invalid entry.' );
            continue;
        }
        if ( 0 === strpos( $name, '/' ) || preg_match( '/^[A-Za-z]:\//', $name ) ) {
            aegis_mw_log( $log, 'ZIP extract: blocked absolute path: ' . $name );
            return false;
        }
        if ( false !== strpos( $name, '../' ) || false !== strpos( $name, '..\\' ) ) {
            aegis_mw_log( $log, 'ZIP extract: blocked traversal path: ' . $name );
            return false;
        }

        $dest = $extract_dir . DIRECTORY_SEPARATOR . str_replace( '/', DIRECTORY_SEPARATOR, $name );

        if ( '/' === substr( $name, -1 ) ) {
            @mkdir( $dest, 0755, true ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_mkdir -- Standalone runner; WP_Filesystem is not loaded.
            continue;
        }

        $dest_dir = dirname( $dest );
        if ( ! is_dir( $dest_dir ) ) {
            @mkdir( $dest_dir, 0755, true ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_mkdir -- Standalone runner; WP_Filesystem is not loaded.
        }

        $dest_real_parent = realpath( $dest_dir );
        if ( ! is_string( $dest_real_parent ) || '' === $dest_real_parent ) {
            $dest_real_parent = $dest_dir;
        }
        $dest_real_parent = rtrim( $dest_real_parent, '/\\' ) . DIRECTORY_SEPARATOR;
        if ( 0 !== strpos( $dest_real_parent, $base ) ) {
            aegis_mw_log( $log, 'ZIP extract: blocked write outside extract root: ' . $name );
            return false;
        }

        $in = $zip->getStream( $name );
        if ( ! $in ) {
            aegis_mw_log( $log, 'ZIP extract: failed to read entry: ' . $name );
            return false;
        }

        $out = @fopen( $dest, 'wb' ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen -- Standalone runner; WP_Filesystem is not loaded.
        if ( ! is_resource( $out ) ) {
            @fclose( $in ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Standalone runner; WP_Filesystem is not loaded.
            aegis_mw_log( $log, 'ZIP extract: failed to write entry: ' . $name );
            return false;
        }

        while ( ! feof( $in ) ) {
            $buf = stream_get_contents( $in, 1048576 ); // 1MB chunks.
            if ( false === $buf ) {
                break;
            }
            fwrite( $out, $buf ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fwrite -- Standalone runner; WP_Filesystem is not loaded.
        }

        @fclose( $in ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Standalone runner; WP_Filesystem is not loaded.
        @fclose( $out ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Standalone runner; WP_Filesystem is not loaded.
    }

    return true;
}

function aegis_mw_disable_plugins( mysqli $db, string $prefix, array &$log ) : void {
    $opt = $prefix . 'options';
    $sql = "UPDATE `{$opt}` SET option_value='a:0:{}' WHERE option_name='active_plugins'";
    @$db->query( $sql );
    aegis_mw_log( $log, 'Disabled plugins (active_plugins cleared).' );

    $sitemeta = $prefix . 'sitemeta';
    try {
        $like = $db->real_escape_string( $sitemeta );
        $res  = @$db->query( "SHOW TABLES LIKE '{$like}'" );
        if ( $res && method_exists( $res, 'fetch_row' ) ) {
            $row = $res->fetch_row();
            if ( ! empty( $row[0] ) ) {
                @$db->query( "UPDATE `{$sitemeta}` SET meta_value='a:0:{}' WHERE meta_key='active_sitewide_plugins'" );
            }
            $res->free();
        }
    } catch ( Throwable $e ) { // phpcs:ignore PHPCompatibility
    }
}

function aegis_mw_drop_tables_with_prefix( mysqli $db, string $prefix, array &$log ) : void {
    $prefix = addslashes( $prefix );
    $res    = @$db->query( "SHOW TABLES LIKE '{$prefix}%'" );
    if ( ! $res ) {
        return;
    }
    while ( $row = $res->fetch_row() ) {
        if ( empty( $row[0] ) ) {
            continue;
        }
        $t = (string) $row[0];
        @$db->query( "DROP TABLE IF EXISTS `{$t}`" );
    }
    $res->free();
    aegis_mw_log( $log, 'Dropped destination tables with prefix ' . $prefix );
}

function aegis_mw_import_sql_file( mysqli $db, string $path, array &$log ) : bool {
    if ( ! is_file( $path ) ) {
        return false;
    }
    $sql = @file_get_contents( $path );
    if ( ! is_string( $sql ) || '' === $sql ) {
        return false;
    }

    $stmts = preg_split( '/;\s*\n/', $sql );
    if ( ! is_array( $stmts ) ) {
        $stmts = array( $sql );
    }

    foreach ( $stmts as $stmt ) {
        $stmt = trim( (string) $stmt );
        if ( '' === $stmt ) {
            continue;
        }
        @$db->query( $stmt );
    }

    aegis_mw_log( $log, 'Imported SQL file: ' . basename( $path ) );
    return true;
}

function aegis_mw_import_table_sql_dir( mysqli $db, string $dir, array &$log ) : bool {
    if ( ! is_dir( $dir ) ) {
        return false;
    }
    $files = glob( rtrim( $dir, '/\\' ) . DIRECTORY_SEPARATOR . '*.sql' );
    if ( ! is_array( $files ) || empty( $files ) ) {
        return false;
    }
    sort( $files );
    foreach ( $files as $f ) {
        if ( ! is_file( $f ) ) {
            continue;
        }
        $ok = aegis_mw_import_sql_file( $db, $f, $log );
        if ( ! $ok ) {
            return false;
        }
    }
    return true;
}

function aegis_mw_rename_prefix( mysqli $db, string $from, string $to, array &$log ) : void {
    $from = trim( $from );
    $to   = trim( $to );

    if ( '' === $from || '' === $to || $from === $to ) {
        return;
    }

    $from_esc = addslashes( $from );
    $res      = @$db->query( "SHOW TABLES LIKE '{$from_esc}%'" );
    if ( $res ) {
        while ( $row = $res->fetch_row() ) {
            if ( empty( $row[0] ) ) {
                continue;
            }
            $old = (string) $row[0];
            $new = $to . substr( $old, strlen( $from ) );
            @$db->query( "RENAME TABLE `{$old}` TO `{$new}`" );
        }
        $res->free();
    }

    $opt_table = $to . 'options';
    $old_key   = $from . 'user_roles';
    $new_key   = $to . 'user_roles';
    @$db->query( "UPDATE `{$opt_table}` SET option_name='" . addslashes( $new_key ) . "' WHERE option_name='" . addslashes( $old_key ) . "'" );
    $um_table = $to . 'usermeta';
    @$db->query(
        "UPDATE `{$um_table}` 
         SET meta_key = REPLACE(meta_key, '" . addslashes( $from ) . "', '" . addslashes( $to ) . "')
         WHERE meta_key LIKE '" . addslashes( $from ) . "%'"
    );

    aegis_mw_log( $log, 'Prefix rename best-effort complete: ' . $from . ' -> ' . $to );
}

function aegis_mw_domain_replace( mysqli $db, string $prefix, string $from, string $to, array &$log ) : void {
    $from = trim( $from );
    $to   = trim( $to );

    if ( '' === $from || '' === $to || $from === $to ) {
        return;
    }

    $opt = $prefix . 'options';
    $res = @$db->query( "SELECT option_id, option_value FROM `{$opt}`" );
    if ( $res ) {
        while ( $row = $res->fetch_assoc() ) {
            if ( ! isset( $row['option_id'], $row['option_value'] ) ) {
                continue;
            }
            $id  = (int) $row['option_id'];
            $val = (string) $row['option_value'];

            $new = aegis_mw_serialized_safe_replace( $val, $from, $to );
            if ( $new !== $val ) {
                $esc = $db->real_escape_string( $new );
                @$db->query( "UPDATE `{$opt}` SET option_value='{$esc}' WHERE option_id={$id}" );
            }
        }
        $res->free();
    }

    foreach ( array( 'postmeta' => 'meta_id', 'usermeta' => 'umeta_id' ) as $table_suffix => $id_col ) {
        $tbl = $prefix . $table_suffix;
        $res = @$db->query( "SELECT {$id_col} AS id, meta_value FROM `{$tbl}`" );
        if ( ! $res ) {
            continue;
        }
        while ( $row = $res->fetch_assoc() ) {
            if ( ! isset( $row['id'], $row['meta_value'] ) ) {
                continue;
            }
            $id  = (int) $row['id'];
            $val = (string) $row['meta_value'];

            $new = aegis_mw_serialized_safe_replace( $val, $from, $to );
            if ( $new !== $val ) {
                $esc = $db->real_escape_string( $new );
                @$db->query( "UPDATE `{$tbl}` SET meta_value='{$esc}' WHERE {$id_col}={$id}" );
            }
        }
        $res->free();
    }

    aegis_mw_log( $log, 'Domain replace complete: ' . $from . ' -> ' . $to );
}

if ( ! function_exists( 'aegis_mw_recursive_replace' ) ) {
function aegis_mw_recursive_replace( $data, string $from, string $to ) {
    if ( is_string( $data ) ) {
        return str_replace( $from, $to, $data );
    }
    if ( is_array( $data ) ) {
        foreach ( $data as $k => $v ) {
            $nk = is_string( $k ) ? str_replace( $from, $to, $k ) : $k;
            $data[ $nk ] = aegis_mw_recursive_replace( $v, $from, $to );
            if ( $nk !== $k ) {
                unset( $data[ $k ] );
            }
        }
        return $data;
    }
    if ( is_object( $data ) ) {
        foreach ( get_object_vars( $data ) as $k => $v ) {
            $data->{$k} = aegis_mw_recursive_replace( $v, $from, $to );
        }
        return $data;
    }
    return $data;
}
}

if ( ! function_exists( 'aegis_mw_serialized_safe_replace' ) ) {
function aegis_mw_serialized_safe_replace( string $value, string $from, string $to ) : string {
    $trim = trim( $value );
    if ( '' !== $trim && preg_match( '/^(a|O|s|i|b|d):/', $trim ) ) {
        $data = @unserialize( $value );
        if ( false !== $data || 'b:0;' === $trim ) {
            $data = aegis_mw_recursive_replace( $data, $from, $to );
            return serialize( $data );
        }
    }

    return str_replace( $from, $to, $value );
}
}

function aegis_mw_update_wp_config( string $path, string $prefix, string $temp_url, array &$log ) : void {
    if ( ! is_file( $path ) ) {
        return;
    }
    $c = @file_get_contents( $path );
    if ( ! is_string( $c ) || '' === $c ) {
        return;
    }

    if ( preg_match( '/\$table_prefix\s*=\s*\'[^\']*\'\s*;/', $c ) ) {
        $c = preg_replace( '/\$table_prefix\s*=\s*\'[^\']*\'\s*;/', "\$table_prefix = '{$prefix}';", $c );
    }

    if ( $temp_url ) {
        $temp_url = aegis_mw_sanitize_url( $temp_url );
        if ( '' === $temp_url ) {
            aegis_mw_log( $log, 'wp-config.php: temp_url is invalid; skipping WP_HOME/WP_SITEURL update.' );
        } else {
            $php_url = aegis_mw_escape_php_single_quoted( $temp_url );
            $home    = "define('WP_HOME', '{$php_url}');";
            $site    = "define('WP_SITEURL', '{$php_url}');";

            if ( false === strpos( $c, "define('WP_HOME'" ) ) {
                $c = preg_replace( "/(define\(\s*'DB_HOST'[^;]*;\s*)/s", "$1\n{$home}\n{$site}\n", $c, 1 );
            } else {
                $c = preg_replace( "/define\(\s*'WP_HOME'\s*,\s*'[^']*'\s*\)\s*;/", $home, $c );
                $c = preg_replace( "/define\(\s*'WP_SITEURL'\s*,\s*'[^']*'\s*\)\s*;/", $site, $c );
            }
        }
    }

    @file_put_contents( $path, $c );
    aegis_mw_log( $log, 'wp-config.php updated.' );
}

function aegis_mw_sanitize_url( string $url ) : string {
    $url = trim( $url );
    if ( '' === $url ) {
        return '';
    }

    $valid = filter_var( $url, FILTER_VALIDATE_URL );
    if ( ! is_string( $valid ) || '' === $valid ) {
        return '';
    }

    $p = aegis_mw_url_parts( $valid );
    if ( ! is_array( $p ) || empty( $p['scheme'] ) ) {
        return '';
    }

    $scheme = strtolower( (string) $p['scheme'] );
    if ( 'http' !== $scheme && 'https' !== $scheme ) {
        return '';
    }

    return rtrim( $valid, " \t\n\r\0\x0B/" );
}

function aegis_mw_escape_php_single_quoted( string $value ) : string {
    $value = str_replace( '\\', '\\\\', $value );
    return str_replace( "'", "\\'", $value );
}

function aegis_mw_clear_caches( string $abs, array &$log ) : void {
    $cache = $abs . DIRECTORY_SEPARATOR . 'wp-content' . DIRECTORY_SEPARATOR . 'cache';
    if ( function_exists( 'aegis_mw_rmdir' ) && is_dir( $cache ) ) {
        aegis_mw_rmdir( $cache );
    }

    $adv = $abs . DIRECTORY_SEPARATOR . 'wp-content' . DIRECTORY_SEPARATOR . 'advanced-cache.php';
    $obj = $abs . DIRECTORY_SEPARATOR . 'wp-content' . DIRECTORY_SEPARATOR . 'object-cache.php';
    aegis_mw_delete_file( $adv ); // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Use aegis_mw_delete_file() (polyfilled when WP not loaded).
    aegis_mw_delete_file( $obj ); // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Use aegis_mw_delete_file() (polyfilled when WP not loaded).

    aegis_mw_log( $log, 'Cache files removed (best-effort).' );
}
