<?php
declare( strict_types=1 );

if ( ! defined( 'ABSPATH' ) ) {
	define( 'ABSPATH', dirname( __DIR__ ) . '/' );
}
defined( 'ABSPATH' ) || exit;


if ( ! function_exists( 'wp_unslash' ) ) {
	function wp_unslash( $value ) {
		return is_array( $value ) ? array_map( 'wp_unslash', $value ) : stripslashes( (string) $value );
	}
}

if ( ! function_exists( 'sanitize_text_field' ) ) {
	function sanitize_text_field( $text ) {
		$text = (string) $text;
		$text = preg_replace( '/[\x00-\x1F\x7F]/u', '', $text );
		$text = str_replace( '<', '&lt;', $text );
		$text = preg_replace( '/<[^>]*>/', '', $text );
		$text = preg_replace( '/[\r\n\t ]+/', ' ', $text );
		$text = preg_replace( '/%[0-9A-Fa-f]{2}/', '', $text );
		return trim( $text );
	}
}

if ( ! function_exists( 'sanitize_key' ) ) {
	function sanitize_key( $key ) {
		$key = strtolower( (string) $key );
		return preg_replace( '/[^a-z0-9_\-]/', '', $key );
	}
}

if ( ! function_exists( 'absint' ) ) {
	function absint( $maybeint ) {
		return (int) abs( (int) $maybeint );
	}
}

if ( ! function_exists( 'esc_html' ) ) {
	function esc_html( $text ) {
		return htmlspecialchars( (string) $text, ENT_QUOTES, 'UTF-8' );
	}
}

if ( ! function_exists( 'esc_attr' ) ) {
	function esc_attr( $text ) {
		return htmlspecialchars( (string) $text, ENT_QUOTES, 'UTF-8' );
	}
}

if ( ! function_exists( 'esc_url' ) ) {
	function esc_url( $url ) {
		$url = (string) $url;
		$url = preg_replace( '/[\x00-\x1F\x7F]/u', '', $url );
		return htmlspecialchars( $url, ENT_QUOTES, 'UTF-8' );
	}
}

if ( ! function_exists( 'wp_create_nonce' ) ) {
	function wp_create_nonce( $action = -1 ) {
		$action = (string) $action;
		$secret = '';
		$token_file = __DIR__ . '/token.php';
		if ( is_file( $token_file ) ) {
			$token = include $token_file;
			if ( is_string( $token ) ) {
				$secret = $token;
			}
		}
		if ( '' === $secret ) {
			$secret = __FILE__;
		}
		$tick = (int) floor( time() / 43200 );
		return substr( hash_hmac( 'sha256', $action . '|' . (string) $tick, $secret ), 0, 12 );
	}
}

if ( ! function_exists( 'wp_verify_nonce' ) ) {
	function wp_verify_nonce( $nonce, $action = -1 ) {
		$nonce  = (string) $nonce;
		$action = (string) $action;
		if ( '' === $nonce ) {
			return false;
		}
		$secret = '';
		$token_file = __DIR__ . '/token.php';
		if ( is_file( $token_file ) ) {
			$token = include $token_file;
			if ( is_string( $token ) ) {
				$secret = $token;
			}
		}
		if ( '' === $secret ) {
			$secret = __FILE__;
		}
		$tick = (int) floor( time() / 43200 );
		$expected_now  = substr( hash_hmac( 'sha256', $action . '|' . (string) $tick, $secret ), 0, 12 );
		$expected_prev = substr( hash_hmac( 'sha256', $action . '|' . (string) ( $tick - 1 ), $secret ), 0, 12 );
		return hash_equals( $expected_now, $nonce ) || hash_equals( $expected_prev, $nonce );
	}
}

if ( ! function_exists( 'wp_nonce_field' ) ) {
	function wp_nonce_field( $action = -1, $name = '_wpnonce' ) {
		$nonce = wp_create_nonce( $action );
		echo '<input type="hidden" name="' . esc_attr( (string) $name ) . '" value="' . esc_attr( $nonce ) . '" />';
	}
}


if ( ! function_exists( 'aegis_mw_wp_unslash' ) ) {
	function aegis_mw_wp_unslash( $value ) {
		if ( function_exists( 'wp_unslash' ) ) {
			return wp_unslash( $value );
		}
		return is_array( $value ) ? array_map( 'aegis_mw_wp_unslash', $value ) : stripslashes( (string) $value );
	}
}

if ( ! function_exists( 'aegis_mw_sanitize_text_field' ) ) {
	function aegis_mw_sanitize_text_field( $text ) {
		if ( function_exists( 'sanitize_text_field' ) ) {
			return sanitize_text_field( $text );
		}
		$text = (string) $text;
		$text = preg_replace( '/[\x00-\x1F\x7F]/u', '', $text );
		$text = preg_replace( '/<[^>]*>/', '', $text );
		return trim( $text );
	}
}

if ( ! function_exists( 'aegis_mw_sanitize_key' ) ) {
	function aegis_mw_sanitize_key( $key ) {
		if ( function_exists( 'sanitize_key' ) ) {
			return sanitize_key( $key );
		}
		$key = strtolower( (string) $key );
		return preg_replace( '/[^a-z0-9_\-]/', '', $key );
	}
}

if ( ! function_exists( 'aegis_mw_absint' ) ) {
	function aegis_mw_absint( $maybeint ) {
		if ( function_exists( 'absint' ) ) {
			return absint( $maybeint );
		}
		return (int) abs( (int) $maybeint );
	}
}

if ( ! function_exists( 'aegis_mw_esc_html' ) ) {
	function aegis_mw_esc_html( $text ) {
		if ( function_exists( 'esc_html' ) ) {
			return esc_html( $text );
		}
		return htmlspecialchars( (string) $text, ENT_QUOTES, 'UTF-8' );
	}
}

if ( ! function_exists( 'aegis_mw_esc_attr' ) ) {
	function aegis_mw_esc_attr( $text ) {
		if ( function_exists( 'esc_attr' ) ) {
			return esc_attr( $text );
		}
		return htmlspecialchars( (string) $text, ENT_QUOTES, 'UTF-8' );
	}
}

if ( ! function_exists( 'aegis_mw_esc_url' ) ) {
	function aegis_mw_esc_url( $url ) {
		if ( function_exists( 'esc_url' ) ) {
			return esc_url( $url );
		}
		$url = trim( (string) $url );
		if ( '' === $url ) {
			return '';
		}
		// Allow only http/https for this runner's links.
		if ( preg_match( '#^([a-z][a-z0-9+\-.]*):\/\/#i', $url, $m ) ) {
			$scheme = strtolower( (string) $m[1] );
			if ( 'http' !== $scheme && 'https' !== $scheme ) {
				return '';
			}
			return $url;
		}
		return esc_attr( $url );
	}
}

if ( ! function_exists( 'aegis_mw_wp_create_nonce' ) ) {
	function aegis_mw_wp_create_nonce( $action = -1 ) {
		if ( function_exists( 'wp_create_nonce' ) ) {
			return wp_create_nonce( $action );
		}
		$action = (string) $action;
		$tick   = (int) floor( time() / 43200 );
		$seed   = $action . '|' . (string) $tick;
		$secret = '';
		$token_file = __DIR__ . '/token.php';
		if ( is_file( $token_file ) ) {
			$token = include $token_file;
			if ( is_string( $token ) ) {
				$secret = $token;
			}
		}
		if ( '' === $secret ) {
			$secret = __FILE__;
		}
		return substr( hash_hmac( 'sha256', $seed, $secret ), 0, 12 );
	}
}

if ( ! function_exists( 'aegis_mw_wp_verify_nonce' ) ) {
	function aegis_mw_wp_verify_nonce( $nonce, $action = -1 ) {
		if ( function_exists( 'wp_verify_nonce' ) ) {
			return (bool) wp_verify_nonce( $nonce, $action );
		}
		$nonce  = (string) $nonce;
		$action = (string) $action;
		if ( '' === $nonce ) {
			return false;
		}

		$secret = '';
		$token_file = __DIR__ . '/token.php';
		if ( is_file( $token_file ) ) {
			$token = include $token_file;
			if ( is_string( $token ) ) {
				$secret = $token;
			}
		}
		if ( '' === $secret ) {
			$secret = __FILE__;
		}
		$tick = (int) floor( time() / 43200 );
		$expected_now  = substr( hash_hmac( 'sha256', $action . '|' . (string) $tick, $secret ), 0, 12 );
		$expected_prev = substr( hash_hmac( 'sha256', $action . '|' . (string) ( $tick - 1 ), $secret ), 0, 12 );
		return hash_equals( $expected_now, $nonce ) || hash_equals( $expected_prev, $nonce );
	}
}

if ( ! function_exists( 'aegis_mw_wp_nonce_field' ) ) {
	function aegis_mw_wp_nonce_field( $action = -1, $name = '_wpnonce' ) {
		$nonce = aegis_mw_wp_create_nonce( $action );
		echo '<input type="hidden" name="' . esc_attr( (string) $name ) . '" value="' . esc_attr( $nonce ) . '" />';
	}
}

header( 'Content-Type: text/html; charset=utf-8' );

function aegis_mw_h( $s ) : string {
    return htmlspecialchars( (string) $s, ENT_QUOTES, 'UTF-8' );
}

function aegis_mw_base_url() : string {
    $https_val = isset( $_SERVER['HTTPS'] ) ? strtolower( (string) sanitize_text_field( wp_unslash( $_SERVER['HTTPS'] ) ) ) : '';
    $port_val  = isset( $_SERVER['SERVER_PORT'] ) ? (string) absint( wp_unslash( $_SERVER['SERVER_PORT'] ) ) : '';
    $https     = ( '' !== $https_val && 'off' !== $https_val ) || ( '443' === $port_val );
    $scheme    = $https ? 'https' : 'http';
    $host_raw = isset( $_SERVER['HTTP_HOST'] ) ? (string) sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : '';
    $host_raw = preg_replace( '/:\d+$/', '', $host_raw );
    $host     = preg_replace( '/[^A-Za-z0-9\.\-]/', '', $host_raw );
    $uri  = isset( $_SERVER['REQUEST_URI'] ) ? (string) sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '/';

    $path = $uri;
    $qpos = strpos( $path, '?' );
    if ( false !== $qpos ) {
        $path = substr( $path, 0, $qpos );
    }

    $dir = rtrim( $path, '/' ) . '/';

    return $scheme . '://' . $host . $dir;
}

function aegis_mw_http_403( string $message ) : void {
    http_response_code( 403 );
    echo '<!doctype html><html><head><meta charset="utf-8"><title>403 Forbidden</title>';
    echo '<style>body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Ubuntu,Cantarell,Noto Sans,sans-serif;margin:24px;line-height:1.4}code{background:#f6f8fa;padding:6px 8px;border-radius:8px}</style>';
    echo '</head><body>';
    echo '<h1>403 - Authorization Required</h1>';
    echo '<p>' . esc_html( $message ) . '</p>';
    echo '</body></html>';
    exit;
}

require_once __DIR__ . DIRECTORY_SEPARATOR . 'lib.php';
require_once __DIR__ . DIRECTORY_SEPARATOR . 'engine.php';


function aegis_mw_ensure_config() : array {
    $mw_dir   = __DIR__;
    $abs_root = realpath( dirname( __DIR__ ) );
    if ( ! $abs_root ) {
        $abs_root = dirname( __DIR__ );
    }

    $aegis_mw_cfg_file = $mw_dir . DIRECTORY_SEPARATOR . 'config.php';
    if ( is_file( $aegis_mw_cfg_file ) ) {
        $aegis_mw_cfg = include $aegis_mw_cfg_file;
        return is_array( $aegis_mw_cfg ) ? $aegis_mw_cfg : array();
    }

    $wp_config = $abs_root . DIRECTORY_SEPARATOR . 'wp-config.php';
    $parsed    = function_exists( 'aegis_mw_parse_wp_config' ) ? aegis_mw_parse_wp_config( $wp_config ) : array();

    $aegis_mw_cfg = array(
        'DB_NAME'      => (string) ( $parsed['DB_NAME'] ?? '' ),
        'DB_USER'      => (string) ( $parsed['DB_USER'] ?? '' ),
        'DB_PASSWORD'  => (string) ( $parsed['DB_PASSWORD'] ?? '' ),
        'DB_HOST'      => (string) ( $parsed['DB_HOST'] ?? 'localhost' ),
        'table_prefix' => (string) ( $parsed['table_prefix'] ?? 'wp_' ),
    );

    $php = "<?php\n// Auto-generated by AegisBackup Migration Wizard runner.\n// This file contains DB connection details read from wp-config.php.\n\nreturn " . var_export( $aegis_mw_cfg, true ) . ";\n"; // phpcs:ignore 

    @file_put_contents( $aegis_mw_cfg_file, $php );
    @chmod( $aegis_mw_cfg_file, 0644 ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_chmod -- Standalone runner; WP_Filesystem is not loaded.

    return $aegis_mw_cfg;
}

$aegis_mw_cfg = aegis_mw_ensure_config();
$aegis_mw_base       = aegis_mw_base_url();
$aegis_mw_runner_url = $aegis_mw_base;
$aegis_mw_report_url = $aegis_mw_base . 'report/';
$aegis_mw_result = null;
$aegis_mw_method = isset( $_SERVER['REQUEST_METHOD'] ) ? strtoupper( (string) sanitize_key( wp_unslash( $_SERVER['REQUEST_METHOD'] ) ) ) : '';
if ( 'POST' === $aegis_mw_method && isset( $_POST['do_restore'] ) ) {
    // Nonce verification to protect against CSRF.
    $aegis_mw_nonce = isset( $_POST['_wpnonce'] ) ? (string) sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ) : '';
	if ( ! aegis_mw_wp_verify_nonce( $aegis_mw_nonce, 'aegis_mw_restore' ) ) {
		$aegis_mw_result = array(
			'ok'      => false,
			'message' => 'Security check failed. Please reload and try again.',
		);
	} else {
		$aegis_mw_result = aegis_mw_run();
	}
}

?><!doctype html>
<html>
<head>
    <meta charset="utf-8">
    <title>AegisBackup Migration Wizard Runner</title>
    <style>
        body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Ubuntu,Cantarell,Noto Sans,sans-serif;margin:20px;line-height:1.4}
        .wrap{max-width:980px}
        .notice{border:1px solid #e5e7eb;border-left-width:4px;border-radius:10px;padding:12px 14px;margin:14px 0;background:#fff}
        .notice-info{border-left-color:#2563eb}
        .notice-ok{border-left-color:#16a34a}
        .notice-fail{border-left-color:#b00020}
        code,pre{background:#f6f8fa;padding:8px;border-radius:8px;display:block;overflow:auto}
        label{font-weight:600;display:block;margin-top:12px}
        input[type="text"], input[type="url"]{width:100%;max-width:720px;padding:8px;border:1px solid #d1d5db;border-radius:10px}
        input[type="file"]{margin-top:6px}
        .row{margin:10px 0}
        .btn{display:inline-block;background:#2563eb;color:#fff;border:0;border-radius:10px;padding:10px 14px;font-weight:600;cursor:pointer}
        .btn:disabled{opacity:.55;cursor:not-allowed}
		.btn-secondary{display:inline-block;background:#111827;color:#fff;border:0;border-radius:10px;padding:10px 14px;font-weight:600;cursor:pointer}
        .btn-secondary:disabled{opacity:.55;cursor:not-allowed}		
        .muted{color:#6b7280}
        .inline{display:flex;gap:14px;flex-wrap:wrap}
        .inline > div{flex:1 1 320px}

        :root{--aegis-mw-hero-height:100%}
        .hero-placeholder{width:100%;height:var(--aegis-mw-hero-height);border:2px dashed #d1d5db;border-radius:12px;background:#f9fafb;display:flex;align-items:center;justify-content:flex-start;padding:0 14px;color:#6b7280;box-sizing:border-box;margin:10px 0 18px}
        .upload-progress{display:none;margin-top:12px}
        .upload-progress .bar{height:10px;background:#e5e7eb;border-radius:999px;overflow:hidden}
        .upload-progress .bar > div{height:100%;width:0%;background:#2563eb}
        .upload-progress .meta{display:flex;justify-content:space-between;gap:10px;font-size:13px;color:#6b7280;margin-top:6px}
        .upload-progress .meta strong{color:#111827}
		.hero-image{
			width:650px;
			height:var(--aegis-mw-hero-height);
			object-fit:cover;
			display:block;
			margin:10px 0 18px;
			border-radius:12px;
		}

    </style>
</head>
<body>
<div class="wrap">
    <h1>Migration Wizard</h1>

    <img src="transfer.png" alt="Aegisify Transfer" class="hero-image" />

    <div class="notice notice-info">
        <p class="muted" style="margin:0;">
            This runner restores a Full Backup (DR) package into this WordPress install without loading WordPress during the database overwrite.
        </p>
    </div>

    <h2>Important URLs</h2>
    <div class="notice">
        <div><strong>Runner URL (use this):</strong> <code><?php echo esc_url( $aegis_mw_runner_url ); ?></code></div>
        <div><strong>Report URL:</strong> <code><?php echo esc_url( $aegis_mw_report_url ); ?></code></div>
    </div>

    <?php if ( is_array( $aegis_mw_result ) ) : ?>
        <?php $aegis_mw_ok = ! empty( $aegis_mw_result['ok'] ); ?>
        <div class="notice <?php echo $aegis_mw_ok ? 'notice-ok' : 'notice-fail'; ?>">
            <strong><?php echo $aegis_mw_ok ? 'SUCCESS' : 'FAILED'; ?></strong>
            <div><?php echo esc_html( (string) ( $aegis_mw_result['message'] ?? '' ) ); ?></div>
            <div style="margin-top:10px;">
                <a class="btn btn-secondary" href="<?php echo esc_url( $aegis_mw_report_url ); ?>">View Report</a>
            </div>
        </div>

        <?php if ( ! empty( $aegis_mw_result['log'] ) && is_array( $aegis_mw_result['log'] ) ) : ?>
            <h3>Log</h3>
            <pre><?php echo esc_html( implode( "\n", $aegis_mw_result['log'] ) ); ?></pre>
        <?php endif; ?>
    <?php endif; ?>

    <h2>Restore a Full Backup (DR) package</h2>

    <form method="post" enctype="multipart/form-data">
        <input type="hidden" name="do_restore" value="1"/>
        <?php aegis_mw_wp_nonce_field( 'aegis_mw_restore' ); ?>

        <label for="package_zip">Package ZIP</label>
        <input id="package_zip" type="file" name="package_zip" accept=".zip" required />

        <div class="upload-progress" id="upload_progress" aria-live="polite">
            <div class="bar"><div id="upload_progress_bar"></div></div>
            <div class="meta">
                <div>Upload: <strong><span id="upload_progress_pct">0</span>%</strong></div>
                <div id="upload_progress_status">Waiting…</div>
            </div>
        </div>


        <div class="row">
            <label style="font-weight:600;">
                <input type="checkbox" name="drop_dest_tables" value="1"/>
                Drop destination tables with this site's prefix before import (destructive)
            </label>
            <div class="muted">Use this if you're restoring into a clean install and want to ensure no leftover tables remain.</div>
        </div>

        <div class="inline">
            <div>
                <label for="source_url">Source Site URL</label>
                <input id="source_url" type="url" name="source_url" placeholder="https://source-site.com" />
            </div>
            <div>
                <label for="temp_url">Temporary / Destination URL (optional)</label>
                <input id="temp_url" type="url" name="temp_url" placeholder="https://destination-site.com" />
            </div>
        </div>

        <div class="inline">
            <div>
                <label for="source_prefix">Source DB Prefix (optional)</label>
                <input id="source_prefix" type="text" name="source_prefix" placeholder="wp_" />
            </div>
            <div>
                <label for="dest_prefix">Destination DB Prefix (optional)</label>
                <input id="dest_prefix" type="text" name="dest_prefix" placeholder="<?php echo esc_attr( (string) ( $aegis_mw_cfg['table_prefix'] ?? 'wp_' ) ); ?>" />
            </div>
        </div>

        <p style="margin-top:16px;">
            <button class="btn" type="submit">Run Restore</button>
            <a class="btn btn-secondary" href="<?php echo esc_url( $aegis_mw_report_url ); ?>">Open Report</a>
        </p>

        <p class="muted">
            Tip: After restore completes, log into <code>/wp-admin</code> and verify site settings, permalinks, and plugins.
        </p>
    </form>
</div>

<script>
(function(){
    var form = document.querySelector('form');
    if(!form) return;

    var fileInput = document.getElementById('package_zip');
    var progressWrap = document.getElementById('upload_progress');
    var bar = document.getElementById('upload_progress_bar');
    var pct = document.getElementById('upload_progress_pct');
    var statusEl = document.getElementById('upload_progress_status');
    var submitBtn = form.querySelector('button[type="submit"]');
    if(fileInput){
        fileInput.addEventListener('change', function(){
            if(!progressWrap) return;
            if(fileInput.files && fileInput.files.length){
                progressWrap.style.display = 'block';
                if(bar) bar.style.width = '0%';
                if(pct) pct.textContent = '0';
                if(statusEl) statusEl.textContent = 'Ready to upload';
            } else {
                progressWrap.style.display = 'none';
            }
        });
    }

    var xhrTest = null;
    try { xhrTest = new XMLHttpRequest(); } catch(e) {}
    if(!xhrTest || !('upload' in xhrTest) || !window.FormData) return;

    form.addEventListener('submit', function(e){
        if(fileInput && (!fileInput.files || !fileInput.files.length)){
            return;
        }

        e.preventDefault();

        if(progressWrap) progressWrap.style.display = 'block';
        if(statusEl) statusEl.textContent = 'Uploading…';
        if(submitBtn) submitBtn.disabled = true;

        var xhr = new XMLHttpRequest();
        xhr.open('POST', form.getAttribute('action') || window.location.href, true);

        xhr.upload.addEventListener('progress', function(evt){
            if(!evt.lengthComputable) return;
            var percent = Math.round((evt.loaded / evt.total) * 100);
            if(percent < 0) percent = 0;
            if(percent > 100) percent = 100;

            if(bar) bar.style.width = percent + '%';
            if(pct) pct.textContent = String(percent);
        });

        xhr.addEventListener('load', function(){
            if(statusEl) statusEl.textContent = 'Upload complete. Processing…';
            if(bar) bar.style.width = '100%';
            if(pct) pct.textContent = '100';

            var html = xhr.responseText || '';
            if(html.trim().length){
                document.open();
                document.write(html);
                document.close();
            } else {
                window.location.reload();
            }
        });

        xhr.addEventListener('error', function(){
            if(statusEl) statusEl.textContent = 'Upload failed (network error).';
            if(submitBtn) submitBtn.disabled = false;
        });

        xhr.addEventListener('abort', function(){
            if(statusEl) statusEl.textContent = 'Upload cancelled.';
            if(submitBtn) submitBtn.disabled = false;
        });

        var fd = new FormData(form);
        xhr.send(fd);
    });
})();
</script>

</body>
</html>