<?php
declare( strict_types=1 );

// Prevent direct access if someone hits this file directly in a WP context.
// This library is meant to be included by engine.php.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Minimal polyfills when WordPress is available but not fully loaded (defensive).
if ( ! function_exists( 'wp_unslash' ) ) {
    function wp_unslash( $value ) {
        return is_array( $value ) ? array_map( 'wp_unslash', $value ) : stripslashes( (string) $value );
    }
}
if ( ! function_exists( 'sanitize_text_field' ) ) {
    function sanitize_text_field( $str ) : string {
        $str = (string) $str;
        $str = preg_replace( '/[\r\n\t\0\x0B]+/', ' ', $str );
        // Plugin Check prefers wp_strip_all_tags(), but this file must also run standalone.
        $str = preg_replace( '/<[^>]*?>/', '', $str );
        return trim( $str );
    }
}
if ( ! function_exists( 'wp_mkdir_p' ) ) {
    function wp_mkdir_p( $target ) : bool {
        $target = (string) $target;
        if ( '' === $target ) {
            return false;
        }
        if ( is_dir( $target ) ) {
            return true;
        }
        return @mkdir( $target, 0755, true ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_mkdir
    }
}
if ( ! function_exists( 'wp_delete_file' ) ) {
    function wp_delete_file( $file ) : bool {
        $file = (string) $file;
        if ( '' === $file || ! is_file( $file ) ) {
            return false;
        }
        return @unlink( $file ); // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink
    }
}

/**
 * Delete a directory tree using WP_Filesystem when available, otherwise PHP fallbacks.
 */
function aegis_mw_fs_delete_dir( string $dir ) : bool {
    if ( function_exists( 'WP_Filesystem' ) ) {
        global $wp_filesystem;
        if ( ! $wp_filesystem && defined( 'ABSPATH' ) && is_dir( ABSPATH . 'wp-admin/includes' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
        }
        if ( $wp_filesystem ) {
            return (bool) $wp_filesystem->delete( $dir, true, 'd' );
        }
    }
    return false;
}

/**
 * Small helper library for the standalone Migration Wizard.
 *
 * IMPORTANT: This file must not rely on WordPress being loaded.
 */

function aegis_mw_log( array &$log, string $m ) : void {
    $log[] = '[' . gmdate( 'c' ) . '] ' . $m;
}

function aegis_mw_rmdir( string $dir ) : void {
    if ( ! is_dir( $dir ) ) {
        return;
    }
    $items = @scandir( $dir );
    if ( ! is_array( $items ) ) {
        return;
    }
    foreach ( $items as $it ) {
        if ( '.' === $it || '..' === $it ) {
            continue;
        }
        $p = $dir . DIRECTORY_SEPARATOR . $it;
        if ( is_dir( $p ) ) {
            aegis_mw_rmdir( $p );
        } else {
            wp_delete_file( $p );
        }
    }
    if ( ! aegis_mw_fs_delete_dir( $dir ) ) {
        @rmdir( $dir ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_rmdir
    }
}

function aegis_mw_copydir( string $src, string $dst, array $opts = array() ) : void {
    $skip_prefixes = isset( $opts['skip_prefixes'] ) && is_array( $opts['skip_prefixes'] ) ? $opts['skip_prefixes'] : array();
    $src = rtrim( $src, '/\\' );
    $dst = rtrim( $dst, '/\\' );
    if ( ! is_dir( $src ) ) {
        return;
    }
    if ( ! is_dir( $dst ) ) {
        wp_mkdir_p( $dst );
    }
    $items = @scandir( $src );
    if ( ! is_array( $items ) ) {
        return;
    }
    foreach ( $items as $it ) {
        if ( '.' === $it || '..' === $it ) {
            continue;
        }
        $from = $src . DIRECTORY_SEPARATOR . $it;
        $to   = $dst . DIRECTORY_SEPARATOR . $it;

        $rel = str_replace( array( '\\', $src ), array( '/', '' ), $from );
        $rel = ltrim( str_replace( '\\', '/', $rel ), '/' );

        $skip = false;
        foreach ( $skip_prefixes as $pref ) {
            $pref = (string) $pref;
            if ( '' !== $pref && 0 === strpos( $rel, $pref ) ) {
                $skip = true;
                break;
            }
        }
        if ( $skip ) {
            continue;
        }

        if ( is_dir( $from ) ) {
            aegis_mw_copydir( $from, $to, $opts );
        } else {
            @copy( $from, $to );
        }
    }
}

/**
 * Back-compat alias.
 * Some runner builds referenced aegis_mw_copy_dir().
 */
function aegis_mw_copy_dir( string $src, string $dst, array $skip = array(), array &$log = array() ) : void {
    $opts = array( 'skip_prefixes' => $skip );
    aegis_mw_copydir( $src, $dst, $opts );
    if ( is_array( $log ) ) {
        $log[] = '[copy] ' . $src . ' -> ' . $dst;
    }
}

function aegis_mw_parse_wp_config( string $config_path ) : array {
    $out = array(
        'DB_NAME' => '',
        'DB_USER' => '',
        'DB_PASSWORD' => '',
        'DB_HOST' => 'localhost',
        'table_prefix' => 'wp_',
    );
    if ( ! is_file( $config_path ) ) {
        return $out;
    }
    $c = @file_get_contents( $config_path );
    if ( ! is_string( $c ) || '' === $c ) {
        return $out;
    }

    foreach ( array( 'DB_NAME', 'DB_USER', 'DB_PASSWORD', 'DB_HOST' ) as $k ) {
        if ( preg_match( "/define\\(\\s*'" . preg_quote( $k, '/' ) . "'\\s*,\\s*'([^']*)'\\s*\\)\\s*;/", $c, $m ) ) {
            $out[ $k ] = (string) $m[1];
        }
    }
    if ( preg_match( '/\$table_prefix\s*=\s*\'([^\']+)\'\s*;/', $c, $m ) ) {
        $out['table_prefix'] = (string) $m[1];
    }
    return $out;
}

function aegis_mw_db_connect( array $cfg ) {
    // phpcs:ignore WordPress.DB.RestrictedClasses.mysql__mysqli -- Standalone runner may execute without WP loaded.
    $mysqli = @new mysqli(
        (string) ( $cfg['DB_HOST'] ?? 'localhost' ),
        (string) ( $cfg['DB_USER'] ?? '' ),
        (string) ( $cfg['DB_PASSWORD'] ?? '' ),
        (string) ( $cfg['DB_NAME'] ?? '' )
    );
    if ( $mysqli && empty( $mysqli->connect_errno ) ) {
        $mysqli->set_charset( 'utf8mb4' );
    }
    return $mysqli;
}

function aegis_mw_is_serialized( $data ) : bool {
    if ( ! is_string( $data ) ) {
        return false;
    }
    $data = trim( $data );
    if ( '' === $data ) {
        return false;
    }
    if ( 'N;' === $data ) {
        return true;
    }
    if ( ! preg_match( '/^[adObis]:/', $data ) ) {
        return false;
    }
    return false !== @unserialize( $data );
}

function aegis_mw_recursive_replace( $data, string $search, string $replace ) {
    if ( is_string( $data ) ) {
        return str_replace( $search, $replace, $data );
    }
    if ( is_array( $data ) ) {
        foreach ( $data as $k => $v ) {
            $data[ $k ] = aegis_mw_recursive_replace( $v, $search, $replace );
        }
        return $data;
    }
    if ( is_object( $data ) ) {
        foreach ( get_object_vars( $data ) as $k => $v ) {
            $data->$k = aegis_mw_recursive_replace( $v, $search, $replace );
        }
        return $data;
    }
    return $data;
}

function aegis_mw_serialized_safe_replace( $value, string $search, string $replace ) {
    if ( ! is_string( $value ) ) {
        return $value;
    }
    if ( ! aegis_mw_is_serialized( $value ) ) {
        return str_replace( $search, $replace, $value );
    }
    $un = @unserialize( $value );
    if ( false === $un && 'b:0;' !== $value ) {
        // Fallback.
        return str_replace( $search, $replace, $value );
    }
    $un = aegis_mw_recursive_replace( $un, $search, $replace );
    return serialize( $un );
}

/**
 * Reporting
 * Writes JSON reports to /aegismw/report so users can review success/fail per phase.
 */

function aegis_mw_report_dir( string $mw_dir ) : string {
    $d = rtrim( $mw_dir, '/\\' ) . DIRECTORY_SEPARATOR . 'report';
    if ( ! is_dir( $d ) ) {
        wp_mkdir_p( $d );
    }
    return $d;
}

function aegis_mw_report_init( string $mw_dir, string $run_id ) : array {
    $now = gmdate( 'c' );
    return array(
        'run_id'   => $run_id,
        'started'  => $now,
        'ended'    => '',
        'ok'       => false,
        'message'  => '',
        'phases'   => array(),
        'log'      => array(),
        'meta'     => array(
            'php' => PHP_VERSION,
        ),
    );
}

function aegis_mw_report_phase_start( array &$report, string $key, string $label ) : void {
    $report['phases'][ $key ] = array(
        'label'   => $label,
        'started' => gmdate( 'c' ),
        'ended'   => '',
        'ok'      => false,
        'message' => '',
        'errors'  => array(),
        'warnings'=> array(),
    );
}

function aegis_mw_report_phase_end( array &$report, string $key, bool $ok, string $message = '', array $errors = array(), array $warnings = array() ) : void {
    if ( empty( $report['phases'][ $key ] ) || ! is_array( $report['phases'][ $key ] ) ) {
        return;
    }
    $report['phases'][ $key ]['ended']   = gmdate( 'c' );
    $report['phases'][ $key ]['ok']      = (bool) $ok;
    $report['phases'][ $key ]['message'] = (string) $message;
    $report['phases'][ $key ]['errors']  = is_array( $errors ) ? $errors : array();
    $report['phases'][ $key ]['warnings']= is_array( $warnings ) ? $warnings : array();
}

function aegis_mw_report_write( string $mw_dir, array $report ) : void {
    $dir = aegis_mw_report_dir( $mw_dir );
    $json = json_encode( $report, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES );
    if ( ! is_string( $json ) ) {
        $json = '{}';
    }
    $run_file = $dir . DIRECTORY_SEPARATOR . $report['run_id'] . '.json';
    @file_put_contents( $run_file, $json );
    @file_put_contents( $dir . DIRECTORY_SEPARATOR . 'latest.json', $json );
}

function aegis_mw_report_finalize( string $mw_dir, array &$report, bool $ok, string $message ) : void {
    $report['ended']   = gmdate( 'c' );
    $report['ok']      = (bool) $ok;
    $report['message'] = (string) $message;
    aegis_mw_report_write( $mw_dir, $report );
}

/**
 * DB option helpers (no WP bootstrap)
 */

function aegis_mw_db_get_option( mysqli $db, string $prefix, string $name ) : string {
    $opt = $prefix . 'options';
    $name_esc = $db->real_escape_string( $name );
    $res = @$db->query( "SELECT option_value FROM `{$opt}` WHERE option_name='{$name_esc}' LIMIT 1" );
    if ( $res && ( $row = $res->fetch_assoc() ) && isset( $row['option_value'] ) ) {
        return (string) $row['option_value'];
    }
    return '';
}

function aegis_mw_db_set_option( mysqli $db, string $prefix, string $name, string $value ) : bool {
    $opt = $prefix . 'options';
    $name_esc  = $db->real_escape_string( $name );
    $value_esc = $db->real_escape_string( $value );
    // Upsert-ish: update then insert if missing.
    @$db->query( "UPDATE `{$opt}` SET option_value='{$value_esc}' WHERE option_name='{$name_esc}'" );
    if ( 0 === (int) $db->affected_rows ) {
        return (bool) @$db->query( "INSERT INTO `{$opt}` (option_name, option_value, autoload) VALUES ('{$name_esc}', '{$value_esc}', 'yes')" );
    }
    return true;
}

function aegis_mw_current_base_url() : string {
    // Sanitize immediately when assigning superglobal values (Plugin Check requirement).
    $host = isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : '';
    $host = preg_replace( '/[^a-zA-Z0-9\.\-:\[\]]/', '', $host );
    if ( '' === $host ) {
        return '';
    }
    $https_raw = isset( $_SERVER['HTTPS'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTPS'] ) ) : '';
    $https     = ( '' !== $https_raw ) && ( 'off' !== strtolower( (string) $https_raw ) );
    $scheme = $https ? 'https://' : 'http://';
    return rtrim( $scheme . $host, '/' );
}

/**
 * Post-restore fixer: attempt to bootstrap WP (plugins disabled) to flush rewrite rules and send email.
 */
function aegis_mw_wp_post_fixer( string $abs_root, array $cfg, string $prefix, array &$log, array &$report ) : void {
    $wp_load = rtrim( $abs_root, '/\\' ) . DIRECTORY_SEPARATOR . 'wp-load.php';
    if ( ! is_file( $wp_load ) ) {
        aegis_mw_log( $log, 'Post-fixer: wp-load.php not found; skipped.' );
        return;
    }

    // Avoid theme output.
    if ( ! defined( 'WP_USE_THEMES' ) ) {
        define( 'WP_USE_THEMES', false ); // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedConstantFound
    }

    // Prevent fatal output from breaking runner response.
    $fatal = '';
    $prev = set_error_handler( // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_set_error_handler -- Used to capture errors during standalone post-fixer.
        static function( $errno, $errstr, $errfile, $errline ) use ( &$fatal ) {
            $fatal = "PHP error {$errno}: {$errstr} @ {$errfile}:{$errline}";
            return false;
        }
    );

    try {
        require_once $wp_load;

        if ( function_exists( 'flush_rewrite_rules' ) ) {
            flush_rewrite_rules( false );
            aegis_mw_log( $log, 'Post-fixer: rewrite rules flushed.' );
        }

        // Email admin.
        $admin_email = '';
        if ( function_exists( 'get_option' ) ) {
            $admin_email = (string) get_option( 'admin_email', '' );
        }
        if ( ! $admin_email ) {
            // Fallback to DB.
            $db = aegis_mw_db_connect( $cfg );
            if ( $db ) {
                $admin_email = aegis_mw_db_get_option( $db, $prefix, 'admin_email' );
            }
        }

        $subject = 'AegisBackup Migration Wizard: Restore Completed';
        $body    = "Your restore run ({$report['run_id']}) has completed.\n\nStatus: " . ( $report['ok'] ? 'SUCCESS' : 'FAIL' ) . "\nMessage: {$report['message']}\n\nYou can view the full report here:\n" . aegis_mw_current_base_url() . "/aegismw/report/\n";
        $sent = false;

        if ( $admin_email ) {
            if ( function_exists( 'wp_mail' ) ) {
                $sent = (bool) wp_mail( $admin_email, $subject, $body );
            } else {
                $sent = (bool) @mail( $admin_email, $subject, $body );
            }
        }

        aegis_mw_log( $log, 'Post-fixer: completion email ' . ( $sent ? 'sent' : 'not sent' ) . ( $admin_email ? ' to ' . $admin_email : ' (admin_email unavailable)' ) . '.' );
    } catch ( Throwable $e ) { // phpcs:ignore PHPCompatibility
        aegis_mw_log( $log, 'Post-fixer failed: ' . $e->getMessage() );
    } finally {
        if ( $prev ) {
            restore_error_handler();
        }
    }
}
