/* global jQuery, AegisBackupPPUpload */
(function($){
	'use strict';

	function fmtBytes(bytes){
		bytes = parseInt(bytes, 10) || 0;
		var units = ['B','KB','MB','GB','TB'];
		var i = 0;
		while(bytes >= 1024 && i < units.length-1){
			bytes = bytes / 1024;
			i++;
		}
		return (i === 0 ? bytes : bytes.toFixed(1)) + ' ' + units[i];
	}

	function openUploaderPopup(target){
		var w = window.open('', 'ab_pp_uploader_' + target, 'width=560,height=320,resizable=yes,scrollbars=yes');
		if(!w){
			alert('Popup blocked. Please allow popups for this site to upload large backups.');
			return;
		}

		var title = (target === 'db') ? 'Database Backup Upload' : 'Files & Folders Backup Upload';
		var doc = w.document;
		doc.open();
		doc.write('<!doctype html><html><head><meta charset="utf-8" />');
		doc.write('<title>' + title.replace(/</g,'&lt;') + '</title>');
		doc.write('<style>body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;margin:16px;} .row{margin:12px 0;} .bar{width:100%;height:14px;background:#e5e5e5;border-radius:999px;overflow:hidden;} .bar>div{height:14px;width:0%;background:#2271b1;} .muted{color:#666;font-size:12px;} .btn{display:inline-block;padding:6px 10px;border:1px solid #c3c4c7;border-radius:4px;background:#f6f7f7;cursor:pointer;} .btn-primary{background:#2271b1;border-color:#2271b1;color:#fff;} .btn:disabled{opacity:.6;cursor:not-allowed;} .mono{font-family:ui-monospace,SFMono-Regular,Menlo,Consolas,monospace;font-size:12px;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;max-width:520px;display:block;} </style>');
		doc.write('</head><body>');
		doc.write('<h2 style="margin:0 0 6px 0;">' + title.replace(/</g,'&lt;') + '</h2>');
		doc.write('<div class="muted">Large ZIP uploads are sent in chunks with progress. Keep this window open until completed.</div>');
		doc.write('<div class="row"><input type="file" id="ab_file" accept=".zip,application/zip" /></div>');
		doc.write('<div class="row"><span class="mono" id="ab_name"></span></div>');
		doc.write('<div class="row"><div class="bar"><div id="ab_bar"></div></div></div>');
		doc.write('<div class="row muted" id="ab_status">Waiting for a .zip file...</div>');
		doc.write('<div class="row" style="display:flex;gap:8px;align-items:center;">');
		doc.write('<button class="btn btn-primary" id="ab_start" disabled>Start Upload</button>');
		doc.write('<button class="btn" id="ab_close">Close</button>');
		doc.write('</div>');
		doc.write('<script>window.__AB_TARGET__=' + JSON.stringify(target) + ';</script>');
		doc.write('</body></html>');
		doc.close();

		// Attach logic after DOM ready.
		w.addEventListener('load', function(){
			try{
				// Ensure popup has access to the parent window's localized vars + jQuery.
				if(typeof w.AegisBackupPPUpload === 'undefined' && w.opener && w.opener.AegisBackupPPUpload){
					w.AegisBackupPPUpload = w.opener.AegisBackupPPUpload;
				}
				if(typeof w.jQuery === 'undefined' && w.opener && w.opener.jQuery){
					w.jQuery = w.opener.jQuery;
				}
				var fileInput = w.document.getElementById('ab_file');
				var startBtn = w.document.getElementById('ab_start');
				var closeBtn = w.document.getElementById('ab_close');
				var nameEl = w.document.getElementById('ab_name');
				var barEl = w.document.getElementById('ab_bar');
				var statusEl = w.document.getElementById('ab_status');

				var selectedFile = null;
				fileInput.addEventListener('change', function(){
					selectedFile = fileInput.files && fileInput.files[0] ? fileInput.files[0] : null;
					if(!selectedFile){
						startBtn.disabled = true;
						nameEl.textContent = '';
						statusEl.textContent = 'Waiting for a .zip file...';
						return;
					}
					nameEl.textContent = selectedFile.name + ' — ' + fmtBytes(selectedFile.size);
					startBtn.disabled = false;
					statusEl.textContent = 'Ready to upload.';
				});

				closeBtn.addEventListener('click', function(){
					w.close();
				});

				startBtn.addEventListener('click', function(){
					if(!selectedFile){return;}
					startBtn.disabled = true;
					fileInput.disabled = true;
					statusEl.textContent = 'Initializing upload...';
					uploadFile(selectedFile);
				});

				function postToOpener(payload){
					try{
						if(w.opener && !w.opener.closed){
							w.opener.postMessage(payload, '*');
						}
					}catch(e){}
				}

				function setProgress(pct, text){
					pct = Math.max(0, Math.min(100, parseInt(pct,10)||0));
					barEl.style.width = pct + '%';
					statusEl.textContent = text || ('Uploading... ' + pct + '%');
					postToOpener({type:'ab_pp_upload_progress', target: target, progress: pct, text: statusEl.textContent});
				}

				function ajax(action, data, fileField){
					return new Promise(function(resolve, reject){
						var cfg = (typeof w.AegisBackupPPUpload !== 'undefined' && w.AegisBackupPPUpload) ? w.AegisBackupPPUpload : ((w.opener && w.opener.AegisBackupPPUpload) ? w.opener.AegisBackupPPUpload : null);
						var jq = (typeof w.jQuery !== 'undefined' && w.jQuery) ? w.jQuery : ((w.opener && w.opener.jQuery) ? w.opener.jQuery : null);
						if(!jq || !jq.ajax){
							reject('Upload error: jQuery.ajax not available in uploader window');
							return;
						}
						var fd = new w.FormData();
						fd.append('action', action);
						fd.append('nonce', (cfg && cfg.nonce) ? cfg.nonce : '');
						Object.keys(data||{}).forEach(function(k){ fd.append(k, data[k]); });
						if(fileField){
							fd.append('chunk', fileField, 'chunk.bin');
						}
						jq.ajax({
							url: (cfg && cfg.ajaxurl) ? cfg.ajaxurl : '',
							type: 'POST',
							data: fd,
							processData: false,
							contentType: false,
							success: function(resp){
								if(resp && resp.success){
									resolve(resp.data||{});
								}else{
									reject((resp && resp.data && resp.data.message) ? resp.data.message : 'Upload error');
								}
							},
							error: function(xhr){
								reject('HTTP ' + (xhr && xhr.status ? xhr.status : 'error'));
							}
						});
					});
				}

				async function uploadFile(file){
					try{
						var cfg2 = (typeof w.AegisBackupPPUpload !== 'undefined' && w.AegisBackupPPUpload) ? w.AegisBackupPPUpload : ((w.opener && w.opener.AegisBackupPPUpload) ? w.opener.AegisBackupPPUpload : null);
							var chunkSize = (cfg2 && cfg2.chunksize) ? parseInt(cfg2.chunksize,10) : 5242880;
						chunkSize = Math.max(1048576, chunkSize);
						var totalChunks = Math.ceil(file.size / chunkSize);

						var init = await ajax('aegisbackup_pp_resumable_upload_init', {
							target: target,
							filename: file.name,
							filesize: file.size
						});
						var uploadId = init.upload_id;
						setProgress(0, 'Uploading... 0%');

						for(var i=0;i<totalChunks;i++){
							var start = i * chunkSize;
							var end = Math.min(file.size, start + chunkSize);
							var blob = file.slice(start, end);
							await ajax('aegisbackup_pp_resumable_upload_chunk', {
								upload_id: uploadId,
								chunk_index: i,
								total_chunks: totalChunks
							}, blob);
							var pct = Math.floor(((i+1)/totalChunks)*100);
							setProgress(pct, 'Uploading... ' + pct + '%');
						}

						setProgress(100, 'Finalizing...');
						var fin = await ajax('aegisbackup_pp_resumable_upload_finalize', { upload_id: uploadId });
						setProgress(100, fin.message || 'Upload complete.');
						postToOpener({type:'ab_pp_upload_done', target: target, message: fin.message || 'Upload complete.'});
						startBtn.disabled = true;
						statusEl.textContent = (fin.message || 'Upload complete.') + ' You can close this window.';
					}catch(err){
						barEl.style.width = '0%';
						statusEl.textContent = 'Error: ' + err;
						postToOpener({type:'ab_pp_upload_error', target: target, message: String(err)});
						startBtn.disabled = false;
						fileInput.disabled = false;
					}
				}

			}catch(e){
				// no-op
			}
		});
	}

	function setInlineStatus(target, text, pct){
		var id = (target === 'db') ? '#ab_pp_upload_status_db' : '#ab_pp_upload_status_file';
		var $el = $(id);
		if(!$el.length){return;}
		var safeText = String(text || '');
		var percent = (typeof pct !== 'undefined' && pct !== null) ? parseInt(pct,10) : null;
		var html = '';
		if(percent !== null && !isNaN(percent)){
			html += '<div style="display:flex;align-items:center;gap:10px;">';
			html += '<div style="flex:1;height:10px;background:#e5e5e5;border-radius:999px;overflow:hidden;">';
			html += '<div style="height:10px;width:' + Math.max(0,Math.min(100,percent)) + '%;background:#2271b1;"></div>';
			html += '</div>';
			html += '<div style="min-width:42px;text-align:right;">' + percent + '%</div>';
			html += '</div>';
		}
		if(safeText){
			html += '<div style="margin-top:6px;color:#646970;">' + $('<div/>').text(safeText).html() + '</div>';
		}
		$el.html(html).show();
	}

	$(function(){
		// Button opens background popup uploader.
		$(document).on('click', '.ab-pp-upload-btn', function(e){
			e.preventDefault();
			var target = $(this).data('ab-upload-target');
			setInlineStatus(target, 'Upload started in a separate window. You can keep working here while it uploads.', 0);
			openUploaderPopup(target);
		});

		// Receive updates from popup.
		window.addEventListener('message', function(ev){
			if(!ev || !ev.data || !ev.data.type){return;}
			var d = ev.data;
			if(d.type === 'ab_pp_upload_progress'){
				setInlineStatus(d.target, d.text || 'Uploading...', d.progress);
			}
			if(d.type === 'ab_pp_upload_done'){
				setInlineStatus(d.target, d.message || 'Upload complete. Refresh to see it listed.', 100);
				// Soft refresh after a moment to show uploaded packages.
				setTimeout(function(){
					try{ window.location.reload(); }catch(e){}
				}, 1200);
			}
			if(d.type === 'ab_pp_upload_error'){
				setInlineStatus(d.target, 'Upload failed: ' + (d.message || 'Unknown error'), null);
			}
		});
	});

})(jQuery);
