<?php
namespace AegisBackup\Admin\Pages;

use AegisBackup\AB_Plugin;
use AegisBackup\Backup\AB_Backup_Manager;
use AegisBackup\Admin\Panels\AB_Panel_DB_Tools;

defined( 'ABSPATH' ) || exit;

class AB_Page_Backups {
    protected $plugin;

    public function __construct( AB_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function render() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'aegisbackup' ) );
        }

        $tab_raw = filter_input( INPUT_GET, 'tab', FILTER_SANITIZE_FULL_SPECIAL_CHARS );
        $tab = $tab_raw ? sanitize_key( (string) $tab_raw ) : 'migration';
        $ab_msg_raw = filter_input( INPUT_GET, 'ab_msg', FILTER_SANITIZE_FULL_SPECIAL_CHARS );
        $ab_msg = $ab_msg_raw ? sanitize_key( (string) $ab_msg_raw ) : '';
        $tabs = array(
            'migration'  => __( 'Full Backup (DR)', 'aegisbackup' ),
            'db'         => __( 'Database Backups', 'aegisbackup' ),
            'files'      => __( 'File Backups', 'aegisbackup' ),
        );

        if ( ! isset( $tabs[ $tab ] ) ) {
            $tab = 'migration';
        }

        ?>
        <div class="wrap aegisbackup-wrap">
            <h1><?php esc_html_e( 'AegisBackup — Backups', 'aegisbackup' ); ?></h1>

            <h2 class="nav-tab-wrapper">
                <?php foreach ( $tabs as $slug => $label ) : ?>
                    <?php
                    $url = add_query_arg(
                        array(
                            'page' => 'aegisbackup',
                            'tab'  => $slug,
                        ),
                        admin_url( 'admin.php' )
                    );
                    $active = ( $slug === $tab ) ? 'nav-tab-active' : '';
                    ?>
                    <a href="<?php echo esc_url( $url ); ?>" class="nav-tab <?php echo esc_attr( $active ); ?>">
                        <?php echo esc_html( $label ); ?>
                    </a>
                <?php endforeach; ?>
            </h2>

            <?php
            $is_pro = false;
            if ( isset( $this->plugin ) && isset( $this->plugin->license ) && is_object( $this->plugin->license ) && method_exists( $this->plugin->license, 'is_pro_active' ) ) {
                $is_pro = (bool) $this->plugin->license->is_pro_active();
            }
            $license_url = admin_url( 'admin.php?page=aegisbackup-license' );

            if ( 'migration' === $tab && ! $is_pro ) :
                ?>
                <div class="notice notice-warning" style="margin:12px 0 16px;">
                    <p style="margin:0;">
                        <strong><?php esc_html_e( 'PRO Feature:', 'aegisbackup' ); ?></strong>
                        <?php esc_html_e( 'Full Backup (DR), scheduling, DR links and more is available in AegisBackup Pro. Upgrade to unlock Disaster Recovery packaging + fast restores.', 'aegisbackup' ); ?>
                        <a href="<?php echo esc_url( $license_url ); ?>" class="button button-primary" style="margin-left:10px;">
                            <?php esc_html_e( 'Upgrade / Activate License', 'aegisbackup' ); ?>
                        </a>
                    </p>
                </div>
                <style>
                    .aegisbackup-pro-locked{
                        opacity:0.45 !important;
                        pointer-events:none !important;
                        filter:grayscale(1) !important;
                        user-select:none !important;
                    }
                </style>
                <?php
            endif;

            switch ( $tab ) {
                case 'db':
                    $this->render_db_backups();
                    break;
                case 'files':
                    $this->render_file_backups();
                    break;
                case 'wordpress':
                    $this->render_wordpress_backups();
                    break;
                case 'migration':
                default:
                    $this->render_migration_package();
                    break;
            }
            ?>
        </div>
        <?php
    }

    protected function render_migration_package() {
        $mgr = new AB_Backup_Manager();
        $list = $mgr->list_packages( 5, true );
        $dr_base   = $mgr->get_dr_base_dir();
        $dr_tokens = $mgr->get_dr_tokens_file();
        $dr_rows   = $mgr->list_dr_packages( 0 );

        $tokens = array();
        if ( is_file( $dr_tokens ) ) {
            $raw = @file_get_contents( $dr_tokens );
            $tmp = json_decode( (string) $raw, true );
            if ( is_array( $tmp ) ) {
                $tokens = $tmp;
            }
        }

        $is_pro = false;
        if ( isset( $this->plugin ) && isset( $this->plugin->license ) && is_object( $this->plugin->license ) && method_exists( $this->plugin->license, 'is_pro_active' ) ) {
            $is_pro = (bool) $this->plugin->license->is_pro_active();
        }

        ?>
        <?php if ( ! $is_pro ) : ?>
            <div class="aegisbackup-pro-locked" aria-disabled="true">
        <?php endif; ?>

        <?php if ( 'dr_created' === $ab_msg ) : ?>
        <div class="notice notice-success"><p><?php esc_html_e( 'DR Recovery link created. Your DR backup is building in the background. Keep this link somewhere safe.', 'aegisbackup' ); ?></p></div>
        <?php endif; ?>

        <div class="aegisbackup-card">
            <h2><?php esc_html_e( 'Create Full Backups or Disaster Recovery Links (DR)', 'aegisbackup' ); ?></h2>
            <p class="description"><?php esc_html_e( 'Important: Full Backup vs. Disaster Recovery (DR) Link. - Creating a Full Backup and Generating a Disaster Recovery (DR) Link serve two different purposes and are processed differently.', 'aegisbackup' ); ?></p>
            <p class="description1"><?php esc_html_e( '1. ** A Full Backup builds a portable backup package containing your site files, database, and a restore manifest. This package is designed for manual downloads, migrations, and standard restores, and works reliably on shared hosting environments.', 'aegisbackup' ); ?></p>
			<p class="description1"><?php esc_html_e( '2. ** A Disaster Recovery (DR) Link, on the other hand, creates a self-contained recovery package stored separately and generates a secure recovery endpoint with a unique token. This allows you to restore your site even if WordPress admin access is unavailable (for example, after a fatal error, lockout, or white screen).  Use only as a last resort. This process restores your site to the most recent known-good state that was successfully created.', 'aegisbackup' ); ?></p>
            <table class="form-table" role="presentation">
                <tbody>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Include', 'aegisbackup' ); ?></th>
                        <td>
                            <label><input type="checkbox" id="ab_include_files" checked> <?php esc_html_e( 'Files (wp-content)', 'aegisbackup' ); ?></label><br>
                            <label><input type="checkbox" id="ab_include_db" checked> <?php esc_html_e( 'Database (table-per-file)', 'aegisbackup' ); ?></label><br>
                            <label><input type="checkbox" id="ab_include_htaccess" checked> <?php esc_html_e( '.htaccess (recommended)', 'aegisbackup' ); ?></label><br>
                            <label><input type="checkbox" id="ab_include_config"> <?php esc_html_e( 'wp-config.php (optional)', 'aegisbackup' ); ?></label><br>
                            <label><input type="checkbox" id="ab_include_core"> <?php esc_html_e( 'WordPress core (optional)', 'aegisbackup' ); ?></label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Disaster Recovery', 'aegisbackup' ); ?></th>
                        <td>
                            <label><input type="checkbox" id="ab_dr_generate_link" value="1"> <?php esc_html_e( 'Generate DR Recovery Link (creates recovery endpoint + token)', 'aegisbackup' ); ?></label>
                            <p class="description"><?php esc_html_e( 'If checked, AegisBackup stores this backup separately (not affected by retention/purge) and generates a recovery link that works even if wp-admin is inaccessible.', 'aegisbackup' ); ?></p>
                        </td>
                    </tr>


                    <tr>
                        <th scope="row"><?php esc_html_e( 'DB Export Mode', 'aegisbackup' ); ?></th>
                        <td>
                            <select id="ab_db_export_mode">
                                <option value="table"><?php esc_html_e( 'Table-per-file (recommended)', 'aegisbackup' ); ?></option>
                                <option value="php"><?php esc_html_e( 'PHP streaming (compatibility)', 'aegisbackup' ); ?></option>
                                <option value="mysqldump"><?php esc_html_e( 'Fast (mysqldump if available)', 'aegisbackup' ); ?></option>
                            </select>
                            <p class="description"><?php esc_html_e( 'If mysqldump is not available, AegisBackup automatically falls back to safe PHP export.', 'aegisbackup' ); ?></p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><?php esc_html_e( 'Exclude Paths', 'aegisbackup' ); ?></th>
                        <td>
                            <textarea id="ab_excludes" rows="5" class="large-text code" placeholder="wp-content/cache\nwp-content/uploads/aegisbackup"></textarea>
                            <p class="description"><?php esc_html_e( 'One path per line (relative to site root). Cache/log/backups are recommended excludes.', 'aegisbackup' ); ?></p>
                        </td>
                    </tr>
                </tbody>
            </table>

            <p>
                <button class="button button-primary" id="ab_start_backup"><?php esc_html_e( 'Create Backup (DR)', 'aegisbackup' ); ?></button>
                <span class="spinner" style="float:none;"></span>
            </p>
            <p class="description"><div id="ab_backup_status"></div></p>
            <div id="ab_backup_checklist" style="margin:10px 0 6px 0;"></div>
            <div id="ab_backup_download" style="margin:8px 0 0 0;"></div>
            <div id="ab_progress_wrap" style="display:none;">
                <div id="ab_live_updates_header" style="margin:10px 0 6px 0;">
                    <h3 style="margin:0 0 4px 0; padding:0;"><?php esc_html_e( 'WordPress Migration Backup Updates', 'aegisbackup' ); ?></h3>
                    <p class="description" style="margin:0;"><?php esc_html_e( 'Important: Please do not navigate away from this page while the backup is running. The migration package is generating in real time and leaving this page may interrupt the process. This process takes only a few minutes.', 'aegisbackup' ); ?></p>
                </div>
                <div class="aegisbackup-progress-bar"><div class="aegisbackup-progress" id="ab_progress_bar"></div></div>
                <pre class="aegisbackup-log" id="ab_progress_log"></pre>
            </div>
        
            <hr style="margin:18px 0;">

            <h3><?php esc_html_e( 'Active DR Recovery Links', 'aegisbackup' ); ?></h3>
            <p class="description"><?php esc_html_e( 'DR links are only created when “Generate DR Recovery Link” is checked above. They are stored separately and are not affected by retention/purge.', 'aegisbackup' ); ?></p>

            <?php if ( empty( $tokens ) ) : ?>
                <p class="description"><?php esc_html_e( 'No DR recovery links yet.', 'aegisbackup' ); ?></p>
            <?php else : ?>
                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th><?php esc_html_e( 'Created', 'aegisbackup' ); ?></th>
                            <th><?php esc_html_e( 'Status', 'aegisbackup' ); ?></th>
                            <th><?php esc_html_e( 'Recovery Link', 'aegisbackup' ); ?></th>
                            <th><?php esc_html_e( 'Recovery Zip', 'aegisbackup' ); ?></th>
                            <th><?php esc_html_e( 'Package', 'aegisbackup' ); ?></th>
                            <th><?php esc_html_e( 'Actions', 'aegisbackup' ); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        uasort(
                            $tokens,
                            function ( $a, $b ) {
                                $ta = isset( $a['created'] ) ? (int) $a['created'] : 0;
                                $tb = isset( $b['created'] ) ? (int) $b['created'] : 0;
                                return $tb - $ta;
                            }
                        );

                        foreach ( $tokens as $tok => $row ) :
                            $created = ! empty( $row['created'] ) ? (int) $row['created'] : 0;
                            $status  = ! empty( $row['status'] ) ? (string) $row['status'] : '';
                            $link    = ! empty( $row['link'] ) ? (string) $row['link'] : '';
                            $zip     = ! empty( $row['zip_path'] ) ? (string) $row['zip_path'] : '';
                            $pkg     = ! empty( $row['package'] ) ? (string) $row['package'] : '';
                            ?>
                            <tr>
                                <td><?php echo $created ? esc_html( gmdate( 'Y-m-d H:i:s', $created ) ) : ''; ?></td>
                                <td><?php echo esc_html( ucfirst( $status ) ); ?></td>
                                <td><?php if ( $link ) : ?><input type="text" readonly value="<?php echo esc_attr( $link ); ?>" style="width:100%;max-width:520px;"><?php endif; ?></td>
                                <td><?php if ( $zip ) : ?><code style="font-size:12px;"><?php echo esc_html( $zip ); ?></code><?php endif; ?></td>
                                <td><?php echo esc_html( $pkg ); ?></td>
                                <td>
                                    <?php
                                    $actions = array();
                                    if ( $pkg && 'ready' === strtolower( $status ) ) {
                                        $actions['download'] = wp_nonce_url(
                                            add_query_arg( array( 'action' => 'aegisbackup_download_package', 'pkg' => $pkg ), admin_url( 'admin-post.php' ) ),
                                            'aegisbackup_download_package',
                                            'aegisbackup_nonce'
                                        );
                                    }
                                    $actions['delete'] = wp_nonce_url(
                                        add_query_arg( array( 'action' => 'aegisbackup_delete_dr_token', 'token' => $tok ), admin_url( 'admin-post.php' ) ),
                                        'aegisbackup_delete_dr_token',
                                        'ab_dr_del_nonce'
                                    );
                                    ?>
                                    <?php if ( ! empty( $actions['download'] ) ) : ?>
                                        <a href="<?php echo esc_url( $actions['download'] ); ?>"><?php esc_html_e( 'Download', 'aegisbackup' ); ?></a> |
                                    <?php endif; ?>
                                    <a href="<?php echo esc_url( $actions['delete'] ); ?>" onclick="return confirm('<?php echo esc_js( __( 'Delete this DR token and its associated package?', 'aegisbackup' ) ); ?>');"><?php esc_html_e( 'Delete', 'aegisbackup' ); ?></a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>

            <p class="description" style="margin-top:12px;">
                <?php esc_html_e( 'IMPORTANT: Look at the latest Disaster Recovery Link, copy the entire URL and store it in a safe place.  Note your recovery endpoint path:', 'aegisbackup' ); ?>
                <code><?php echo esc_html( trailingslashit( site_url() ) . 'aegisbackup-recovery/?token=...' ); ?></code>
            </p>

</div>

        <div class="aegisbackup-card">
            <h2><?php esc_html_e( 'Recent Packages', 'aegisbackup' ); ?></h2>
            <div id="ab_recent_packages_area">
            <?php if ( empty( $list ) ) : ?>
                <p class="description"><?php esc_html_e( 'No packages found yet.', 'aegisbackup' ); ?></p>
            <?php else : ?>
                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th><?php esc_html_e( 'Name', 'aegisbackup' ); ?></th>
                            <th><?php esc_html_e( 'Created', 'aegisbackup' ); ?></th>
                            <th><?php esc_html_e( 'Size', 'aegisbackup' ); ?></th>
                            <th><?php esc_html_e( 'Action', 'aegisbackup' ); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ( $list as $pkg ) : ?>
                            <?php
                            $name = isset( $pkg['name'] ) ? (string) $pkg['name'] : '';
                            $zip  = isset( $pkg['package'] ) ? (string) $pkg['package'] : '';
                            $download = wp_nonce_url(
                                add_query_arg( array( 'action' => 'aegisbackup_download_package', 'pkg' => $name ), admin_url( 'admin-post.php' ) ),
                                'aegisbackup_download_package',
                                'aegisbackup_nonce'
                            );
                            $delete = wp_nonce_url(
                                add_query_arg( array( 'action' => 'aegisbackup_delete_package', 'pkg' => $name, 'redirect' => 'migration' ), admin_url( 'admin-post.php' ) ),
                                'aegisbackup_delete_package',
                                'aegisbackup_nonce'
                            );
                            ?>
                            <tr>
                                <td><code><?php echo esc_html( 'AegisBackup-' . $name . '.zip' ); ?></code></td>
                                <td><?php echo esc_html( isset( $pkg['created'] ) ? (string) $pkg['created'] : '' ); ?></td>
                                <td><?php echo esc_html( isset( $pkg['size'] ) ? (string) $pkg['size'] : '' ); ?></td>
                                <td>
                                    <?php if ( $zip ) : ?>
                                        <a class="button button-small" href="<?php echo esc_url( $download ); ?>"><?php esc_html_e( 'Download', 'aegisbackup' ); ?></a>
                                        <a class="button button-small" href="<?php echo esc_url( $delete ); ?>" onclick="return confirm('Delete this package?');"><?php esc_html_e( 'Delete', 'aegisbackup' ); ?></a>
                                    <?php else : ?>
                                        —
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
            </div>
        </div>
        
        <div class="aegisbackup-card">
            <h2><?php esc_html_e( 'Local Backup Retention', 'aegisbackup' ); ?></h2>
            <p class="description"><?php esc_html_e( 'Automatically manage local migration packages after a number of days.', 'aegisbackup' ); ?></p>

            <?php
            $ret = get_option( 'aegisbackup_local_retention', array() );
            $days = isset( $ret['migration']['days'] ) ? absint( $ret['migration']['days'] ) : 30;
            $act  = isset( $ret['migration']['action'] ) ? sanitize_key( $ret['migration']['action'] ) : 'purge';
            if ( $days < 1 ) { $days = 1; }
            if ( $days > 90 ) { $days = 90; }
            ?>
            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                <?php wp_nonce_field( 'aegisbackup_save_settings', 'aegisbackup_settings_nonce' ); ?>
                <input type="hidden" name="action" value="aegisbackup_save_settings" />
                <input type="hidden" name="settings_section" value="retention" />
                <input type="hidden" name="retention_context" value="migration" />
                <input type="hidden" name="retention_redirect" value="<?php echo esc_url( admin_url( 'admin.php?page=aegisbackup&tab=migration' ) ); ?>" />

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Days', 'aegisbackup' ); ?></th>
                        <td>
                            <select name="retention_days">
                                <?php for ( $i = 1; $i <= 90; $i++ ) : ?>
                                    <option value="<?php echo esc_attr( $i ); ?>" <?php selected( $days, $i ); ?>><?php echo esc_html( $i ); ?></option>
                                <?php endfor; ?>
                            </select>
                            <p class="description"><?php esc_html_e( 'Keep local migration packages for this many days.', 'aegisbackup' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Action', 'aegisbackup' ); ?></th>
                        <td>
                            <select name="retention_action">
                                <option value="purge" <?php selected( 'purge', $act ); ?>><?php esc_html_e( 'Delete / Purge', 'aegisbackup' ); ?></option>
                                <option value="offsite" <?php selected( 'offsite', $act ); ?>><?php esc_html_e( 'Move to Offsite Remote Storage', 'aegisbackup' ); ?></option>
                            </select>
                            <p class="description"><?php esc_html_e( 'If “Move to Offsite” is selected, AegisBackup will attempt an offsite upload (if enabled) before deleting local copies.', 'aegisbackup' ); ?></p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <button type="submit" class="button button-primary"><?php esc_html_e( 'Save Retention', 'aegisbackup' ); ?></button>
                </p>
            </form>
        </div>
        <?php if ( ! $is_pro ) : ?>
            </div>
        <?php endif; ?>
        <?php
    }

    protected function render_db_backups() {
        require_once AEGISBACKUP_DIR . 'includes/admin/panels/class-ab-panel-db-tools.php';

        $panel = new AB_Panel_DB_Tools( $this->plugin );
        $panel->render();
    }

    protected function render_file_backups() {
		$mgr = $this->plugin->file_backup;
		$plans = $mgr->get_plans();
		$backups = $mgr->list_backups();
		$msg_raw = filter_input( INPUT_GET, 'ab_msg', FILTER_SANITIZE_FULL_SPECIAL_CHARS );
		$msg = $msg_raw ? sanitize_key( (string) $msg_raw ) : '';
		$err_detail_raw = filter_input( INPUT_GET, 'ab_err', FILTER_SANITIZE_FULL_SPECIAL_CHARS );
		$err_detail = $err_detail_raw ? (string) $err_detail_raw : '';
		$err_detail = $err_detail ? rawurldecode( $err_detail ) : '';
		$is_pro = false;
		if ( isset( $this->plugin ) && isset( $this->plugin->license ) && is_object( $this->plugin->license ) && method_exists( $this->plugin->license, 'is_pro_active' ) ) {
			$is_pro = (bool) $this->plugin->license->is_pro_active();
		}
		$license_url = admin_url( 'admin.php?page=aegisbackup-license' );

		if ( $msg ) {
			$map = array(
				'plan_saved' => array( 'success', __( 'Backup plan saved.', 'aegisbackup' ) ),
				'plan_deleted' => array( 'success', __( 'Backup plan deleted.', 'aegisbackup' ) ),
				'backup_created' => array( 'success', __( 'Backup created.', 'aegisbackup' ) ),
				'backup_failed' => array( 'error', __( 'Backup failed. Check server permissions and selected paths.', 'aegisbackup' ) ),
				'backup_deleted' => array( 'success', __( 'Backup deleted.', 'aegisbackup' ) ),
				'backup_restored' => array( 'success', __( 'Backup restored.', 'aegisbackup' ) ),
				'restore_failed' => array( 'error', __( 'Restore failed. Check permissions and backup integrity.', 'aegisbackup' ) ),
				'plan_locked' => array( 'warning', __( 'Scheduling is a PRO feature. Upgrade to save plans (manual Run Now is free).', 'aegisbackup' ) ),
			);
			if ( isset( $map[ $msg ] ) ) {
				list( $type, $text ) = $map[ $msg ];
				echo '<div class="notice notice-' . esc_attr( $type ) . ' inline"><p>' . esc_html( $text ) . '</p>';
				if ( 'backup_failed' === $msg && $err_detail ) {
					echo '<p><code>' . esc_html( $err_detail ) . '</code></p>';
				}
				echo '</div>';
			}
		}

		?>
		<div class="aegisbackup-card">
			<h2><?php esc_html_e( 'File Backup Plans', 'aegisbackup' ); ?></h2>
			<?php
			?>
			<style id="ab-filetree-inline-css">
				.aegisbackup-wrap .ab-filetree{border:1px solid #dcdcde;background:#fff;padding:6px;max-width:100%;max-height:600px;overflow:auto;border-radius:6px;}
				.aegisbackup-wrap .ab-filetree .ab-ft-tree{list-style:none;margin:0;padding:0;}
				.aegisbackup-wrap .ab-filetree .ab-ft-details > summary{list-style:none;}
				.aegisbackup-wrap .ab-filetree .ab-ft-details > summary::-webkit-details-marker{display:none;}
				.aegisbackup-wrap .ab-filetree .ab-ft-row{display:flex;align-items:flex-start;gap:8px;padding:5px 6px;border-radius:4px;}
				.aegisbackup-wrap .ab-filetree .ab-ft-row:hover{background:#f6f7f7;}
				.aegisbackup-wrap .ab-filetree .ab-ft-row::before{content:'';width:14px;flex:0 0 14px;display:inline-block;margin-top:1px;}
				.aegisbackup-wrap .ab-filetree .ab-ft-folder > .ab-ft-details > summary.ab-ft-row::before{content:'▸';color:#646970;}
				.aegisbackup-wrap .ab-filetree .ab-ft-folder > .ab-ft-details[open] > summary.ab-ft-row::before{content:'▾';color:#646970;}
				.aegisbackup-wrap .ab-filetree .ab-ft-check{margin:2px 0 0 0;}
				.aegisbackup-wrap .ab-filetree .ab-ft-text{display:flex;flex-direction:column;gap:2px;}
				.aegisbackup-wrap .ab-filetree .ab-ft-name{font-size:13px;line-height:1.15;}
				.aegisbackup-wrap .ab-filetree .ab-ft-subpath code{color:#646970;font-size:11px;background:transparent;padding:0;}
				.aegisbackup-wrap .ab-filetree .ab-ft-children{margin:2px 0 0 0;}
			</style>
			<p class="description">
				<?php esc_html_e( 'Create backup plans for selected directories/files under /public_html/ (WordPress root). Backups are saved as zip files in:', 'aegisbackup' ); ?>
				<?php
					$display = trailingslashit( $mgr->backups_dir() );
				?>
				<code><?php echo esc_html( str_replace( ABSPATH, '/public_html/', $display ) ); ?></code>
			</p>

			<form id="ab_file_plan_form" method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
				<input type="hidden" name="action" value="aegisbackup_save_file_plan" />
				<?php wp_nonce_field( 'aegisbackup_save_file_plan' ); ?>

				<table class="form-table" role="presentation">
					<tbody>
						<tr>
							<th scope="row"><?php esc_html_e( 'Plan Name', 'aegisbackup' ); ?></th>
							<td>
								<input type="text" name="plan_name" class="regular-text" placeholder="Weekly Plugins + Uploads" />
								<p class="description"><?php esc_html_e( 'Optional. If blank, a default name is generated.', 'aegisbackup' ); ?></p>
							</td>
						</tr>

						<tr>
							<th scope="row"><?php esc_html_e( 'Directories / Files', 'aegisbackup' ); ?></th>
							<td>
								<p class="description">
									<?php esc_html_e( 'Select folders and/or files. Click folders to expand like a tree (no buttons, no ajax).', 'aegisbackup' ); ?>
								</p>

								<div class="ab-filetree">
									<?php
										$this->ab_render_tree( '', 0, 19 );
									?>
								</div>

								<p class="description" style="margin-top:8px;">
									<?php esc_html_e( 'Tip: Selecting a folder backs up everything inside that folder. Selecting a file backs up only that file.', 'aegisbackup' ); ?>
								</p>
							</td>
						</tr>

						<tr>
							<th scope="row"><?php esc_html_e( 'Backup Type', 'aegisbackup' ); ?></th>
							<td>
								<select name="plan_type">
									<option value="full"><?php esc_html_e( 'Full', 'aegisbackup' ); ?></option>
									<option value="incremental"><?php esc_html_e( 'Incremental', 'aegisbackup' ); ?></option>
									<option value="differential"><?php esc_html_e( 'Differential', 'aegisbackup' ); ?></option>
								</select>
								<p class="description"><?php esc_html_e( 'Incremental compares against the last backup. Differential compares against the last full backup.', 'aegisbackup' ); ?></p>
							</td>
						</tr>

						<tr>
							<th scope="row"><?php esc_html_e( 'Schedule', 'aegisbackup' ); ?></th>
							<td>
								<select name="plan_schedule" id="ab_plan_schedule">
									<option value="daily"><?php esc_html_e( 'Daily', 'aegisbackup' ); ?></option>
									<option value="weekly"><?php esc_html_e( 'Weekly', 'aegisbackup' ); ?></option>
									<option value="monthly"><?php esc_html_e( 'Monthly', 'aegisbackup' ); ?></option>
								</select>
								<input type="time" name="plan_time" value="02:00" style="margin-left:10px;" id="ab_plan_time">
								<span class="description" style="margin-left:6px;"><?php esc_html_e( 'Local WP timezone.', 'aegisbackup' ); ?></span>
								<div style="margin-top:8px; display:flex; gap:12px; flex-wrap:wrap;" id="ab_schedule_fields">
									<label id="ab_sched_weekly">
										<?php esc_html_e( 'Weekly day', 'aegisbackup' ); ?>
										<select name="plan_dow" style="width:140px;">
											<option value="1" selected><?php esc_html_e( 'Mon', 'aegisbackup' ); ?></option>
											<option value="2"><?php esc_html_e( 'Tue', 'aegisbackup' ); ?></option>
											<option value="3"><?php esc_html_e( 'Wed', 'aegisbackup' ); ?></option>
											<option value="4"><?php esc_html_e( 'Thu', 'aegisbackup' ); ?></option>
											<option value="5"><?php esc_html_e( 'Fri', 'aegisbackup' ); ?></option>
											<option value="6"><?php esc_html_e( 'Sat', 'aegisbackup' ); ?></option>
											<option value="0"><?php esc_html_e( 'Sun', 'aegisbackup' ); ?></option>
										</select>
									</label>
									<label id="ab_sched_monthly"><?php esc_html_e( 'Monthly day (1-31)', 'aegisbackup' ); ?> <input type="number" name="plan_dom" min="1" max="31" value="1" style="width:70px;"></label>
									<label><input type="checkbox" name="plan_enabled" value="1" checked> <?php esc_html_e( 'Enabled', 'aegisbackup' ); ?></label>
								</div>

								<script>
								(function(){
									var sel = document.getElementById('ab_plan_schedule');
									if(!sel){ return; }
									var weekly = document.getElementById('ab_sched_weekly');
									var monthly = document.getElementById('ab_sched_monthly');
									function apply(){
										var v = sel.value || 'daily';
										if(weekly){ weekly.style.display = (v === 'weekly') ? '' : 'none'; }
										if(monthly){ monthly.style.display = (v === 'monthly') ? '' : 'none'; }
									}
									sel.addEventListener('change', apply);
									apply();
								})();
								</script>
							</td>
						</tr>
					</tbody>
				</table>

				<p>
					<button type="submit" class="button button-primary" name="ab_file_submit" value="save" <?php disabled( ! $is_pro ); ?>><?php esc_html_e( 'Save Schedule', 'aegisbackup' ); ?></button>
					<button type="submit" class="button" name="ab_file_submit" value="run_now"><?php esc_html_e( 'Run Now', 'aegisbackup' ); ?></button>

					<?php if ( ! $is_pro ) : ?>
						<span style="margin-left:10px; color:#b32d2e; font-weight:600;">
							<?php esc_html_e( 'Scheduling is a PRO feature.', 'aegisbackup' ); ?>
						</span>
						<a href="<?php echo esc_url( $license_url ); ?>" class="button" style="margin-left:8px;">
							<?php esc_html_e( 'Upgrade', 'aegisbackup' ); ?>
						</a>
					<?php endif; ?>
				</p>
			</form>
		</div>

		<div class="aegisbackup-card">
			<h2><?php esc_html_e( 'Scheduled Jobs', 'aegisbackup' ); ?></h2>
			<?php if ( empty( $plans ) ) : ?>
				<p class="description"><?php esc_html_e( 'No file backup plans yet.', 'aegisbackup' ); ?></p>
			<?php else : ?>
				<table class="widefat striped">
					<thead>
						<tr>
							<th><?php esc_html_e( 'Name', 'aegisbackup' ); ?></th>
							<th><?php esc_html_e( 'Type', 'aegisbackup' ); ?></th>
							<th><?php esc_html_e( 'Schedule', 'aegisbackup' ); ?></th>
							<th><?php esc_html_e( 'Paths', 'aegisbackup' ); ?></th>
							<th><?php esc_html_e( 'Next Run', 'aegisbackup' ); ?></th>
							<th><?php esc_html_e( 'Actions', 'aegisbackup' ); ?></th>
						</tr>
					</thead>
					<tbody>
						<?php foreach ( $plans as $p ) : ?>
							<?php
								$next = ! empty( $p['enabled'] ) && ! empty( $p['next_run'] ) ? date_i18n( 'Y-m-d H:i', (int) $p['next_run'] ) : __( 'Disabled', 'aegisbackup' );
								$paths_txt = isset( $p['paths'] ) && is_array( $p['paths'] ) ? implode( "\n", $p['paths'] ) : '';
								$run_url = wp_nonce_url( add_query_arg( array( 'action' => 'aegisbackup_run_file_plan_now', 'plan_id' => $p['id'] ), admin_url( 'admin-post.php' ) ), 'aegisbackup_run_file_plan_now' );
								$del_url = wp_nonce_url( add_query_arg( array( 'action' => 'aegisbackup_delete_file_plan', 'plan_id' => $p['id'] ), admin_url( 'admin-post.php' ) ), 'aegisbackup_delete_file_plan' );
							?>
							<tr>
								<td><strong><?php echo esc_html( $p['name'] ?? $p['id'] ); ?></strong><br><code><?php echo esc_html( $p['id'] ); ?></code></td>
								<td><?php echo esc_html( strtoupper( (string) ( $p['type'] ?? 'full' ) ) ); ?></td>
								<td><?php echo esc_html( ucfirst( (string) ( $p['schedule'] ?? 'daily' ) ) . ' @ ' . ( $p['time'] ?? '02:00' ) ); ?></td>
								<td><textarea class="large-text code" rows="3" readonly><?php echo esc_textarea( $paths_txt ); ?></textarea></td>
								<td><?php echo esc_html( $next ); ?></td>
								<td>
									<a class="button button-secondary" href="<?php echo esc_url( $run_url ); ?>"><?php esc_html_e( 'Run Now', 'aegisbackup' ); ?></a>
									<a class="button" href="<?php echo esc_url( $del_url ); ?>" onclick="return confirm('Delete this plan?');"><?php esc_html_e( 'Delete', 'aegisbackup' ); ?></a>
								</td>
							</tr>
						<?php endforeach; ?>
					</tbody>
				</table>
			<?php endif; ?>
		</div>

		<div class="aegisbackup-card">
			<h2><?php esc_html_e( 'Backups', 'aegisbackup' ); ?></h2>
			<?php if ( empty( $backups ) ) : ?>
				<p class="description"><?php esc_html_e( 'No file backups found yet.', 'aegisbackup' ); ?></p>
			<?php else : ?>
				<table class="widefat striped">
					<thead>
						<tr>
							<th><?php esc_html_e( 'Backup ID', 'aegisbackup' ); ?></th>
							<th><?php esc_html_e( 'Created', 'aegisbackup' ); ?></th>
							<th><?php esc_html_e( 'Type', 'aegisbackup' ); ?></th>
							<th><?php esc_html_e( 'Files Added', 'aegisbackup' ); ?></th>
							<th><?php esc_html_e( 'Size', 'aegisbackup' ); ?></th>
							<th><?php esc_html_e( 'Actions', 'aegisbackup' ); ?></th>
						</tr>
					</thead>
					<tbody>
						<?php foreach ( $backups as $b ) : ?>
							<?php
								$m = is_array( $b['manifest'] ) ? $b['manifest'] : array();
								$type = isset( $m['type'] ) ? strtoupper( (string) $m['type'] ) : 'ZIP';
								$cnt = isset( $m['file_count_added'] ) ? (int) $m['file_count_added'] : 0;
								$download_url = wp_nonce_url( add_query_arg( array( 'action' => 'aegisbackup_download_file_backup', 'backup_id' => $b['id'] ), admin_url( 'admin-post.php' ) ), 'aegisbackup_download_file_backup' );
								$restore_url = wp_nonce_url( add_query_arg( array( 'action' => 'aegisbackup_restore_file_backup', 'backup_id' => $b['id'] ), admin_url( 'admin-post.php' ) ), 'aegisbackup_restore_file_backup' );
								$delete_url = wp_nonce_url( add_query_arg( array( 'action' => 'aegisbackup_delete_file_backup', 'backup_id' => $b['id'] ), admin_url( 'admin-post.php' ) ), 'aegisbackup_delete_file_backup' );
							?>
							<tr>
								<td><strong><?php echo esc_html( $b['id'] ); ?></strong><br><code><?php echo esc_html( $b['file'] ); ?></code></td>
								<td><?php echo esc_html( $b['created'] ); ?></td>
								<td><?php echo esc_html( $type ); ?></td>
								<td><?php echo esc_html( number_format_i18n( $cnt ) ); ?></td>
								<td><?php echo esc_html( $b['size'] ); ?></td>
								<td>
									<a class="button button-secondary" href="<?php echo esc_url( $download_url ); ?>"><?php esc_html_e( 'Download', 'aegisbackup' ); ?></a>
									<a class="button button-secondary" href="<?php echo esc_url( $restore_url ); ?>" onclick="return confirm('This will overwrite files in /public_html/. Continue?');"><?php esc_html_e( 'Restore', 'aegisbackup' ); ?></a>
									<a class="button" href="<?php echo esc_url( $delete_url ); ?>" onclick="return confirm('Delete this backup zip?');"><?php esc_html_e( 'Delete', 'aegisbackup' ); ?></a>
								</td>
							</tr>
						<?php endforeach; ?>
					</tbody>
				</table>
			<?php endif; ?>
		</div>
		
        <div class="aegisbackup-card">
            <h2><?php esc_html_e( 'Local Backup Retention', 'aegisbackup' ); ?></h2>
            <p class="description"><?php esc_html_e( 'Automatically manage local file backups after a number of days.', 'aegisbackup' ); ?></p>

            <?php
            $ret = get_option( 'aegisbackup_local_retention', array() );
            $days = isset( $ret['files']['days'] ) ? absint( $ret['files']['days'] ) : 30;
            $act  = isset( $ret['files']['action'] ) ? sanitize_key( $ret['files']['action'] ) : 'purge';
            if ( $days < 1 ) { $days = 1; }
            if ( $days > 90 ) { $days = 90; }
            ?>
            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                <?php wp_nonce_field( 'aegisbackup_save_settings', 'aegisbackup_settings_nonce' ); ?>
                <input type="hidden" name="action" value="aegisbackup_save_settings" />
                <input type="hidden" name="settings_section" value="retention" />
                <input type="hidden" name="retention_context" value="files" />
                <input type="hidden" name="retention_redirect" value="<?php echo esc_url( admin_url( 'admin.php?page=aegisbackup&tab=files' ) ); ?>" />

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Days', 'aegisbackup' ); ?></th>
                        <td>
                            <select name="retention_days">
                                <?php for ( $i = 1; $i <= 90; $i++ ) : ?>
                                    <option value="<?php echo esc_attr( $i ); ?>" <?php selected( $days, $i ); ?>><?php echo esc_html( $i ); ?></option>
                                <?php endfor; ?>
                            </select>
                            <p class="description"><?php esc_html_e( 'Keep local file backups for this many days.', 'aegisbackup' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Action', 'aegisbackup' ); ?></th>
                        <td>
                            <select name="retention_action">
                                <option value="purge" <?php selected( 'purge', $act ); ?>><?php esc_html_e( 'Delete / Purge', 'aegisbackup' ); ?></option>
                                <option value="offsite" <?php selected( 'offsite', $act ); ?>><?php esc_html_e( 'Move to Offsite Remote Storage', 'aegisbackup' ); ?></option>
                            </select>
                            <p class="description"><?php esc_html_e( 'If “Move to Offsite” is selected, AegisBackup will attempt an offsite upload (if enabled) before deleting local copies.', 'aegisbackup' ); ?></p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <button type="submit" class="button button-primary"><?php esc_html_e( 'Save Retention', 'aegisbackup' ); ?></button>
                </p>
            </form>
        </div>

		<?php
    }

    protected function render_wordpress_backups() {
        $plans = $this->plugin->wp_backup ? $this->plugin->wp_backup->get_plans() : array();
        $packages = $this->plugin->backup->list_packages();
        $wp_packages = array();
        foreach ( (array) $packages as $pkg ) {
            $m = isset( $pkg['manifest'] ) && is_array( $pkg['manifest'] ) ? $pkg['manifest'] : array();
            $args = isset( $m['args'] ) && is_array( $m['args'] ) ? $m['args'] : array();
            $purpose = isset( $args['package_purpose'] ) ? (string) $args['package_purpose'] : '';
            $snapshot = ! empty( $args['snapshot'] );
            if ( 'wordpress' === $purpose || $snapshot ) {
                $wp_packages[] = $pkg;
            }
        }

    }

    protected function render_root_folder_checklist() {
        $root = trailingslashit( ABSPATH );
        $items = array();
        if ( is_dir( $root ) && is_readable( $root ) ) {
            $dh = opendir( $root );
            if ( $dh ) {
                while ( false !== ( $entry = readdir( $dh ) ) ) {
                    if ( '.' === $entry || '..' === $entry ) {
                        continue;
                    }
                    $full = $root . $entry;
                    if ( is_dir( $full ) ) {
                        $items[] = $entry;
                    }
                }
                closedir( $dh );
            }
        }
        sort( $items );

        echo '<div style="display:grid; grid-template-columns: repeat(3, minmax(0, 1fr)); gap:8px; max-width: 900px;">';
        foreach ( $items as $dir ) {
            echo '<label style="display:flex; gap:6px; align-items:center;"><input type="checkbox" disabled> <code>' . esc_html( $dir ) . '</code></label>';
        }
        echo '</div>';
        echo '<p class="description">' . esc_html__( 'Folder-level selection will be enabled in the next release.', 'aegisbackup' ) . '</p>';
    }

	private function ab_ft_state() {
		$uid = get_current_user_id();
		$rel = (string) get_user_meta( $uid, 'aegisbackup_ft_rel', true );
		$sel = get_user_meta( $uid, 'aegisbackup_ft_sel', true );

		$rel = str_replace( array( '\\', chr( 0 ) ), array( '/', '' ), $rel );
		$rel = ltrim( $rel, '/' );

		if ( ! is_array( $sel ) ) {
			$sel = array();
		}
		$clean = array();
		foreach ( $sel as $p ) {
			$p = (string) $p;
			$p = str_replace( array( '\\', chr( 0 ) ), array( '/', '' ), $p );
			$p = ltrim( $p, '/' );
			if ( '' !== $p ) {
				$clean[ $p ] = true;
			}
		}

		return array(
			'rel' => $rel,
			'sel' => array_keys( $clean ),
		);
	}

	private function ab_render_tree( $rel, $depth, $max_depth ) {
		$depth = (int) $depth;
		$pad_left = 10 + ( $depth * 18 );
		$rel = (string) $rel;
		$rel = str_replace( array( '\\', chr( 0 ) ), array( '/', '' ), $rel );
		$rel = ltrim( $rel, '/' );

		$root_real = realpath( (string) ABSPATH );
		if ( false === $root_real ) {
			echo '<div class="ab-ft-error">' . esc_html__( 'ABSPATH not accessible.', 'aegisbackup' ) . '</div>';
			return;
		}
		$root_real = rtrim( $root_real, "/\\" );

		$target = $root_real . ( $rel ? ( DIRECTORY_SEPARATOR . str_replace( '/', DIRECTORY_SEPARATOR, $rel ) ) : '' );
		$target_real = realpath( $target );
		if ( false === $target_real ) {
			echo '<div class="ab-ft-error">' . esc_html__( 'Path not found.', 'aegisbackup' ) . '</div>';
			return;
		}
		$target_real = rtrim( $target_real, "/\\" );

		if ( 0 !== strpos( $target_real, $root_real ) ) {
			echo '<div class="ab-ft-error">' . esc_html__( 'Invalid path.', 'aegisbackup' ) . '</div>';
			return;
		}

		if ( ! is_dir( $target_real ) || ! is_readable( $target_real ) ) {
			echo '<div class="ab-ft-error">' . esc_html__( 'Directory not readable.', 'aegisbackup' ) . '</div>';
			return;
		}

		$items = @scandir( $target_real );
		if ( ! is_array( $items ) ) {
			echo '<div class="ab-ft-error">' . esc_html__( 'Failed to scan directory.', 'aegisbackup' ) . '</div>';
			return;
		}

		$dirs  = array();
		$files = array();
		foreach ( $items as $name ) {
			if ( '.' === $name || '..' === $name ) {
				continue;
			}
			$abs = $target_real . DIRECTORY_SEPARATOR . $name;
			$child_rel = $rel ? ( $rel . '/' . $name ) : $name;
			$child_rel = str_replace( array( '\\', chr( 0 ) ), array( '/', '' ), (string) $child_rel );
			$child_rel = ltrim( $child_rel, '/' );
			if ( @is_dir( $abs ) ) {
				$dirs[] = array( 'name' => $name, 'rel' => $child_rel );
			} else {
				$files[] = array( 'name' => $name, 'rel' => $child_rel );
			}
		}
		usort( $dirs, static function( $a, $b ) { return strcasecmp( (string) $a['name'], (string) $b['name'] ); } );
		usort( $files, static function( $a, $b ) { return strcasecmp( (string) $a['name'], (string) $b['name'] ); } );

		echo '<ul class="ab-ft-tree" data-depth="' . esc_attr( $depth ) . '">';

		foreach ( $dirs as $d ) {
			$folder_rel  = (string) $d['rel'];
			$folder_name = (string) $d['name'];
			$disabled    = ( $depth >= $max_depth );

			echo '<li class="ab-ft-node ab-ft-folder">';
			if ( $disabled ) {
				echo '<div class="ab-ft-row" style="padding-left:' . esc_attr( (int) $pad_left ) . 'px;">';
					echo '<input class="ab-ft-check" type="checkbox" name="plan_paths[]" value="' . esc_attr( $folder_rel ) . '" onclick="event.stopPropagation();" />';
				echo '<div class="ab-ft-text">';
				echo '<div class="ab-ft-name">📁 ' . esc_html( $folder_name ) . ' <span class="ab-ft-muted">' . esc_html__( '(max depth)', 'aegisbackup' ) . '</span></div>';
				echo '<div class="ab-ft-subpath"><code>' . esc_html( $folder_rel ) . '</code></div>';
				echo '</div>';
				echo '</div>';
			} else {
				echo '<details class="ab-ft-details">';
				echo '<summary class="ab-ft-row" style="padding-left:' . esc_attr( (int) $pad_left ) . 'px;">';
				echo '<input class="ab-ft-check" type="checkbox" name="plan_paths[]" value="' . esc_attr( $folder_rel ) . '" onclick="event.stopPropagation();" />';
				echo '<div class="ab-ft-text">';
				echo '<div class="ab-ft-name">📁 ' . esc_html( $folder_name ) . '</div>';
				echo '<div class="ab-ft-subpath"><code>' . esc_html( $folder_rel ) . '</code></div>';
				echo '</div>';
				echo '</summary>';
				echo '<div class="ab-ft-children">';
				$this->ab_render_tree( $folder_rel, $depth + 1, $max_depth );
				echo '</div>';
				echo '</details>';
			}
			echo '</li>';
		}

		foreach ( $files as $f ) {
			$file_rel  = (string) $f['rel'];
			$file_name = (string) $f['name'];
			echo '<li class="ab-ft-node ab-ft-file">';
			echo '<div class="ab-ft-row" style="padding-left:' . esc_attr( (int) $pad_left ) . 'px;">';
			echo '<input class="ab-ft-check" type="checkbox" name="plan_paths[]" value="' . esc_attr( $file_rel ) . '" onclick="event.stopPropagation();" />';
			echo '<div class="ab-ft-text">';
			echo '<div class="ab-ft-name">📄 ' . esc_html( $file_name ) . '</div>';
			echo '<div class="ab-ft-subpath"><code>' . esc_html( $file_rel ) . '</code></div>';
			echo '</div>';
			echo '</div>';
			echo '</li>';
		}

		echo '</ul>';
	}

	private function ab_ft_list_dir( $rel ) {
		$rel = (string) $rel;
		$rel = str_replace( array( '\\', chr( 0 ) ), array( '/', '' ), $rel );
		$rel = ltrim( $rel, '/' );

		$root_real = realpath( (string) ABSPATH );
		if ( false === $root_real ) {
			return array( 'ok' => false, 'rel' => $rel, 'abs' => '', 'dirs' => array(), 'files' => array(), 'error' => 'ABSPATH not accessible.' );
		}
		$root_real = rtrim( $root_real, "/\\" );

		$target = $root_real . ( $rel ? ( DIRECTORY_SEPARATOR . str_replace( '/', DIRECTORY_SEPARATOR, $rel ) ) : '' );
		$target_real = realpath( $target );
		if ( false === $target_real ) {
			return array( 'ok' => false, 'rel' => $rel, 'abs' => '', 'dirs' => array(), 'files' => array(), 'error' => 'Path not found.' );
		}
		$target_real = rtrim( $target_real, "/\\" );

		if ( 0 !== strpos( $target_real, $root_real ) ) {
			return array( 'ok' => false, 'rel' => $rel, 'abs' => '', 'dirs' => array(), 'files' => array(), 'error' => 'Invalid path.' );
		}

		if ( ! is_dir( $target_real ) || ! is_readable( $target_real ) ) {
			return array( 'ok' => false, 'rel' => $rel, 'abs' => $target_real, 'dirs' => array(), 'files' => array(), 'error' => 'Directory not readable.' );
		}

		$items = @scandir( $target_real );
		if ( ! is_array( $items ) ) {
			return array( 'ok' => false, 'rel' => $rel, 'abs' => $target_real, 'dirs' => array(), 'files' => array(), 'error' => 'Failed to scan directory.' );
		}

		$dirs  = array();
		$files = array();

		$max = 1500;
		$count = 0;

		foreach ( $items as $name ) {
			if ( '.' === $name || '..' === $name ) {
				continue;
			}
			$count++;
			if ( $count > $max ) {
				break;
			}

			$abs = $target_real . DIRECTORY_SEPARATOR . $name;
			$is_dir = @is_dir( $abs );
			$child_rel = $rel ? ( $rel . '/' . $name ) : $name;

			if ( $is_dir ) {
				$dirs[] = array( 'name' => $name, 'rel' => $child_rel );
			} else {
				$files[] = array( 'name' => $name, 'rel' => $child_rel );
			}
		}

		usort( $dirs, static function( $a, $b ) { return strcmp( (string) $a['name'], (string) $b['name'] ); } );
		usort( $files, static function( $a, $b ) { return strcmp( (string) $a['name'], (string) $b['name'] ); } );

		return array(
			'ok' => true,
			'rel' => $rel,
			'abs' => $target_real,
			'dirs' => $dirs,
			'files' => $files,
			'error' => '',
		);
	}
}
