<?php
namespace AegisBackup\Admin\Pages;

use AegisBackup\AB_Plugin;

defined( 'ABSPATH' ) || exit;

class AB_Page_Dashboard {

	protected $plugin;

	public function __construct( AB_Plugin $plugin ) {
		$this->plugin = $plugin;
	}

	public function render() {
		if ( ! current_user_can( 'manage_options' ) && ! current_user_can( 'aegisbackup_manage_backups' ) ) {
			wp_die( esc_html__( 'Insufficient permissions.', 'aegisbackup' ), 403 );
		}

		require_once AEGISBACKUP_DIR . 'includes/backup/class-ab-backup-manager.php';
		require_once AEGISBACKUP_DIR . 'includes/libs/class-ab-activity-log.php';
		$mgr  = new \AegisBackup\Backup\AB_Backup_Manager();
		$pkgs = $mgr->list_packages( 12, false );
		$now  = (int) current_time( 'timestamp' );
		$recent_packages = array();
		$pkg_labels = array();
		$pkg_sizes  = array();
		$last_backup_ts = 0;
		foreach ( (array) $pkgs as $p ) {
			$label = isset( $p['name'] ) ? (string) $p['name'] : '';
			$zip   = isset( $p['package'] ) ? (string) $p['package'] : '';
			$bytes = ( $zip && is_file( $zip ) ) ? (int) @filesize( $zip ) : 0;
			$created_str = isset( $p['created'] ) ? (string) $p['created'] : '';
			$created_ts  = $created_str ? strtotime( $created_str ) : 0;
			if ( $created_ts > $last_backup_ts ) {
				$last_backup_ts = $created_ts;
			}
			$recent_packages[] = array(
				'name'       => $label ? $label : ( $zip ? basename( $zip ) : '(package)' ),
				'package'    => $zip,
				'bytes'      => $bytes,
				'created'    => $created_str,
				'created_ts' => $created_ts,
			);
			if ( $label ) {
				$pkg_labels[] = $label;
				$pkg_sizes[]  = $bytes;
			}
		}
		// Sort newest-first and keep only last 5 for the dashboard list.
		usort(
			$recent_packages,
			static function( $a, $b ) {
				$at = isset( $a['created_ts'] ) ? (int) $a['created_ts'] : 0;
				$bt = isset( $b['created_ts'] ) ? (int) $b['created_ts'] : 0;
				return $bt <=> $at;
			}
		);

		$recent_packages_top5 = array_slice( $recent_packages, 0, 5 );
		$rows = \AegisBackup\Libs\AB_Activity_Log::get( 700 );
		$days = 14;
		$day_labels = array();
		$backup_counts  = array();
		$failure_counts = array();
		$backup_success_counts = array();
		$backup_failed_counts  = array();
		$backup_error_counts   = array();
		$backup_size_sum = array();
		$backup_size_n   = array();
		$backup_avg_size_mb = array();
		$restore_success_counts = array();
		$restore_failed_counts  = array();
		$restore_error_counts   = array();

		for ( $i = $days - 1; $i >= 0; $i-- ) {
			$dts = $now - ( $i * DAY_IN_SECONDS );
			$key = gmdate( 'Y-m-d', $dts );
			$day_labels[] = gmdate( 'M j', $dts );
			$backup_counts[ $key ]  = 0;
			$failure_counts[ $key ] = 0;
			$backup_success_counts[ $key ] = 0;
			$backup_failed_counts[ $key ]  = 0;
			$backup_error_counts[ $key ]   = 0;
			$backup_size_sum[ $key ]       = 0;
			$backup_size_n[ $key ]         = 0;
			$backup_avg_size_mb[ $key ]    = 0;
			$restore_success_counts[ $key ] = 0;
			$restore_failed_counts[ $key ]  = 0;
			$restore_error_counts[ $key ]   = 0;
		}
		
		$restore_ok = 0;
		$restore_fail = 0;
		$backup_ok = 0;
		$backup_fail = 0;
		$backup_err = 0; 
		$cut_restore = $now - ( 30 * DAY_IN_SECONDS );
		foreach ( (array) $rows as $r ) {
			$ts = isset( $r['ts'] ) ? (int) $r['ts'] : 0;
			$type = isset( $r['type'] ) ? (string) $r['type'] : '';
			$status = isset( $r['status'] ) ? (string) $r['status'] : '';
			if ( 'backup' === $type ) {
				if ( 'success' === $status ) {
					$backup_ok++;
				} elseif ( 'failed' === $status ) {
					$backup_fail++;
				}
			}
			if ( 'restore' === $type && $ts >= $cut_restore ) {
				if ( 'success' === $status ) {
					$restore_ok++;
				} elseif ( 'failed' === $status ) {
					$restore_fail++;
				}
			}
			if ( $ts > 0 ) {
				$dkey = gmdate( 'Y-m-d', $ts );

				if ( 'backup' === $type && array_key_exists( $dkey, $backup_counts ) ) {

					$backup_counts[ $dkey ]++;
					$bytes = 0;
					if ( isset( $r['bytes'] ) ) {
						$bytes = (int) $r['bytes'];
					} elseif ( isset( $r['size'] ) ) {
						$bytes = (int) $r['size'];
					}

					if ( 'success' === $status ) {
						$backup_success_counts[ $dkey ]++;
						if ( $bytes > 0 ) {
							$backup_size_sum[ $dkey ] += $bytes;
							$backup_size_n[ $dkey ]++;
						}
					} elseif ( 'failed' === $status ) {
						$backup_failed_counts[ $dkey ]++;
						$failure_counts[ $dkey ]++; // legacy
					} elseif ( 'error' === $status || 'err' === $status ) {
						$backup_error_counts[ $dkey ]++;
						$backup_err++;
					}
				}

				if ( 'restore' === $type && array_key_exists( $dkey, $restore_success_counts ) ) {
					if ( 'success' === $status ) {
						$restore_success_counts[ $dkey ]++;
					} elseif ( 'failed' === $status ) {
						$restore_failed_counts[ $dkey ]++;
					} elseif ( 'error' === $status || 'err' === $status ) {
						$restore_error_counts[ $dkey ]++;
					}
				}
			}

		}

		$user_id = get_current_user_id();
		$connections = get_user_meta( $user_id, 'aegisbackup_pp_connections', true );
		if ( ! is_array( $connections ) ) {
			$connections = array();
		}
		foreach ( $backup_avg_size_mb as $k => $v ) {
			$n = isset( $backup_size_n[ $k ] ) ? (int) $backup_size_n[ $k ] : 0;
			$sum = isset( $backup_size_sum[ $k ] ) ? (int) $backup_size_sum[ $k ] : 0;
			$backup_avg_size_mb[ $k ] = ( $n > 0 && $sum > 0 ) ? round( ( $sum / $n ) / ( 1024 * 1024 ), 2 ) : 0;
		}

		$fleet_labels = array();
		$fleet_days_since = array();
		$fleet_rows = array();
		foreach ( (array) $connections as $c ) {
			if ( ! is_array( $c ) ) {
				continue;
			}
			$name = isset( $c['name'] ) ? (string) $c['name'] : '';
			$dest = isset( $c['dest_url'] ) ? (string) $c['dest_url'] : '';
			$connected = ! empty( $c['connected'] ) ? 1 : 0;
			$last_activity = isset( $c['last_activity'] ) ? (int) $c['last_activity'] : 0;
			$age_days = $last_activity ? round( max( 0, ( $now - $last_activity ) / DAY_IN_SECONDS ), 1 ) : null;
			$fleet_labels[] = $name ? $name : ( $dest ? $dest : 'Connection' );
			$fleet_days_since[] = ( null === $age_days ) ? null : $age_days;
			$fleet_rows[] = array(
				'name' => $name,
				'dest_url' => $dest,
				'connected' => $connected,
				'last_activity' => $last_activity,
				'age_days' => $age_days,
			);
		}

		$remote = get_option( 'aegisbackup_remote_storage', array() );
		if ( ! is_array( $remote ) ) { $remote = array(); }
		$smtp = get_option( 'aegisbackup_smtp_settings', array() );
		if ( ! is_array( $smtp ) ) { $smtp = array(); }

		$wp_health = array( 'good' => 0, 'recommended' => 0, 'critical' => 0, 'score' => 0 );
		if ( is_admin() ) {

			if ( ! class_exists( 'WP_Site_Health' ) && file_exists( ABSPATH . 'wp-admin/includes/class-wp-site-health.php' ) ) {
				require_once ABSPATH . 'wp-admin/includes/class-wp-site-health.php';
			}
			if ( class_exists( 'WP_Site_Health' ) ) {
				try {
					$sh = \WP_Site_Health::get_instance();
					$tests = $sh->get_tests();
					if ( ! empty( $tests['direct'] ) && is_array( $tests['direct'] ) ) {
						foreach ( $tests['direct'] as $t ) {
							if ( empty( $t['test'] ) || ! is_callable( $t['test'] ) ) { continue; }
							$r = call_user_func( $t['test'] );
							$st = ( is_array( $r ) && isset( $r['status'] ) ) ? (string) $r['status'] : '';
							if ( 'good' === $st ) { $wp_health['good']++; }
							elseif ( 'recommended' === $st ) { $wp_health['recommended']++; }
							elseif ( 'critical' === $st ) { $wp_health['critical']++; }
						}
					}
					$score = 100 - ( (int) $wp_health['critical'] * 20 ) - ( (int) $wp_health['recommended'] * 10 );
					$wp_health['score'] = max( 0, min( 100, (int) $score ) );
				} catch ( \Throwable $e ) {

				}
			}
		}

		global $wpdb;
		$db_health = array( 'ok' => 1, 'warn' => 0, 'critical' => 0, 'score' => 100, 'size_mb' => 0, 'overhead_mb' => 0 );
		try {
			$cache_group = 'aegisbackup';
			$blog_id     = function_exists( 'get_current_blog_id' ) ? (int) get_current_blog_id() : 0;
			$ping_key    = 'ab_dash_db_ping_' . $blog_id;

			$ping = wp_cache_get( $ping_key, $cache_group );
			if ( false === $ping ) {
				$wpdb->get_var( 'SELECT 1' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				wp_cache_set( $ping_key, 1, $cache_group, 30 );
			}
		} catch ( \Throwable $e ) {
			$db_health['ok'] = 0;
			$db_health['critical'] = 1;
			$db_health['score'] = 0;
		}
		$size_bytes = 0;
		$overhead_bytes = 0;
		$schema = $wpdb->dbname;
		if ( $schema ) {
			$sum_key = 'ab_dash_db_sum_' . md5( (string) $schema . '|' . (string) $blog_id );
			$sum     = wp_cache_get( $sum_key, $cache_group );

			if ( false === $sum ) {
				$sum = $wpdb->get_row( $wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
					"SELECT SUM(DATA_LENGTH+INDEX_LENGTH) AS total_bytes, SUM(DATA_FREE) AS free_bytes FROM information_schema.TABLES WHERE TABLE_SCHEMA = %s",
					$schema
				), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cached for 5 minutes.

				wp_cache_set( $sum_key, $sum, $cache_group, 300 );
			}

			if ( is_array( $sum ) && isset( $sum['total_bytes'] ) ) {
				$size_bytes     = (int) $sum['total_bytes'];
				$overhead_bytes = isset( $sum['free_bytes'] ) ? (int) $sum['free_bytes'] : 0;
			}
		if ( $size_bytes <= 0 ) {
			$tbls_key = 'ab_dash_db_tables_' . md5( (string) $blog_id );
			$tbls     = wp_cache_get( $tbls_key, $cache_group );

			if ( false === $tbls ) {
				$tbls = $wpdb->get_results( 'SHOW TABLE STATUS', ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				wp_cache_set( $tbls_key, $tbls, $cache_group, 300 );
			}

			if ( is_array( $tbls ) ) {
				foreach ( $tbls as $t ) {
					$size_bytes     += (int) ( $t['Data_length'] ?? 0 ) + (int) ( $t['Index_length'] ?? 0 );
					$overhead_bytes += (int) ( $t['Data_free'] ?? 0 );
				}
			}
		}
		$db_health['size_mb'] = $size_bytes ? round( $size_bytes / ( 1024 * 1024 ), 1 ) : 0;
		$db_health['overhead_mb'] = $overhead_bytes ? round( $overhead_bytes / ( 1024 * 1024 ), 1 ) : 0;
		if ( $db_health['size_mb'] > 1024 ) { $db_health['warn']++; $db_health['score'] -= 15; }
		if ( $db_health['overhead_mb'] > 200 ) { $db_health['warn']++; $db_health['score'] -= 15; }
		$db_health['score'] = max( 0, min( 100, (int) $db_health['score'] ) );

		$svc_health = array( 'ok' => 0, 'warn' => 0, 'critical' => 0, 'score' => 100 );
		$upload = wp_upload_dir();
		// Use WP_Filesystem for filesystem checks (plugin checker requirement).
		global $wp_filesystem;
		if ( ! function_exists( 'WP_Filesystem' ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}
		WP_Filesystem();

		$u_ok = false;
		if ( is_array( $upload ) && ! empty( $upload['basedir'] ) && is_dir( $upload['basedir'] ) && $wp_filesystem && method_exists( $wp_filesystem, 'is_writable' ) ) {
			$u_ok = (bool) $wp_filesystem->is_writable( $upload['basedir'] );
		}
		if ( $u_ok ) { $svc_health['ok']++; } else { $svc_health['critical']++; $svc_health['score'] -= 40; }
		$disk_free = @disk_free_space( ABSPATH );
		$disk_total = @disk_total_space( ABSPATH );
		$free_pct = ( $disk_free && $disk_total ) ? ( ( $disk_free / $disk_total ) * 100 ) : null;
		if ( null !== $free_pct && $free_pct < 10 ) { $svc_health['critical']++; $svc_health['score'] -= 30; }
		elseif ( null !== $free_pct && $free_pct < 20 ) { $svc_health['warn']++; $svc_health['score'] -= 15; }
		else { $svc_health['ok']++; }
		if ( ! empty( $remote['enabled'] ) ) { $svc_health['ok']++; } else { $svc_health['warn']++; $svc_health['score'] -= 10; }
		if ( ! empty( $smtp['enabled'] ) ) { $svc_health['ok']++; } else { $svc_health['warn']++; $svc_health['score'] -= 10; }
		if ( defined( 'DISABLE_WP_CRON' ) && DISABLE_WP_CRON ) { $svc_health['warn']++; $svc_health['score'] -= 10; } else { $svc_health['ok']++; }
		$svc_health['score'] = max( 0, min( 100, (int) $svc_health['score'] ) );
		$total_attempts = (int) $backup_ok + (int) $backup_fail + (int) $backup_err;
		$success_rate = $total_attempts ? round( ( (int) $backup_ok / $total_attempts ) * 100 ) : 0;
		$age_days = $last_backup_ts ? max( 0, ( $now - $last_backup_ts ) / DAY_IN_SECONDS ) : 999;
		$recency = ( $age_days >= 14 ) ? 0 : (int) round( 100 - ( $age_days / 14 ) * 100 );
		$storage_score = ( null === $free_pct ) ? 50 : (int) max( 0, min( 100, round( $free_pct ) ) );
		$remote_score = ! empty( $remote['enabled'] ) ? 100 : 30;
		$svc_score = (int) $svc_health['score'];
		$license_snap = array(
			'active' => false,
			'is_pro' => false,
			'plan'   => '',
			'status' => '',
		);

		if ( isset( $this->plugin->license ) && is_object( $this->plugin->license ) ) {
			$lic = $this->plugin->license;

			$status = array();
			if ( method_exists( $lic, 'get_status' ) ) {
				$status = $lic->get_status();
			} elseif ( method_exists( $lic, 'get_cached_status' ) ) {
				$status = $lic->get_cached_status();
			}

			if ( is_array( $status ) ) {
				$is_pro    = ! empty( $status['is_pro'] );
				$plan      = isset( $status['plan'] ) ? (string) $status['plan'] : '';
				$lic_state = isset( $status['status'] ) ? (string) $status['status'] : '';
				$is_active = $is_pro || ( 'active' === strtolower( $lic_state ) );

				$license_snap['active'] = $is_active ? true : false;
				$license_snap['is_pro'] = $is_pro ? true : false;
				$license_snap['plan']   = $plan;
				$license_snap['status'] = $lic_state;
			}
		}

		$payload = array(
			'meta' => array(
				'site' => home_url(),
				'version' => defined( 'AEGISBACKUP_VERSION' ) ? AEGISBACKUP_VERSION : '',
				'last_backup_ts' => $last_backup_ts,
			),
			'license' => $license_snap,
			'charts' => array(
				'days' => $day_labels,
				'wp_health'  => $wp_health,
				'db_health'  => $db_health,
				'svc_health' => $svc_health,
				'backup_success_14d' => array_values( $backup_success_counts ),
				'backup_failed_14d'  => array_values( $backup_failed_counts ),
				'backup_error_14d'   => array_values( $backup_error_counts ),
				'backup_avg_size_mb' => array_values( $backup_avg_size_mb ),
				'restore_success_14d' => array_values( $restore_success_counts ),
				'restore_failed_14d'  => array_values( $restore_failed_counts ),
				'restore_error_14d'   => array_values( $restore_error_counts ),
				'pkg_labels' => $pkg_labels,
				'pkg_sizes'  => $pkg_sizes,
				'fleet_labels' => $fleet_labels,
				'fleet_days_since' => $fleet_days_since,
				'backup_radar' => array(
					'labels' => array( 'Success Rate', 'Recency', 'Free Disk', 'Remote Storage', 'Service Health' ),
					'values' => array( (int) $success_rate, (int) $recency, (int) $storage_score, (int) $remote_score, (int) $svc_score ),
				),
			),
			'fleet' => $fleet_rows,
			'recent_packages' => $recent_packages,
			'settings' => array(
				'remote_enabled' => ! empty( $remote['enabled'] ) ? 1 : 0,
				'remote_provider' => isset( $remote['provider'] ) ? (string) $remote['provider'] : 'generic',
				'smtp_enabled' => ! empty( $smtp['enabled'] ) ? 1 : 0,
			),
		);

		$download_logs_url = wp_nonce_url(
			add_query_arg(
				array( 'action' => 'aegisbackup_dashboard_download_logs' ),
				admin_url( 'admin-post.php' )
			),
			'aegisbackup_dashboard_download_logs',
			'aegisbackup_nonce'
		);

		$last_backup_text  = __( 'No backups yet', 'aegisbackup' );
		$last_backup_class = 'ab-badge';

		if ( $last_backup_ts > 0 ) {
			$last_backup_text  = date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $last_backup_ts );
			$last_backup_class = 'ab-badge ok';
		}

		$ver_text = ( defined( 'AEGISBACKUP_VERSION' ) && AEGISBACKUP_VERSION ) ? AEGISBACKUP_VERSION : '';

		$remote_enabled = ! empty( $remote['enabled'] );
		$remote_provider = isset( $remote['provider'] ) ? (string) $remote['provider'] : 'generic';
		/* translators: %s: remote storage provider name (e.g., S3, SFTP). */
		$remote_text  = $remote_enabled ? sprintf( __( 'Enabled (%s)', 'aegisbackup' ), $remote_provider ) : __( 'Disabled', 'aegisbackup' );
		$remote_class = $remote_enabled ? 'ab-badge ok' : 'ab-badge';

		$smtp_enabled = ! empty( $smtp['enabled'] );
		$smtp_text  = $smtp_enabled ? __( 'Enabled', 'aegisbackup' ) : __( 'Disabled', 'aegisbackup' );
		$smtp_class = $smtp_enabled ? 'ab-badge ok' : 'ab-badge';

		$lic_text  = __( 'Inactive', 'aegisbackup' );
		$lic_class = 'ab-badge fail';

		if ( ! empty( $license_snap['active'] ) ) {
			$lic_text  = ! empty( $license_snap['is_pro'] ) ? __( 'Active (Pro)', 'aegisbackup' ) : __( 'Active', 'aegisbackup' );
			if ( ! empty( $license_snap['plan'] ) ) {
				$lic_text .= ' • ' . (string) $license_snap['plan'];
			}
			$lic_class = 'ab-badge ok';
		}

		?>
		<style>
		  /* Hard-enforce 3x2 chart grid on desktop regardless of cache/minifiers. */
		  .ab-dash-charts{
			display:grid;
			grid-template-columns:repeat(3,minmax(0,1fr));
			gap:14px;
			align-items:start; /* IMPORTANT: do NOT stretch row items */
		  }

		  /* IMPORTANT: fixed card height prevents Chart.js resize feedback loops */
		  .ab-dash-chart{
			height:260px !important;
			min-height:260px !important;
			max-height:260px !important;
			overflow:hidden !important;
		  }

		  /* IMPORTANT: fixed canvas display height (Chart.js can't expand it) */
		  .ab-dash-chart canvas{
			display:block !important;
			width:100% !important;
			height:190px !important;
			max-height:190px !important;
		  }

		  @media (max-width:782px){
			.ab-dash-charts{grid-template-columns:1fr;}
		  }
		</style>
		<script>
			window.AegisBackupDashPayload = <?php echo wp_json_encode( $payload ); ?>;
		</script>
		<script>
		(function(){
		  function abWaitForChart(tries){
			tries = tries || 0;

			if (typeof Chart === 'undefined') {
			  if (tries < 60) { // ~15 seconds max
				return setTimeout(function(){ abWaitForChart(tries + 1); }, 250);
			  }
			  return;
			}

			if (!window.AegisBackupDashPayload || !window.AegisBackupDashPayload.charts) {
			  return;
			}

			var c = window.AegisBackupDashPayload.charts;

			function mount(id, cfg){
			  var el = document.getElementById(id);
			  if (!el) return;

			  try {
				if (!cfg.type && cfg.data && cfg.data.datasets && cfg.data.datasets.length) {
				  cfg.type = cfg.data.datasets[0].type || 'bar';
				}
				cfg.options = cfg.options || {};
				cfg.options.animation = false;
				cfg.options.responsive = false;          
				cfg.options.maintainAspectRatio = true;  
				cfg.options.devicePixelRatio = 1;        

				var parentW = (el.parentNode && el.parentNode.clientWidth) ? el.parentNode.clientWidth : 600;
				el.width = parentW;  
				el.height = 190;      

				new Chart(el.getContext('2d'), cfg);
			  } catch(e) {
			  }
			}

			var wh = c.wp_health || {};
			mount('ab_dash_chart_1', {
			  type: 'doughnut',
			  data: {
				labels: ['Good', 'Recommended', 'Critical'],
				datasets: [{
				  label: 'WordPress Health',
				  data: [wh.good || 0, wh.recommended || 0, wh.critical || 0]
				}]
			  },
			  options: {
				responsive: true,
				maintainAspectRatio: false,
				plugins: {
				  legend: { display: true },
				  title: { display: true, text: 'WP Health Score: ' + (wh.score || 0) + '/100' }
				}
			  }
			});

			var dh = c.db_health || {};
			mount('ab_dash_chart_2', {
			  type: 'doughnut',
			  data: {
				labels: ['OK', 'Warnings', 'Critical'],
				datasets: [{
				  label: 'Database Health',
				  data: [dh.ok || 0, dh.warn || 0, dh.critical || 0]
				}]
			  },
			  options: {
				responsive: true,
				maintainAspectRatio: false,
				plugins: {
				  legend: { display: true },
				  title: { display: true, text: 'DB Score: ' + (dh.score || 0) + '/100 • Size: ' + (dh.size_mb || 0) + 'MB' }
				}
			  }
			});

			var sh = c.svc_health || {};
			mount('ab_dash_chart_3', {
			  type: 'doughnut',
			  data: {
				labels: ['OK', 'Warnings', 'Critical'],
				datasets: [{
				  label: 'Services Health',
				  data: [sh.ok || 0, sh.warn || 0, sh.critical || 0]
				}]
			  },
			  options: {
				responsive: true,
				maintainAspectRatio: false,
				plugins: {
				  legend: { display: true },
				  title: { display: true, text: 'Services Score: ' + (sh.score || 0) + '/100' }
				}
			  }
			});

			var days = c.days || [];
			mount('ab_dash_chart_4', {
			  type: 'bar',
			  data: {
				labels: days,
				datasets: [
				  { type: 'bar',  label: 'Success', data: c.backup_success_14d || [], stack: 'bk' },
				  { type: 'bar',  label: 'Failed',  data: c.backup_failed_14d  || [], stack: 'bk' },
				  { type: 'bar',  label: 'Error',   data: c.backup_error_14d   || [], stack: 'bk' },
				  { type: 'line', label: 'Avg Size (MB)', data: c.backup_avg_size_mb || [], yAxisID: 'y1', tension: 0.25 }
				]
			  },
			  options: {
				responsive: true,
				maintainAspectRatio: false,
				plugins: { legend: { display: true } },
				scales: {
				  y:  { stacked: true, beginAtZero: true },
				  y1: { beginAtZero: true, position: 'right', grid: { drawOnChartArea: false } }
				}
			  }
			});

			mount('ab_dash_chart_5', {
			  type: 'bar',
			  data: {
				labels: days,
				datasets: [
				  { label: 'Success', data: c.restore_success_14d || [], stack: 'rs' },
				  { label: 'Failed',  data: c.restore_failed_14d  || [], stack: 'rs' },
				  { label: 'Error',   data: c.restore_error_14d   || [], stack: 'rs' }
				]
			  },
			  options: {
				responsive: true,
				maintainAspectRatio: false,
				plugins: { legend: { display: true } },
				scales: { y: { stacked: true, beginAtZero: true } }
			  }
			});

			var br = c.backup_radar || {};
			mount('ab_dash_chart_6', {
			  type: 'radar',
			  data: {
				labels: br.labels || [],
				datasets: [{
				  label: 'Backup KPI (0-100)',
				  data: br.values || [],
				  fill: true
				}]
			  },
			  options: {
				responsive: true,
				maintainAspectRatio: false,
				plugins: { legend: { display: true } },
				scales: { r: { beginAtZero: true, suggestedMax: 100 } }
			  }
			});
		  }

		  if (document.readyState === 'loading') {
			document.addEventListener('DOMContentLoaded', function(){ abWaitForChart(0); });
		  } else {
			abWaitForChart(0);
		  }
		})();
		</script>
		<div class="wrap aegisbackup-wrap">
			<h1><?php esc_html_e( 'AegisBackup Dashboard', 'aegisbackup' ); ?></h1>

			<div class="aegisbackup-grid">
				<div class="aegisbackup-card">
					<h2 style="margin-top:0;"><?php esc_html_e( 'Central Master Dashboard', 'aegisbackup' ); ?></h2>
					<p class="description"><?php esc_html_e( 'At-a-glance health, recent activity, and quick actions for backups, restores, and migrations.', 'aegisbackup' ); ?></p>

					<div class="ab-grid-3" style="max-width:none;max-height:100%;height:100%">
						<div>
							<div class="<?php echo esc_attr( $last_backup_class ); ?>" id="ab_dash_last_backup"><?php echo esc_html( $last_backup_text ); ?></div>
							<div style="margin-top:6px; color:#646970;"><?php esc_html_e( 'Last backup time', 'aegisbackup' ); ?></div>
						</div>
						<div>
							<div class="ab-badge" id="ab_dash_version"><?php echo esc_html( $ver_text ? $ver_text : '—' ); ?></div>
							<div style="margin-top:6px; color:#646970;"><?php esc_html_e( 'Plugin version', 'aegisbackup' ); ?></div>
						</div>
						<div>
							<div class="<?php echo esc_attr( $remote_class ); ?>" id="ab_dash_remote"><?php echo esc_html( $remote_text ); ?></div>
							<div style="margin-top:6px; color:#646970;"><?php esc_html_e( 'Remote storage', 'aegisbackup' ); ?></div>
						</div>
						<div>
							<div class="<?php echo esc_attr( $smtp_class ); ?>" id="ab_dash_smtp"><?php echo esc_html( $smtp_text ); ?></div>
							<div style="margin-top:6px; color:#646970;"><?php esc_html_e( 'Email alerts', 'aegisbackup' ); ?></div>
						</div>
						<div>
							<div class="<?php echo esc_attr( $lic_class ); ?>" id="ab_dash_license"><?php echo esc_html( $lic_text ); ?></div>
							<div style="margin-top:6px; color:#646970;"><?php esc_html_e( 'Licensing', 'aegisbackup' ); ?></div>
						</div>
					</div>
					<div style="margin-top:16px; width:100%;">
						<h3 style="margin:0 0 8px;"><?php esc_html_e( 'Bulk Actions', 'aegisbackup' ); ?></h3>
						<p class="description" style="margin-top:0;"><?php esc_html_e( 'Run one-click actions for operational workflows. This uses the same job engine as the Backups page.', 'aegisbackup' ); ?></p>

						<div style="margin:8px 0 14px;">
							<a class="button" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisbackup' ) ); ?>"><?php esc_html_e( 'Open Backups', 'aegisbackup' ); ?></a>
							<a class="button" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisbackup-restore' ) ); ?>"><?php esc_html_e( 'Open Restore', 'aegisbackup' ); ?></a>
							<a class="button" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisbackup-pushpull' ) ); ?>"><?php esc_html_e( 'Open Transfer Wizard', 'aegisbackup' ); ?></a>
							<a class="button" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisbackup-logs' ) ); ?>"><?php esc_html_e( 'Open Settings', 'aegisbackup' ); ?></a>
						</div>

						<div class="ab-box">
							<div style="display:flex; gap:10px; flex-wrap:wrap; align-items:center;">
								<a href="#" class="button" id="ab_dash_prune_btn"><?php esc_html_e( 'Prune Old Packages (Keep 10)', 'aegisbackup' ); ?></a>
								<a class="button" href="<?php echo esc_url( $download_logs_url ); ?>"><?php esc_html_e( 'Download Diagnostics Logs', 'aegisbackup' ); ?></a>
								<span id="ab_dash_prune_msg" style="color:#646970;"></span>
							</div>
							<p class="description" style="margin-top:8px;"><?php esc_html_e( 'Prune removes older package folders from wp-content/uploads/aegisbackup to save disk space.', 'aegisbackup' ); ?></p>
						</div>
					</div>
				</div>

				<div class="aegisbackup-card">
					<h2 style="margin-top:0;"><?php esc_html_e( 'Backups', 'aegisbackup' ); ?></h2>
					<p class="description"><?php esc_html_e( 'Run a full backup package and manage your most recent backups.', 'aegisbackup' ); ?></p>

					<div class="ab-box">
						<div style="font-weight:600; margin-bottom:8px;"><?php esc_html_e( 'Run a Full Backup Now', 'aegisbackup' ); ?></div>
						<p class="description" style="margin-top:0;"><?php esc_html_e( 'Creates a full migration package (files + database + wp-config). Progress and sanity checks will appear below.', 'aegisbackup' ); ?></p>

						<label style="margin-right:12px;"><input type="checkbox" id="ab_include_files" checked="checked" /> <?php esc_html_e( 'Files', 'aegisbackup' ); ?></label>
						<label style="margin-right:12px;"><input type="checkbox" id="ab_include_db" checked="checked" /> <?php esc_html_e( 'Database', 'aegisbackup' ); ?></label>
						<label style="margin-right:12px;"><input type="checkbox" id="ab_include_config" checked="checked" /> <?php esc_html_e( 'wp-config', 'aegisbackup' ); ?></label>
						<label style="margin-right:12px;"><input type="checkbox" id="ab_include_core" /> <?php esc_html_e( 'Core', 'aegisbackup' ); ?></label>
						<input type="hidden" id="ab_db_export_mode" value="table" />
						<input type="hidden" id="ab_excludes" value="" />

						<div style="margin-top:10px;">
							<a href="#" class="button button-primary" id="ab_start_backup"><?php esc_html_e( 'Run Backup Now', 'aegisbackup' ); ?></a>
							<span class="aegisbackup-status" id="ab_backup_status"></span>
						</div>

						<div id="ab_progress_wrap" style="display:none; margin-top:12px;">
							<div class="aegisbackup-progress"><div class="aegisbackup-progress-bar" id="ab_progress_bar"></div></div>
							<div id="ab_backup_checklist" style="margin:10px 0 12px;"></div>
							<div id="ab_backup_download" style="margin:10px 0 12px;"></div>
							<pre class="aegisbackup-log" id="ab_progress_log"></pre>
						</div>
					</div>

					<div style="margin-top:16px; width:100%;">
					<h3 style="margin:0 0 8px;"><?php esc_html_e( 'Recent Backups (Last 5)', 'aegisbackup' ); ?></h3>
					<p class="description" style="margin-top:0;">
						<?php
						esc_html_e(
							'Packages are stored in wp-content/uploads/aegisbackup/. Use Download to fetch the ZIP, Restore to open Migration Restores, or Delete to remove the package.',
							'aegisbackup'
						);
						?>
					</p>

					<table class="widefat striped" style="width:100%;">
						<thead>
							<tr>
								<th><?php esc_html_e( 'Name', 'aegisbackup' ); ?></th>
								<th style="width:200px;"><?php esc_html_e( 'Date', 'aegisbackup' ); ?></th>
								<th style="width:260px;"><?php esc_html_e( 'Actions', 'aegisbackup' ); ?></th>
							</tr>
						</thead>
						<tbody>
						<?php if ( empty( $recent_packages_top5 ) ) : ?>
							<tr>
								<td colspan="3"><?php esc_html_e( 'No backups found yet. Click “Run Backup Now” to create your first package.', 'aegisbackup' ); ?></td>
							</tr>
						<?php else : ?>
							<?php foreach ( (array) $recent_packages_top5 as $rp ) : ?>
								<?php
								// Package folder name is the directory under uploads/aegisbackup/.
								$pkg_folder = isset( $rp['name'] ) ? sanitize_text_field( (string) $rp['name'] ) : '';
								$created    = isset( $rp['created'] ) ? sanitize_text_field( (string) $rp['created'] ) : '';

								$download_url = wp_nonce_url(
									add_query_arg(
										array(
											'action' => 'aegisbackup_download_package',
											'pkg'    => $pkg_folder,
										),
										admin_url( 'admin-post.php' )
									),
									'aegisbackup_download_package',
									'aegisbackup_nonce'
								);

								$delete_url = wp_nonce_url(
									add_query_arg(
										array(
											'action'   => 'aegisbackup_delete_package',
											'pkg'      => $pkg_folder,
											'redirect' => 'wordpress',
										),
										admin_url( 'admin-post.php' )
									),
									'aegisbackup_delete_package',
									'aegisbackup_nonce'
								);

								$restore_url = admin_url( 'admin.php?page=aegisbackup-restore&tab=migrate' );
								?>
								<tr>
									<td><strong><?php echo esc_html( $pkg_folder ); ?></strong></td>
									<td><?php echo esc_html( $created ); ?></td>
									<td>
										<a class="button button-small" href="<?php echo esc_url( $download_url ); ?>">
											<?php esc_html_e( 'Download', 'aegisbackup' ); ?>
										</a>
										<a class="button button-small" href="<?php echo esc_url( $restore_url ); ?>">
											<?php esc_html_e( 'Restore', 'aegisbackup' ); ?>
										</a>
										<a class="button button-small" href="<?php echo esc_url( $delete_url ); ?>"
										   onclick="return confirm('<?php echo esc_js( __( 'Delete this backup package?', 'aegisbackup' ) ); ?>');">
											<?php esc_html_e( 'Delete', 'aegisbackup' ); ?>
										</a>
									</td>
								</tr>
							<?php endforeach; ?>
						<?php endif; ?>
						</tbody>
					</table>
				</div>
				</div>
			</div>
			
			<div class="aegisbackup-card">
				<h2 style="margin-top:0;"><?php esc_html_e( 'Dashboard Charts', 'aegisbackup' ); ?></h2>
				<p class="description"><?php esc_html_e( 'Visual health and operational reporting for WordPress, database, services, backups, and restores.', 'aegisbackup' ); ?></p>

				<div class="ab-dash-charts">
					<div class="ab-dash-chart">
						<div style="font-weight:600; margin:4px 0 8px;">WordPress Health</div>
						<canvas id="ab_dash_chart_1" height="180"></canvas>
					</div>

					<div class="ab-dash-chart">
						<div style="font-weight:600; margin:4px 0 8px;">Database Health</div>
						<canvas id="ab_dash_chart_2" height="180"></canvas>
					</div>

					<div class="ab-dash-chart">
						<div style="font-weight:600; margin:4px 0 8px;">AegisBackup & Services</div>
						<canvas id="ab_dash_chart_3" height="180"></canvas>
					</div>

					<div class="ab-dash-chart">
						<div style="font-weight:600; margin:4px 0 8px;">Backups (14 days)</div>
						<canvas id="ab_dash_chart_4" height="180"></canvas>
					</div>

					<div class="ab-dash-chart">
						<div style="font-weight:600; margin:4px 0 8px;">Restores (14 days)</div>
						<canvas id="ab_dash_chart_5" height="180"></canvas>
					</div>

					<div class="ab-dash-chart">
						<div style="font-weight:600; margin:4px 0 8px;">Backup KPI Snapshot</div>
						<canvas id="ab_dash_chart_6" height="180"></canvas>
					</div>
				</div>
			</div>		
			
			<div class="aegisbackup-card">
				<h2 style="margin-top:0;"><?php esc_html_e( 'Recent Packages Report', 'aegisbackup' ); ?></h2>
				<p class="description"><?php esc_html_e( 'Latest WordPress migration/backup packages detected in wp-content/uploads/aegisbackup.', 'aegisbackup' ); ?></p>

				<table class="widefat striped">
					<thead>
						<tr>
							<th><?php esc_html_e( 'Package', 'aegisbackup' ); ?></th>
							<th style="width:180px;"><?php esc_html_e( 'Created', 'aegisbackup' ); ?></th>
							<th style="width:140px;"><?php esc_html_e( 'Size', 'aegisbackup' ); ?></th>
							<th><?php esc_html_e( 'Path', 'aegisbackup' ); ?></th>
						</tr>
					</thead>
					<tbody id="ab_dash_pkg_rows">
					<?php if ( empty( $recent_packages ) ) : ?>
						<tr><td colspan="4"><?php esc_html_e( 'No packages found yet. Create a backup to populate this report.', 'aegisbackup' ); ?></td></tr>
					<?php else : ?>
						<?php foreach ( (array) $recent_packages as $rp ) : ?>
							<tr>
								<td><strong><?php echo esc_html( isset( $rp['name'] ) ? (string) $rp['name'] : '(package)' ); ?></strong></td>
								<td><?php echo esc_html( isset( $rp['created'] ) ? (string) $rp['created'] : '' ); ?></td>
								<td><?php echo esc_html( isset( $rp['bytes'] ) ? size_format( (int) $rp['bytes'] ) : '' ); ?></td>
								<td><code style="white-space:nowrap;"><?php echo esc_html( isset( $rp['package'] ) ? (string) $rp['package'] : '' ); ?></code></td>
							</tr>
						<?php endforeach; ?>
					<?php endif; ?>
					</tbody>
				</table>

				<p class="description" style="margin-top:10px;"><?php esc_html_e( 'Tip: Use Prune to keep only the newest packages and free disk space.', 'aegisbackup' ); ?></p>
			</div>


<div class="aegisbackup-card">
				<h2 style="margin-top:0;"><?php esc_html_e( 'Fleet Reporting', 'aegisbackup' ); ?></h2>
				<p class="description"><?php esc_html_e( 'Shows saved Migration Wizard connections for this admin user and their last activity timestamps.', 'aegisbackup' ); ?></p>
				<table class="widefat striped">
					<thead>
						<tr>
							<th style="width:220px;"><?php esc_html_e( 'Connection', 'aegisbackup' ); ?></th>
							<th><?php esc_html_e( 'Destination URL', 'aegisbackup' ); ?></th>
							<th style="width:110px;"><?php esc_html_e( 'Connected', 'aegisbackup' ); ?></th>
							<th style="width:170px;"><?php esc_html_e( 'Last Activity', 'aegisbackup' ); ?></th>
							<th style="width:140px;"><?php esc_html_e( 'Days Since', 'aegisbackup' ); ?></th>
						</tr>
					</thead>
				<tbody id="ab_dash_fleet_rows">
					<?php if ( empty( $fleet_rows ) ) : ?>
						<tr><td colspan="5"><?php esc_html_e( 'No saved connections yet. Create one in Migration Wizard.', 'aegisbackup' ); ?></td></tr>
					<?php else : ?>
						<?php foreach ( (array) $fleet_rows as $fr ) : ?>
							<tr>
								<td><strong><?php echo esc_html( isset( $fr['name'] ) ? (string) $fr['name'] : '' ); ?></strong></td>
								<td><code><?php echo esc_html( isset( $fr['dest_url'] ) ? (string) $fr['dest_url'] : '' ); ?></code></td>
								<td><?php echo ! empty( $fr['connected'] ) ? esc_html__( 'Yes', 'aegisbackup' ) : esc_html__( 'No', 'aegisbackup' ); ?></td>
								<td><?php echo ! empty( $fr['last_activity'] ) ? esc_html( date_i18n( 'n/j/Y, g:i:s A', (int) $fr['last_activity'] ) ) : '—'; ?></td>
								<td><?php echo ( null !== $fr['age_days'] ) ? esc_html( (string) $fr['age_days'] . 'd' ) : '—'; ?></td>
							</tr>
						<?php endforeach; ?>
					<?php endif; ?>
				</tbody>
				</table>
			</div>
		</div>
		<?php
		}
	}
}
