<?php
namespace AegisBackup\Admin\Pages;

use AegisBackup\AB_Plugin;
use AegisBackup\Utils\License;

defined( 'ABSPATH' ) || exit;

class AB_Page_License {

    protected $plugin;

    public function __construct( AB_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function render() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'aegisbackup' ) );
        }

        $license = $this->plugin->license;
        if ( ! ( $license instanceof License ) ) {
            echo '<div class="wrap"><h1>' . esc_html__( 'AegisBackup', 'aegisbackup' ) . '</h1>';
            echo '<div class="notice notice-error"><p>' . esc_html__( 'License module not available.', 'aegisbackup' ) . '</p></div></div>';
            return;
        }

        $free_optin  = (int) get_option( License::OPTION_FREE_OPTIN, 0 );
        $free_status = (string) get_option( License::OPTION_FREE_STATUS, 'unregistered' );

        if ( ! empty( $_POST['aegisbackup_license_action'] ) ) {
            check_admin_referer( 'aegisbackup_license_action' );

            $key   = isset( $_POST['license_key'] ) ? sanitize_text_field( (string) wp_unslash( $_POST['license_key'] ) ) : '';
            $email = isset( $_POST['license_email'] ) ? sanitize_email( (string) wp_unslash( $_POST['license_email'] ) ) : '';

            $license->set_key( $key );
            update_option( License::OPTION_LICENSE_EMAIL, $email, false );

            $msg = 'saved';

            if ( isset( $_POST['aegisbackup_activate_license'] ) ) {
                $resp = $license->activate_now( defined( 'AEGISBACKUP_VERSION' ) ? AEGISBACKUP_VERSION : '' );
                $msg  = ! empty( $resp['success'] ) ? 'activated' : 'activate_failed';
            } elseif ( isset( $_POST['aegisbackup_deactivate_license'] ) ) {
                $resp = $license->deactivate_now( defined( 'AEGISBACKUP_VERSION' ) ? AEGISBACKUP_VERSION : '' );
                $msg  = ! empty( $resp['success'] ) ? 'deactivated' : 'deactivate_failed';
            } elseif ( isset( $_POST['aegisbackup_check_license'] ) ) {
                $license->check_now( defined( 'AEGISBACKUP_VERSION' ) ? AEGISBACKUP_VERSION : '' );
                $msg = 'checked';
            } elseif ( isset( $_POST['aegisbackup_register_free_install'] ) ) {
                $consent_raw = isset( $_POST['aegisbackup_free_reg_consent'] ) ? sanitize_text_field( wp_unslash( $_POST['aegisbackup_free_reg_consent'] ) ) : '';
                $consent     = ( '1' === (string) $consent_raw );

                if ( ! $consent ) {
                    $msg = 'consent_required';
                } else {
                    update_option( License::OPTION_FREE_OPTIN, 1, false );
                    update_option( License::OPTION_FREE_STATUS, 'unregistered', false );
                    delete_transient( 'aegisbackup_clm_free_reg_sent' );
                    $license->maybe_send_free_register( defined( 'AEGISBACKUP_VERSION' ) ? AEGISBACKUP_VERSION : '' );

                    $free_status = (string) get_option( License::OPTION_FREE_STATUS, 'unregistered' );
                    $msg = ( 'registered' === $free_status ) ? 'free_registered' : 'free_register_failed';
                }
            } elseif ( isset( $_POST['aegisbackup_unregister_free_install'] ) ) {
                update_option( License::OPTION_FREE_OPTIN, 0, false );
                update_option( License::OPTION_FREE_STATUS, 'unregistered', false );
                $msg = 'free_unregistered';
            }

            $_GET['aegisbackup_msg'] = $msg;

            $free_optin  = (int) get_option( License::OPTION_FREE_OPTIN, 0 );
            $free_status = (string) get_option( License::OPTION_FREE_STATUS, 'unregistered' );
        }

		$status = method_exists( $license, 'get_status' ) ? $license->get_status() : $license->get_cached_status();

		// Aegisify Integration: suite license overrides local status
		$suite_pro = ( function_exists( 'aegisify_is_pro' ) && aegisify_is_pro( true ) );

		$is_pro    = ( $suite_pro || ! empty( $status['is_pro'] ) );

		$plan      = isset( $status['plan'] ) ? (string) $status['plan'] : '';
		if ( $suite_pro && '' === $plan ) {
			$plan = 'pro';
		}

		$expires = '';
		if ( isset( $status['expires_at'] ) && '' !== (string) $status['expires_at'] ) {
			$expires = (string) $status['expires_at'];
		} elseif ( isset( $status['expires'] ) && '' !== (string) $status['expires'] ) {
			$expires = (string) $status['expires'];
		}

		$lic_state = isset( $status['status'] ) ? (string) $status['status'] : '';
		$is_active = $suite_pro || $is_pro || ( 'active' === strtolower( $lic_state ) );

        echo '<div class="wrap">';
        echo '<h1>' . esc_html__( 'AegisBackup', 'aegisbackup' ) . '</h1>';

        echo '<h2 style="margin-top:18px;">' . esc_html__( 'License Status', 'aegisbackup' ) . '</h2>';

        echo '<table class="widefat striped" style="max-width:600px;margin-bottom:20px;">';
        echo '<tbody>';

        echo '<tr><th scope="row">' . esc_html__( 'Status', 'aegisbackup' ) . '</th><td>';
        if ( $is_active ) {
            echo '<span style="color:#46b450;font-weight:600;">' . esc_html( $is_pro ? __( 'Active (Pro enabled)', 'aegisbackup' ) : __( 'Active', 'aegisbackup' ) ) . '</span>';
        } else {
            echo '<span style="color:#dc3232;font-weight:600;">' . esc_html__( 'Inactive', 'aegisbackup' ) . '</span>';
        }
        echo '</td></tr>';

        echo '<tr><th scope="row">' . esc_html__( 'Plan', 'aegisbackup' ) . '</th><td>' . esc_html( '' !== $plan ? $plan : '-' ) . '</td></tr>';
        echo '<tr><th scope="row">' . esc_html__( 'Expires At', 'aegisbackup' ) . '</th><td>' . esc_html( '' !== $expires ? $expires : '-' ) . '</td></tr>';
        echo '</tbody></table>';

        if ( isset( $_GET['aegisbackup_msg'] ) ) {
            $m = sanitize_key( (string) wp_unslash( $_GET['aegisbackup_msg'] ) );
            $map = array(
                'saved'                => array( 'success', __( 'License saved.', 'aegisbackup' ) ),
                'activated'            => array( 'success', __( 'License activated.', 'aegisbackup' ) ),
                'activate_failed'      => array( 'error', __( 'Activation failed. Please verify the key/email and try again.', 'aegisbackup' ) ),
                'deactivated'          => array( 'success', __( 'License deactivated.', 'aegisbackup' ) ),
                'deactivate_failed'    => array( 'error', __( 'Deactivation failed. Please try again.', 'aegisbackup' ) ),
                'checked'              => array( 'success', __( 'License status refreshed.', 'aegisbackup' ) ),
                'consent_required'     => array( 'error', __( 'Please confirm consent before registering this installation.', 'aegisbackup' ) ),
                'free_registered'      => array( 'success', __( 'Registered successfully with Aegisify.', 'aegisbackup' ) ),
                'free_register_failed' => array( 'error', __( 'Registration failed. Please try again later.', 'aegisbackup' ) ),
                'free_unregistered'    => array( 'success', __( 'Free-install registration set to Unregistered.', 'aegisbackup' ) ),
            );
            if ( isset( $map[ $m ] ) ) {
                echo '<div class="notice notice-' . esc_attr( $map[ $m ][0] ) . '"><p>' . esc_html( $map[ $m ][1] ) . '</p></div>';
            }
        }

        echo '<div style="margin:10px 0 14px;color:#50575e;font-size:13px;">'
            . esc_html__( 'Enter your license key and the email used at checkout. Activate to unlock PRO features and updates.', 'aegisbackup' )
            . '</div>';

        echo '<form method="post">';
        wp_nonce_field( 'aegisbackup_license_action' );
        echo '<input type="hidden" name="aegisbackup_license_action" value="1" />';

        echo '<table class="form-table" role="presentation"><tbody>';

        echo '<tr><th scope="row"><label for="license_key">' . esc_html__( 'License Key', 'aegisbackup' ) . '</label></th>';
        echo '<td><input type="text" id="license_key" name="license_key" class="regular-text" style="width:420px;" value="' . esc_attr( $license->get_key() ) . '" autocomplete="off" /></td></tr>';

        echo '<tr><th scope="row"><label for="license_email">' . esc_html__( 'License Email', 'aegisbackup' ) . '</label></th>';
        echo '<td><input type="email" id="license_email" name="license_email" class="regular-text" style="width:420px;" value="' . esc_attr( (string) get_option( License::OPTION_LICENSE_EMAIL, '' ) ) . '" autocomplete="email" /></td></tr>';

        echo '</tbody></table>';

        $confirm = esc_js( __( 'Deactivate this license on this site?', 'aegisbackup' ) );
        echo '<p>';
        echo '<button type="submit" class="button button-primary" name="aegisbackup_activate_license" value="1">' . esc_html__( 'Activate License', 'aegisbackup' ) . '</button> ';
        $onclick = "return confirm('" . $confirm . "');";
        echo '<button type="submit" class="button" name="aegisbackup_deactivate_license" value="1" onclick="' . esc_attr( $onclick ) . '">' . esc_html__( 'Deactivate', 'aegisbackup' ) . '</button> ';
        echo '<button type="submit" class="button button-secondary" name="aegisbackup_check_license" value="1">' . esc_html__( 'Check Status', 'aegisbackup' ) . '</button>';
        echo '</p>';

        echo '<hr style="margin:18px 0;" />';

        echo '<h2 style="margin:0 0 8px 0;">' . esc_html__( 'Optional: Register this free installation', 'aegisbackup' ) . '</h2>';
        echo '<div style="margin:0 0 10px;color:#50575e;font-size:13px; max-width:1100px;">'
            . esc_html__( 'This is optional and if you want update checks and license-related features through Aegisify, you can register this free installation. This will send your site domain, site URL, and the WordPress admin email to Aegisify. Your information will never be sold to any 3rd party.', 'aegisbackup' )
            . '</div>';

        echo '<label style="display:block; margin:10px 0;">';
        echo '<input type="checkbox" name="aegisbackup_free_reg_consent" value="1" ' . checked( 1, $free_optin, false ) . ' /> ';
        echo esc_html__( 'I consent to send the data described above to Aegisify.', 'aegisbackup' );
        echo '</label>';

        echo '<div style="margin:6px 0 10px 0;">';
        echo '<strong>' . esc_html__( 'Registration Status:', 'aegisbackup' ) . '</strong> ';
        echo '<span style="font-weight:600;">' . esc_html( ucfirst( $free_status ) ) . '</span>';
        echo '</div>';

        echo '<p style="margin:0 0 10px 0;">';
        echo '<button type="submit" class="button" name="aegisbackup_register_free_install" value="1">' . esc_html__( 'Register Free Installation', 'aegisbackup' ) . '</button> ';
        echo '<button type="submit" class="button" name="aegisbackup_unregister_free_install" value="1">' . esc_html__( 'Unregister Free Installation', 'aegisbackup' ) . '</button>';
        echo '</p>';

        echo '<hr style="margin:18px 0;" />';

        echo '<h2 style="margin:0 0 8px 0;">' . esc_html__( 'Need a license?', 'aegisbackup' ) . '</h2>';
        echo '<p style="margin:0;">' . sprintf(
            /* translators: %s: Aegisify account registration URL. */
            esc_html__( 'Create an account at %s and buy a license to unlock PRO features.', 'aegisbackup' ),
            '<a href="https://aegisify.com/register/" target="_blank" rel="noopener">https://aegisify.com/register/</a>'
        ) . '</p>';

        echo '</form>';
        echo '</div>';
    }
}
