<?php
namespace AegisBackup\Admin\Pages;

use AegisBackup\AB_Plugin;

defined( 'ABSPATH' ) || exit;

// phpcs:disable WordPress.Security.NonceVerification.Recommended

class AB_Page_Restore {
    protected $plugin;

    public function __construct( AB_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function render() {
        $tab = isset( $_GET['tab'] ) ? sanitize_key( (string) wp_unslash( $_GET['tab'] ) ) : 'migrate';
        if ( ! in_array( $tab, array( 'restore', 'migrate' ), true ) ) {
            $tab = 'restore';
        }

        $is_pro = false;
        if ( isset( $this->plugin ) && isset( $this->plugin->license ) && is_object( $this->plugin->license ) && method_exists( $this->plugin->license, 'is_pro_active' ) ) {
            $is_pro = (bool) $this->plugin->license->is_pro_active();
        }
        $license_url = admin_url( 'admin.php?page=aegisbackup-license' );
        $needs_lock  = ( ! $is_pro && in_array( $tab, array( 'migrate', 'restore' ), true ) );
                            ?>
        <div class="wrap aegisbackup-wrap">
            <h1><?php esc_html_e( 'Restore / Migrate', 'aegisbackup' ); ?></h1>

            <h2 class="nav-tab-wrapper">
                <a href="<?php echo esc_url( admin_url( 'admin.php?page=aegisbackup-restore&tab=migrate' ) ); ?>" class="nav-tab <?php echo ( 'migrate' === $tab ) ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'Migration Restores', 'aegisbackup' ); ?></a>
                <a href="<?php echo esc_url( admin_url( 'admin.php?page=aegisbackup-restore&tab=restore' ) ); ?>" class="nav-tab <?php echo ( 'restore' === $tab ) ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'WordPress Restores', 'aegisbackup' ); ?></a>
            </h2>

            <?php if ( $needs_lock ) : ?>
                <div class="notice notice-warning" style="margin:12px 0 16px;">
                    <p style="margin:0;">
                        <strong><?php esc_html_e( 'PRO Feature:', 'aegisbackup' ); ?></strong>
                        <?php esc_html_e( 'Restore workflows are available in AegisBackup Pro. Upgrade to unlock Migration Restores and WordPress Restore Wizards.', 'aegisbackup' ); ?>
                        <a href="<?php echo esc_url( $license_url ); ?>" class="button button-primary" style="margin-left:10px;">
                            <?php esc_html_e( 'Upgrade / Activate License', 'aegisbackup' ); ?>
                        </a>
                    </p>
                </div>

                <style>
                    .aegisbackup-pro-locked{
                        opacity:0.45 !important;
                        pointer-events:none !important;
                        filter:grayscale(1) !important;
                        user-select:none !important;
                    }
                </style>
                <div class="aegisbackup-pro-locked" aria-disabled="true">
            <?php endif; ?>

            <?php
            if ( 'migrate' === $tab ) {
                $this->render_migrate_tab();
            } else {
                $this->render_restore_tab();
            }
            ?>

            <?php if ( $needs_lock ) : ?>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    private function render_restore_tab() {
        $packages = $this->plugin->backup->list_packages();

        $is_wizard = ! empty( $_GET['ab_wizard'] );
        if ( $is_wizard ) {
            $this->render_restore_process_inventory( 'restore' );

            $wiz_kind = isset( $_GET['ab_kind'] ) ? sanitize_key( (string) wp_unslash( $_GET['ab_kind'] ) ) : 'package';

            if ( 'file_backup' === $wiz_kind ) {
                $id = isset( $_GET['id'] ) ? sanitize_key( (string) wp_unslash( $_GET['id'] ) ) : '';
                $this->render_simple_file_restore_wizard( $id );
                return;
            }

            if ( 'db_tools_snapshot' === $wiz_kind || 'db_snapshot' === $wiz_kind ) {
                $file = isset( $_GET['file'] ) ? sanitize_file_name( (string) wp_unslash( $_GET['file'] ) ) : '';
                $this->render_simple_db_snapshot_restore_wizard( $file );
                return;
            }

            if ( 'table_backup' === $wiz_kind ) {
                $id = isset( $_GET['id'] ) ? sanitize_text_field( (string) wp_unslash( $_GET['id'] ) ) : '';
                $this->render_simple_table_backup_restore_wizard( $id );
                return;
            }

            $selected_name = isset( $_GET['pkg'] ) ? sanitize_text_field( (string) wp_unslash( $_GET['pkg'] ) ) : '';
            $selected_pkg  = null;

            foreach ( (array) $packages as $p ) {
                if ( isset( $p['name'] ) && (string) $p['name'] === $selected_name ) {
                    $selected_pkg = $p;
                    break;
                }
            }

            if ( empty( $selected_pkg ) ) {
                echo '<div class="notice notice-error"><p>' . esc_html__( 'Restore Wizard: selected backup was not found.', 'aegisbackup' ) . '</p></div>';
                echo '<p><a class="button" href="' . esc_url( admin_url( 'admin.php?page=aegisbackup-restore&tab=restore' ) ) . '">' . esc_html__( 'Back to WordPress Restores', 'aegisbackup' ) . '</a></p>';
                return;
            }

            $pkg_path = ! empty( $selected_pkg['package'] ) ? (string) $selected_pkg['package'] : ( ! empty( $selected_pkg['dir'] ) ? (string) $selected_pkg['dir'] : '' );
            $created  = isset( $selected_pkg['created'] ) ? (string) $selected_pkg['created'] : '';
            $m        = isset( $selected_pkg['manifest'] ) && is_array( $selected_pkg['manifest'] ) ? (array) $selected_pkg['manifest'] : array();
            $label    = isset( $selected_pkg['name'] ) ? (string) $selected_pkg['name'] : '';
            if ( ! empty( $m['uploaded_filename'] ) ) {
                $label = (string) $m['uploaded_filename'];
            }

            $this->render_simple_restore_wizard( $pkg_path, $label, $created, $m );
            return;
        }

        $wp_pkgs   = array();
        $file_pkgs = array();
        $db_pkgs   = array();

        foreach ( (array) $packages as $pkg ) {
            $m = isset( $pkg['manifest'] ) && is_array( $pkg['manifest'] ) ? (array) $pkg['manifest'] : array();
            $flags = isset( $m['flags'] ) && is_array( $m['flags'] ) ? (array) $m['flags'] : array();
            $include_files = ! empty( $flags['include_files'] );
            $include_db    = ! empty( $flags['include_db'] );
            $include_core  = ! empty( $flags['include_core'] );

            if ( $include_files && $include_db && $include_core ) {
                $pkg['kind'] = 'package_wp';
                $wp_pkgs[] = $pkg;
            } elseif ( $include_files && ! $include_db ) {
                $pkg['kind'] = 'package_files';
                $file_pkgs[] = $pkg;
            } elseif ( $include_db && ! $include_files ) {
                $pkg['kind'] = 'package_db';
                $db_pkgs[] = $pkg;
            } else {
                $pkg['kind'] = 'package_wp';
                $wp_pkgs[] = $pkg;
            }
        }

        if ( isset( $this->plugin->file_backup ) && $this->plugin->file_backup ) {
            $file_backups = $this->plugin->file_backup->list_backups();
            foreach ( (array) $file_backups as $fb ) {
                $file_pkgs[] = array(
                    'kind'    => 'file_backup',
                    'id'      => isset( $fb['id'] ) ? (string) $fb['id'] : '',
                    'name'    => isset( $fb['file'] ) ? basename( (string) $fb['file'] ) : ( isset( $fb['id'] ) ? (string) $fb['id'] . '.zip' : '' ),
                    'created' => isset( $fb['created'] ) ? (string) $fb['created'] : '',
                    'size'    => isset( $fb['size'] ) ? (string) $fb['size'] : '',
                    'path'    => isset( $fb['file'] ) ? (string) $fb['file'] : '',
                    'manifest'=> isset( $fb['manifest'] ) && is_array( $fb['manifest'] ) ? (array) $fb['manifest'] : array(),
                );
            }
        }

        if ( method_exists( $this->plugin, 'db_tools' ) && $this->plugin->db_tools() ) {
            $mod = $this->plugin->db_tools();
            $dir = method_exists( $mod, 'get_db_tools_backup_dir' ) ? (string) $mod->get_db_tools_backup_dir() : '';
            if ( $dir && is_dir( $dir ) ) {
                $files = glob( trailingslashit( $dir ) . '*.sql' );
                if ( is_array( $files ) ) {
                    foreach ( $files as $f ) {
                        if ( ! is_file( $f ) ) { continue; }
                        $db_pkgs[] = array(
                            'kind'    => 'db_tools_snapshot',
                            'name'    => basename( (string) $f ),
                            'created' => gmdate( 'Y-m-d H:i:s', (int) filemtime( $f ) ),
                            'size'    => size_format( (int) filesize( $f ) ),
                            'path'    => (string) $f,
                            'file'    => basename( (string) $f ),
                        );
                    }
                }
            }
        }

        if ( isset( $this->plugin->table_backup ) && $this->plugin->table_backup ) {
            $tbs = $this->plugin->table_backup->list_backups_sorted();
            foreach ( (array) $tbs as $tb ) {
                $db_pkgs[] = array(
                    'kind'    => 'table_backup',
                    'id'      => isset( $tb['id'] ) ? (string) $tb['id'] : '',
                    'name'    => isset( $tb['name'] ) ? (string) $tb['name'] : ( isset( $tb['id'] ) ? (string) $tb['id'] : '' ),
                    'created' => isset( $tb['created'] ) ? (string) $tb['created'] : '',
                    'size'    => isset( $tb['size_human'] ) ? (string) $tb['size_human'] : ( isset( $tb['size'] ) ? size_format( (int) $tb['size'] ) : '' ),
                    'zip'     => isset( $tb['zip'] ) ? (string) $tb['zip'] : '',
                );
            }
        }

        ?>
        <?php $this->render_restore_process_inventory( 'restore' ); ?>
        <?php $this->render_disaster_recovery_box( $wp_pkgs ); ?>
        <div class="aegisbackup-card">
            <h2><?php esc_html_e( 'Restore Backups', 'aegisbackup' ); ?></h2>
            <p class="description"><?php esc_html_e( 'Select a backup package below and restore it into this site. Packages are grouped by type.', 'aegisbackup' ); ?></p>
            <?php $this->render_restore_table( __( 'WordPress Restores', 'aegisbackup' ), $wp_pkgs, 'wp' ); ?>
            <?php $this->render_restore_table( __( 'File & Folder Restores', 'aegisbackup' ), $file_pkgs, 'files' ); ?>
            <?php $this->render_restore_table( __( 'Database Restores', 'aegisbackup' ), $db_pkgs, 'db' ); ?>
        </div>


        <?php
    }

	private function render_restore_process_inventory( $tab ) {
		$tab = (string) $tab;
		$index = get_option( \AegisBackup\Restore\AB_Restore_Manager::JOBS_INDEX_OPTION, array() );
		if ( ! is_array( $index ) ) {
			$index = array();
		}

		$rows = array();
		foreach ( $index as $jid => $row ) {
			if ( ! is_array( $row ) ) {
				continue;
			}
			$type = isset( $row['type'] ) ? (string) $row['type'] : 'migration';
			if ( 'migrate' === $tab && 'migration' !== $type ) {
				continue;
			}
			$rows[ (string) $jid ] = $row;
		}

		if ( empty( $rows ) ) {
			return;
		}

		uasort( $rows, function( $a, $b ) {
			$au = isset( $a['updated'] ) ? (int) $a['updated'] : 0;
			$bu = isset( $b['updated'] ) ? (int) $b['updated'] : 0;
			if ( $au === $bu ) { return 0; }
			return ( $au > $bu ) ? -1 : 1;
		} );

		?>
		<div class="aegisbackup-card">
			<h2><?php esc_html_e( 'Restore Processes', 'aegisbackup' ); ?></h2>
			<p class="description"><?php esc_html_e( 'Running and recent restore jobs are listed here so you can leave the page and return later without losing visibility. Use Open to view the live log.', 'aegisbackup' ); ?></p>
			<table class="widefat striped" style="margin-top:10px;">
				<thead>
				<tr>
					<th><?php esc_html_e( 'Job ID', 'aegisbackup' ); ?></th>
					<th><?php esc_html_e( 'Status', 'aegisbackup' ); ?></th>
					<th><?php esc_html_e( 'Phase', 'aegisbackup' ); ?></th>
					<th><?php esc_html_e( 'Progress', 'aegisbackup' ); ?></th>
					<th><?php esc_html_e( 'Last update', 'aegisbackup' ); ?></th>
					<th><?php esc_html_e( 'Last message', 'aegisbackup' ); ?></th>
					<th><?php esc_html_e( 'Actions', 'aegisbackup' ); ?></th>
				</tr>
				</thead>
				<tbody>
				<?php foreach ( $rows as $jid => $row ) :
					$status = isset( $row['status'] ) ? (string) $row['status'] : '';
					$phase = isset( $row['phase'] ) ? (string) $row['phase'] : '';
					$progress = isset( $row['progress'] ) ? (int) $row['progress'] : 0;
					$updated = isset( $row['updated'] ) ? (int) $row['updated'] : 0;
					$last_tick = isset( $row['last_tick'] ) ? (int) $row['last_tick'] : 0;
					$last_log = isset( $row['last_log'] ) ? (string) $row['last_log'] : '';
					$stalled = ( 'running' === $status && $last_tick > 0 && ( time() - $last_tick ) > 300 );
					$proc_base = admin_url( 'admin-post.php?action=aegisbackup_restore_process_action' );
					$nonce = wp_create_nonce( 'aegisbackup_restore_process_action' );
					$open_url = add_query_arg( array( 'job_id' => $jid, 'do' => 'open', 'tab' => $tab, 'ab_restore_proc_nonce' => $nonce ), $proc_base );
					$restart_url = add_query_arg( array( 'job_id' => $jid, 'do' => 'restart', 'tab' => $tab, 'ab_restore_proc_nonce' => $nonce ), $proc_base );
					$stop_url = add_query_arg( array( 'job_id' => $jid, 'do' => 'stop', 'tab' => $tab, 'ab_restore_proc_nonce' => $nonce ), $proc_base );
					$delete_url = add_query_arg( array( 'job_id' => $jid, 'do' => 'delete', 'tab' => $tab, 'ab_restore_proc_nonce' => $nonce ), $proc_base );
					?>
					<tr>
						<td><code><?php echo esc_html( $jid ); ?></code></td>
						<td>
							<?php
							echo esc_html( $status ? ucfirst( $status ) : '-' );
							if ( $stalled ) {
								echo ' <span class="description" style="color:#b32d2e; font-weight:600;">' . esc_html__( '(Stalled?)', 'aegisbackup' ) . '</span>';
							}
							?>
						</td>
						<td><?php echo esc_html( $phase ? $phase : '-' ); ?></td>
						<td><?php echo esc_html( (string) $progress ); ?>%</td>
						<td>
							<?php
							if ( $updated > 0 ) {
								echo esc_html( human_time_diff( $updated, time() ) . ' ago' );
							} else {
								echo esc_html__( '-', 'aegisbackup' );
							}
							?>
						</td>
						<td><?php echo esc_html( $last_log ? $last_log : '-' ); ?></td>
						<td>
							<a class="button button-small" href="<?php echo esc_url( $open_url ); ?>"><?php esc_html_e( 'Open', 'aegisbackup' ); ?></a>
							<a class="button button-small" href="<?php echo esc_url( $restart_url ); ?>" onclick="return confirm('Restart this restore job?');"><?php esc_html_e( 'Restart', 'aegisbackup' ); ?></a>
							<a class="button button-small" href="<?php echo esc_url( $stop_url ); ?>" onclick="return confirm('Stop this restore job?');"><?php esc_html_e( 'Stop', 'aegisbackup' ); ?></a>
							<a class="button button-small" href="<?php echo esc_url( $delete_url ); ?>" onclick="return confirm('Delete this restore job and its stored logs?');"><?php esc_html_e( 'Delete', 'aegisbackup' ); ?></a>
						</td>
					</tr>
				<?php endforeach; ?>
				</tbody>
			</table>
		</div>
		<?php
	}

    private function render_migration_report_ui( $report ) {
        if ( empty( $report ) || ! is_array( $report ) ) {
            echo '<p class="description">' . esc_html__( 'No migration report available yet. Run a restore or click “Finalize Migration & Automate Fix” to generate a report.', 'aegisbackup' ) . '</p>';
            return;
        }

        $counts  = isset( $report['counts'] ) && is_array( $report['counts'] ) ? (array) $report['counts'] : array();
        $samples = isset( $report['samples'] ) && is_array( $report['samples'] ) ? (array) $report['samples'] : array();

        $attachments_verified  = isset( $counts['attachments_verified'] ) ? (int) $counts['attachments_verified'] : 0;
        $missing_files         = isset( $counts['attachments_missing_files'] ) ? (int) $counts['attachments_missing_files'] : 0;
        $corrupted_meta        = isset( $counts['attachments_corrupted_meta'] ) ? (int) $counts['attachments_corrupted_meta'] : 0;
        $thumbs_regenerated    = isset( $counts['thumbnails_regenerated'] ) ? (int) $counts['thumbnails_regenerated'] : 0;
        $zero_byte_files       = isset( $counts['zero_byte_files'] ) ? (int) $counts['zero_byte_files'] : 0;

        $failed_writes         = isset( $counts['failed_writes'] ) ? (int) $counts['failed_writes'] : 0;
        $hash_mismatch         = isset( $counts['hash_mismatch'] ) ? (int) $counts['hash_mismatch'] : 0;
        $caches_cleared        = isset( $counts['caches_cleared'] ) ? (int) $counts['caches_cleared'] : 0;

        $missing_samples = isset( $samples['missing_files'] ) && is_array( $samples['missing_files'] ) ? (array) $samples['missing_files'] : array();
        $corrupt_samples = isset( $samples['corrupted_meta'] ) && is_array( $samples['corrupted_meta'] ) ? (array) $samples['corrupted_meta'] : array();
        $zero_samples    = isset( $samples['zero_byte_files'] ) && is_array( $samples['zero_byte_files'] ) ? (array) $samples['zero_byte_files'] : array();
        $fw_samples      = isset( $samples['failed_writes'] ) && is_array( $samples['failed_writes'] ) ? (array) $samples['failed_writes'] : array();
        $hm_samples      = isset( $samples['hash_mismatch'] ) && is_array( $samples['hash_mismatch'] ) ? (array) $samples['hash_mismatch'] : array();

        ?>
        <style>
            .ab-report-grid { display:grid; grid-template-columns: repeat(2, minmax(220px, 1fr)); gap:10px; margin-top:10px; }
            .ab-report-stat { background:#fff; border:1px solid #ccd0d4; border-radius:4px; padding:10px 12px; }
            .ab-report-stat strong { display:block; font-size:13px; margin-bottom:4px; }
            .ab-report-stat .ab-num { font-size:20px; font-weight:600; }
            .ab-report-stat .ab-sub { color:#646970; font-size:12px; margin-top:4px; }
            details.ab-wp-accordion { background:#fff; border:1px solid #ccd0d4; border-radius:4px; margin:10px 0; }
            details.ab-wp-accordion > summary { padding:10px 12px; cursor:pointer; user-select:none; }
            details.ab-wp-accordion > summary:hover { background:#f6f7f7; }
            details.ab-wp-accordion[open] > summary { border-bottom:1px solid #ccd0d4; }
            .ab-wp-accordion .ab-accordion-body { padding:12px; }
            .ab-report-samples { background:#f6f7f7; border:1px solid #dcdcde; padding:10px; max-height:220px; overflow:auto; white-space:pre-wrap; }
        </style>

        <div class="ab-report-grid">
            <div class="ab-report-stat">
                <strong>✔ <?php esc_html_e( 'Attachments verified', 'aegisbackup' ); ?></strong>
                <div class="ab-num"><?php echo esc_html( (string) $attachments_verified ); ?></div>
                <div class="ab-sub"><?php esc_html_e( 'Checked Media Library attachments vs files on disk.', 'aegisbackup' ); ?></div>
            </div>
            <div class="ab-report-stat">
                <strong>✔ <?php esc_html_e( 'Thumbnails regenerated', 'aegisbackup' ); ?></strong>
                <div class="ab-num"><?php echo esc_html( (string) $thumbs_regenerated ); ?></div>
                <div class="ab-sub"><?php esc_html_e( 'Only when enabled (optional).', 'aegisbackup' ); ?></div>
            </div>
            <div class="ab-report-stat">
                <strong>⚠ <?php esc_html_e( 'Missing files', 'aegisbackup' ); ?></strong>
                <div class="ab-num"><?php echo esc_html( (string) $missing_files ); ?></div>
                <div class="ab-sub"><?php esc_html_e( 'Attachments that reference a file that is not present on disk.', 'aegisbackup' ); ?></div>
            </div>
            <div class="ab-report-stat">
                <strong>⚠ <?php esc_html_e( 'Corrupted metadata', 'aegisbackup' ); ?></strong>
                <div class="ab-num"><?php echo esc_html( (string) $corrupted_meta ); ?></div>
                <div class="ab-sub"><?php esc_html_e( 'Attachments where metadata could not be read/parsed (common after unsafe search/replace).', 'aegisbackup' ); ?></div>
            </div>
            <div class="ab-report-stat">
                <strong>⚠ <?php esc_html_e( 'Zero-byte files', 'aegisbackup' ); ?></strong>
                <div class="ab-num"><?php echo esc_html( (string) $zero_byte_files ); ?></div>
                <div class="ab-sub"><?php esc_html_e( 'Files that exist but have size 0 (interrupted copy/extract/disk quota).', 'aegisbackup' ); ?></div>
            </div>
            <div class="ab-report-stat">
                <strong>✔ <?php esc_html_e( 'Local caches cleared', 'aegisbackup' ); ?></strong>
                <div class="ab-num"><?php echo esc_html( (string) $caches_cleared ); ?></div>
                <div class="ab-sub"><?php esc_html_e( 'Object cache flush + common cache plugin hooks (no CDN/host purge).', 'aegisbackup' ); ?></div>
            </div>
        </div>

        <details class="ab-wp-accordion">
            <summary><strong><?php esc_html_e( 'During restore integrity', 'aegisbackup' ); ?></strong> — <?php echo esc_html( 'Failed writes: ' . $failed_writes . ' | Hash mismatches: ' . $hash_mismatch ); ?></summary>
            <div class="ab-accordion-body">
                <p class="description"><?php esc_html_e( 'AegisBackup copies files in binary-safe mode and verifies a SHA-256 hash after each copy. Files that cannot be written are retried and recorded as failed writes.', 'aegisbackup' ); ?></p>

                <?php if ( ! empty( $fw_samples ) ) : ?>
                    <details class="ab-wp-accordion">
                        <summary><strong><?php esc_html_e( 'View samples: failed writes', 'aegisbackup' ); ?></strong></summary>
                        <div class="ab-accordion-body"><div class="ab-report-samples"><?php echo esc_html( wp_json_encode( $fw_samples, JSON_PRETTY_PRINT ) ); ?></div></div>
                    </details>
                <?php endif; ?>

                <?php if ( ! empty( $hm_samples ) ) : ?>
                    <details class="ab-wp-accordion">
                        <summary><strong><?php esc_html_e( 'View samples: hash mismatches', 'aegisbackup' ); ?></strong></summary>
                        <div class="ab-accordion-body"><div class="ab-report-samples"><?php echo esc_html( wp_json_encode( $hm_samples, JSON_PRETTY_PRINT ) ); ?></div></div>
                    </details>
                <?php endif; ?>
            </div>
        </details>

        <?php if ( ! empty( $missing_samples ) ) : ?>
            <details class="ab-wp-accordion">
                <summary><strong><?php esc_html_e( 'View samples', 'aegisbackup' ); ?></strong> — <?php esc_html_e( 'Missing files', 'aegisbackup' ); ?></summary>
                <div class="ab-accordion-body"><div class="ab-report-samples"><?php echo esc_html( wp_json_encode( array_slice( $missing_samples, 0, 25 ), JSON_PRETTY_PRINT ) ); ?></div></div>
            </details>
        <?php endif; ?>

        <?php if ( ! empty( $zero_samples ) ) : ?>
            <details class="ab-wp-accordion">
                <summary><strong><?php esc_html_e( 'View samples', 'aegisbackup' ); ?></strong> — <?php esc_html_e( 'Zero-byte files', 'aegisbackup' ); ?></summary>
                <div class="ab-accordion-body"><div class="ab-report-samples"><?php echo esc_html( wp_json_encode( array_slice( $zero_samples, 0, 25 ), JSON_PRETTY_PRINT ) ); ?></div></div>
            </details>
        <?php endif; ?>

        <?php if ( ! empty( $corrupt_samples ) ) : ?>
            <details class="ab-wp-accordion">
                <summary><strong><?php esc_html_e( 'View samples', 'aegisbackup' ); ?></strong> — <?php esc_html_e( 'Corrupted metadata', 'aegisbackup' ); ?></summary>
                <div class="ab-accordion-body"><div class="ab-report-samples"><?php echo esc_html( wp_json_encode( array_slice( $corrupt_samples, 0, 25 ), JSON_PRETTY_PRINT ) ); ?></div></div>
            </details>
        <?php endif; ?>

        <?php
        ?>
        <details class="ab-wp-accordion">
            <summary><strong><?php esc_html_e( 'Raw report JSON (debug)', 'aegisbackup' ); ?></strong></summary>
            <div class="ab-accordion-body"><div class="ab-report-samples"><?php echo esc_html( wp_json_encode( $report, JSON_PRETTY_PRINT ) ); ?></div></div>
        </details>
        <?php
    }

    private function render_migrate_tab() {
        global $wpdb;

        $packages = $this->plugin->backup->list_packages();

        $selected_name = isset( $_GET['pkg'] ) ? sanitize_text_field( (string) wp_unslash( $_GET['pkg'] ) ) : '';
		$preupdate_mode = ! empty( $_GET['ab_preupdate'] );
		if ( $preupdate_mode ) {
			$last = get_option( 'aegisbackup_preupdate_last_snapshot', array() );
			if ( is_array( $last ) && ! empty( $last['package'] ) ) {
				$selected_name = basename( (string) $last['package'] );
			}
		}

        $uploaded_pkg = isset( $_GET['ab_pkg'] ) ? sanitize_text_field( (string) wp_unslash( $_GET['ab_pkg'] ) ) : '';
        if ( ! $selected_name && $uploaded_pkg ) {
            $selected_name = $uploaded_pkg;
        }

        $job_id = isset( $_GET['ab_job'] ) ? sanitize_text_field( (string) wp_unslash( $_GET['ab_job'] ) ) : '';
        $job_done = ! empty( $_GET['ab_done'] );
		$this->render_restore_process_inventory( 'migrate' );

        ?>
            <div class="aegisbackup-card">
                <h2><?php esc_html_e( 'Upload a Package', 'aegisbackup' ); ?></h2>
                <p class="description"><?php esc_html_e( 'Upload a migration package ZIP (for example received from another site).', 'aegisbackup' ); ?></p>
                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" enctype="multipart/form-data">
                    <?php wp_nonce_field( 'aegisbackup_upload_package', 'aegisbackup_nonce' ); ?>
                    <input type="hidden" name="action" value="aegisbackup_upload_package" />
                    <input type="file" name="ab_package_zip" accept=".zip" required />
                    <button class="button button-primary"><?php esc_html_e( 'Upload', 'aegisbackup' ); ?></button>
                </form>

                <?php
                if ( $uploaded_pkg ) :
                    $uploaded_label = $uploaded_pkg;
                    foreach ( (array) $packages as $pck ) {
                        if ( isset( $pck['name'] ) && (string) $pck['name'] === $uploaded_pkg ) {
                            $m = isset( $pck['manifest'] ) && is_array( $pck['manifest'] ) ? (array) $pck['manifest'] : array();
                            if ( ! empty( $m['uploaded_filename'] ) ) {
                                $uploaded_label = (string) $m['uploaded_filename'];
                            }
                            break;
                        }
                    }
                    ?>
                    <div class="notice notice-success inline" style="margin-top:10px;"><p>
                        <?php
                        printf(
/* translators: %s: uploaded package name */
                            esc_html__( 'Uploaded package: %s. Now select it in Step 1 below.', 'aegisbackup' ),
                            '<code>' . esc_html( $uploaded_label ) . '</code>'
                        );
                        ?>
                    </p></div>
                <?php endif; ?>
            </div>

            <div class="aegisbackup-card">
                <h2><?php esc_html_e( 'Migration Wizard', 'aegisbackup' ); ?></h2>
                <p class="description"><?php esc_html_e( 'This process is ideal for moves from shared hosting, VPS, dedicated servers, local development environments, or staging sites. Upload a migration backup package exported from your source WordPress AegisBackup Plugin and AegisBackup Migration will restore the database, files, and configuration into this installation.  During the restore, you can choose to overwrite the existing database or restore into a new database, adjust table prefixes, and optionally update WordPress domain URLs and configuration settings.', 'aegisbackup' ); ?></p>

                <?php
                $ab_js_enq = function_exists( 'wp_script_is' ) ? wp_script_is( 'aegisbackup-admin', 'enqueued' ) : false;
                if ( ! $ab_js_enq ) :
                    ?>
                    <div class="notice notice-warning inline"><p>
                        <?php esc_html_e( 'Diagnostics: aegisbackup/assets/admin.js is not enqueued on this page. If your host/minifier/WAF blocks plugin scripts, UI toggles and buttons may not respond.', 'aegisbackup' ); ?>
                    </p></div>
                    <?php
                endif;
                ?>

                <?php if ( ! empty( $_GET['ab_err'] ) ) : ?>
                    <div class="notice notice-error"><p>
                        <?php
                        $err = sanitize_key( (string) wp_unslash( $_GET['ab_err'] ) );
                        if ( 'confirm' === $err ) {
                            esc_html_e( 'Confirmation required. Please check the box and type RESTORE to continue.', 'aegisbackup' );
                        } elseif ( 'db_creds' === $err ) {
                            esc_html_e( 'Database credentials are required when restoring into a new database.', 'aegisbackup' );
                        } elseif ( 'no_pkg' === $err ) {
                            esc_html_e( 'Please select a package before starting restore.', 'aegisbackup' );
                        } elseif ( 'start_failed' === $err ) {
                            esc_html_e( 'Failed to start restore job.', 'aegisbackup' );
                        } elseif ( 'no_job' === $err ) {
                            esc_html_e( 'Missing restore job id.', 'aegisbackup' );
                        } else {
                            esc_html_e( 'Restore request failed.', 'aegisbackup' );
                        }
                        ?>
                    </p></div>
                <?php endif; ?>

                <?php if ( ! empty( $_GET['ab_msg'] ) ) : ?>
                    <div class="notice notice-success"><p>
                        <?php
                        $msg = sanitize_key( (string) wp_unslash( $_GET['ab_msg'] ) );
                        if ( 'tb_deleted' === $msg ) {
                            esc_html_e( 'Table backup deleted.', 'aegisbackup' );
                        } elseif ( 'dbsnap_deleted' === $msg ) {
                            esc_html_e( 'Database snapshot deleted.', 'aegisbackup' );
                        } elseif ( 'file_deleted' === $msg ) {
                            esc_html_e( 'File backup deleted.', 'aegisbackup' );
                        } else {
                            esc_html_e( 'Action completed.', 'aegisbackup' );
                        }
                        ?>
                    </p></div>
                <?php endif; ?>

                <?php
                if ( $job_id ) :
					$opt_key = \AegisBackup\Restore\AB_Restore_Manager::JOB_OPTION_PREFIX . $job_id;
					$state   = get_option( $opt_key, array() );

					$log_lines = array();
					if ( is_array( $state ) && ! empty( $state['log'] ) && is_array( $state['log'] ) ) {
						$log_lines = $state['log'];
					}

					$last = ( is_array( $state ) && ! empty( $state['last'] ) && is_array( $state['last'] ) ) ? $state['last'] : array();
                    ?>

					<div class="notice notice-info"><p>
						<?php
						if ( $job_done ) {
							esc_html_e( 'DB & Files Restore completed. Please run Step3 to Finalize Migration. Review the Migration Report below.', 'aegisbackup' );
						} else {
							esc_html_e( 'Depending on your server performance and the size of your WordPress package, the process typically completes within 2 hours. You may safely leave this page and return later to review the results.', 'aegisbackup' );
						}
						?>
					</p></div>

					<div id="ab_restore_job_meta" data-jobid="<?php echo esc_attr( $job_id ); ?>"></div>
<script>
(function(){
    try {
        var meta = document.getElementById('ab_restore_job_meta');
        if (!meta) { return; }
        var jobId = meta.getAttribute('data-jobid') || '';
        if (!jobId) { return; }

        var ajaxUrl = "<?php echo esc_js( admin_url( 'admin-ajax.php' ) ); ?>";
        var nonce   = "<?php echo esc_js( wp_create_nonce( 'aegisbackup_nonce' ) ); ?>";

        var bar = document.getElementById('ab_restore_progress_bar');
        var elStatus  = document.getElementById('ab_restore_status');
        var elPercent = document.getElementById('ab_restore_percent');
        var elTask    = document.getElementById('ab_restore_task');
        var elLog     = document.getElementById('ab_restore_progress_log');
        var elPre     = document.getElementById('ab_preflight_results');

        function escapeHtml(s){
            return String(s || '').replace(/[&<>"']/g, function(c){
                return {'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[c];
            });
        }

        function renderPreflight(pre){
            if (!elPre) { return; }
            if (!pre || typeof pre !== 'object') { return; }
            var out = [];
            for (var i=0;i<pre.length;i++){
                var r = pre[i];
                if (!r) { continue; }
                var ok = (r.ok === true);
                var icon = ok ? '✅' : (r.severity === 'warn' ? '⚠️' : '❌');
                var msg = (r.label || r.key || 'Check');
                var det = (r.details || '');
                out.push('<div style="margin:2px 0;">' + icon + ' <strong>' + escapeHtml(msg) + '</strong>' + (det ? ' <span class="description">' + escapeHtml(det) + '</span>' : '') + '</div>');
            }
            if (out.length) {
                elPre.innerHTML = out.join('');
            }
        }

        function setBar(pct){
            pct = Math.max(0, Math.min(100, parseInt(pct, 10) || 0));
            if (bar) { bar.style.width = pct + '%'; }
            if (elPercent) { elPercent.textContent = pct + '%'; }
        }

        function setStatusText(status, done){
            if (!elStatus) { return; }
            var s = status || '';
            if (done) {
                if (s === 'error' || s === 'failed' || s === 'stopped') {
                    elStatus.textContent = 'Restore Failed';
                } else {
                    elStatus.textContent = 'Restore Complete!';
                }
            } else {
                elStatus.textContent = s ? s : 'Running…';
            }
        }

        function update(){
            var form = new FormData();
            form.append('action', 'aegisbackup_restore_status');
            form.append('nonce', nonce);
            form.append('job_id', jobId);

            fetch(ajaxUrl, { method:'POST', credentials:'same-origin', body: form })
            .then(function(r){ return r.json(); })
            .then(function(resp){
                if (!resp || !resp.success || !resp.data) { return; }
                var d = resp.data;

                setBar(d.progress);

                // Show a human-ish task line.
                if (elTask) {
                    var phase = d.phase || '';
                    var last  = d.last_log || '';
                    elTask.textContent = last ? last : (phase ? phase : 'Working…');
                }

                if (elLog && Array.isArray(d.log)) {
                    elLog.textContent = d.log.join("\n");
                    elLog.scrollTop = elLog.scrollHeight;
                }

                renderPreflight(d.preflight);

                setStatusText(d.status, !!d.done);

                if (d.done) {
                    setBar(100);
                    clearInterval(window.__abRestorePoll);
                    window.__abRestorePoll = null;
                }
            }).catch(function(){});
        }

        update();
        window.__abRestorePoll = setInterval(update, 2000);
    } catch (e) {}
})();
</script>

					<div class="ab-box" style="margin-top:10px;">
						<strong><?php esc_html_e( 'Progress Log', 'aegisbackup' ); ?></strong>

						<div style="margin-top:10px;">
							<div style="height:14px; background:#dcdcde; border-radius:4px; overflow:hidden;">
								<div id="ab_restore_progress_bar" style="height:14px; width:1%; background:#2271b1;"></div>
							</div>
							<div class="description" style="margin-top:8px;">
								<?php esc_html_e( 'Status:', 'aegisbackup' ); ?>
								<strong id="ab_restore_status"><?php echo $job_done ? esc_html__( 'Done', 'aegisbackup' ) : esc_html__( 'Running…', 'aegisbackup' ); ?></strong>
								&nbsp;&nbsp;<?php esc_html_e( 'Progress:', 'aegisbackup' ); ?>
								<strong id="ab_restore_percent"><?php echo $job_done ? '100%' : '1%'; ?></strong>
							</div>
							<div class="description" style="margin-top:6px;">
								<?php esc_html_e( 'Current task:', 'aegisbackup' ); ?>
								<strong id="ab_restore_current_task"><?php echo ! empty( $last['step'] ) ? esc_html( (string) $last['step'] ) : ( $job_done ? esc_html__( 'Finished', 'aegisbackup' ) : esc_html__( 'Starting…', 'aegisbackup' ) ); ?></strong>
							</div>
						</div>

						<pre class="aegisbackup-log" style="max-height:260px; overflow:auto; margin-top:10px;" id="ab_restore_progress_log"><?php echo esc_html( implode( "\n", $log_lines ) ); ?></pre>

						<?php if ( ! empty( $last ) ) : ?>
							<div class="description" style="margin-top:8px;">
								<?php
								if ( ! empty( $last['step'] ) ) {
/* translators: %s: last completed step name */
									printf( esc_html__( 'Last step: %s', 'aegisbackup' ), esc_html( (string) $last['step'] ) );
								}
								?>
							</div>
						<?php endif; ?>
					</div>

					<div class="ab-box" style="margin-top:12px;">
						<strong><?php esc_html_e( 'Restore Progress Details', 'aegisbackup' ); ?></strong>
						<div id="ab_preflight_results" style="margin-top:8px;"></div>
								<p class="description" style="margin:6px 0 0;"><?php esc_html_e( 'This checklist updates live while the restore runs in the background.', 'aegisbackup' ); ?></p>
					</div>


                    <?php if ( $job_done ) : ?>
                        <?php $report = get_option( 'aegisbackup_last_migration_report', array() ); ?>
                        <div class="ab-box" style="margin-top:12px;">
                            <strong><?php esc_html_e( 'Migration Report', 'aegisbackup' ); ?></strong>
                            <div style="margin-top:8px;">
                                <?php $this->render_migration_report_ui( $report ); ?>
                            </div>
                        </div>
                    <?php endif; ?>

                    <hr>
                <?php endif; ?>

                <?php
                $s3_job_id = isset( $_GET['ab_s3'] ) ? sanitize_text_field( wp_unslash( $_GET['ab_s3'] ) ) : '';
                $s3_data = array();
                $s3_done = false;
                $s3_log_lines = array();
                if ( ! empty( $s3_job_id ) ) {
                    $s3_store_key = 'ab_step3_job_' . $s3_job_id . '_' . (int) get_current_user_id();
                    $s3_data = get_transient( $s3_store_key );
                    if ( is_array( $s3_data ) ) {
                        $s3_done = ! empty( $s3_data['done'] );
                        $s3_log_lines = ! empty( $s3_data['log'] ) && is_array( $s3_data['log'] ) ? $s3_data['log'] : array();
                    }
                }
                ?>

                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                    <?php wp_nonce_field( 'aegisbackup_restore_start', 'aegisbackup_restore_nonce' ); ?>
                    <input type="hidden" name="action" value="aegisbackup_restore_start" />

                <h3><?php esc_html_e( 'Step 1 — Select the AegisBackup Migration Package', 'aegisbackup' ); ?></h3>
				<?php if ( ! empty( $preupdate_mode ) ) : ?>
					<div class="notice notice-warning inline">
						<p><?php esc_html_e( 'Rollback Mode: The last pre-update snapshot has been pre-selected below. Restoring will roll your site back to the state just BEFORE the update ran.', 'aegisbackup' ); ?></p>
					</div>
				<?php endif; ?>

                <p class="description"><?php esc_html_e( 'Select your migration package zip file below.', 'aegisbackup' ); ?></p>

                <select id="ab_restore_package" name="package_path" class="large-text">
                    <option value=""><?php esc_html_e( '— Select —', 'aegisbackup' ); ?></option>
                    <?php foreach ( $packages as $pkg ) : ?>
                        <?php $pkg_path = ! empty( $pkg['package'] ) ? $pkg['package'] : ( ! empty( $pkg['dir'] ) ? $pkg['dir'] : '' ); ?>
                        <?php $sel = ( $selected_name && isset( $pkg['name'] ) && (string) $pkg['name'] === $selected_name ) ? 'selected' : ''; ?>
                        <?php
                            $label = isset( $pkg['name'] ) ? (string) $pkg['name'] : '';
                            $m = isset( $pkg['manifest'] ) && is_array( $pkg['manifest'] ) ? (array) $pkg['manifest'] : array();
                            if ( ! empty( $m['uploaded_filename'] ) ) {
                                $label = (string) $m['uploaded_filename'];
                            }
                        ?>
                        <option value="<?php echo esc_attr( $pkg_path ); ?>" <?php echo esc_attr( $sel ); ?>><?php echo esc_html( $label ); ?></option>
                    <?php endforeach; ?>
                </select>

                <hr>

                <h2><?php esc_html_e( 'Step 2 — Choose Database Restore Mode', 'aegisbackup' ); ?></h2>
                <p class="description">
                    <?php esc_html_e( 'Recommended: restore into the existing WordPress installation (uses current wp-config.php DB credentials). Advanced: restore into a different database.', 'aegisbackup' ); ?>
                </p>

                <p>
                    <label>
                        <input type="radio" name="ab_restore_mode" value="existing" checked>
                        <strong><?php esc_html_e( 'Restore into existing WordPress installation (recommended)', 'aegisbackup' ); ?></strong>
                    </label><br>
                    <span class="description">
                        <?php
/* translators: %1$s: database name, %2$s: table prefix */
                        printf( esc_html__( 'This will DROP and REPLACE all tables in database %1$s with prefix %2$s.', 'aegisbackup' ), '<code>' . esc_html( defined( 'DB_NAME' ) ? DB_NAME : '' ) . '</code>', '<code>' . esc_html( $wpdb->prefix ) . '</code>' );
                        ?>
                    </span>
                </p>

                <p>
                    <label>
                        <input type="radio" name="ab_restore_mode" value="newdb">
                        <strong><?php esc_html_e( 'Restore into a new database (advanced)', 'aegisbackup' ); ?></strong>
                    </label><br>
                    <span class="description"><?php esc_html_e( 'Use this if you created a new database/user in your hosting panel and want AegisBackup to restore into that database.', 'aegisbackup' ); ?></span>
                </p>

                <?php

                ?>
                <script>
                (function(){
                  function getMode(){
                    var els = document.querySelectorAll('input[name="ab_restore_mode"]');
                    for(var i=0;i<els.length;i++){ if(els[i].checked){ return els[i].value; } }
                    return 'existing';
                  }
                  function toggle(){
                    var mode = getMode();
                    var sec = document.getElementById('ab_wpconfig_section');
                    if(!sec){ return; }
                    if(mode === 'newdb'){
                      sec.style.display = 'block';
                    }else{
                      sec.style.display = 'none';
                      var cb1 = document.getElementById('ab_wpconfig_update');
                      if(cb1){ cb1.checked = false; }
                      var cb2 = document.getElementById('ab_wpconfig_regen_salts');
                      if(cb2){ cb2.checked = false; }
                      var ids = ['ab_db_name','ab_db_user','ab_db_pass','ab_db_host'];
                      for(var j=0;j<ids.length;j++){
                        var inp = document.getElementById(ids[j]);
                        if(inp){ inp.value = ''; }
                      }
                    }
                  }
                  document.addEventListener('change', function(e){
                    if(e && e.target && e.target.name === 'ab_restore_mode'){ toggle(); }
                  });
                  if(document.readyState === 'loading'){
                    document.addEventListener('DOMContentLoaded', toggle);
                  }else{
                    toggle();
                  }
                })();
                </script>

                <div id="ab_wpconfig_section" style="display:none; margin-top:10px;">

                <h2><?php esc_html_e( 'wp-config.php Rewrite (optional)', 'aegisbackup' ); ?></h2>
                <p class="description"><?php esc_html_e( 'For migrations, AegisBackup can update DB credentials, $table_prefix, and optionally regenerate salts. A backup of wp-config.php will be created first.', 'aegisbackup' ); ?></p>

                <p>
                    <label>
                        <input type="checkbox" id="ab_wpconfig_update" name="wpconfig_update" value="1">
                        <?php esc_html_e( 'Update wp-config.php after restore', 'aegisbackup' ); ?>
                    </label>
                </p>

                <div class="ab-grid-2">
                    <div>
                        <label for="ab_db_name"><strong><?php esc_html_e( 'DB Name', 'aegisbackup' ); ?></strong></label>
                        <input type="text" id="ab_db_name" name="db_name" class="regular-text" placeholder="database_name">
                    </div>
                    <div>
                        <label for="ab_db_user"><strong><?php esc_html_e( 'DB User', 'aegisbackup' ); ?></strong></label>
                        <input type="text" id="ab_db_user" name="db_user" class="regular-text" placeholder="database_user">
                    </div>
                    <div>
                        <label for="ab_db_pass"><strong><?php esc_html_e( 'DB Password', 'aegisbackup' ); ?></strong></label>
                        <input type="password" id="ab_db_pass" name="db_pass" class="regular-text" placeholder="••••••••">
                    </div>
                    <div>
                        <label for="ab_db_host"><strong><?php esc_html_e( 'DB Host', 'aegisbackup' ); ?></strong></label>
                        <input type="text" id="ab_db_host" name="db_host" class="regular-text" placeholder="localhost">
                    </div>
                </div>

                <p>
                    <label>
                        <input type="checkbox" id="ab_wpconfig_regen_salts" name="wpconfig_regen_salts" value="1">
                        <?php esc_html_e( 'Regenerate salts (AUTH_KEY, NONCE_SALT, etc.)', 'aegisbackup' ); ?>
                    </label>
                </p>

                <div class="notice notice-warning inline">
                    <p><?php esc_html_e( 'Note: Updating DB credentials can affect the site immediately after the restore completes. If wp-config.php is not writable, AegisBackup will show manual instructions in the Migration Report.', 'aegisbackup' ); ?></p>
                </div>

                

                </div>



                <div class="aegisbackup-grid">
                    <div>
                        <h3><?php esc_html_e( 'DB Prefix', 'aegisbackup' ); ?></h3>
                        <label><?php esc_html_e( 'New prefix (optional):', 'aegisbackup' ); ?></label>
                        <input type="text" id="ab_new_prefix" name="new_prefix" class="regular-text" placeholder="wp_">
                        <p class="description"><?php esc_html_e( 'If provided, AegisBackup will import DB then apply collision-safe prefix migration (tables + options/usermeta keys).', 'aegisbackup' ); ?></p>
                    </div>
                    <div>
                        <h3><?php esc_html_e( 'Domain / URL', 'aegisbackup' ); ?></h3>
                        <label><?php esc_html_e( 'Old domain (optional):', 'aegisbackup' ); ?></label>
                        <input type="text" id="ab_old_domain" name="old_domain" class="regular-text" placeholder="example.com">
                        <br>
                        <label><?php esc_html_e( 'New domain (optional):', 'aegisbackup' ); ?></label>
                        <input type="text" id="ab_new_domain" name="new_domain" class="regular-text" placeholder="newsite.com">
                        <p class="description"><?php esc_html_e( 'Phase 3 implements serialized-safe search/replace. v1 drafts the pipeline and UI only.', 'aegisbackup' ); ?></p>
                    </div>
                </div>

                <div class="aegisbackup-card" style="border-left:4px solid #d63638;">
                    <h3 style="margin-top:0;"><?php esc_html_e( 'Confirmation (required)', 'aegisbackup' ); ?></h3>
                    <p class="description"><?php esc_html_e( 'Restoring will overwrite the current site. Please confirm before continuing.', 'aegisbackup' ); ?></p>
                    <label>
                        <input type="checkbox" id="ab_confirm_drop" name="confirm_drop" value="1">
                        <?php esc_html_e( 'I understand this will delete/overwrite the current WordPress content (database tables + files).', 'aegisbackup' ); ?>
                    </label>
                    <p style="margin-top:10px;">
                        <label for="ab_confirm_text"><strong><?php esc_html_e( 'Type RESTORE to confirm:', 'aegisbackup' ); ?></strong></label><br>
                        <input type="text" id="ab_confirm_text" name="confirm_text" class="regular-text" placeholder="RESTORE">
                    </p>
                </div>

                <p style="margin-top:12px;">
                    <button class="button button-primary" type="submit"><?php esc_html_e( 'Run Restore', 'aegisbackup' ); ?></button>
                </p>

                </form>

                <div class="aegisbackup-card" style="border-left:4px solid #2271b1;">
                    <h3 style="margin-top:0;"><?php esc_html_e( 'Step 3 — Finalize Migration & Automate Fix', 'aegisbackup' ); ?></h3>
                    <p class="description"><?php esc_html_e( 'Step3 is required to run after a successful database and file restoration to fix the list below.', 'aegisbackup' ); ?></p>

                    <ul class="ul-disc" style="margin-top:8px;">
                        <li><?php esc_html_e( 'Serialized-safe search/replace (Old Domain → New Domain).', 'aegisbackup' ); ?></li>
                        <li><?php esc_html_e( 'Normalize upload options (upload_path / upload_url_path) to prevent broken media links.', 'aegisbackup' ); ?></li>
                        <li><?php esc_html_e( 'Sanity-check attachment metadata and report missing/corrupted items.', 'aegisbackup' ); ?></li>
                        <li><?php esc_html_e( 'Optional thumbnail regeneration (best-effort) to repair missing sizes.', 'aegisbackup' ); ?></li>
                        <li><?php esc_html_e( 'Uploads permission sanity check (best-effort).', 'aegisbackup' ); ?></li>
                    </ul>

                    <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin-top:12px;">
                        <?php wp_nonce_field( 'aegisbackup_step3_start', 'aegisbackup_step3_nonce' ); ?>
                        <input type="hidden" name="action" value="aegisbackup_step3_start" />

                        <div style="margin:10px 0 0 0;">
                            <label style="margin-right:14px;"><input type="checkbox" name="regen_thumbs" value="1" /> <?php esc_html_e( 'Regenerate thumbnails (optional)', 'aegisbackup' ); ?></label>
                            <label style="margin-right:14px;"><input type="checkbox" name="fix_perms" value="1" checked="checked" /> <?php esc_html_e( 'Fix uploads permissions (best-effort)', 'aegisbackup' ); ?></label>
                            <label><input type="checkbox" name="update_guid" value="1" /> <?php esc_html_e( 'Update GUIDs (advanced / not recommended)', 'aegisbackup' ); ?></label>
                        </div>

                        <p style="margin:12px 0 0 0;">
                            <button class="button button-secondary" type="submit">
                                <?php esc_html_e( 'Finalize Migration & Automate Fix', 'aegisbackup' ); ?>
                            </button>
                        </p>
                    </form>

                    <p class="description" style="margin-top:8px;"><?php esc_html_e( 'Results appear in the Migration Report once the job finishes.', 'aegisbackup' ); ?></p>

                    <div id="ab_step3_live_progress" style="margin-top:8px;">
                        <?php
                        if ( ! empty( $s3_job_id ) && is_array( $s3_data ) ) {
                            echo '<strong>' . esc_html__( 'Live Progress', 'aegisbackup' ) . '</strong>';
                            echo '<pre class="aegisbackup-log" style="max-height:220px; overflow:auto; margin-top:8px;" id="ab_step3_progress_log_inline">' . esc_html( implode( "\n", $s3_log_lines ) ) . '</pre>';
                            if ( ! $s3_done ) {
                                echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '" id="ab_step3_auto_continue_form" style="margin-top:10px;">';
                                wp_nonce_field( 'aegisbackup_step3_continue', 'aegisbackup_step3_nonce' );
                                echo '<input type="hidden" name="action" value="aegisbackup_step3_continue" />';
                                echo '<input type="hidden" name="s3_job_id" value="' . esc_attr( $s3_job_id ) . '" />';
                                echo '<button class="button button-secondary">' . esc_html__( 'Continue Step 3', 'aegisbackup' ) . '</button>';
                                echo '</form>';
                                echo '<script>document.addEventListener("DOMContentLoaded",function(){var f=document.getElementById("ab_step3_auto_continue_form");if(f){setTimeout(function(){try{f.submit();}catch(e){}},1500);}});</script>';
                            }
                        }
                        ?>
                    </div>
                </div>

                <hr>

                <h2><?php esc_html_e( 'Migration Report', 'aegisbackup' ); ?></h2>
                <p class="description"><?php esc_html_e( 'After completion, AegisBackup generates a report of what changed.', 'aegisbackup' ); ?></p>
                <div id="ab_migration_report" class="ab-box">
                    <?php $report = get_option( 'aegisbackup_last_migration_report', array() ); ?>
                    <?php $this->render_migration_report_ui( $report ); ?>
                </div>

                <hr>

                <h2><?php esc_html_e( 'Post-restore checklist', 'aegisbackup' ); ?></h2>
                <p class="description"><?php esc_html_e( 'Use the steps below after a restore completes to bring the site online cleanly and safely. Expand each section for detailed instructions.', 'aegisbackup' ); ?></p>

                <div class="ab-box" style="padding:12px;">

                    <details open style="margin:8px 0;">
                        <summary><strong><?php esc_html_e( '1) Verify the restore completed successfully', 'aegisbackup' ); ?></strong></summary>
                        <ul class="ul-disc" style="margin-top:10px;">
                            <li><?php esc_html_e( 'Review the Migration Report / logs for errors or failed steps.', 'aegisbackup' ); ?></li>
                            <li><?php esc_html_e( 'Visit the site frontend and wp-admin to confirm pages load and you can log in.', 'aegisbackup' ); ?></li>
                            <li><?php esc_html_e( 'If you restored into a new database, confirm the DB user has correct privileges (SELECT, INSERT, UPDATE, DELETE, CREATE, ALTER, DROP, INDEX).', 'aegisbackup' ); ?></li>
                        </ul>
                    </details>

                    <details style="margin:8px 0;">
                        <summary><strong><?php esc_html_e( '2) Flush permalinks (required)', 'aegisbackup' ); ?></strong></summary>
                        <div style="margin-top:10px;">
                            <ol style="margin-left:18px;">
                                <li><?php esc_html_e( 'Go to wp-admin → Settings → Permalinks.', 'aegisbackup' ); ?></li>
                                <li><?php esc_html_e( 'Do not change anything unless needed; simply click “Save Changes”.', 'aegisbackup' ); ?></li>
                                <li><?php esc_html_e( 'This rebuilds rewrite rules and fixes 404 errors after migrations/restores.', 'aegisbackup' ); ?></li>
                            </ol>
                        </div>
                    </details>

                    <details style="margin:8px 0;">
                        <summary><strong><?php esc_html_e( '3) Clear all caches (plugin + server/host + CDN)', 'aegisbackup' ); ?></strong></summary>
                        <ul class="ul-disc" style="margin-top:10px;">
                            <li><?php esc_html_e( 'Clear any WordPress caching plugin cache (examples: WP Rocket, LiteSpeed Cache, W3 Total Cache, etc.).', 'aegisbackup' ); ?></li>
                            <li><?php esc_html_e( 'Clear your hosting/server cache (many hosts have a “Purge Cache” button in the hosting control panel).', 'aegisbackup' ); ?></li>
                            <li><?php esc_html_e( 'If using a CDN (Cloudflare, etc.), purge CDN cache. If you changed DNS, consider “Purge Everything” once.', 'aegisbackup' ); ?></li>
                            <li><?php esc_html_e( 'If you use object cache (Redis/Memcached), flush it from the host panel or plugin.', 'aegisbackup' ); ?></li>
                        </ul>
                    </details>

                    <details style="margin:8px 0;">
                        <summary><strong><?php esc_html_e( '4) DNS changes (when moving to a new server)', 'aegisbackup' ); ?></strong></summary>
                        <div style="margin-top:10px;">
                            <p class="description"><?php esc_html_e( 'If the restore is moving the site to a new server, update DNS at your domain registrar or DNS provider. The exact UI differs by provider, but the steps below are universal.', 'aegisbackup' ); ?></p>
                            <ol style="margin-left:18px;">
                                <li><?php esc_html_e( 'Find your new hosting/server IP address (or the target record your host provides).', 'aegisbackup' ); ?></li>
                                <li><?php esc_html_e( 'Log in to your Domain Registrar / DNS provider (GoDaddy, Namecheap, Cloudflare, Google Domains, etc.).', 'aegisbackup' ); ?></li>
                                <li><?php esc_html_e( 'Locate DNS settings for your domain.', 'aegisbackup' ); ?></li>
                                <li><?php esc_html_e( 'Update the A record for the root domain (@) to the new server IP. If you use “www”, update the www record (A or CNAME) as well.', 'aegisbackup' ); ?></li>
                                <li><?php esc_html_e( 'Wait for DNS propagation. It can take minutes to 24–48 hours depending on TTL and provider.', 'aegisbackup' ); ?></li>
                            </ol>
                            <p class="description"><?php esc_html_e( 'Tip: Before changing DNS, set TTL lower (e.g., 300 seconds) a few hours in advance to speed up propagation.', 'aegisbackup' ); ?></p>
                        </div>
                    </details>

                    <details style="margin:8px 0;">
                        <summary><strong><?php esc_html_e( '5) Update site URL / search & replace (domain changes)', 'aegisbackup' ); ?></strong></summary>
                        <ul class="ul-disc" style="margin-top:10px;">
                            <li><?php esc_html_e( 'If your domain changed (oldsite.com → newsite.com), ensure WordPress “Site Address (URL)” and “WordPress Address (URL)” are correct.', 'aegisbackup' ); ?></li>
                            <li><?php esc_html_e( 'Run a serialized-safe search/replace for old domain → new domain to update links in content, widgets, and settings.', 'aegisbackup' ); ?></li>
                            <li><?php esc_html_e( 'After search/replace, re-save permalinks and clear caches again.', 'aegisbackup' ); ?></li>
                        </ul>
                    </details>

                    <details style="margin:8px 0;">
                        <summary><strong><?php esc_html_e( '6) SSL / HTTPS checks', 'aegisbackup' ); ?></strong></summary>
                        <ul class="ul-disc" style="margin-top:10px;">
                            <li><?php esc_html_e( 'Confirm an SSL certificate is installed for the domain on the new host (Let’s Encrypt or a paid cert).', 'aegisbackup' ); ?></li>
                            <li><?php esc_html_e( 'If using Cloudflare or a proxy, confirm SSL mode is correct (avoid “Flexible” unless necessary).', 'aegisbackup' ); ?></li>
                            <li><?php esc_html_e( 'Ensure WordPress URLs are https:// and fix mixed-content warnings (images/scripts still loading over http).', 'aegisbackup' ); ?></li>
                        </ul>
                    </details>

                    <details style="margin:8px 0;">
                        <summary><strong><?php esc_html_e( '7) Verify wp-config.php, database, and table prefix', 'aegisbackup' ); ?></strong></summary>
                        <ul class="ul-disc" style="margin-top:10px;">
                            <li><?php esc_html_e( 'If you restored into a new database, confirm DB_NAME, DB_USER, DB_PASSWORD, and DB_HOST are correct.', 'aegisbackup' ); ?></li>
                            <li><?php esc_html_e( 'Confirm $table_prefix matches the restored tables. If you migrated prefixes, verify tables exist with the new prefix.', 'aegisbackup' ); ?></li>
                            <li><?php esc_html_e( 'If wp-config.php was not writable, apply the manual steps shown in the Migration Report.', 'aegisbackup' ); ?></li>
                        </ul>
                    </details>

                    <details style="margin:8px 0;">
                        <summary><strong><?php esc_html_e( '8) Re-check cron jobs, email, and integrations', 'aegisbackup' ); ?></strong></summary>
                        <ul class="ul-disc" style="margin-top:10px;">
                            <li><?php esc_html_e( 'Confirm scheduled tasks (WP-Cron or server cron) are running.', 'aegisbackup' ); ?></li>
                            <li><?php esc_html_e( 'Test critical forms (contact forms, checkout, login/registration) and ensure email delivery works.', 'aegisbackup' ); ?></li>
                            <li><?php esc_html_e( 'If using SMTP/SES, confirm credentials and sending domain/DKIM settings are still valid.', 'aegisbackup' ); ?></li>
                        </ul>
                    </details>

                    <details style="margin:8px 0;">
                        <summary><strong><?php esc_html_e( '9) Security + performance quick checks', 'aegisbackup' ); ?></strong></summary>
                        <ul class="ul-disc" style="margin-top:10px;">
                            <li><?php esc_html_e( 'Confirm admin login works and user roles/capabilities look correct.', 'aegisbackup' ); ?></li>
                            <li><?php esc_html_e( 'Re-scan for malware and confirm file permissions are correct on the new host.', 'aegisbackup' ); ?></li>
                            <li><?php esc_html_e( 'If you use a firewall/WAF, confirm it is not blocking legitimate wp-admin requests after restore.', 'aegisbackup' ); ?></li>
                        </ul>
                    </details>

                    <details style="margin:8px 0;">
                        <summary><strong><?php esc_html_e( '10) Make a fresh backup after go-live', 'aegisbackup' ); ?></strong></summary>
                        <ul class="ul-disc" style="margin-top:10px;">
                            <li><?php esc_html_e( 'Once the site is confirmed stable, create a new full backup on the new host.', 'aegisbackup' ); ?></li>
                            <li><?php esc_html_e( 'Store at least one copy offsite (cloud storage / remote).', 'aegisbackup' ); ?></li>
                        </ul>
                    </details>

                </div>

            </div>
        <?php
    }

private function render_disaster_recovery_box( array $wp_pkgs ) {
    if ( empty( $wp_pkgs ) ) {
        return;
    }

    usort( $wp_pkgs, function( $a, $b ) {
        $ac = isset( $a['created'] ) ? strtotime( (string) $a['created'] ) : 0;
        $bc = isset( $b['created'] ) ? strtotime( (string) $b['created'] ) : 0;
        if ( $ac === $bc ) { return 0; }
        return ( $ac > $bc ) ? -1 : 1;
    } );

    $latest = $wp_pkgs[0];
    $latest_name = isset( $latest['name'] ) ? (string) $latest['name'] : '';
    $latest_created = isset( $latest['created'] ) ? (string) $latest['created'] : '';

    $latest_url = add_query_arg(
        array(
            'page'      => 'aegisbackup-restore',
            'tab'       => 'restore',
            'ab_wizard' => 1,
            'pkg'       => $latest_name,
            'ab_dr'     => 1,
        ),
        admin_url( 'admin.php' )
    );

    ?>
    <div class="aegisbackup-card" style="margin-bottom:14px;">
        <h2 style="margin-top:0;"><?php esc_html_e( 'Disaster Recovery Mode', 'aegisbackup' ); ?></h2>
        <p class="description">
            <?php esc_html_e( 'Use a dated recovery point link to launch a safe, step-by-step Restore Wizard. This is designed for non-technical admins to restore the correct snapshot without changing advanced settings.', 'aegisbackup' ); ?>
        </p>

        <div class="notice notice-warning inline" style="margin:10px 0;">
            <p><strong><?php esc_html_e( 'Warning:', 'aegisbackup' ); ?></strong>
                <?php esc_html_e( 'Restoring a WordPress package will overwrite your current database + files (themes/plugins/uploads) based on what the package contains. Always confirm you selected the correct date.', 'aegisbackup' ); ?>
            </p>
        </div>

        <p>
            <a class="button button-primary" href="<?php echo esc_url( $latest_url ); ?>">
                <?php esc_html_e( 'Restore Latest Recovery Point', 'aegisbackup' ); ?>
            </a>
            <span class="description" style="margin-left:10px;">
                <?php
                if ( $latest_created ) {
                    printf(
/* translators: %s: date/time of latest backup */
                        esc_html__( 'Latest backup date: %s', 'aegisbackup' ),
                        '<code>' . esc_html( $latest_created ) . '</code>'
                    );
                }
                ?>
            </span>
        </p>

        <details class="ab-wp-accordion">
            <summary><strong><?php esc_html_e( 'Recovery point links (dated)', 'aegisbackup' ); ?></strong></summary>
            <div class="ab-accordion-body">
                <table class="widefat striped" style="table-layout:fixed; width:100%;">
                    <colgroup>
                        <col style="width:55%;" />
                        <col style="width:20%;" />
                        <col style="width:25%;" />
                    </colgroup>
                    <thead>
                        <tr>
                            <th><?php esc_html_e( 'Backup', 'aegisbackup' ); ?></th>
                            <th><?php esc_html_e( 'Created', 'aegisbackup' ); ?></th>
                            <th><?php esc_html_e( 'Link', 'aegisbackup' ); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        $max = 12;
                        $i = 0;
                        foreach ( $wp_pkgs as $pkg ) :
                            if ( $i >= $max ) { break; }
                            $name = isset( $pkg['name'] ) ? (string) $pkg['name'] : '';
                            if ( '' === $name ) { continue; }
                            $created = isset( $pkg['created'] ) ? (string) $pkg['created'] : '';
                            $url = add_query_arg(
                                array(
                                    'page'      => 'aegisbackup-restore',
                                    'tab'       => 'restore',
                                    'ab_wizard' => 1,
                                    'pkg'       => $name,
                                    'ab_dr'     => 1,
                                ),
                                admin_url( 'admin.php' )
                            );
                            $i++;
                            ?>
                            <tr>
                                <td><code style="display:block;white-space:normal;word-break:break-all;"><?php echo esc_html( $name ); ?></code></td>
                                <td><?php echo esc_html( $created ); ?></td>
                                <td><a class="button button-small" href="<?php echo esc_url( $url ); ?>"><?php esc_html_e( 'Open Restore Wizard', 'aegisbackup' ); ?></a></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <p class="description" style="margin-top:10px;">
                    <?php esc_html_e( 'Tip: Bookmark the specific recovery point link you trust. It will always open the Restore Wizard with that package pre-selected.', 'aegisbackup' ); ?>
                </p>
            </div>
        </details>
    </div>
    <?php
}

private function render_simple_restore_wizard( $pkg_path, $label, $created, array $manifest ) {
    $pkg_path = (string) $pkg_path;
    $label    = (string) $label;
    $created  = (string) $created;

    $return_url = admin_url( 'admin.php?page=aegisbackup-restore&tab=restore' );
    $dr_mode = ! empty( $_GET['ab_dr'] );

    $flags = isset( $manifest['flags'] ) && is_array( $manifest['flags'] ) ? (array) $manifest['flags'] : array();
    $include_files = ! empty( $flags['include_files'] );
    $include_db    = ! empty( $flags['include_db'] );
    $include_core  = ! empty( $flags['include_core'] );

    $summary_bits = array();
    if ( $include_core ) { $summary_bits[] = __( 'Core', 'aegisbackup' ); }
    if ( $include_files ) { $summary_bits[] = __( 'Files', 'aegisbackup' ); }
    if ( $include_db ) { $summary_bits[] = __( 'Database', 'aegisbackup' ); }
    if ( empty( $summary_bits ) ) { $summary_bits[] = __( 'Package contents unknown', 'aegisbackup' ); }

    ?>
    <div class="aegisbackup-card" style="margin-top:14px;">
        <h2 style="margin-top:0;">
            <?php echo $dr_mode ? esc_html__( 'Disaster Recovery — Restore Wizard', 'aegisbackup' ) : esc_html__( 'Restore Wizard', 'aegisbackup' ); ?>
        </h2>

        <p class="description">
            <?php esc_html_e( 'This guided restore is designed for non-technical admins. It uses safe defaults and prevents accidental changes to advanced migration options.', 'aegisbackup' ); ?>
        </p>

        <div class="notice notice-warning inline" style="margin:10px 0;">
            <p><strong><?php esc_html_e( 'Important:', 'aegisbackup' ); ?></strong>
                <?php esc_html_e( 'This restore will overwrite your current site based on the selected recovery point. Double-check the date before continuing.', 'aegisbackup' ); ?>
            </p>
        </div>

        <table class="widefat striped" style="table-layout:fixed; width:100%; margin-top:10px;">
            <colgroup>
                <col style="width:25%;" />
                <col style="width:75%;" />
            </colgroup>
            <tbody>
                <tr>
                    <th><?php esc_html_e( 'Recovery Point', 'aegisbackup' ); ?></th>
                    <td><code style="display:block;white-space:normal;word-break:break-all;"><?php echo esc_html( $label ); ?></code></td>
                </tr>
                <tr>
                    <th><?php esc_html_e( 'Date', 'aegisbackup' ); ?></th>
                    <td><?php echo $created ? '<code>' . esc_html( $created ) . '</code>' : esc_html__( '(unknown)', 'aegisbackup' ); ?></td>
                </tr>
                <tr>
                    <th><?php esc_html_e( 'Includes', 'aegisbackup' ); ?></th>
                    <td><?php echo esc_html( implode( ', ', $summary_bits ) ); ?></td>
                </tr>
            </tbody>
        </table>

        <hr>

        <?php if ( ! empty( $_GET['ab_err'] ) ) : ?>
            <div class="notice notice-error"><p>
                <?php
                $err = sanitize_key( (string) wp_unslash( $_GET['ab_err'] ) );
                if ( 'confirm' === $err ) {
                    esc_html_e( 'Confirmation required. Please check the box and type RESTORE to continue.', 'aegisbackup' );
                } elseif ( 'no_pkg' === $err ) {
                    esc_html_e( 'Missing package path.', 'aegisbackup' );
                } elseif ( 'start_failed' === $err ) {
                    esc_html_e( 'Failed to start restore job.', 'aegisbackup' );
                } else {
                    esc_html_e( 'Restore request failed.', 'aegisbackup' );
                }
                ?>
            </p></div>
        <?php endif; ?>

        <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin-top:10px;">
            <?php wp_nonce_field( 'aegisbackup_restore_start', 'aegisbackup_restore_nonce' ); ?>
            <input type="hidden" name="action" value="aegisbackup_restore_start" />
            <input type="hidden" name="package_path" value="<?php echo esc_attr( $pkg_path ); ?>" />
            <input type="hidden" name="ab_restore_mode" value="existing" />
            <input type="hidden" name="ab_redirect" value="<?php echo esc_attr( $return_url ); ?>" />

            <h3 style="margin-top:0;"><?php esc_html_e( 'Step 1 — Confirm overwrite', 'aegisbackup' ); ?></h3>

            <p>
                <label>
                    <input type="checkbox" name="confirm_drop" value="1">
                    <strong><?php esc_html_e( 'Yes, overwrite my current site using this recovery point', 'aegisbackup' ); ?></strong>
                </label>
            </p>

            <p>
                <label for="ab_confirm_text"><strong><?php esc_html_e( 'Type RESTORE to confirm', 'aegisbackup' ); ?></strong></label><br>
                <input type="text" id="ab_confirm_text" name="confirm_text" class="regular-text" placeholder="RESTORE" autocomplete="off" />
                <span class="description" style="margin-left:8px;"><?php esc_html_e( 'This prevents accidental restores.', 'aegisbackup' ); ?></span>
            </p>

            <p style="margin-top:12px;">
                <button class="button button-primary"><?php esc_html_e( 'Start Restore in Background', 'aegisbackup' ); ?></button>
                <a class="button" href="<?php echo esc_url( $return_url ); ?>" style="margin-left:6px;"><?php esc_html_e( 'Cancel', 'aegisbackup' ); ?></a>
            </p>

            <p class="description">
                <?php esc_html_e( 'After you start, the restore runs in the background. You can safely leave this page and return later. Your restore process will appear in “Restore Processes” with a live log.', 'aegisbackup' ); ?>
            </p>

            <p class="description">
                <?php
                $adv_url = add_query_arg(
                    array(
                        'page' => 'aegisbackup-restore',
                        'tab'  => 'migrate',
                        'pkg'  => isset( $_GET['pkg'] ) ? sanitize_text_field( (string) wp_unslash( $_GET['pkg'] ) ) : '',
                    ),
                    admin_url( 'admin.php' )
                );
                ?>
                <a href="<?php echo esc_url( $adv_url ); ?>"><?php esc_html_e( 'Need advanced options (new DB, prefix change, domain replace)? Use the Advanced Migration Wizard.', 'aegisbackup' ); ?></a>
            </p>
        </form>
    </div>
    <?php
}

    private function render_simple_file_restore_wizard( $backup_id ) {
        $backup_id = sanitize_text_field( (string) $backup_id );
		if ( $backup_id && ! preg_match( '/^[A-Za-z0-9._-]+$/', $backup_id ) ) {
			$backup_id = '';
		}
        $return_url = admin_url( 'admin.php?page=aegisbackup-restore&tab=restore' );

        if ( empty( $backup_id ) || empty( $this->plugin->file_backup ) ) {
            echo '<div class="notice notice-error"><p>' . esc_html__( 'Restore Wizard: selected file backup was not found.', 'aegisbackup' ) . '</p></div>';
            echo '<p><a class="button" href="' . esc_url( $return_url ) . '">' . esc_html__( 'Back to WordPress Restores', 'aegisbackup' ) . '</a></p>';
            return;
        }

        $found = null;
        $list = $this->plugin->file_backup->list_backups();
        foreach ( (array) $list as $b ) {
            if ( isset( $b['id'] ) && (string) $b['id'] === $backup_id ) {
                $found = $b;
                break;
            }
        }

        if ( empty( $found ) ) {
            echo '<div class="notice notice-error"><p>' . esc_html__( 'Restore Wizard: selected file backup was not found.', 'aegisbackup' ) . '</p></div>';
            echo '<p><a class="button" href="' . esc_url( $return_url ) . '">' . esc_html__( 'Back to WordPress Restores', 'aegisbackup' ) . '</a></p>';
            return;
        }

        $label   = isset( $found['file'] ) ? basename( (string) $found['file'] ) : $backup_id;
        $created = isset( $found['created'] ) ? (string) $found['created'] : '';
        $size    = isset( $found['size'] ) ? (string) $found['size'] : '';

        ?>
        <div class="aegisbackup-card" style="margin-top:14px;">
            <h2 style="margin-top:0;"><?php esc_html_e( 'Restore Wizard — Files & Folders', 'aegisbackup' ); ?></h2>
            <p class="description"><?php esc_html_e( 'This guided restore is designed for non-technical admins. It uses safe defaults and prevents accidental restores.', 'aegisbackup' ); ?></p>

            <div class="notice notice-warning inline" style="margin:10px 0;">
                <p><strong><?php esc_html_e( 'Important:', 'aegisbackup' ); ?></strong>
                    <?php esc_html_e( 'This restore overwrites existing files on this site for the folders/files included in the backup.', 'aegisbackup' ); ?>
                </p>
            </div>

            <table class="widefat striped" style="table-layout:fixed; width:100%; margin-top:10px;">
                <colgroup>
                    <col style="width:25%;" />
                    <col style="width:75%;" />
                </colgroup>
                <tbody>
                    <tr>
                        <th><?php esc_html_e( 'Backup', 'aegisbackup' ); ?></th>
                        <td><code style="display:block;white-space:normal;word-break:break-all;"><?php echo esc_html( $label ); ?></code></td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e( 'Date', 'aegisbackup' ); ?></th>
                        <td><?php echo $created ? '<code>' . esc_html( $created ) . '</code>' : esc_html__( '(unknown)', 'aegisbackup' ); ?></td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e( 'Size', 'aegisbackup' ); ?></th>
                        <td><?php echo $size ? esc_html( $size ) : esc_html__( '(unknown)', 'aegisbackup' ); ?></td>
                    </tr>
                </tbody>
            </table>

            <hr>

            <?php if ( ! empty( $_GET['ab_msg'] ) && 'file_restore_done' === sanitize_key( (string) wp_unslash( $_GET['ab_msg'] ) ) ) : ?>
                <div class="notice notice-success inline" style="margin:10px 0;"><p>
                    <?php
                    $restored = isset( $_GET['ab_restored'] ) ? (int) $_GET['ab_restored'] : 0;
                    $skipped  = isset( $_GET['ab_skipped'] ) ? (int) $_GET['ab_skipped'] : 0;
                    printf(
/* translators: %1$d: number of restored files, %2$d: number of skipped items */
                        esc_html__( 'File restore complete. Restored %1$d files. Skipped %2$d items.', 'aegisbackup' ),
                        (int) $restored,
                        (int) $skipped
                    );
                    ?>
                </p></div>
            <?php endif; ?>

            <?php if ( ! empty( $_GET['ab_err'] ) ) : ?>
                <div class="notice notice-error"><p>
                    <?php
                    $err = sanitize_key( (string) wp_unslash( $_GET['ab_err'] ) );
                    if ( 'confirm' === $err ) {
                        esc_html_e( 'Confirmation required. Please check the box and type RESTORE to continue.', 'aegisbackup' );
                    } elseif ( 'restore_failed' === $err ) {
                        esc_html_e( 'File restore failed to start or failed to complete.', 'aegisbackup' );
                        if ( ! empty( $_GET['ab_emsg'] ) ) {
                             $emsg = sanitize_text_field( rawurldecode( (string) wp_unslash( $_GET['ab_emsg'] ) ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized via sanitize_text_field().
                            echo ' ' . esc_html( $emsg );
                        }
                    } else {
                        esc_html_e( 'Restore request failed.', 'aegisbackup' );
                    }
                    ?>
                </p></div>
            <?php endif; ?>

            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin-top:10px;">
                <?php wp_nonce_field( 'aegisbackup_restore_file_backup_wizard', 'aegisbackup_restore_file_backup_wizard_nonce' ); ?>
                <input type="hidden" name="action" value="aegisbackup_restore_file_backup_wizard" />
                <input type="hidden" name="backup_id" value="<?php echo esc_attr( $backup_id ); ?>" />
                <input type="hidden" name="ab_redirect" value="<?php echo esc_attr( $return_url ); ?>" />

                <h3 style="margin-top:0;"><?php esc_html_e( 'Step 1 — Confirm overwrite', 'aegisbackup' ); ?></h3>
                <p>
                    <label>
                        <input type="checkbox" name="confirm_drop" value="1">
                        <strong><?php esc_html_e( 'Yes, overwrite existing files using this backup', 'aegisbackup' ); ?></strong>
                    </label>
                </p>
                <p>
                    <label for="ab_confirm_text_files"><strong><?php esc_html_e( 'Type RESTORE to confirm', 'aegisbackup' ); ?></strong></label><br>
                    <input type="text" id="ab_confirm_text_files" name="confirm_text" class="regular-text" placeholder="RESTORE" autocomplete="off" />
                    <span class="description" style="margin-left:8px;"><?php esc_html_e( 'This prevents accidental restores.', 'aegisbackup' ); ?></span>
                </p>

                <p style="margin-top:12px;">
                    <button class="button button-primary"><?php esc_html_e( 'Start File Restore', 'aegisbackup' ); ?></button>
                    <a class="button" href="<?php echo esc_url( $return_url ); ?>" style="margin-left:6px;"><?php esc_html_e( 'Cancel', 'aegisbackup' ); ?></a>
                </p>
            </form>
        </div>
        <?php
    }

    private function render_simple_db_snapshot_restore_wizard( $file ) {
        $file = sanitize_file_name( (string) $file );
        $return_url = admin_url( 'admin.php?page=aegisbackup-restore&tab=restore' );

        $wizard_url = add_query_arg( array( 'page' => 'aegisbackup-restore', 'tab' => 'restore', 'ab_wizard' => 1, 'ab_kind' => 'db_tools_snapshot', 'file' => $file ), admin_url( 'admin.php' ) );

        if ( empty( $file ) || false === strpos( $file, '.sql' ) || ! method_exists( $this->plugin, 'db_tools' ) || ! $this->plugin->db_tools() ) {
            echo '<div class="notice notice-error"><p>' . esc_html__( 'Restore Wizard: selected database snapshot was not found.', 'aegisbackup' ) . '</p></div>';
            echo '<p><a class="button" href="' . esc_url( $return_url ) . '">' . esc_html__( 'Back to WordPress Restores', 'aegisbackup' ) . '</a></p>';
            return;
        }

        $module = $this->plugin->db_tools();
        $dir = method_exists( $module, 'get_db_tools_backup_dir' ) ? (string) $module->get_db_tools_backup_dir() : '';
        $path = $dir ? trailingslashit( $dir ) . $file : '';
        $created = ( $path && is_file( $path ) ) ? gmdate( 'Y-m-d H:i:s', (int) filemtime( $path ) ) : '';
        $size = ( $path && is_file( $path ) ) ? size_format( (int) filesize( $path ) ) : '';

        if ( ! $path || ! is_file( $path ) ) {
            echo '<div class="notice notice-error"><p>' . esc_html__( 'Restore Wizard: selected database snapshot was not found.', 'aegisbackup' ) . '</p></div>';
            echo '<p><a class="button" href="' . esc_url( $return_url ) . '">' . esc_html__( 'Back to WordPress Restores', 'aegisbackup' ) . '</a></p>';
            return;
        }

        ?>
        <div class="aegisbackup-card" style="margin-top:14px;">
            <h2 style="margin-top:0;"><?php esc_html_e( 'Restore Wizard — Database Snapshot', 'aegisbackup' ); ?></h2>
            <p class="description"><?php esc_html_e( 'This guided restore is designed for non-technical admins. It uses safe defaults and prevents accidental restores.', 'aegisbackup' ); ?></p>

            <div class="notice notice-warning inline" style="margin:10px 0;">
                <p><strong><?php esc_html_e( 'Important:', 'aegisbackup' ); ?></strong>
                    <?php esc_html_e( 'This restore overwrites existing database tables for this site prefix. Always verify the snapshot date before proceeding.', 'aegisbackup' ); ?>
                </p>
            </div>


            <?php
            $ab_msg = isset( $_GET['ab_msg'] ) ? sanitize_key( (string) wp_unslash( $_GET['ab_msg'] ) ) : '';
            $ab_err = isset( $_GET['ab_err'] ) ? sanitize_key( (string) wp_unslash( $_GET['ab_err'] ) ) : '';
            if ( 'confirm' === $ab_err ) :
                ?>
                <div class="notice notice-warning inline" style="margin:10px 0;">
                    <p><?php esc_html_e( 'Please confirm the restore checkbox and type RESTORE to proceed.', 'aegisbackup' ); ?></p>
                </div>
                <?php
            elseif ( $ab_msg ) :
                if ( 'snapshot_restored' === $ab_msg ) :
                    ?>
                    <div class="notice notice-success inline" style="margin:10px 0;">
                        <p><?php esc_html_e( 'Database snapshot restore complete.', 'aegisbackup' ); ?></p>
                    </div>
                    <?php
                elseif ( 'restore_failed' === $ab_msg ) :
                    ?>
                    <div class="notice notice-error inline" style="margin:10px 0;">
                        <p><?php esc_html_e( 'Database snapshot restore failed. Please check Settings → Logs for details.', 'aegisbackup' ); ?></p>
                    </div>
                    <?php
                endif;
            endif;
            ?>

            <table class="widefat striped" style="table-layout:fixed; width:100%; margin-top:10px;">
                <colgroup>
                    <col style="width:25%;" />
                    <col style="width:75%;" />
                </colgroup>
                <tbody>
                    <tr>
                        <th><?php esc_html_e( 'Snapshot', 'aegisbackup' ); ?></th>
                        <td><code style="display:block;white-space:normal;word-break:break-all;"><?php echo esc_html( $file ); ?></code></td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e( 'Date', 'aegisbackup' ); ?></th>
                        <td><?php echo $created ? '<code>' . esc_html( $created ) . '</code>' : esc_html__( '(unknown)', 'aegisbackup' ); ?></td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e( 'Size', 'aegisbackup' ); ?></th>
                        <td><?php echo $size ? esc_html( $size ) : esc_html__( '(unknown)', 'aegisbackup' ); ?></td>
                    </tr>
                </tbody>
            </table>

            <hr>

            <?php if ( ! empty( $_GET['ab_err'] ) ) : ?>
                <div class="notice notice-error"><p>
                    <?php
                    $err = sanitize_key( (string) wp_unslash( $_GET['ab_err'] ) );
                    if ( 'confirm' === $err ) {
                        esc_html_e( 'Confirmation required. Please check the box and type RESTORE to continue.', 'aegisbackup' );
                    } elseif ( 'restore_failed' === $err ) {
                        esc_html_e( 'Database snapshot restore failed to start or failed to complete.', 'aegisbackup' );
                    } else {
                        esc_html_e( 'Restore request failed.', 'aegisbackup' );
                    }
                    ?>
                </p></div>
            <?php endif; ?>

            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin-top:10px;">
                <?php wp_nonce_field( 'aegisbackup_dbtools_restore_snapshot_wizard', 'aegisbackup_dbtools_restore_snapshot_wizard_nonce' ); ?>
                <input type="hidden" name="action" value="aegisbackup_dbtools_restore_snapshot_wizard" />
                <input type="hidden" name="file" value="<?php echo esc_attr( $file ); ?>" />
                <input type="hidden" name="ab_redirect" value="<?php echo esc_attr( $wizard_url ); ?>" />

                <h3 style="margin-top:0;"><?php esc_html_e( 'Step 1 — Confirm overwrite', 'aegisbackup' ); ?></h3>
                <p>
                    <label>
                        <input type="checkbox" name="confirm_drop" value="1">
                        <strong><?php esc_html_e( 'Yes, overwrite existing database tables using this snapshot', 'aegisbackup' ); ?></strong>
                    </label>
                </p>
                <p>
                    <label for="ab_confirm_text_db"><strong><?php esc_html_e( 'Type RESTORE to confirm', 'aegisbackup' ); ?></strong></label><br>
                    <input type="text" id="ab_confirm_text_db" name="confirm_text" class="regular-text" placeholder="RESTORE" autocomplete="off" />
                    <span class="description" style="margin-left:8px;"><?php esc_html_e( 'This prevents accidental restores.', 'aegisbackup' ); ?></span>
                </p>

                <p style="margin-top:12px;">
                    <button class="button button-primary"><?php esc_html_e( 'Start Database Restore', 'aegisbackup' ); ?></button>
                    <a class="button" href="<?php echo esc_url( $return_url ); ?>" style="margin-left:6px;"><?php esc_html_e( 'Cancel', 'aegisbackup' ); ?></a>
                </p>
            </form>
        </div>
        <?php
    }

    private function render_simple_table_backup_restore_wizard( $backup_id ) {
        $backup_id = sanitize_text_field( (string) $backup_id );
        if ( $backup_id && ! preg_match( '/^[A-Za-z0-9._-]+$/', $backup_id ) ) {
            $backup_id = '';
        }

        $return_url = isset( $_GET['ab_return'] ) ? wp_validate_redirect( esc_url_raw( (string) wp_unslash( $_GET['ab_return'] ) ), '' ) : '';
        if ( ! $return_url || 0 !== strpos( $return_url, admin_url() ) ) {
            $return_url = admin_url( 'admin.php?page=aegisbackup-restore&tab=restore' );
        }

        if ( empty( $backup_id ) || empty( $this->plugin->table_backup ) ) {
            echo '<div class="notice notice-error"><p>' . esc_html__( 'Restore Wizard: selected table backup was not found.', 'aegisbackup' ) . '</p></div>';
            echo '<p><a class="button" href="' . esc_url( $return_url ) . '">' . esc_html__( 'Back to WordPress Restores', 'aegisbackup' ) . '</a></p>';
            return;
        }

        $meta = null;
        if ( method_exists( $this->plugin->table_backup, 'get_backups' ) ) {
            $all = $this->plugin->table_backup->get_backups();
            if ( is_array( $all ) && isset( $all[ $backup_id ] ) && is_array( $all[ $backup_id ] ) ) {
                $meta = $all[ $backup_id ];
            }
        }

        $label = $meta && ! empty( $meta['name'] ) ? (string) $meta['name'] : $backup_id;
        $created = $meta && ! empty( $meta['created'] ) ? (string) $meta['created'] : '';
        $size = $meta && ! empty( $meta['size_human'] ) ? (string) $meta['size_human'] : ( $meta && ! empty( $meta['size'] ) ? size_format( (int) $meta['size'] ) : '' );

        $advanced_url = admin_url( 'admin.php?page=aegisbackup&tab=db&dbtab=table_backups' );

        $restore_action = admin_url( 'admin-post.php' );

        ?>
        <div class="aegisbackup-card" style="margin-top:14px;">
            <h2 style="margin-top:0;"><?php esc_html_e( 'Restore Wizard — Table Backup', 'aegisbackup' ); ?></h2>
            <p class="description"><?php esc_html_e( 'Table Backups restore selected database tables into this site. Confirm below to start the restore job.', 'aegisbackup' ); ?></p>

            <div class="notice notice-warning inline" style="margin:10px 0;">
                <p><strong><?php esc_html_e( 'Important:', 'aegisbackup' ); ?></strong>
                    <?php esc_html_e( 'Restoring a table backup will overwrite the selected tables on this site.', 'aegisbackup' ); ?>
                </p>
            </div>

            <table class="widefat striped" style="table-layout:fixed; width:100%; margin-top:10px;">
                <colgroup>
                    <col style="width:25%;" />
                    <col style="width:75%;" />
                </colgroup>
                <tbody>
                    <tr>
                        <th><?php esc_html_e( 'Backup', 'aegisbackup' ); ?></th>
                        <td><code style="display:block;white-space:normal;word-break:break-all;"><?php echo esc_html( $label ); ?></code></td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e( 'Date', 'aegisbackup' ); ?></th>
                        <td><?php echo $created ? '<code>' . esc_html( $created ) . '</code>' : esc_html__( '(unknown)', 'aegisbackup' ); ?></td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e( 'Size', 'aegisbackup' ); ?></th>
                        <td><?php echo $size ? esc_html( $size ) : esc_html__( '(unknown)', 'aegisbackup' ); ?></td>
                    </tr>
                </tbody>
            </table>


            <?php
            $ab_msg = isset( $_GET['ab_msg'] ) ? sanitize_key( (string) wp_unslash( $_GET['ab_msg'] ) ) : '';
            if ( $ab_msg ) :
                if ( 'confirm_required' === $ab_msg ) :
                    ?>
                    <div class="notice notice-warning inline" style="margin:10px 0;">
                        <p><?php esc_html_e( 'Please confirm the restore checkbox and type RESTORE to proceed.', 'aegisbackup' ); ?></p>
                    </div>
                    <?php
                elseif ( 'table_restore_done' === $ab_msg ) :
                    ?>
                    <div class="notice notice-success inline" style="margin:10px 0;">
                        <p><?php esc_html_e( 'Table restore complete.', 'aegisbackup' ); ?></p>
                    </div>
                    <?php
                elseif ( 'table_restore_started' === $ab_msg ) :
                    ?>
                    <div class="notice notice-info inline" style="margin:10px 0;">
                        <p><?php esc_html_e( 'Table restore started. If it is not finished yet, it will continue automatically in the background.', 'aegisbackup' ); ?></p>
                    </div>
                    <?php
                elseif ( 'table_restore_failed' === $ab_msg ) :
                    ?>
                    <div class="notice notice-error inline" style="margin:10px 0;">
                        <p><?php esc_html_e( 'Table restore failed. Please check Settings → Logs for details.', 'aegisbackup' ); ?></p>
                    </div>
                    <?php
                endif;
            endif;
            ?>

            <form method="post" action="<?php echo esc_url( $restore_action ); ?>" style="margin-top:12px;">
                <input type="hidden" name="action" value="aegisbackup_tablebacks_restore_start_wizard" />
                <input type="hidden" name="id" value="<?php echo esc_attr( $backup_id ); ?>" />
                <input type="hidden" name="ab_return" value="<?php echo esc_attr( $return_url ); ?>" />
                <?php wp_nonce_field( 'aegisbackup_tablebacks_restore_start_wizard', 'aegisbackup_tb_restore_wizard_nonce' ); ?>

                <p style="margin-top:10px;">
                    <label>
                        <input type="checkbox" name="confirm_drop" value="1">
                        <strong><?php esc_html_e( 'Yes, restore (overwrite) database tables using this table backup', 'aegisbackup' ); ?></strong>
                    </label>
                </p>
                <p>
                    <label for="ab_confirm_text_tb"><strong><?php esc_html_e( 'Type RESTORE to confirm', 'aegisbackup' ); ?></strong></label><br>
                    <input type="text" id="ab_confirm_text_tb" name="confirm_text" class="regular-text" placeholder="RESTORE" autocomplete="off" />
                    <span class="description" style="margin-left:8px;"><?php esc_html_e( 'This prevents accidental restores.', 'aegisbackup' ); ?></span>
                </p>

                <p style="margin-top:12px;">
                    <button class="button button-primary"><?php esc_html_e( 'Restore', 'aegisbackup' ); ?></button>
                    <a class="button" href="<?php echo esc_url( $advanced_url ); ?>" style="margin-left:6px;"><?php esc_html_e( 'Advanced', 'aegisbackup' ); ?></a>
                    <a class="button" href="<?php echo esc_url( $return_url ); ?>" style="margin-left:6px;"><?php esc_html_e( 'Back', 'aegisbackup' ); ?></a>
                </p>
            </form>

            <p class="description"><?php esc_html_e( 'Click Restore to start the database restore job. You can leave this page — progress will appear in Restore Processes.', 'aegisbackup' ); ?></p>
        </div>
        <?php
    }

    private function render_restore_table( $title, array $pkgs, $section = 'wp' ) {
        $section = sanitize_key( (string) $section );
        if ( ! $section ) {
            $section = 'wp';
        }

        $per_page = 14;
        $paged_param = 'ab_paged_' . $section;
        $paged = isset( $_GET[ $paged_param ] ) ? max( 1, (int) $_GET[ $paged_param ] ) : 1;
        $total_items = count( $pkgs );
        $total_pages = (int) ceil( $total_items / $per_page );
        if ( $paged > $total_pages && $total_pages > 0 ) {
            $paged = $total_pages;
        }

        $offset = ( $paged - 1 ) * $per_page;
        $page_items = array_slice( $pkgs, $offset, $per_page );

        $anchor = '';
        if ( 'wp' === $section ) {
            $anchor = '#wp-restores';
        } elseif ( 'files' === $section ) {
            $anchor = '#file-restores';
        } elseif ( 'db' === $section ) {
            $anchor = '#db-restores';
        }

        $base_return = admin_url( 'admin.php?page=aegisbackup-restore&tab=restore' );
        $keep = array( 'ab_paged_wp', 'ab_paged_files', 'ab_paged_db' );
        $keep_args = array();
        foreach ( $keep as $k ) {
            if ( isset( $_GET[ $k ] ) ) {
                $keep_args[ $k ] = max( 1, (int) $_GET[ $k ] );
            }
        }

        $return_to_restore = add_query_arg( $keep_args, $base_return ) . $anchor;

        $card_id = '';
        if ( 'wp' === $section ) {
            $card_id = 'wp-restores';
        } elseif ( 'files' === $section ) {
            $card_id = 'file-restores';
        } elseif ( 'db' === $section ) {
            $card_id = 'db-restores';
        }
        ?>
        <div class="aegisbackup-card" id="<?php echo esc_attr( $card_id ); ?>" style="margin-top:14px;">
            <h3 style="margin-top:0;"><?php echo esc_html( $title ); ?></h3>

            <?php if ( empty( $pkgs ) ) : ?>
                <p class="description"><?php esc_html_e( 'No backups found for this category yet.', 'aegisbackup' ); ?></p>
            <?php else : ?>
                <table class="widefat striped" style="table-layout:fixed; width:100%;">
                    <colgroup>
                        <col style="width:18%;" />
                        <col style="width:42%;" />
                        <col style="width:15%;" />
                        <col style="width:10%;" />
                        <col style="width:15%;" />
                    </colgroup>
                    <thead>
                        <tr>
                            <th><?php esc_html_e( 'Name', 'aegisbackup' ); ?></th>
                            <th><?php esc_html_e( 'Package', 'aegisbackup' ); ?></th>
                            <th><?php esc_html_e( 'Created', 'aegisbackup' ); ?></th>
                            <th><?php esc_html_e( 'Size', 'aegisbackup' ); ?></th>
                            <th><?php esc_html_e( 'Actions', 'aegisbackup' ); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ( $page_items as $pkg ) : ?>
                            <?php
                            $kind = isset( $pkg['kind'] ) ? (string) $pkg['kind'] : 'package_wp';

$name = isset( $pkg['name'] ) ? (string) $pkg['name'] : '';
if ( '' === $name && isset( $pkg['id'] ) ) {
    $name = (string) $pkg['id'];
}

$label = '';
if ( isset( $pkg['plan_name'] ) ) {
    $label = (string) $pkg['plan_name'];
} elseif ( isset( $pkg['schedule_name'] ) ) {
    $label = (string) $pkg['schedule_name'];
} elseif ( isset( $pkg['label'] ) ) {
    $label = (string) $pkg['label'];
} elseif ( isset( $pkg['manifest'] ) && is_array( $pkg['manifest'] ) ) {
    $m2 = (array) $pkg['manifest'];
    if ( isset( $m2['plan_name'] ) ) {
        $label = (string) $m2['plan_name'];
    } elseif ( isset( $m2['schedule_name'] ) ) {
        $label = (string) $m2['schedule_name'];
    } elseif ( isset( $m2['label'] ) ) {
        $label = (string) $m2['label'];
    } elseif ( isset( $m2['plan'] ) && is_array( $m2['plan'] ) && isset( $m2['plan']['name'] ) ) {
        $label = (string) $m2['plan']['name'];
    }
}

$pkg_display = isset( $pkg['name'] ) ? (string) $pkg['name'] : '';
if ( '' === $pkg_display && isset( $pkg['id'] ) ) {
    $pkg_display = (string) $pkg['id'];
}

if ( 'table_backup' === $kind ) {
    if ( '' === $label && isset( $pkg['name'] ) ) {
        $label = (string) $pkg['name'];
    }
    if ( isset( $pkg['zip'] ) && $pkg['zip'] ) {
        $pkg_display = basename( (string) $pkg['zip'] );
    } elseif ( isset( $pkg['id'] ) && $pkg['id'] ) {
        $pkg_display = (string) $pkg['id'];
    }
}

if ( 'file_backup' === $kind ) {
    if ( isset( $pkg['path'] ) && $pkg['path'] ) {
        $pkg_display = basename( (string) $pkg['path'] );
    }
}

                            $created = isset( $pkg['created'] ) ? (string) $pkg['created'] : '';
                            if ( '' === $created && isset( $pkg['timestamp'] ) ) {
                                $created = (string) $pkg['timestamp'];
                            }

                            $size = isset( $pkg['size'] ) ? (string) $pkg['size'] : '';
                            if ( '' === $size && isset( $pkg['bytes'] ) ) {
                                $size = size_format( (int) $pkg['bytes'] );
                            }

                            $actions = array();

                            if ( 0 === strpos( $kind, 'package_' ) ) {
                                $download = wp_nonce_url(
                                    add_query_arg(
                                        array(
                                            'action' => 'aegisbackup_download_package',
                                            'pkg'    => $name,
                                        ),
                                        admin_url( 'admin-post.php' )
                                    ),
                                    'aegisbackup_download_package',
                                    'aegisbackup_nonce'
                                );

                                $delete = wp_nonce_url(
                                    add_query_arg(
                                        array(
                                            'action' => 'aegisbackup_delete_package',
                                            'pkg'    => $name,
                                        ),
                                        admin_url( 'admin-post.php' )
                                    ),
                                    'aegisbackup_delete_package',
                                    'aegisbackup_nonce'
                                );

                                $restore_wiz = add_query_arg(
                                    array(
                                        'page'      => 'aegisbackup-restore',
                                        'tab'       => 'restore',
                                        'ab_wizard' => 1,
                                        'pkg'       => $name,
                                    ),
                                    admin_url( 'admin.php' )
                                );

                                $dr_wiz = $restore_wiz;

                                $advanced = add_query_arg(
                                    array(
                                        'page' => 'aegisbackup-restore',
                                        'tab'  => 'migrate',
                                        'pkg'  => $name,
                                    ),
                                    admin_url( 'admin.php' )
                                );

                                $actions[] = array( 'label' => __( 'Restore Wizard', 'aegisbackup' ), 'url' => $restore_wiz, 'confirm' => '' );
                                $actions[] = array( 'label' => __( 'Disaster Recovery', 'aegisbackup' ), 'url' => $dr_wiz, 'confirm' => '' );
                                $actions[] = array( 'label' => __( 'Advanced', 'aegisbackup' ), 'url' => $advanced, 'confirm' => '' );
                                $actions[] = array( 'label' => __( 'Download', 'aegisbackup' ), 'url' => $download, 'confirm' => '' );
                                $actions[] = array( 'label' => __( 'Delete', 'aegisbackup' ), 'url' => $delete, 'confirm' => __( 'Delete this backup package?', 'aegisbackup' ) );
                            } elseif ( 'file_backup' === $kind ) {

                                $id = isset( $pkg['id'] ) ? (string) $pkg['id'] : '';

                                $download = wp_nonce_url(
                                    add_query_arg(
                                        array(
                                            'action'    => 'aegisbackup_download_file_backup',
                                            'backup_id' => $id,
                                        ),
                                        admin_url( 'admin-post.php' )
                                    ),
                                    'aegisbackup_download_file_backup'
                                );

                                $delete = wp_nonce_url(
                                    add_query_arg(
                                        array(
                                            'action'     => 'aegisbackup_delete_file_backup',
                                            'backup_id'  => $id,
                                            'ab_return'  => $return_to_restore,
                                        ),
                                        admin_url( 'admin-post.php' )
                                    ),
                                    'aegisbackup_delete_file_backup'
                                );

                                $restore_wiz = add_query_arg(
                                    array(
                                        'page'      => 'aegisbackup-restore',
                                        'tab'       => 'restore',
                                        'ab_wizard' => 1,
                                        'ab_kind'   => 'file_backup',
                                        'id'        => $id,
                                    ),
                                    admin_url( 'admin.php' )
                                );

                                $advanced = wp_nonce_url(
                                    add_query_arg(
                                        array(
                                            'action'     => 'aegisbackup_restore_file_backup',
                                            'backup_id'  => $id,
                                            'ab_return'  => $return_to_restore,
                                        ),
                                        admin_url( 'admin-post.php' )
                                    ),
                                    'aegisbackup_restore_file_backup'
                                );

                                $actions[] = array( 'label' => __( 'Restore Wizard', 'aegisbackup' ), 'url' => $restore_wiz, 'confirm' => '' );
                                $actions[] = array( 'label' => __( 'Advanced', 'aegisbackup' ), 'url' => $advanced, 'confirm' => __( 'This will overwrite existing files in-place. Continue?', 'aegisbackup' ) );
                                $actions[] = array( 'label' => __( 'Download', 'aegisbackup' ), 'url' => $download, 'confirm' => '' );
                                $actions[] = array( 'label' => __( 'Delete', 'aegisbackup' ), 'url' => $delete, 'confirm' => __( 'Delete this file backup?', 'aegisbackup' ) );
                            } elseif ( 'db_tools_snapshot' === $kind ) {
                                $file = '';
                                if ( isset( $pkg['file'] ) ) {
                                    $file = basename( (string) $pkg['file'] );
                                } elseif ( isset( $pkg['path'] ) ) {
                                    $file = basename( (string) $pkg['path'] );
                                }

                                $download = wp_nonce_url(
                                    add_query_arg(
                                        array(
                                            'action' => 'aegisbackup_dbtools_download_snapshot',
                                            'file'   => $file,
                                        ),
                                        admin_url( 'admin-post.php' )
                                    ),
                                    'aegisbackup_dbtools_snapshot'
                                );

                                $delete = wp_nonce_url(
                                    add_query_arg(
                                        array(
                                            'action'    => 'aegisbackup_dbtools_delete_snapshot',
                                            'file'      => $file,
                                            'ab_return' => $return_to_restore,
                                        ),
                                        admin_url( 'admin-post.php' )
                                    ),
                                    'aegisbackup_dbtools_snapshot'
                                );

                                $restore_wiz = add_query_arg(
                                    array(
                                        'page'      => 'aegisbackup-restore',
                                        'tab'       => 'restore',
                                        'ab_wizard' => 1,
                                        'ab_kind'   => 'db_tools_snapshot',
                                        'file'      => $file,
                                    ),
                                    admin_url( 'admin.php' )
                                );

                                $advanced = admin_url( 'admin.php?page=aegisbackup&tab=db&dbtab=backups' );

                                $actions[] = array( 'label' => __( 'Restore Wizard', 'aegisbackup' ), 'url' => $restore_wiz, 'confirm' => '' );
                                $actions[] = array( 'label' => __( 'Advanced', 'aegisbackup' ), 'url' => $advanced, 'confirm' => '' );
                                $actions[] = array( 'label' => __( 'Download', 'aegisbackup' ), 'url' => $download, 'confirm' => '' );
                            } elseif ( 'table_backup' === $kind ) {

                                $id = isset( $pkg['id'] ) ? (string) $pkg['id'] : '';

                                $restore_wiz = add_query_arg(
                                    array(
                                        'page'      => 'aegisbackup-restore',
                                        'tab'       => 'restore',
                                        'ab_wizard' => 1,
                                        'ab_kind'   => 'table_backup',
                                        'id'        => $id,
                                        'ab_return' => $return_to_restore,
                                    ),
                                    admin_url( 'admin.php' )
                                );

                                $restore = wp_nonce_url(
                                    add_query_arg(
                                        array(
                                            'action'    => 'aegisbackup_tablebacks_restore_start',
                                            'id'        => $id,
                                            'ab_return' => $return_to_restore,
                                        ),
                                        admin_url( 'admin-post.php' )
                                    ),
                                    'aegisbackup_tablebacks_restore_start'
                                );

                                $advanced = admin_url( 'admin.php?page=aegisbackup&tab=db&dbtab=table_backups' );

                                $can_download = false;
                                if ( isset( $this->plugin->table_backup ) && $this->plugin->table_backup && method_exists( $this->plugin->table_backup, 'get_backup' ) ) {
                                    $b = $this->plugin->table_backup->get_backup( $id );
                                    if ( is_array( $b ) && ! empty( $b['zip'] ) && is_file( (string) $b['zip'] ) ) {
                                        $can_download = true;
                                    }
                                }

                                if ( $can_download ) {
                                    $download = wp_nonce_url(
                                        add_query_arg(
                                            array(
                                                'action' => 'aegisbackup_tablebacks_download',
                                                'id'     => $id,
                                            ),
                                            admin_url( 'admin-post.php' )
                                        ),
                                        'aegisbackup_tablebacks_download'
                                    );
                                } else {
                                    $download = '';
                                }

                                $delete = wp_nonce_url(
                                    add_query_arg(
                                        array(
                                            'action' => 'aegisbackup_tablebacks_backup_delete',
                                            'id'     => $id,
                                            'return' => $return_to_restore,
                                        ),
                                        admin_url( 'admin-post.php' )
                                    ),
                                    'aegisbackup_tablebacks_backup_delete'
                                );

                                $actions[] = array( 'label' => __( 'Restore Wizard', 'aegisbackup' ), 'url' => $restore_wiz, 'confirm' => '' );
                                $actions[] = array( 'label' => __( 'Advanced', 'aegisbackup' ), 'url' => $advanced, 'confirm' => '' );
                                if ( $download ) {
                                    $actions[] = array( 'label' => __( 'Download', 'aegisbackup' ), 'url' => $download, 'confirm' => '' );
                                }
                            }
                            ?>
                            <tr>
                                <td><?php echo esc_html( $label ? $label : '-' ); ?></td>
                                <td><code style="display:block;white-space:normal;word-break:break-all;"><?php echo esc_html( $pkg_display ); ?></code></td>
                                <td><?php echo esc_html( $created ); ?></td>
                                <td><?php echo esc_html( $size ); ?></td>
                                <td>
                                    <?php foreach ( $actions as $a ) : ?>
                                        <?php
                                        $confirm_msg = isset( $a['confirm'] ) ? (string) $a['confirm'] : '';
                                        ?>
                                        <a class="button button-small" href="<?php echo esc_url( (string) $a['url'] ); ?>"<?php if ( '' !== $confirm_msg ) : ?> onclick="return confirm(<?php echo esc_attr( wp_json_encode( $confirm_msg ) ); ?>);"<?php endif; ?>><?php echo esc_html( (string) $a['label'] ); ?></a>
                                    <?php endforeach; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <?php if ( $total_pages > 1 ) : ?>
                    <div class="tablenav" style="margin-top:10px;">
                        <div class="tablenav-pages">
                            <?php
                            $base_args = $keep_args;
                            $base_args[ $paged_param ] = '%#%';
                            $base = add_query_arg( $base_args, $base_return ) . $anchor;

                            echo wp_kses_post( paginate_links( array(
                                'base'      => $base,
                                'format'    => '',
                                'current'   => $paged,
                                'total'     => $total_pages,
                                'prev_text' => '&laquo;',
                                'next_text' => '&raquo;',
                            ) ) );
                            ?>
                        </div>
                    </div>
                <?php endif; ?>

                <p class="description">
                    <?php esc_html_e( 'Packages open the Migration Restore wizard. File backups and DB snapshots restore immediately (overwrite existing content in-place).', 'aegisbackup' ); ?>
                </p>
            <?php endif; ?>
        </div>
        <?php
    }

}