<?php
namespace AegisBackup\Backup;

defined( 'ABSPATH' ) || exit;

require_once AEGISBACKUP_DIR . 'includes/backup/class-ab-manifest.php';

class AB_Backup_Manager {
    const JOB_OPTION_PREFIX = 'aegisbackup_job_';
	const COMPLETION_HOOK = 'aegisbackup_run_backup_completed_hooks';

	public function __construct() {
		add_action( self::COMPLETION_HOOK, array( $this, 'handle_run_backup_completed_hooks' ), 10, 2 );
	}

	public function handle_run_backup_completed_hooks( $zip_path, $state ) {
		$zip_path = (string) $zip_path;
		$state    = is_array( $state ) ? $state : array();

		try {
			do_action( 'aegisbackup_backup_completed', $zip_path, $state );
		} catch ( \Throwable $e ) {
			$this->log( 'ERROR backup_completed hook exception: ' . $e->getMessage() . ' @ ' . $e->getFile() . ':' . $e->getLine() );
		}
	}

	private function schedule_completion_hooks( $zip_path, array $state ) {
		$zip_path = (string) $zip_path;
		$payload = array(
			'job_id'      => isset( $state['job_id'] ) ? (string) $state['job_id'] : '',
			'is_dr'       => ! empty( $state['is_dr'] ) ? 1 : 0,
			'dr_token'    => isset( $state['dr_token'] ) ? (string) $state['dr_token'] : '',
			'pkg_dir'     => isset( $state['pkg_dir'] ) ? (string) $state['pkg_dir'] : '',
			'zip_path'    => isset( $state['zip_path'] ) ? (string) $state['zip_path'] : '',
			'created_at'  => isset( $state['created_at'] ) ? (string) $state['created_at'] : '',
		);

		if ( ! wp_next_scheduled( self::COMPLETION_HOOK, array( $zip_path, $payload ) ) ) {
			wp_schedule_single_event( time() + 5, self::COMPLETION_HOOK, array( $zip_path, $payload ) );
		}
	}

    private function zip_error_message( $code ) {
        $map = array(
            \ZipArchive::ER_EXISTS => 'File already exists.',
            \ZipArchive::ER_INCONS => 'Zip archive inconsistent.',
            \ZipArchive::ER_INVAL => 'Invalid argument.',
            \ZipArchive::ER_MEMORY => 'Malloc failure.',
            \ZipArchive::ER_NOENT => 'No such file.',
            \ZipArchive::ER_NOZIP => 'Not a zip archive.',
            \ZipArchive::ER_OPEN => 'Cannot open file.',
            \ZipArchive::ER_READ => 'Read error.',
            \ZipArchive::ER_SEEK => 'Seek error.',
            \ZipArchive::ER_WRITE => 'Write error.',
            \ZipArchive::ER_CRC => 'CRC error.',
            \ZipArchive::ER_ZIPCLOSED => 'Containing zip archive was closed.',
        );
        return isset( $map[ $code ] ) ? $map[ $code ] : 'Zip error code: ' . (int) $code;
    }

    private function open_zip( \ZipArchive $zip, $zip_path, $allow_create = true ) {
        $zip_path = (string) $zip_path;
        $flags = $allow_create ? \ZipArchive::CREATE : 0;

        $code = $zip->open( $zip_path, $flags );
        if ( true === $code ) {
            return array( 'ok' => true, 'code' => 0, 'message' => 'OK' );
        }

        if ( $allow_create ) {
            $code2 = $zip->open( $zip_path, \ZipArchive::OVERWRITE );
            if ( true === $code2 ) {
                return array( 'ok' => true, 'code' => 0, 'message' => 'OK' );
            }
            $code = $code2;
        }

        return array( 'ok' => false, 'code' => (int) $code, 'message' => $this->zip_error_message( (int) $code ) );
    }

    private function zip_has_prefix( \ZipArchive $zip, $prefix, $max_scan = 4000 ) {
        $prefix = (string) $prefix;
        $n = (int) $zip->numFiles;
        if ( $n <= 0 ) {
            return false;
        }
        $limit = min( $n, (int) $max_scan );
        for ( $i = 0; $i < $limit; $i++ ) {
            $name = $zip->getNameIndex( $i );
            if ( is_string( $name ) && 0 === strpos( $name, $prefix ) ) {
                return true;
            }
        }
        return false;
    }

    public function get_package_sanity( $package_name ) {
        $package_name = (string) $package_name;
        $items = array();
        $upload = wp_upload_dir();
        $base = trailingslashit( $upload['basedir'] ) . 'aegisbackup';

        if ( ! empty( $args['base_dir'] ) ) {
            $cand = (string) $args['base_dir'];
            if ( $cand ) {
                $base = untrailingslashit( $cand );
            }
        }

        if ( ! empty( $args['base_dir'] ) ) {
            $custom = (string) $args['base_dir'];
            if ( $custom ) {
                $base = untrailingslashit( $custom );
            }
        }
        $pkg_dir = trailingslashit( $base ) . $package_name;
        $zip_path = trailingslashit( $pkg_dir ) . 'AegisBackup-' . $package_name . '.zip';
        if ( ! is_file( $zip_path ) ) {
            $zip_path = trailingslashit( $pkg_dir ) . 'package.zip';
        }
        $manifest_path = trailingslashit( $pkg_dir ) . 'manifest.json';

        $items['zip_exists'] = array(
            'ok' => is_file( $zip_path ) && is_readable( $zip_path ),
            'detail' => $zip_path,
        );

        $manifest = array();
        if ( is_file( $manifest_path ) ) {
            $raw = @file_get_contents( $manifest_path );
            $m = json_decode( (string) $raw, true );
            $manifest = is_array( $m ) ? $m : array();
        }
        $flags = isset( $manifest['flags'] ) && is_array( $manifest['flags'] ) ? (array) $manifest['flags'] : array();
        $want_files = ! empty( $flags['include_files'] );
        $want_db    = ! empty( $flags['include_db'] );

        if ( empty( $items['zip_exists']['ok'] ) ) {
            return array( 'ok' => false, 'items' => $items );
        }

        $zip = new \ZipArchive();
        $z = $this->open_zip( $zip, $zip_path, false );
        $items['zip_open'] = array(
            'ok' => (bool) $z['ok'],
            'detail' => $z['ok'] ? 'OK' : (string) $z['message'],
        );

        if ( ! $z['ok'] ) {
            $this->log( 'DEBUG open_zip failed (manifest): message=' . ( isset( $z['message'] ) ? (string) $z['message'] : '' ) . ' code=' . ( isset( $z['code'] ) ? (string) $z['code'] : '' ) );
            return array( 'ok' => false, 'items' => $items );
        }

        $items['has_manifest_json'] = array(
            'ok' => ( false !== $zip->locateName( 'manifest.json' ) ),
            'detail' => 'manifest.json',
        );

        $items['has_checksums'] = array(
            'ok' => ( false !== $zip->locateName( 'checksums.sha256' ) ),
            'detail' => 'checksums.sha256',
        );

        if ( $want_db ) {
            $items['has_db_dump'] = array(
                'ok' => ( false !== $zip->locateName( 'db/db.sql' ) ),
                'detail' => 'db/db.sql',
            );
        } else {
            $items['has_db_dump'] = array(
                'ok' => true,
                'detail' => 'Skipped (DB not selected)',
            );
        }

        if ( $want_files ) {
            $has_wp_content = $this->zip_has_prefix( $zip, 'files/wp-content/' );
            $items['has_wp_content'] = array(
                'ok' => (bool) $has_wp_content,
                'detail' => 'files/wp-content/*',
            );
        } else {
            $items['has_wp_content'] = array(
                'ok' => true,
                'detail' => 'Skipped (Files not selected)',
            );
        }

        $zip->close();
        $this->log( 'DEBUG zip close after manifest. zip_size=' . ( is_file( $zip_path ) ? (string) @filesize( $zip_path ) : 'na' ) );

        $items['manifest_has_urls'] = array(
            'ok' => ( ! empty( $manifest['site_url'] ) && ! empty( $manifest['home_url'] ) ),
            'detail' => 'site_url + home_url',
        );

        $ok = true;
        foreach ( $items as $it ) {
            if ( empty( $it['ok'] ) ) {
                $ok = false;
                break;
            }
        }

        return array( 'ok' => $ok, 'items' => $items );
    }

    public function list_packages( $limit = 0, $purge = false ) {
        $upload = wp_upload_dir();
        $base = trailingslashit( $upload['basedir'] ) . 'aegisbackup';
        if ( ! is_dir( $base ) ) {
            return array();
        }

        $items = glob( $base . '/package-*/AegisBackup-*.zip' );
        if ( empty( $items ) ) {
            $items = glob( $base . '/package-*/package.zip' );
        }
        if ( empty( $items ) ) {
            $items = array();
            try {
                $it = new \DirectoryIterator( $base );
                foreach ( $it as $f ) {
                    if ( $f->isDot() || ! $f->isDir() ) {
                        continue;
                    }
                    $name = $f->getFilename();
                    if ( 0 !== strpos( $name, 'package-' ) ) {
                        continue;
                    }
                    $zip = trailingslashit( $f->getPathname() ) . 'AegisBackup-' . $name . '.zip';
                    if ( is_file( $zip ) ) {
                        $items[] = $zip;
                    } else {
                        $legacy = trailingslashit( $f->getPathname() ) . 'package.zip';
                        if ( is_file( $legacy ) ) {
                            $items[] = $legacy;
                        }
                    }
                }
            } catch ( \Exception $e ) {
                // ignore
            }
        }

        if ( empty( $items ) ) {
            $items = glob( $base . '/package-*/manifest.json' );
        }

        $out = array();
        foreach ( (array) $items as $path ) {
            $dir = dirname( $path );
            $name = basename( $dir );
            $created = gmdate( 'Y-m-d H:i:s', (int) @filemtime( $dir ) );
            $size = is_file( $path ) ? @filesize( $path ) : 0;

            $manifest_path = trailingslashit( $dir ) . 'manifest.json';
            $manifest = array();
            if ( is_file( $manifest_path ) ) {
                $raw = file_get_contents( $manifest_path );
                $manifest = json_decode( (string) $raw, true );
                if ( ! is_array( $manifest ) ) {
                    $manifest = array();
                }
            }

            $pkg_zip = trailingslashit( $dir ) . 'AegisBackup-' . $name . '.zip';
            if ( ! is_file( $pkg_zip ) ) {
                $pkg_zip = trailingslashit( $dir ) . 'package.zip';
            }

            $out[] = array(
                'name' => $name,
                'dir'  => $dir,
                'package' => is_file( $pkg_zip ) ? $pkg_zip : '',
                'created' => $created,
                'size' => $this->human_bytes( (int) $size ),
                'manifest' => $manifest,
            );
        }

        usort(
            $out,
            static function( $a, $b ) {
                return strcmp( (string) $b['created'], (string) $a['created'] );
            }
        );

        $limit = (int) $limit;
        if ( $limit > 0 ) {
            if ( $purge && count( $out ) > $limit ) {
                $drop = array_slice( $out, $limit );
                foreach ( $drop as $d ) {
                    if ( empty( $d['dir'] ) ) {
                        continue;
                    }
                    $dir = (string) $d['dir'];
                    if ( ! is_dir( $dir ) ) {
                        continue;
                    }

                    $it = new \RecursiveIteratorIterator(
                        new \RecursiveDirectoryIterator( $dir, \FilesystemIterator::SKIP_DOTS ),
                        \RecursiveIteratorIterator::CHILD_FIRST
                    );
                    $fs = $this->ab_get_fs();
                foreach ( $it as $f ) {
                    if ( $f->isDir() ) {
                        if ( $fs ) {
                            $fs->rmdir( $f->getPathname(), false );
                        }
                    } else {
                        wp_delete_file( $f->getPathname() );
                    }
                }
                if ( $fs ) {
                    $fs->rmdir( $dir, false );
                }
}
            }
            $out = array_slice( $out, 0, $limit );
        }

        return $out;
    }

    public function get_dr_base_dir() {
        $upload = wp_upload_dir();
        return trailingslashit( $upload['basedir'] ) . 'aegisbackup-dr';
    }

    public function get_dr_tokens_file() {
        return trailingslashit( $this->get_dr_base_dir() ) . 'dr-tokens.json';
    }

    private function maybe_update_dr_token_registry_on_done( array $state ) {
        $args = isset( $state['args'] ) ? (array) $state['args'] : array();

        $this->log( 'DEBUG backup job step: job_id=' . (string) $job_id . ' phase=' . (string) $state['phase'] . ' progress=' . (string) $state['progress'] . ' zip=' . (string) ( isset( $state['zip_path'] ) ? $state['zip_path'] : '' ) . ' pkg=' . (string) ( isset( $state['pkg_dir'] ) ? $state['pkg_dir'] : '' ) . ' work=' . (string) ( isset( $state['work_dir'] ) ? $state['work_dir'] : '' ) );
        if ( function_exists( 'memory_get_usage' ) ) { $this->log( 'DEBUG memory: ' . (int) memory_get_usage( true ) ); }
        if ( function_exists( 'disk_free_space' ) && ! empty( $state['pkg_dir'] ) ) { $this->log( 'DEBUG disk_free_space(pkg): ' . (string) @disk_free_space( (string) $state['pkg_dir'] ) ); }

        if ( empty( $args['dr_token'] ) || empty( $args['dr_tokens_file'] ) ) {
            return;
        }

        $token      = (string) $args['dr_token'];
        $tokens_file = (string) $args['dr_tokens_file'];

        if ( ! $token || ! $tokens_file ) {
            return;
        }

        $tokens_dir = dirname( $tokens_file );
        if ( $tokens_dir && ! is_dir( $tokens_dir ) ) {
            @wp_mkdir_p( $tokens_dir );
        }

        $tokens = array();
        if ( is_file( $tokens_file ) ) {
            $raw = @file_get_contents( $tokens_file );
            $tmp = json_decode( (string) $raw, true );
            if ( is_array( $tmp ) ) {
                $tokens = $tmp;
            }
        }

        if ( empty( $tokens[ $token ] ) || ! is_array( $tokens[ $token ] ) ) {
            return;
        }

        $zip_path = ! empty( $state['zip_path'] ) ? (string) $state['zip_path'] : '';
        $pkg_dir  = ! empty( $state['pkg_dir'] ) ? (string) $state['pkg_dir'] : '';

        if ( $zip_path ) {
            $tokens[ $token ]['zip_path'] = $zip_path;
        }
        if ( $pkg_dir ) {
            $tokens[ $token ]['package'] = basename( rtrim( $pkg_dir, "/\\" ) );
        } elseif ( $zip_path ) {
            $tokens[ $token ]['package'] = basename( dirname( $zip_path ) );
        }

        $tokens[ $token ]['status'] = 'ready';
        $tokens[ $token ]['updated'] = time();

        @file_put_contents( $tokens_file, wp_json_encode( $tokens ) );
    }

    public function list_dr_packages( $limit = 0 ) {
        $base = $this->get_dr_base_dir();
        if ( ! is_dir( $base ) ) {
            return array();
        }

        $items = glob( $base . '/package-*/AegisBackup-*.zip' );
        if ( empty( $items ) ) {
            $items = glob( $base . '/package-*/package.zip' );
        }
        if ( empty( $items ) ) {
            $items = array();
            try {
                $it = new \DirectoryIterator( $base );
                foreach ( $it as $f ) {
                    if ( $f->isDot() || ! $f->isDir() ) {
                        continue;
                    }
                    if ( 0 !== strpos( $f->getFilename(), 'package-' ) ) {
                        continue;
                    }
                    $pkg_dir = trailingslashit( $base ) . $f->getFilename();
                    $z = glob( $pkg_dir . '/AegisBackup-*.zip' );
                    if ( empty( $z ) ) {
                        $z = glob( $pkg_dir . '/package.zip' );
                    }
                    if ( ! empty( $z[0] ) && is_file( $z[0] ) ) {
                        $items[] = $z[0];
                    }
                }
            } catch ( \Exception $e ) {
            }
        }

        $rows = array();
        foreach ( $items as $zip ) {
            $zip = (string) $zip;
            $rows[] = array(
                'package' => basename( dirname( $zip ) ),
                'created' => (int) @filemtime( $zip ),
                'size'    => (int) @filesize( $zip ),
                'path'    => $zip,
            );
        }

        usort(
            $rows,
            function ( $a, $b ) {
                return (int) $b['created'] - (int) $a['created'];
            }
        );

        if ( $limit > 0 ) {
            $rows = array_slice( $rows, 0, (int) $limit );
        }

        return $rows;
    }

public function start_backup_job( $args ) {
        $upload = wp_upload_dir();
        $base = trailingslashit( $upload['basedir'] ) . 'aegisbackup';

        if ( ! empty( $args['base_dir'] ) ) {
            $cand = (string) $args['base_dir'];
            if ( $cand ) {
                $base = untrailingslashit( $cand );
            }
        }

        wp_mkdir_p( $base );

        $job_id = ! empty( $args['job_id_override'] ) ? (string) $args['job_id_override'] : ( 'ab_' . wp_generate_password( 12, false, false ) );
        $package_name = '';
        if ( ! empty( $args['package_name'] ) ) {
            $package_name = (string) $args['package_name'];

            if ( method_exists( $this, 'get_package_sanity' ) ) {
                $maybe = $this->get_package_sanity( $package_name );
                if ( is_string( $maybe ) && $maybe !== '' ) {
                    $package_name = $maybe;
                } else {
                    $package_name = sanitize_file_name( $package_name );
                }
            } else {
                $package_name = sanitize_file_name( $package_name );
            }
        }
        if ( ! $package_name ) {
            $package_name = 'package-' . gmdate( 'Ymd_His' ) . '-' . substr( $job_id, -6 );
        }

        $pkg_dir = trailingslashit( $base ) . $package_name;
        $work_dir = trailingslashit( $pkg_dir ) . 'work';
        wp_mkdir_p( $work_dir );

        $zip_filename = 'AegisBackup-' . basename( $pkg_dir ) . '.zip';
        $zip_path = trailingslashit( $pkg_dir ) . $zip_filename;
        $checksum_path = trailingslashit( $work_dir ) . 'checksums.sha256';

        $state = array(
            'job_id' => $job_id,
            'status' => 'running',
            'created'=> time(),
            'pkg_dir'=> $pkg_dir,
            'work_dir' => $work_dir,
            'zip_path' => $zip_path,
            'checksum_path' => $checksum_path,
            'progress' => 0,
            'phase' => 'init',
            'args' => array(
                'include_files'  => ! empty( $args['include_files'] ),
                'include_db'     => ! empty( $args['include_db'] ),
                'include_config' => ! empty( $args['include_config'] ),
                'include_core'   => ! empty( $args['include_core'] ),
                'backup_type'    => isset( $args['backup_type'] ) ? sanitize_key( (string) $args['backup_type'] ) : 'full',
                'package_purpose'=> isset( $args['package_purpose'] ) ? sanitize_key( (string) $args['package_purpose'] ) : '',
                'snapshot'       => ! empty( $args['snapshot'] ),
                'base_dir'       => isset( $args['base_dir'] ) ? (string) $args['base_dir'] : '',
                'dr_tokens_file' => isset( $args['dr_tokens_file'] ) ? (string) $args['dr_tokens_file'] : '',
                'dr_token'       => isset( $args['dr_token'] ) ? (string) $args['dr_token'] : '',
                'excludes'       => isset( $args['excludes'] ) ? (string) $args['excludes'] : '',
                'db_export_mode' => isset( $args['db_export_mode'] ) ? sanitize_key( (string) $args['db_export_mode'] ) : 'auto', // auto|fast|php|table
            ),
            'files' => array(
                'queue_file' => '',
                'queue_total' => 0,
                'idx' => 0,
                'added' => 0,
                'queue_built' => false,
            ),
            'db' => array(
                'mode_used' => '',
                'tables' => array(),
                'idx' => 0,
                'table_offset' => 0,
                'attempted_fast' => false,
            ),
        );

		$zip = new \ZipArchive();
		$z = $this->open_zip( $zip, $zip_path, true );
		if ( ! $z['ok'] ) {
			$this->log( 'Failed to create zip: ' . $z['message'] . ' | path=' . $zip_path );
			do_action( 'aegisbackup_backup_failed', array( 'code' => 'zip_open_failed', 'message' => $z['message'], 'path' => $zip_path ), array() );
			return array( 'error' => 'zip_open_failed', 'message' => $z['message'], 'path' => $zip_path );
		}
		$zip->setArchiveComment( 'AegisBackup package ' . gmdate( 'c' ) );
		$zip->close();

        @file_put_contents( $checksum_path, '' );

        update_option( self::JOB_OPTION_PREFIX . $job_id, $state, false );
        $this->log( "Backup job started: {$job_id}" );

        return array(
            'job_id' => $job_id,
            'package_dir' => $pkg_dir,
            'zip_path' => $zip_path,
        );
    }

    public function process_backup_job( $job_id ) {
        $this->ab_dbg( $job_id, 'process_backup_job enter' );
        $state = get_option( self::JOB_OPTION_PREFIX . $job_id, array() );
        $this->ab_dbg( $job_id, 'state loaded. type=' . ( is_array( $state ) ? 'array' : gettype( $state ) ) . ' phase=' . ( is_array( $state ) && isset( $state['phase'] ) ? (string) $state['phase'] : '' ) . ' progress=' . ( is_array( $state ) && isset( $state['progress'] ) ? (string) $state['progress'] : '' ) );
        if ( empty( $state ) || empty( $state['job_id'] ) ) {
            $this->ab_dbg( $job_id, 'Job not found.' );
            return array( 'done' => true, 'progress' => 100, 'log' => 'Job not found.' );
        }

        $args = isset( $state['args'] ) ? (array) $state['args'] : array();

        if ( 'init' === $state['phase'] ) {
            $state['phase'] = ! empty( $args['include_files'] ) ? 'files' : ( ! empty( $args['include_db'] ) ? 'db' : 'manifest' );
            $state['progress'] = 2;
            update_option( self::JOB_OPTION_PREFIX . $job_id, $state, false );
            $this->ab_dbg( $job_id, 'init -> ' . (string) $state['phase'] . ' progress=' . (string) $state['progress'] );
            return array( 'done' => false, 'progress' => $state['progress'], 'log' => 'Initialized backup package.' );
        }

        if ( 'files' === $state['phase'] ) {
            $this->ab_dbg( $job_id, 'phase files start. files_idx=' . ( isset( $state['files_idx'] ) ? (string) $state['files_idx'] : '' ) . ' files_total=' . ( isset( $state['files_total'] ) ? (string) $state['files_total'] : '' ) );
            $step = $this->process_files_phase( $state );
            $this->ab_dbg( $job_id, 'phase files step returned. type=' . ( is_array( $step ) ? 'array' : gettype( $step ) ) . ' done=' . ( is_array( $step ) && ! empty( $step['done'] ) ? '1' : '0' ) . ' progress=' . ( is_array( $step ) && isset( $step['progress'] ) ? (string) $step['progress'] : '' ) . ' log=' . ( is_array( $step ) && isset( $step['log'] ) ? (string) $step['log'] : '' ) );
            update_option( self::JOB_OPTION_PREFIX . $job_id, $state, false );
            return $step;
        }

        if ( 'db' === $state['phase'] ) {
            $this->ab_dbg( $job_id, 'phase db start. db_table=' . ( isset( $state['db_table'] ) ? (string) $state['db_table'] : '' ) . ' db_offset=' . ( isset( $state['db_offset'] ) ? (string) $state['db_offset'] : '' ) );
            $step = $this->process_db_phase( $state );
            $this->ab_dbg( $job_id, 'phase db step returned. type=' . ( is_array( $step ) ? 'array' : gettype( $step ) ) . ' done=' . ( is_array( $step ) && ! empty( $step['done'] ) ? '1' : '0' ) . ' progress=' . ( is_array( $step ) && isset( $step['progress'] ) ? (string) $step['progress'] : '' ) . ' log=' . ( is_array( $step ) && isset( $step['log'] ) ? (string) $step['log'] : '' ) );
            update_option( self::JOB_OPTION_PREFIX . $job_id, $state, false );
            return $step;
        }

        if ( 'manifest' === $state['phase'] ) {
            $this->ab_dbg( $job_id, 'phase manifest start. pkg_dir=' . ( isset( $state['pkg_dir'] ) ? (string) $state['pkg_dir'] : '' ) . ' zip_path=' . ( isset( $state['zip_path'] ) ? (string) $state['zip_path'] : '' ) );
            $this->log( 'DEBUG entering manifest phase: job_id=' . (string) $job_id . ' zip=' . (string) $state['zip_path'] . ' checksum=' . (string) $state['checksum_path'] . ' work_dir=' . (string) $state['work_dir'] );
            $ok = $this->write_manifest_and_checksums( $state );
            $this->log( 'DEBUG manifest write result: job_id=' . (string) $job_id . ' ok=' . ( $ok ? '1' : '0' ) . ' zip_exists=' . ( is_file( (string) $state['zip_path'] ) ? '1' : '0' ) . ' zip_size=' . ( is_file( (string) $state['zip_path'] ) ? (string) @filesize( (string) $state['zip_path'] ) : 'na' ) );
            $state['manifest_written'] = $ok;
            $state['phase'] = 'done';
            $state['progress'] = 100;
            update_option( self::JOB_OPTION_PREFIX . $job_id, $state, false );
            $pkg_dir = (string) $state['pkg_dir'];
            $zip_path = (string) $state['zip_path'];
            $this->maybe_update_dr_token_registry_on_done( $state );
            $manifest_root = trailingslashit( $pkg_dir ) . 'manifest.json';
            if ( is_file( $state['work_dir'] . 'manifest.json' ) ) {
                @copy( $state['work_dir'] . 'manifest.json', $manifest_root );
            }

			$this->schedule_completion_hooks( $zip_path, $state );

			return array( 'done' => true, 'progress' => 100, 'log' => 'Backup package ready.', 'package' => $zip_path );
        }

        return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => 'Working...' );
    }

    private function process_files_phase( array &$state ) {
        $args = (array) $state['args'];
        $files_state = &$state['files'];
        $zip_path = (string) $state['zip_path'];
        $checksum_path = (string) $state['checksum_path'];

        $work_dir = trailingslashit( (string) $state['work_dir'] );
        $queue_file = $work_dir . 'file-queue.json';

        if ( empty( $files_state['queue_built'] ) ) {
            $queue = $this->build_files_queue( $args );
            @file_put_contents( $queue_file, wp_json_encode( array_values( $queue ) ) );
            $files_state['queue_file'] = $queue_file;
            $files_state['queue_total'] = count( $queue );
            $files_state['queue_built'] = true;
            $files_state['idx'] = 0;
            $files_state['added'] = 0;
            $state['progress'] = 5;
            return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => 'File queue indexed: ' . (int) $files_state['queue_total'] . ' items.' );
        }

        $raw = is_file( $files_state['queue_file'] ) ? file_get_contents( $files_state['queue_file'] ) : '[]';
        $queue = json_decode( (string) $raw, true );
        if ( ! is_array( $queue ) ) {
            $queue = array();
        }

        $batch = (int) ( defined( 'AEGISBACKUP_FILES_BATCH' ) ? AEGISBACKUP_FILES_BATCH : apply_filters( 'aegisbackup_files_batch', 120 ) );
        $total = (int) $files_state['queue_total'];
        $start = (int) $files_state['idx'];
        $end = min( $total, $start + $batch );

        if ( $start >= $total ) {
            $state['phase'] = ! empty( $args['include_db'] ) ? 'db' : 'manifest';
            $state['progress'] = 48;
            return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => 'Files phase complete.' );
        }

		$zip = new \ZipArchive();
		$z = $this->open_zip( $zip, $zip_path, true );
		if ( ! $z['ok'] ) {
			$state['status'] = 'error';
			do_action( 'aegisbackup_backup_failed', array( 'code' => 'zip_open_failed', 'message' => $z['message'], 'path' => $zip_path ), $state );
			return array(
				'done' => true,
				'progress' => (int) $state['progress'],
				'log' => 'Failed to open zip archive (' . $z['message'] . '). Path: ' . $zip_path,
				'zip_error' => $z,
			);
		}

        $added_now = 0;
        for ( $i = $start; $i < $end; $i++ ) {
            $rel = isset( $queue[ $i ] ) ? (string) $queue[ $i ] : '';
            if ( '' === $rel ) {
                continue;
            }

            $rel_norm = ltrim( str_replace( '\\', '/', $rel ), '/' );

            $src = ABSPATH . $rel_norm;
            if ( ! is_file( $src ) ) {
                continue;
            }

            $zip_name = 'files/' . $rel_norm;
            $zip->deleteName( $zip_name );
            $zip->addFile( $src, $zip_name );
            $added_now++;
            $hash = @hash_file( 'sha256', $src );
            if ( $hash ) {
                @file_put_contents( $checksum_path, $hash . '  ' . $zip_name . "\n", FILE_APPEND );
            }
        }
        $zip->close();

        $files_state['idx'] = $end;
        $files_state['added'] = (int) $files_state['added'] + $added_now;

        $pct = $total > 0 ? ( $files_state['idx'] / $total ) : 1;
        $state['progress'] = (int) ( 5 + ( $pct * 43 ) ); // 5..48

        return array(
            'done' => false,
            'progress' => (int) $state['progress'],
            'log' => 'Added files: ' . (int) $added_now . ' (total ' . (int) $files_state['added'] . ')',
        );
    }

    private function process_db_phase( array &$state ) {
        global $wpdb;

        $args = (array) $state['args'];
        $work_dir = trailingslashit( (string) $state['work_dir'] );
        $zip_path = (string) $state['zip_path'];
        $checksum_path = (string) $state['checksum_path'];
        $db_state = &$state['db'];
        $db_dir = $work_dir . 'db';
        wp_mkdir_p( $db_dir );
        $master_file = trailingslashit( $db_dir ) . 'db.sql';
        if ( ! file_exists( $master_file ) ) {
            $hdr  = "-- AegisBackup full database dump\n";
            $hdr .= "-- Generated: " . gmdate( 'c' ) . "\n\n";
            $hdr .= "SET FOREIGN_KEY_CHECKS=0;\n\n";
            @file_put_contents( $master_file, $hdr );
        }

        $mode = isset( $args['db_export_mode'] ) ? sanitize_key( (string) $args['db_export_mode'] ) : 'auto';
        if ( 'auto' === $mode ) {
            $mode = 'table';
        }

        if ( 'fast' === $mode && empty( $db_state['attempted_fast'] ) ) {
            $db_state['attempted_fast'] = true;

            $fast = $this->try_mysqldump( $work_dir );
            if ( ! empty( $fast['ok'] ) && ! empty( $fast['file'] ) && is_file( $fast['file'] ) ) {
                $db_state['mode_used'] = 'fast';
                $zip = new \ZipArchive();
                $z = $this->open_zip( $zip, $zip_path, true );
				if ( ! empty( $z['ok'] ) ) {
                    $zip_name = 'db/db.sql';
                    $zip->deleteName( $zip_name );
                    $zip->addFile( $fast['file'], $zip_name );
                    $zip->close();
                    $hash = @hash_file( 'sha256', $fast['file'] );
                    if ( $hash ) {
                        @file_put_contents( $checksum_path, $hash . '  ' . $zip_name . "\n", FILE_APPEND );
                    }
                }

                wp_delete_file( $fast['file'] );
                $state['phase'] = 'manifest';
                $state['progress'] = 90;
                return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => 'DB exported via mysqldump.' );
            }

            $mode = 'table';
        }

        if ( empty( $db_state['tables'] ) ) {
            $prefix = $wpdb->prefix;
            $like = $wpdb->esc_like( $prefix ) . '%';
            $cache_key = 'aegisbackup_db_export_tables_' . md5( (string) $like );
            $tables = wp_cache_get( $cache_key, 'aegisbackup' );
            if ( false === $tables ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Required to enumerate tables for export.
                $tables = $wpdb->get_col( $wpdb->prepare( 'SHOW TABLES LIKE %s', $like ) );
                wp_cache_set( $cache_key, $tables, 'aegisbackup', 300 );
            }
            $db_state['tables'] = is_array( $tables ) ? array_values( $tables ) : array();
            $db_state['idx'] = 0;
            $db_state['table_offset'] = 0;
            $db_state['mode_used'] = ( 'php' === $mode ) ? 'php' : 'table';
            $state['progress'] = 50;
            return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => 'DB tables discovered: ' . count( $db_state['tables'] ) );
        }

        $tidx = (int) $db_state['idx'];
        $tables = (array) $db_state['tables'];

        if ( $tidx >= count( $tables ) ) {
            @file_put_contents( $master_file, "\nSET FOREIGN_KEY_CHECKS=1;\n", FILE_APPEND );
            $zip = new \ZipArchive();
			$z = $this->open_zip( $zip, $zip_path, true );
			if ( ! empty( $z['ok'] ) ) {
                $zip_name = 'db/db.sql';
                $zip->deleteName( $zip_name );
                if ( is_file( $master_file ) ) {
                    $zip->addFile( $master_file, $zip_name );
                    $hash = @hash_file( 'sha256', $master_file );
                    if ( $hash ) {
                        @file_put_contents( $checksum_path, $hash . '  ' . $zip_name . "\n", FILE_APPEND );
                    }
                }
                $zip->close();
            }
            wp_delete_file( $master_file );

            $state['phase'] = 'manifest';
            $state['progress'] = 90;
            return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => 'DB export complete.' );
        }

        $table = (string) $tables[ $tidx ];
        $limit = (int) ( defined( 'AEGISBACKUP_DB_EXPORT_LIMIT' ) ? AEGISBACKUP_DB_EXPORT_LIMIT : apply_filters( 'aegisbackup_db_export_limit', 250 ) );
        $offset = (int) $db_state['table_offset'];
        $safe_name = preg_replace( '/[^a-zA-Z0-9_\-]/', '_', $table );
        $file = trailingslashit( $db_dir ) . $safe_name . '.sql';

        if ( 0 === $offset && ! file_exists( $file ) ) {
            $cache_key = 'aegisbackup_show_create_' . md5( (string) $table );
            $create = wp_cache_get( $cache_key, 'aegisbackup' );
            if ( false === $create ) {
                // Table identifiers should be prepared using %i (WP 6.2+).
                $create = $wpdb->get_row( $wpdb->prepare( 'SHOW CREATE TABLE %i', $table ), ARRAY_N ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange -- Cached for 5 minutes; SHOW CREATE TABLE is required for export header.
                wp_cache_set( $cache_key, $create, 'aegisbackup', 300 );
            }
            $header  = "-- AegisBackup table export: {$table}\n";
            $header .= "-- Generated: " . gmdate( 'c' ) . "\n\n";
            $header .= "DROP TABLE IF EXISTS `" . str_replace( '`', '``', $table ) . "`;\n";
            if ( isset( $create[1] ) ) {
                $header .= $create[1] . ";\n\n";
            }
            @file_put_contents( $file, $header );
            @file_put_contents( $master_file, $header, FILE_APPEND );
        }

        // Table identifiers should be prepared using %i (WP 6.2+).
        $rows = $wpdb->get_results( $wpdb->prepare( 'SELECT * FROM %i LIMIT %d OFFSET %d', $table, $limit, $offset ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Batch export.
        if ( empty( $rows ) ) {
            $zip = new \ZipArchive();
			$z = $this->open_zip( $zip, $zip_path, true );
			if ( ! empty( $z['ok'] ) ) {
                $zip_name = 'db/tables/' . $safe_name . '.sql';
                $zip->deleteName( $zip_name );
                $zip->addFile( $file, $zip_name );
                $zip->close();
                $hash = @hash_file( 'sha256', $file );
                if ( $hash ) {
                    @file_put_contents( $checksum_path, $hash . '  ' . $zip_name . "\n", FILE_APPEND );
                }
            }
            wp_delete_file( $file );

            $db_state['idx'] = $tidx + 1;
            $db_state['table_offset'] = 0;

            $pct = count( $tables ) > 0 ? ( $db_state['idx'] / count( $tables ) ) : 1;
            $state['progress'] = (int) ( 50 + ( $pct * 40 ) );
            return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => 'Exported table: ' . $table );
        }

        $sql = '';
        foreach ( (array) $rows as $row ) {
            $cols = array();
            $vals = array();
            foreach ( (array) $row as $k => $v ) {
                $cols[] = '`' . str_replace( '`', '``', (string) $k ) . '`';
                $vals[] = $this->sql_escape_value( $v );
            }
            $sql .= 'INSERT INTO `' . str_replace( '`', '``', $table ) . '` (' . implode( ',', $cols ) . ') VALUES (' . implode( ',', $vals ) . ");\n";
        }
        @file_put_contents( $file, $sql, FILE_APPEND );
        @file_put_contents( $master_file, $sql, FILE_APPEND );

        $db_state['table_offset'] = $offset + count( $rows );
        $pct = count( $tables ) > 0 ? ( $tidx / count( $tables ) ) : 0;
        $state['progress'] = (int) ( 50 + ( $pct * 40 ) );

        return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => 'DB exporting: ' . $table . ' offset ' . (int) $db_state['table_offset'] );
    }

    private function write_manifest_and_checksums( array &$state ) {
        global $wpdb;

        $args = (array) $state['args'];
        $pkg_dir = (string) $state['pkg_dir'];
        $work_dir = trailingslashit( (string) $state['work_dir'] );
        $zip_path = (string) $state['zip_path'];
        $checksum_path = (string) $state['checksum_path'];

        $manifest = new AB_Manifest();

        $data = array(
            'backup_format_version' => '1.2',
            'package_format' => 'option1',
            'db_dump_path' => 'db/db.sql',
            'site_url' => site_url(),
            'home_url' => home_url(),
            'wp_version' => get_bloginfo( 'version' ),
            'php_version' => phpversion(),
            'mysql_version' => $wpdb->db_version(),
            'table_prefix' => $wpdb->prefix,
            'db_export_mode' => isset( $state['db']['mode_used'] ) ? (string) $state['db']['mode_used'] : '',
			'package_purpose' => isset( $args['package_purpose'] ) ? sanitize_key( (string) $args['package_purpose'] ) : '',
			'update_context' => isset( $args['update_context'] ) && is_array( $args['update_context'] ) ? (array) $args['update_context'] : array(),
            'tables' => $manifest->get_table_sizes( $wpdb->prefix ),
            'plugins' => $manifest->get_plugins_list(),
            'mu_plugins' => $manifest->get_mu_plugins_list(),
            'theme' => $manifest->get_theme_info(),
            'wp_snapshot' => ! empty( $args['snapshot'] ) ? $manifest->get_wp_snapshot() : array(),
            'flags' => array(
                'include_files' => ! empty( $args['include_files'] ),
                'include_db' => ! empty( $args['include_db'] ),
                'include_config' => ! empty( $args['include_config'] ),
                'include_core' => ! empty( $args['include_core'] ),
            ),
        );

                $this->log( 'DEBUG building manifest data: zip=' . (string) $zip_path . ' checksum_path=' . (string) $checksum_path . ' work_dir=' . (string) $work_dir );
        $built = $manifest->build_manifest( $data );
        $manifest_path = $work_dir . 'manifest.json';
        $json = wp_json_encode( $built, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES );
        if ( false === $json ) { $this->log( 'DEBUG manifest json encode failed.' ); }
        $wrote = @file_put_contents( $manifest_path, $json );
        $this->log( 'DEBUG wrote manifest.json: path=' . (string) $manifest_path . ' bytes=' . (string) $wrote . ' exists=' . ( is_file( $manifest_path ) ? '1' : '0' ) . ' size=' . ( is_file( $manifest_path ) ? (string) @filesize( $manifest_path ) : 'na' ) );

		$zip = new \ZipArchive();
		$z = $this->open_zip( $zip, $zip_path, true );
		if ( ! $z['ok'] ) {
			$this->log( 'Failed to open zip for manifest: ' . $z['message'] . ' | path=' . $zip_path );
			return false;
		}

        $zip->deleteName( 'manifest.json' );
        $ok_m =         $this->log( 'DEBUG zip add manifest.json: ok=' . ( $ok_m ? '1' : '0' ) . ' src_exists=' . ( is_file( $manifest_path ) ? '1' : '0' ) );

        
        $zip->addFile( $manifest_path, 'manifest.json' );

        $zip->deleteName( 'checksums.sha256' );
        $ok_c =         $this->log( 'DEBUG zip add checksums.sha256: ok=' . ( $ok_c ? '1' : '0' ) . ' src_exists=' . ( is_file( $checksum_path ) ? '1' : '0' ) . ' size=' . ( is_file( $checksum_path ) ? (string) @filesize( $checksum_path ) : 'na' ) );

        
        $zip->addFile( $checksum_path, 'checksums.sha256' );

        $zip->close();

        // Also keep manifest at package root for listing.
        $dst_manifest = trailingslashit( $pkg_dir ) . 'manifest.json';
        $copied = @copy( $manifest_path, $dst_manifest );
        $this->log( 'DEBUG copy manifest to package root: ok=' . ( $copied ? '1' : '0' ) . ' dst=' . (string) $dst_manifest . ' exists=' . ( is_file( $dst_manifest ) ? '1' : '0' ) );

        return true;
    }

    private function build_files_queue( array $args ) {
        $queue = array();
        $excludes = $this->parse_excludes( isset( $args['excludes'] ) ? (string) $args['excludes'] : '' );

        if ( ! empty( $args['include_htaccess'] ) ) {
            $root_paths = array(
                '.htaccess',
                'web.config',
                '.user.ini',
                'php.ini',
            );

            foreach ( $root_paths as $p ) {
                $abs = ABSPATH . ltrim( $p, "/\\" );
                if ( is_file( $abs ) ) {
                    $queue[] = $p;
                }
            }
        }

        if ( ! empty( $args['include_files'] ) ) {
            $paths = array(
                'wp-content',
            );

            if ( ! empty( $args['include_paths'] ) && is_array( $args['include_paths'] ) ) {
                $paths = array();
                foreach ( $args['include_paths'] as $p ) {
                    $p = ltrim( str_replace( '\\', '/', (string) $p ), '/' );
                    $p = rtrim( $p, '/' );
                    if ( '' === $p ) {
                        continue;
                    }
                    $paths[] = $p;
                }
                if ( empty( $paths ) ) {
                    $paths = array( 'wp-content' );
                }
            }


            if ( ! empty( $args['include_config'] ) ) {
                $paths[] = 'wp-config.php';
            }

            if ( ! empty( $args['include_core'] ) ) {
                $paths[] = '';
            }

	            foreach ( $paths as $p ) {
	                $abs = ABSPATH . ltrim( $p, "/\\" );
                if ( '' === $p ) {
                    $entries = glob( ABSPATH . '*', GLOB_NOSORT );
                    foreach ( (array) $entries as $e ) {
                        $rel = str_replace( ABSPATH, '', $e );
                        $rel = ltrim( str_replace( '\\', '/', $rel ), '/' );
                        if ( 'wp-content' === $rel ) {
                            continue;
                        }
                        if ( $this->is_excluded( $rel, $excludes ) ) {
                            continue;
                        }
                        if ( is_dir( $e ) ) {
                            foreach ( $this->rglob( $e ) as $f ) {
	                                $r = ltrim( str_replace( ABSPATH, '', $f ), "/\\" );
                                $r = str_replace( '\\', '/', $r );
                                if ( $this->is_excluded( $r, $excludes ) ) {
                                    continue;
                                }
                                $queue[] = $r;
                            }
                        } elseif ( is_file( $e ) ) {
                            $queue[] = $rel;
                        }
                    }
                    continue;
                }

                if ( is_file( $abs ) ) {
	                    $rel = ltrim( str_replace( ABSPATH, '', $abs ), "/\\" );
                    $rel = str_replace( '\\', '/', $rel );
                    if ( ! $this->is_excluded( $rel, $excludes ) ) {
                        $queue[] = $rel;
                    }
                    continue;
                }

                if ( is_dir( $abs ) ) {
                    foreach ( $this->rglob( $abs ) as $file ) {
	                        $rel = ltrim( str_replace( ABSPATH, '', $file ), "/\\" );
                        $rel = str_replace( '\\', '/', $rel );
                        if ( $this->is_excluded( $rel, $excludes ) ) {
                            continue;
                        }
                        $queue[] = $rel;
                    }
                }
            }
        }

		if ( ! empty( $args['include_paths'] ) && is_array( $args['include_paths'] ) ) {
			$paths = array();
			foreach ( $args['include_paths'] as $p ) {
				$p = ltrim( str_replace( '\\', '/', (string) $p ), '/' );
				$p = rtrim( $p, '/' );
				if ( '' === $p ) {
					continue;
				}
				$paths[] = $p;
			}
			if ( empty( $paths ) ) {
				$paths = array( 'wp-content' );
			}
		}
        $queue = array_values( array_unique( $queue ) );
        sort( $queue );
        return $queue;
    }

    private function parse_excludes( $raw ) {
        $lines = preg_split( '/\r\n|\r|\n/', (string) $raw );
        $out = array();
        foreach ( (array) $lines as $line ) {
            $line = trim( (string) $line );
            if ( '' === $line ) {
                continue;
            }
            $line = ltrim( str_replace( '\\', '/', $line ), '/' );
            $out[] = $line;
        }

        $builtins = array(
            'wp-content/cache',
            'wp-content/wflogs',
            'wp-content/uploads/aegisbackup',
            'wp-content/uploads/wpvivid_backup',
            'wp-content/updraft',
            'wp-content/ai1wm-backups',
        );
        foreach ( $builtins as $b ) {
            $out[] = $b;
        }

        return array_values( array_unique( $out ) );
    }

    private function is_excluded( $rel, array $excludes ) {
        $rel = ltrim( str_replace( '\\', '/', (string) $rel ), '/' );
        foreach ( $excludes as $ex ) {
            $ex = ltrim( str_replace( '\\', '/', (string) $ex ), '/' );
            if ( '' === $ex ) {
                continue;
            }
            if ( 0 === strpos( $rel, $ex ) ) {
                return true;
            }
        }
        return false;
    }

    private function rglob( $dir ) {
        $files = array();
        $dir = rtrim( (string) $dir, '/\\' );
        if ( ! is_dir( $dir ) ) {
            return $files;
        }
        $it = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator( $dir, \FilesystemIterator::SKIP_DOTS ),
            \RecursiveIteratorIterator::LEAVES_ONLY
        );
        foreach ( $it as $file ) {
            if ( $file->isFile() ) {
                $files[] = $file->getPathname();
            }
        }
        return $files;
    }

    private function human_bytes( $bytes ) {
        $bytes = (int) $bytes;
        $units = array( 'B', 'KB', 'MB', 'GB', 'TB' );
        $i = 0;
        $val = (float) $bytes;
        while ( $val >= 1024 && $i < count( $units ) - 1 ) {
            $val /= 1024;
            $i++;
        }
        if ( 0 === $i ) {
            return (int) $val . ' ' . $units[ $i ];
        }
        return number_format( $val, 2 ) . ' ' . $units[ $i ];
    }

    private function sql_escape_value( $value ) {
        if ( null === $value ) {
            return 'NULL';
        }
        if ( is_bool( $value ) ) {
            return $value ? '1' : '0';
        }
        if ( is_int( $value ) || is_float( $value ) ) {
            return (string) $value;
        }

        $str = (string) $value;
        $escaped = '';
        $len = strlen( $str );
        for ( $i = 0; $i < $len; $i++ ) {
            $ch = $str[ $i ];
            if ( "\n" === $ch ) { $escaped .= "\\n"; continue; }
            if ( "\r" === $ch ) { $escaped .= "\\r"; continue; }
            if ( "\t" === $ch ) { $escaped .= "\\t"; continue; }
            if ( "\0" === $ch ) { $escaped .= "\\0"; continue; }
            if ( "\x1a" === $ch ) { $escaped .= "\\Z"; continue; }
            if ( "\\" === $ch ) { $escaped .= "\\\\"; continue; }
            if ( "'" === $ch ) { $escaped .= "\\'"; continue; }
            $escaped .= $ch;
        }
        return "'" . $escaped . "'";
    }

    private function try_mysqldump( $work_dir ) {
        $disabled = (string) ini_get( 'disable_functions' );
        if ( function_exists( 'exec' ) && false === strpos( $disabled, 'exec' ) ) {
            $mysqldump = 'mysqldump';
            @exec( 'command -v mysqldump 2>/dev/null', $out, $rc );
            if ( 0 === $rc && ! empty( $out[0] ) ) {
                $mysqldump = trim( (string) $out[0] );
            }

            if ( defined( 'DB_NAME' ) && defined( 'DB_USER' ) && defined( 'DB_PASSWORD' ) && defined( 'DB_HOST' ) ) {
                $host = DB_HOST;
                $port = '';
                if ( strpos( $host, ':' ) !== false ) {
                    list( $host, $port ) = explode( ':', $host, 2 );
                }

                $file = trailingslashit( $work_dir ) . 'db-fast.sql';
                $cmd = escapeshellcmd( $mysqldump ) .
                    ' --single-transaction --quick --skip-lock-tables' .
                    ' -h ' . escapeshellarg( $host ) .
                    ( $port ? ' -P ' . escapeshellarg( $port ) : '' ) .
                    ' -u ' . escapeshellarg( DB_USER ) .
                    ' ' . ( DB_PASSWORD !== '' ? ' -p' . escapeshellarg( DB_PASSWORD ) : '' ) .
                    ' ' . escapeshellarg( DB_NAME ) .
                    ' > ' . escapeshellarg( $file ) . ' 2>&1';

                @exec( $cmd, $o, $code );
                if ( 0 === (int) $code && is_file( $file ) && filesize( $file ) > 0 ) {
                    return array( 'ok' => true, 'file' => $file );
                }
            }
        }

        return array( 'ok' => false );
    }

protected function ab_get_fs() {
    global $wp_filesystem;

    if ( ! function_exists( 'WP_Filesystem' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }

    if ( ! is_object( $wp_filesystem ) ) {
        WP_Filesystem();
    }

    return is_object( $wp_filesystem ) ? $wp_filesystem : null;
}

private function log( $message ) {
        $logs = get_option( 'aegisbackup_logs', array() );
        if ( ! is_array( $logs ) ) {
            $logs = array();
        }
        $logs[] = '[' . gmdate( 'Y-m-d H:i:s' ) . '] ' . (string) $message;
        $logs = array_slice( $logs, -500 );
        update_option( 'aegisbackup_logs', $logs, false );
    }

	private function ab_dbg( $job_id, $message ) {
		$job_id = (string) $job_id;

		$msg = '[' . gmdate( 'Y-m-d H:i:s' ) . ' UTC] '
			. ( $job_id ? ( 'job=' . $job_id . ' ' ) : '' )
			. (string) $message;

		$upload = wp_upload_dir();
		$dir    = trailingslashit( (string) $upload['basedir'] ) . 'aegisbackup/logs/backup-jobs';

		if ( ! is_dir( $dir ) ) {
			@wp_mkdir_p( $dir );
		}

		$safe = $job_id ? preg_replace( '/[^a-zA-Z0-9_\-]/', '', $job_id ) : 'unknown';
		$file = trailingslashit( $dir ) . 'job-' . $safe . '.log';

		@file_put_contents( $file, $msg . "\n", FILE_APPEND );

		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( 'AegisBackup DEBUG: ' . $msg );
		}
	}

}