<?php
namespace AegisBackup\Backup;

defined( 'ABSPATH' ) || exit;

class AB_File_Backup_Manager {
	const PLANS_OPTION = 'aegisbackup_file_backup_plans';
	const PLAN_STATE_PREFIX = 'aegisbackup_file_backup_state_';
	const CRON_HOOK = 'aegisbackup_run_file_backup_plan';

	/**
	 * Get an initialized WP_Filesystem instance (Direct/FTP/etc.).
	 * Minimal helper to satisfy WP Plugin Checker filesystem operation rules.
	 *
	 * @return \WP_Filesystem_Base|null
	 */
	private function ab_filesystem() {
		static $fs = null;
		if ( null !== $fs ) {
			return $fs;
		}

		if ( ! function_exists( 'WP_Filesystem' ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}

		$ok = WP_Filesystem();
		if ( $ok ) {
			global $wp_filesystem;
			if ( isset( $wp_filesystem ) && $wp_filesystem ) {
				$fs = $wp_filesystem;
				return $fs;
			}
		}

		$fs = null;
		return null;
	}

	public function backups_dir() {
		$u = wp_upload_dir();
		$base = isset( $u['basedir'] ) ? (string) $u['basedir'] : '';
		$dir = trailingslashit( $base ) . 'aegisbackup/file-backups';
		$legacy_dir = trailingslashit( AEGISBACKUP_DIR ) . 'backups';

		if ( ! is_dir( $dir ) ) {
			wp_mkdir_p( $dir );
		}

		if ( is_dir( $legacy_dir ) && is_dir( $dir ) ) {
			$new_has = glob( trailingslashit( $dir ) . '*.{zip,json}', GLOB_BRACE );
			$old_has = glob( trailingslashit( $legacy_dir ) . '*.{zip,json}', GLOB_BRACE );
			if ( empty( $new_has ) && ! empty( $old_has ) ) {
				foreach ( (array) $old_has as $src ) {
					$dst = trailingslashit( $dir ) . basename( (string) $src );
					if ( ! is_file( $dst ) && is_file( $src ) ) {
						@copy( $src, $dst );
					}
				}
			}
		}

		if ( is_dir( $dir ) && ! wp_is_writable( $dir ) ) {
			$fs = $this->ab_filesystem();
			if ( $fs ) {
				$fs->chmod( $dir, 0755 );
			}
		}

		if ( is_dir( $dir ) ) {
			if ( ! is_file( trailingslashit( $dir ) . 'index.php' ) ) {
				@file_put_contents( trailingslashit( $dir ) . 'index.php', "<?php\n// Silence is golden.\n" );
				$fs = $this->ab_filesystem();
				if ( $fs ) {
					$fs->chmod( trailingslashit( $dir ) . 'index.php', 0640 );
				}
			}
			if ( ! is_file( trailingslashit( $dir ) . '.htaccess' ) ) {
				@file_put_contents(
					trailingslashit( $dir ) . '.htaccess',
					"Options -Indexes\n<IfModule mod_headers.c>\nHeader set X-Robots-Tag \"noindex, nofollow\"\n</IfModule>\n<IfModule mod_authz_core.c>\nRequire all denied\n</IfModule>\n<IfModule !mod_authz_core.c>\nDeny from all\n</IfModule>\n"
				);
				$fs = $this->ab_filesystem();
				if ( $fs ) {
					$fs->chmod( trailingslashit( $dir ) . '.htaccess', 0640 );
				}
			}
		}

		return $dir;
	}

	public function get_plans() {
		$plans = get_option( self::PLANS_OPTION, array() );
		return is_array( $plans ) ? $plans : array();
	}

	public function get_plan( $plan_id ) {
		$needle_raw = (string) $plan_id;
		$needle_key = sanitize_key( $needle_raw );
		foreach ( $this->get_plans() as $p ) {
			if ( ! isset( $p['id'] ) ) {
				continue;
			}
			$pid_raw = (string) $p['id'];

			if ( $pid_raw === $needle_raw || sanitize_key( $pid_raw ) === $needle_key ) {
				return $p;
			}
		}
		return null;
	}

	public function save_plan( array $plan ) {
		$plans = $this->get_plans();
		$needle_raw = isset( $plan['id'] ) ? (string) $plan['id'] : '';
		$needle_key = sanitize_key( $needle_raw );
		$found = false;
		foreach ( $plans as $i => $p ) {
			if ( ! isset( $p['id'] ) ) {
				continue;
			}
			$pid_raw = (string) $p['id'];
			if ( $pid_raw === $needle_raw || sanitize_key( $pid_raw ) === $needle_key ) {
				$plans[ $i ] = $plan;
				$found = true;
				break;
			}
		}
		if ( ! $found ) {
			$plans[] = $plan;
		}
		update_option( self::PLANS_OPTION, array_values( $plans ), false );
		return true;
	}

	public function delete_plan( $plan_id ) {
		$plans = $this->get_plans();
		$needle_raw = (string) $plan_id;
		$needle_key = sanitize_key( $needle_raw );
		$out = array();
		foreach ( $plans as $p ) {
			if ( ! isset( $p['id'] ) ) {
				$out[] = $p;
				continue;
			}
			$pid_raw = (string) $p['id'];
			if ( $pid_raw === $needle_raw || sanitize_key( $pid_raw ) === $needle_key ) {
				continue;
			}
			$out[] = $p;
		}
		update_option( self::PLANS_OPTION, array_values( $out ), false );
		$this->unschedule_plan( $plan_id );
		return true;
	}

	public function list_backups() {
		$dir = $this->backups_dir();
		$items = glob( trailingslashit( $dir ) . '*.zip' );
		if ( ! $items ) {
			return array();
		}
		$out = array();
		foreach ( (array) $items as $zip_path ) {
			$base = basename( $zip_path, '.zip' );
			$manifest_path = trailingslashit( $dir ) . $base . '.json';
			$manifest = array();
			if ( is_file( $manifest_path ) ) {
				$raw = file_get_contents( $manifest_path );
				$m = json_decode( (string) $raw, true );
				if ( is_array( $m ) ) {
					$manifest = $m;
				}
			}
			$out[] = array(
				'id' => $base,
				'file' => $zip_path,
				'size' => $this->human_bytes( (int) @filesize( $zip_path ) ),
				'created' => date_i18n( 'Y-m-d H:i:s', (int) @filemtime( $zip_path ) ),
				'manifest' => $manifest,
			);
		}
		usort(
			$out,
			static function( $a, $b ) {
				return strcmp( (string) $b['created'], (string) $a['created'] );
			}
		);
		return $out;
	}

	public function delete_backup( $backup_id ) {
		$dir = $this->backups_dir();
		$zip = trailingslashit( $dir ) . basename( (string) $backup_id ) . '.zip';
		$json = trailingslashit( $dir ) . basename( (string) $backup_id ) . '.json';
		$ok = true;
		if ( is_file( $zip ) ) {
			$ok = (bool) wp_delete_file( $zip ) && $ok;
		}
		if ( is_file( $json ) ) {
			$ok = (bool) wp_delete_file( $json ) && $ok;
		}
		return $ok;
	}

	public function run_plan_now( array $plan ) {
		$dir = $this->backups_dir();
		if ( ! is_dir( $dir ) ) {
			return array(
				'ok' => false,
				'id' => '',
				'path' => '',
				'message' => 'Backup directory does not exist: ' . $dir,
			);
		}
		if ( ! wp_is_writable( $dir ) ) {
			return array(
				'ok' => false,
				'id' => '',
				'path' => '',
				'message' => 'Backup directory is not writable: ' . $dir . ' (fix server permissions/ownership)',
			);
		}
		if ( ! is_dir( $dir ) || ! wp_is_writable( $dir ) ) {
			return array(
				'ok' => false,
				'id' => '',
				'path' => '',
				'message' => 'Backup storage directory is not writable: ' . $dir,
			);
		}
		$type = isset( $plan['type'] ) ? sanitize_key( (string) $plan['type'] ) : 'full';
		$plan_id = isset( $plan['id'] ) ? sanitize_key( (string) $plan['id'] ) : 'plan';
		$stamp = gmdate( 'Ymd_His' );
		$backup_id = 'filebackup-' . $plan_id . '-' . $stamp . '-' . $type;
		$zip_path = trailingslashit( $dir ) . $backup_id . '.zip';
		$manifest_path = trailingslashit( $dir ) . $backup_id . '.json';

		$paths = isset( $plan['paths'] ) && is_array( $plan['paths'] ) ? $plan['paths'] : array();
		$paths = array_values( array_filter( array_map( array( $this, 'normalize_rel_path' ), $paths ) ) );
		if ( empty( $paths ) ) {
			return array( 'ok' => false, 'id' => '', 'path' => '', 'message' => 'No paths selected.' );
		}

		$state = $this->get_plan_state( $plan_id );
		$baseline = array();
		$baseline_mode = '';
		if ( 'incremental' === $type && ! empty( $state['last_manifest'] ) && is_array( $state['last_manifest'] ) ) {
			$baseline = (array) $state['last_manifest'];
			$baseline_mode = 'incremental';
		} elseif ( 'differential' === $type && ! empty( $state['last_full_manifest'] ) && is_array( $state['last_full_manifest'] ) ) {
			$baseline = (array) $state['last_full_manifest'];
			$baseline_mode = 'differential';
		}

		$zip = new \ZipArchive();
		$ok = $zip->open( $zip_path, \ZipArchive::CREATE | \ZipArchive::OVERWRITE );
		if ( true !== $ok ) {
			return array(
				'ok' => false,
				'id' => '',
				'path' => '',
				'message' => 'Failed to create zip archive (ZipArchive::open code=' . (string) $ok . '). Target: ' . $zip_path,
			);
		}

		$zip->setArchiveComment( 'AegisBackup File Backup ' . gmdate( 'c' ) );
		$root = rtrim( ABSPATH, '/\\' ) . DIRECTORY_SEPARATOR;
		$file_index = array();
		$added = 0;
		$total_bytes = 0;
		$deleted = array();
		$current_manifest = array();
		foreach ( $this->iter_files_for_paths( $paths ) as $rel => $abs ) {
			$stat = @stat( $abs );
			$mtime = $stat ? (int) $stat['mtime'] : 0;
			$size = $stat ? (int) $stat['size'] : 0;
			$hash = '';
			$need_hash = true;
			if ( isset( $baseline[ $rel ] ) && is_array( $baseline[ $rel ] ) ) {
				$b = $baseline[ $rel ];
				if ( isset( $b['mtime'], $b['size'] ) && (int) $b['mtime'] === $mtime && (int) $b['size'] === $size ) {
					$need_hash = false;
					$hash = isset( $b['sha256'] ) ? (string) $b['sha256'] : '';
				}
			}
			if ( $need_hash ) {
				$hash = @hash_file( 'sha256', $abs );
			}
			$current_manifest[ $rel ] = array(
				'sha256' => (string) $hash,
				'mtime' => $mtime,
				'size' => $size,
			);
		}

		if ( ! empty( $baseline ) ) {
			foreach ( $baseline as $rel => $meta ) {
				if ( ! isset( $current_manifest[ $rel ] ) ) {
					$deleted[] = $rel;
				}
			}
		}

		foreach ( $current_manifest as $rel => $meta ) {
			$include = true;
			if ( ! empty( $baseline ) ) {
				if ( isset( $baseline[ $rel ] ) ) {
					$b = (array) $baseline[ $rel ];
					$include = (string) ( $b['sha256'] ?? '' ) !== (string) ( $meta['sha256'] ?? '' );
				} else {
					$include = true;
				}
			}

			if ( ! $include ) {
				continue;
			}
			$abs = $root . str_replace( '/', DIRECTORY_SEPARATOR, $rel );
			if ( ! is_file( $abs ) ) {
				continue;
			}
			$zip_name = 'files/' . $rel;
			$zip->addFile( $abs, $zip_name );
			$file_index[] = $rel;
			$added++;
			$total_bytes += (int) ( $meta['size'] ?? 0 );
		}

		$manifest = array(
			'id' => $backup_id,
			'plan_id' => $plan_id,
			'created_gmt' => gmdate( 'c' ),
			'type' => $type,
			'baseline_mode' => $baseline_mode,
			'root' => rtrim( ABSPATH, '/\\' ),
			'paths' => $paths,
			'added_files' => $file_index,
			'deleted_files' => $deleted,
			'file_count_added' => $added,
			'bytes_added' => $total_bytes,
			'full_manifest' => $current_manifest,
		);
		$zip->addFromString( 'manifest.json', wp_json_encode( $manifest, JSON_PRETTY_PRINT ) );
		$zip->close();

		@file_put_contents( $manifest_path, wp_json_encode( $manifest, JSON_PRETTY_PRINT ) );

		$state['last_backup_id'] = $backup_id;
		$state['last_backup_time'] = time();
		$state['last_manifest'] = $current_manifest;
		if ( 'full' === $type ) {
			$state['last_full_backup_id'] = $backup_id;
			$state['last_full_time'] = time();
			$state['last_full_manifest'] = $current_manifest;
		}
		$this->set_plan_state( $plan_id, $state );

		return array( 'ok' => true, 'id' => $backup_id, 'path' => $zip_path, 'message' => 'Backup created.' );
	}

	public function restore_backup( $backup_id ) {
		$dir = $this->backups_dir();
		$zip_path = trailingslashit( $dir ) . basename( (string) $backup_id ) . '.zip';
		if ( ! is_file( $zip_path ) ) {
			return array( 'ok' => false, 'message' => 'Backup not found.' );
		}

		if ( function_exists( 'ignore_user_abort' ) ) {
			@ignore_user_abort( true );
		}
		if ( function_exists( 'set_time_limit' ) ) {
			// phpcs:ignore Squiz.PHP.DiscouragedFunctions.Discouraged -- Backup/restore may run long on shared hosting.
			@set_time_limit( 0 );
		}

		$zip = new \ZipArchive();
		$ok = $zip->open( $zip_path );
		if ( true !== $ok ) {
			return array( 'ok' => false, 'message' => 'Failed to open backup zip.' );
		}

		$root = rtrim( ABSPATH, '/\\' ) . DIRECTORY_SEPARATOR;
		$restored = 0;
		$skipped = 0;
		$skip_prefixes = array(
			'wp-content/plugins/aegisbackup/',
			'wp-content/plugins/aegisbackup\\',
		);

		for ( $i = 0; $i < $zip->numFiles; $i++ ) {
			$name = (string) $zip->getNameIndex( $i );
			if ( 'manifest.json' === $name ) {
				continue;
			}
			if ( 0 !== strpos( $name, 'files/' ) ) {
				continue;
			}

			if ( '' !== $name && '/' === substr( $name, -1 ) ) {
				continue;
			}
			$rel = substr( $name, 6 ); 
			$rel = ltrim( str_replace( '\\', '/', $rel ), '/' );

			foreach ( $skip_prefixes as $pfx ) {
				$pfx = str_replace( '\\', '/', (string) $pfx );
				if ( 0 === strpos( $rel, $pfx ) ) {
					$skipped++;
					continue 2;
				}
			}

			if ( '' === $rel || false !== strpos( $rel, '../' ) || false !== strpos( $rel, '..\\' ) ) {
				$skipped++;
				continue;
			}
			$dest = $root . str_replace( '/', DIRECTORY_SEPARATOR, $rel );
			$dest_dir = dirname( $dest );
			if ( ! is_dir( $dest_dir ) ) {
				wp_mkdir_p( $dest_dir );
			}

			$in = $zip->getStream( $name );
			if ( ! is_resource( $in ) ) {
				$skipped++;
				continue;
			}
			$fs = $this->ab_filesystem();
			$contents = stream_get_contents( $in );
			// ZipArchive::getStream() returns a PHP stream; close it after reading.
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing a ZipArchive stream resource.
			@fclose( $in );

			if ( false === $contents ) {
				$skipped++;
				continue;
			}

			$wrote = false;
			if ( $fs ) {
				$wrote = (bool) $fs->put_contents( $dest, $contents, defined( 'FS_CHMOD_FILE' ) ? FS_CHMOD_FILE : false );
			} else {
				$wrote = (bool) @file_put_contents( $dest, $contents );
			}

			if ( ! $wrote ) {
				$skipped++;
				continue;
			}

			$restored++;
		}
		$zip->close();

		return array( 'ok' => true, 'message' => 'Restore complete.', 'restored' => $restored, 'skipped' => $skipped );
	}

	public function schedule_plan( array $plan ) {
		$plan_id = isset( $plan['id'] ) ? sanitize_key( (string) $plan['id'] ) : '';
		if ( '' === $plan_id ) {
			return;
		}
		$this->unschedule_plan( $plan_id );
		if ( empty( $plan['enabled'] ) ) {
			return;
		}
		$ts = $this->next_run_timestamp( $plan );
		if ( $ts <= time() ) {
			$ts = time() + 300;
		}
		wp_schedule_single_event( $ts, self::CRON_HOOK, array( $plan_id ) );
	}

	public function unschedule_plan( $plan_id ) {
		$plan_id = sanitize_key( (string) $plan_id );
		$next = wp_next_scheduled( self::CRON_HOOK, array( $plan_id ) );
		while ( $next ) {
			wp_unschedule_event( $next, self::CRON_HOOK, array( $plan_id ) );
			$next = wp_next_scheduled( self::CRON_HOOK, array( $plan_id ) );
		}
	}

	public function cron_run_plan( $plan_id ) {
		$plan = $this->get_plan( $plan_id );
		if ( ! is_array( $plan ) || empty( $plan['enabled'] ) ) {
			return;
		}
		$this->run_plan_now( $plan );
		$plan['last_run'] = time();
		$plan['next_run'] = $this->next_run_timestamp( $plan );
		$this->save_plan( $plan );
		$this->schedule_plan( $plan );
	}

	public function next_run_timestamp( array $plan ) {
		$freq = isset( $plan['schedule'] ) ? sanitize_key( (string) $plan['schedule'] ) : 'daily';
		$time = isset( $plan['time'] ) ? (string) $plan['time'] : '02:00';
		$time = preg_match( '/^\d{2}:\d{2}$/', $time ) ? $time : '02:00';
		list( $hh, $mm ) = array_map( 'intval', explode( ':', $time ) );
		$tz = wp_timezone();
		$now = new \DateTimeImmutable( 'now', $tz );
		$base = $now->setTime( $hh, $mm, 0 );
		if ( $base->getTimestamp() <= $now->getTimestamp() ) {
			$base = $base->modify( '+1 day' );
		}

		if ( 'weekly' === $freq ) {
			$dow = isset( $plan['dow'] ) ? (int) $plan['dow'] : 1; // 1=Mon
			$dow = max( 0, min( 6, $dow ) );
			$cur = (int) $base->format( 'w' );
			$add = ( $dow - $cur + 7 ) % 7;
			if ( 0 === $add ) {
				$add = 7;
			}
			$base = $base->modify( '+' . $add . ' day' );
		}

			if ( 'monthly' === $freq ) {
				$dom = isset( $plan['dom'] ) ? (int) $plan['dom'] : 1;
				$dom = max( 1, min( 31, $dom ) );
				$y = (int) $now->format( 'Y' );
				$m = (int) $now->format( 'm' );

				$first = new \DateTimeImmutable( sprintf( '%04d-%02d-01 %02d:%02d:00', $y, $m, $hh, $mm ), $tz );
				$last_day = (int) $first->modify( 'last day of this month' )->format( 'j' );
				$use_dom = min( $dom, $last_day );

				$target = new \DateTimeImmutable( sprintf( '%04d-%02d-%02d %02d:%02d:00', $y, $m, $use_dom, $hh, $mm ), $tz );

				if ( $target->getTimestamp() <= $now->getTimestamp() ) {
					$next_first = $first->modify( 'first day of next month' );
					$ny = (int) $next_first->format( 'Y' );
					$nm = (int) $next_first->format( 'm' );
					$next_last_day = (int) $next_first->modify( 'last day of this month' )->format( 'j' );
					$use_dom = min( $dom, $next_last_day );
					$target = new \DateTimeImmutable( sprintf( '%04d-%02d-%02d %02d:%02d:00', $ny, $nm, $use_dom, $hh, $mm ), $tz );
				}

				return (int) $target->getTimestamp();
			}

			return (int) $base->getTimestamp();
		}

		private function get_plan_state( $plan_id ) {
		$state = get_option( self::PLAN_STATE_PREFIX . sanitize_key( (string) $plan_id ), array() );
		return is_array( $state ) ? $state : array();
	}

	private function set_plan_state( $plan_id, array $state ) {
		update_option( self::PLAN_STATE_PREFIX . sanitize_key( (string) $plan_id ), $state, false );
	}

	private function iter_files_for_paths( array $paths ) {
		$root = rtrim( ABSPATH, '/\\' ) . DIRECTORY_SEPARATOR;
		$backups_dir_rel = ltrim( str_replace( '\\', '/', str_replace( $root, '', $this->backups_dir() ) ), '/' );

		foreach ( $paths as $rel ) {
			$rel = $this->normalize_rel_path( $rel );
			if ( '' === $rel ) {
				continue;
			}

			if ( '' !== $backups_dir_rel && 0 === strpos( $rel, $backups_dir_rel ) ) {
				continue;
			}
			$abs = $root . str_replace( '/', DIRECTORY_SEPARATOR, $rel );
			if ( is_file( $abs ) ) {
				yield $rel => $abs;
				continue;
			}
			if ( ! is_dir( $abs ) ) {
				continue;
			}

			try {
				$it = new \RecursiveIteratorIterator(
					new \RecursiveDirectoryIterator( $abs, \FilesystemIterator::SKIP_DOTS ),
					\RecursiveIteratorIterator::SELF_FIRST
				);
			} catch ( \Exception $e ) {
				continue;
			}

			foreach ( $it as $f ) {
				if ( ! $f->isFile() ) {
					continue;
				}
				$abs_file = $f->getPathname();
				if ( ! @is_readable( $abs_file ) ) {
					continue;
				}

				$rel_file = ltrim( str_replace( '\\', '/', str_replace( $root, '', $abs_file ) ), '/' );
				if ( '' === $rel_file ) {
					continue;
				}
				if ( '' !== $backups_dir_rel && 0 === strpos( $rel_file, $backups_dir_rel ) ) {
					continue;
				}
				yield $rel_file => $abs_file;
			}
		}
	}

	public function normalize_rel_path( $path ) {
		$path = trim( (string) $path );
		$path = str_replace( "\\", '/', $path );
		$path = ltrim( $path, '/' );
		$path = preg_replace( '#/+#', '/', $path );

		if ( '' === $path || false !== strpos( $path, '../' ) || false !== strpos( $path, '..\\' ) ) {
			return '';
		}
		return $path;
	}

	private function human_bytes( $bytes ) {
		$bytes = (int) $bytes;
		$units = array( 'B', 'KB', 'MB', 'GB', 'TB' );
		$i = 0;
		while ( $bytes >= 1024 && $i < count( $units ) - 1 ) {
			$bytes /= 1024;
			$i++;
		}
		return number_format_i18n( $bytes, ( $i ? 2 : 0 ) ) . ' ' . $units[ $i ];
	}
}
