<?php
namespace AegisBackup\Backup;

defined( 'ABSPATH' ) || exit;

class AB_Manifest {
    public function build_manifest( array $data ) {
        $data['generated_at'] = gmdate( 'c' );
        return $data;
    }

    public function get_table_sizes( $prefix ) {
        global $wpdb;
        $prefix = (string) $prefix;

        // Cache table size snapshot per prefix to reduce repeated SHOW TABLES / STATUS calls.
        $cache_group = 'aegisbackup';
        $cache_key   = 'ab_manifest_table_sizes_' . (string) get_current_blog_id() . '_' . md5( $prefix );
        $cached      = wp_cache_get( $cache_key, $cache_group );
        if ( false !== $cached && is_array( $cached ) ) {
            return $cached;
        }

        $like   = $wpdb->esc_like( $prefix ) . '%';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Required for schema introspection in backups.
        $tables = $wpdb->get_col( $wpdb->prepare( 'SHOW TABLES LIKE %s', $like ) );
        $out    = array();

        if ( is_array( $tables ) ) {
            foreach ( $tables as $t ) {
                // Use prepare() to avoid concatenated SQL; SHOW TABLE STATUS requires the raw table name string.
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Required for schema introspection in backups.
                $rows = $wpdb->get_row( $wpdb->prepare( 'SHOW TABLE STATUS LIKE %s', (string) $t ) );
                $out[] = array(
                    'name'       => $t,
                    'rows'       => isset( $rows->Rows ) ? (int) $rows->Rows : 0,
                    'size_bytes' => isset( $rows->Data_length ) ? (int) $rows->Data_length + (int) $rows->Index_length : 0,
                );
            }
        }

        wp_cache_set( $cache_key, $out, $cache_group, 300 );

        return $out;
    }

    public function get_plugins_list() {
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $plugins = get_plugins();
        $out = array();
        foreach ( (array) $plugins as $file => $info ) {
            $out[] = array(
                'file' => $file,
                'name' => isset( $info['Name'] ) ? (string) $info['Name'] : '',
                'version' => isset( $info['Version'] ) ? (string) $info['Version'] : '',
            );
        }
        return $out;
    }

    public function get_themes_list() {
        $themes = wp_get_themes();
        $out = array();
        foreach ( (array) $themes as $slug => $theme ) {
            $out[] = array(
                'slug' => $slug,
                'name' => $theme->get( 'Name' ),
                'version' => $theme->get( 'Version' ),
            );
        }
        return $out;
    }

    public function get_mu_plugins_list() {
        $out = array();
        $mu = wp_get_mu_plugins();
        foreach ( (array) $mu as $file ) {
            $out[] = array( 'file' => $file );
        }
        return $out;
    }


    public function get_theme_info() {
        $theme = wp_get_theme();
        if ( ! $theme ) {
            return array();
        }
        return array(
            'name' => $theme->get( 'Name' ),
            'stylesheet' => $theme->get_stylesheet(),
            'template' => $theme->get_template(),
            'version' => $theme->get( 'Version' ),
        );
    }

    
    public function get_plugins_info() {
        if ( ! function_exists( 'get_plugin_data' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $plugins = array();

        $active = (array) get_option( 'active_plugins', array() );
        foreach ( $active as $plugin_file ) {
            $path = WP_PLUGIN_DIR . '/' . ltrim( (string) $plugin_file, '/\\' );
            if ( is_file( $path ) ) {
                $data = get_plugin_data( $path, false, false );
                $plugins[] = array(
                    'file' => (string) $plugin_file,
                    'name' => isset( $data['Name'] ) ? (string) $data['Name'] : '',
                    'version' => isset( $data['Version'] ) ? (string) $data['Version'] : '',
                    'plugin_uri' => isset( $data['PluginURI'] ) ? (string) $data['PluginURI'] : '',
                    'author' => isset( $data['Author'] ) ? (string) $data['Author'] : '',
                );
            } else {
                $plugins[] = array(
                    'file' => (string) $plugin_file,
                    'name' => '',
                    'version' => '',
                    'missing' => 1,
                );
            }
        }

        // Must-use plugins
        $mu = array();
        if ( function_exists( 'get_mu_plugins' ) ) {
            foreach ( (array) get_mu_plugins() as $file => $headers ) {
                $mu[] = array(
                    'file' => (string) $file,
                    'name' => isset( $headers['Name'] ) ? (string) $headers['Name'] : '',
                    'version' => isset( $headers['Version'] ) ? (string) $headers['Version'] : '',
                );
            }
        }

        return array(
            'active' => $plugins,
            'mu'     => $mu,
        );
    }

    public function get_settings_snapshot() {
        $keys = array(
            'siteurl',
            'home',
            'blogname',
            'blogdescription',
            'admin_email',
            'users_can_register',
            'default_role',
            'timezone_string',
            'gmt_offset',
            'date_format',
            'time_format',
            'start_of_week',
            'permalink_structure',
            'category_base',
            'tag_base',
            'uploads_use_yearmonth_folders',
            'upload_path',
            'upload_url_path',
            'WPLANG',
        );

        $out = array();
        foreach ( $keys as $k ) {
            $out[ $k ] = get_option( $k, null );
        }

        $theme = wp_get_theme();
        if ( $theme ) {
            $stylesheet = $theme->get_stylesheet();
            $out['theme_mods_stylesheet'] = $stylesheet;
            $out['theme_mods'] = get_option( 'theme_mods_' . $stylesheet, array() );
        }

        global $wp_rewrite;
        if ( isset( $wp_rewrite ) ) {
            $out['rewrite_rules_option_present'] = null !== get_option( 'rewrite_rules', null ) ? 1 : 0;
        }

        return $out;
    }

    public function get_wp_snapshot() {
        $upload = wp_upload_dir();
        $snapshot = array(
            'generated_at' => gmdate( 'c' ),
            'site_url' => site_url(),
            'home_url' => home_url(),
            'wp_version' => get_bloginfo( 'version' ),
            'php_version' => phpversion(),
            'mysql_version' => ( isset( $GLOBALS['wpdb'] ) && is_object( $GLOBALS['wpdb'] ) && method_exists( $GLOBALS['wpdb'], 'db_version' ) ) ? (string) $GLOBALS['wpdb']->db_version() : (string) phpversion( 'mysqli' ),
            'multisite' => is_multisite() ? 1 : 0,
            'plugins' => $this->get_plugins_info(),
            'theme' => $this->get_theme_info(),
            'settings' => $this->get_settings_snapshot(),
            'uploads' => array(
                'basedir' => isset( $upload['basedir'] ) ? (string) $upload['basedir'] : '',
                'baseurl' => isset( $upload['baseurl'] ) ? (string) $upload['baseurl'] : '',
                'subdir'  => isset( $upload['subdir'] ) ? (string) $upload['subdir'] : '',
            ),
        );

        $consts = array(
            'DB_NAME','DB_USER','DB_HOST','DB_CHARSET','DB_COLLATE',
            'WP_DEBUG','WP_DEBUG_LOG','WP_DEBUG_DISPLAY',
            'DISALLOW_FILE_EDIT','DISALLOW_FILE_MODS',
            'WP_MEMORY_LIMIT','WP_MAX_MEMORY_LIMIT',
        );
        $c = array();
        foreach ( $consts as $name ) {
            if ( defined( $name ) ) {
                $c[ $name ] = constant( $name );
            }
        }
        $snapshot['wp_config_constants_loaded'] = $c;

        return $snapshot;
    }


public function build_checksums( $root_dir ) {
        $root_dir = trailingslashit( $root_dir );
        $lines = array();
        $rii = new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($root_dir, \FilesystemIterator::SKIP_DOTS));
        foreach ( $rii as $file ) {
            if ( $file->isDir() ) {
                continue;
            }
            $path = $file->getPathname();
            $rel  = ltrim( str_replace( $root_dir, '', $path ), '/\\' );
            $hash = hash_file( 'sha256', $path );
            $lines[] = $hash . '  ' . $rel;
        }
        sort( $lines );
        return implode( "\n", $lines ) . "\n";
    }
}