<?php
namespace AegisBackup\Backup;

use AegisBackup\Restore\AB_DB_Restorer;

defined( 'ABSPATH' ) || exit;

class AB_Table_Backup_Manager {

    private function append_settings_log( $message, $job_id = '' ) {
        $job_id = (string) $job_id;
        $prefix = $job_id ? ('[TableRestore ' . $job_id . '] ') : '[TableRestore] ';
        $line = '[' . gmdate( 'Y-m-d H:i:s' ) . ' UTC] ' . $prefix . (string) $message;

        $logs = get_option( 'aegisbackup_logs', array() );
        if ( ! is_array( $logs ) ) {
            $logs = array();
        }
        $logs[] = $line;
        if ( count( $logs ) > 1500 ) {
            $logs = array_slice( $logs, -1200 );
        }
        update_option( 'aegisbackup_logs', $logs, false );
    }

    protected function sanitize_backup_id( $id ) {
        $id = sanitize_text_field( (string) $id );
        if ( '' !== $id && preg_match( '/^[A-Za-z0-9._-]+$/', $id ) ) {
            return $id;
        }
        return '';
    }


    const BACKUPS_OPTION = 'aegisbackup_table_backups';
    const PLANS_OPTION   = 'aegisbackup_table_backup_plans';
    const CRON_HOOK      = 'aegisbackup_run_table_backup_plan';
    const RESTORE_OPTION_PREFIX = 'aegisbackup_table_restore_';


    /**
     * Get WP_Filesystem instance (initialized).
     *
     * @return \WP_Filesystem_Base|null
     */
    protected function get_filesystem() {
        global $wp_filesystem;
        if ( isset( $wp_filesystem ) && is_object( $wp_filesystem ) ) {
            return $wp_filesystem;
        }

        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        // Attempt direct filesystem init (no credentials prompt in typical plugin runtime).
        WP_Filesystem();

        if ( isset( $wp_filesystem ) && is_object( $wp_filesystem ) ) {
            return $wp_filesystem;
        }

        return null;
    }

    /**
     * Sanitize a MySQL identifier (database / table / column name).
     * This is NOT for values; identifiers must be whitelisted.
     *
     * @param string $id Identifier.
     * @return string
     */
    protected function sanitize_db_identifier( $id ) {
        $id = (string) $id;
        return preg_replace( '/[^A-Za-z0-9_\$]/', '', $id );
    }

    /**
     * Cache helpers.
     */
    protected function cache_get( $key ) {
        return wp_cache_get( (string) $key, 'aegisbackup' );
    }
    protected function cache_set( $key, $value, $ttl = 300 ) {
        wp_cache_set( (string) $key, $value, 'aegisbackup', (int) $ttl );
    }
    public function __construct() {
        add_action( self::CRON_HOOK, array( $this, 'cron_run_plan' ), 10, 1 );
        add_filter( 'cron_schedules', array( $this, 'add_custom_schedules' ) );
    }

    public function add_custom_schedules( $schedules ) {
        if ( ! is_array( $schedules ) ) {
            $schedules = array();
        }
        if ( ! isset( $schedules['aegisbackup_weekly'] ) ) {
            $schedules['aegisbackup_weekly'] = array(
                'interval' => 7 * DAY_IN_SECONDS,
                'display'  => 'Once Weekly (AegisBackup)',
            );
        }
        if ( ! isset( $schedules['aegisbackup_monthly'] ) ) {
            $schedules['aegisbackup_monthly'] = array(
                'interval' => 30 * DAY_IN_SECONDS,
                'display'  => 'Once Monthly (AegisBackup)',
            );
        }
        return $schedules;
    }

    public function get_backups() {
        $b = get_option( self::BACKUPS_OPTION, array() );
        return is_array( $b ) ? $b : array();
    }

    public function save_backups( array $backups ) {
        update_option( self::BACKUPS_OPTION, $backups, false );
    }

    public function delete_backup( $backup_id ) {
        $backup_id = $this->sanitize_backup_id( $backup_id );
        $all = $this->get_backups();
        if ( isset( $all[ $backup_id ] ) ) {
        if ( '' === $backup_id ) { return; }
            $zip = isset( $all[ $backup_id ]['zip'] ) ? (string) $all[ $backup_id ]['zip'] : '';
            $dir = isset( $all[ $backup_id ]['dir'] ) ? (string) $all[ $backup_id ]['dir'] : '';
            unset( $all[ $backup_id ] );
            $this->save_backups( $all );

            if ( $zip && is_file( $zip ) ) {
                wp_delete_file( $zip );
            }
            if ( $dir && is_dir( $dir ) ) {
                $this->rrmdir( $dir );
            }
            return true;
        }
        return false;
    }

    public function list_databases() {
        global $wpdb;

        $dbs = array();
		$current = (string) $wpdb->dbname;

		if ( '' === $current && defined( 'DB_NAME' ) ) {
			$current = (string) DB_NAME;
		}
        if ( '' !== $current ) {
            $dbs[] = $current;
        }

        $cache_key = 'ab_tb_databases';
        $rows = $this->cache_get( $cache_key );
        if ( false === $rows ) {
            $rows = $wpdb->get_col( 'SHOW DATABASES' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
            $this->cache_set( $cache_key, $rows, 300 );
        }
        if ( is_array( $rows ) ) {
            foreach ( $rows as $d ) {
                $d = $this->sanitize_db_identifier( (string) $d );
                if ( '' === $d ) {
                    continue;
                }
                $dbs[] = $d;
            }
        }

        $dbs = array_values( array_unique( array_filter( $dbs ) ) );
        if ( empty( $dbs ) && '' !== $current ) {
            $dbs = array( $current );
        }
        return $dbs;
    }

    public function list_tables( $db_name ) {
        global $wpdb;

        $db_name = $this->sanitize_db_identifier( (string) $db_name );
        if ( '' === $db_name ) {
            $db_name = (string) $wpdb->dbname;
        }

        $tables = array();

        $cache_key = 'ab_tb_tables_' . $db_name;
        $rows = $this->cache_get( $cache_key );
        if ( false === $rows ) {
            // Use information_schema with a prepared parameter (wpdb::prepare does not support identifier placeholders).
            $rows = $wpdb->get_col( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    'SELECT TABLE_NAME FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = %s',
                    $db_name
                )
            );
            $this->cache_set( $cache_key, $rows, 300 );
        }
        if ( is_array( $rows ) ) {
            foreach ( $rows as $t ) {
                $t = sanitize_text_field( (string) $t );
                if ( '' !== $t ) {
                    $tables[] = $t;
                }
            }
        }

        if ( empty( $tables ) ) {
            $rows2 = $this->cache_get( 'ab_tb_tables_default' );
            if ( false === $rows2 ) {
                $rows2 = $wpdb->get_col( 'SHOW TABLES' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                $this->cache_set( 'ab_tb_tables_default', $rows2, 300 );
            }
            if ( is_array( $rows2 ) ) {
                foreach ( $rows2 as $t ) {
                    $t = sanitize_text_field( (string) $t );
                    if ( '' !== $t ) {
                        $tables[] = $t;
                    }
                }
            }
        }

        sort( $tables );
        return array_values( array_unique( $tables ) );
    }

    public function list_columns( $db_name, $table ) {
        global $wpdb;

        $db_name = $this->sanitize_db_identifier( (string) $db_name );
        if ( '' === $db_name ) {
            $db_name = (string) $wpdb->dbname;
        }
        $table = $this->sanitize_table_identifier( (string) $table );
        if ( '' === $table ) {
            return array();
        }

        $cache_key = 'ab_tb_cols_' . $db_name . '_' . $table;
        $rows = $this->cache_get( $cache_key );
        if ( false === $rows ) {
            // Use information_schema with prepared parameters (avoid identifier placeholders).
            $rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    'SELECT COLUMN_NAME AS Field, COLUMN_TYPE AS Type, IS_NULLABLE AS `Null`, COLUMN_DEFAULT AS `Default`, COLUMN_KEY AS `Key` FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = %s AND TABLE_NAME = %s ORDER BY ORDINAL_POSITION ASC',
                    $db_name,
                    $table
                ),
                ARRAY_A
            );
            $this->cache_set( $cache_key, $rows, 300 );
        }
        if ( ! is_array( $rows ) ) {
            $rows = array();
        }

        $out = array();
        foreach ( $rows as $r ) {
            $out[] = array(
                'name' => isset( $r['Field'] ) ? (string) $r['Field'] : '',
                'type' => isset( $r['Type'] ) ? (string) $r['Type'] : '',
                'nullable' => isset( $r['Null'] ) ? (string) $r['Null'] : '',
                'default' => $r['Default'] ?? '',
                'key' => isset( $r['Key'] ) ? (string) $r['Key'] : '',
            );
        }
        return $out;
    }

    public function start_restore_job( $backup_id ) {
        $backup_id = $this->sanitize_backup_id( $backup_id );
        $all = $this->get_backups();
        if ( empty( $all[ $backup_id ] ) ) {
            if ( '' === $backup_id ) {
                return array( 'ok' => false, 'message' => 'Invalid backup id.' );
            }
            return array( 'ok' => false, 'message' => 'Backup not found.' );
        }

        $zip = isset( $all[ $backup_id ]['zip'] ) ? (string) $all[ $backup_id ]['zip'] : '';
        if ( '' === $zip || ! is_file( $zip ) ) {
            return array( 'ok' => false, 'message' => 'Zip not found.' );
        }

        $job_id = 'tbr_' . wp_generate_password( 10, false, false );
        $state = array(
            'backup_id' => $backup_id,
            'job_id' => $job_id,
            'created' => time(),
            'zip' => $zip,
            'dir' => '',
            'import_state' => array(),
            'done' => 0,
            'progress' => 0,
            'stage' => 'extract',
            'last_log' => 'Restore job created. Starting restore…',
        );

        update_option( self::RESTORE_OPTION_PREFIX . $job_id, $state, false );
        $this->append_settings_log( $state['last_log'], $job_id );
        return array( 'ok' => true, 'message' => 'Restore started.', 'job_id' => $job_id );
    }

    public function process_restore_job( $job_id ) {
        $job_id = $this->sanitize_backup_id( $job_id );
        $state = get_option( self::RESTORE_OPTION_PREFIX . $job_id, array() );
        if ( empty( $state['job_id'] ) ) {
            return array( 'ok' => false, 'done' => true, 'progress' => 100, 'log' => 'Restore job not found.' );
        }

        $dir = isset( $state['dir'] ) ? (string) $state['dir'] : '';
        if ( '' === $dir || ! is_dir( $dir ) ) {
            $zip = isset( $state['zip'] ) ? (string) $state['zip'] : '';
            if ( '' === $zip || ! is_file( $zip ) ) {
                $this->append_settings_log( 'Restore zip missing.', $job_id );
                delete_option( self::RESTORE_OPTION_PREFIX . $job_id );
                return array( 'ok' => false, 'done' => true, 'progress' => 100, 'log' => 'Restore zip missing.' );
            }

            $ex = $this->extract_backup_zip( $zip );
            if ( empty( $ex['ok'] ) || empty( $ex['dir'] ) ) {
                $state['done'] = 1;
                $state['progress'] = 100;
                $state['stage'] = 'failed';
                $state['last_log'] = isset( $ex['message'] ) ? (string) $ex['message'] : 'Failed to extract zip.';
                update_option( self::RESTORE_OPTION_PREFIX . $job_id, $state, false );
                $this->append_settings_log( $state['last_log'], $job_id );
                return array( 'ok' => false, 'done' => true, 'progress' => 100, 'log' => $state['last_log'] );
            }

            $state['dir'] = (string) $ex['dir'];
            $state['stage'] = 'import';
            $state['progress'] = 5;
            $state['last_log'] = 'Extracted restore package. Starting import…';
            update_option( self::RESTORE_OPTION_PREFIX . $job_id, $state, false );
            $this->append_settings_log( $state['last_log'], $job_id );

            return array( 'ok' => true, 'done' => false, 'progress' => 5, 'log' => $state['last_log'] );
        }

        $restorer = new AB_DB_Restorer();
        $import_state = isset( $state['import_state'] ) && is_array( $state['import_state'] ) ? $state['import_state'] : array();
        $step = $restorer->import_db_step( $dir, $import_state, 80, 1.2 );

        $state['import_state'] = $import_state;
        $state['stage'] = 'import';
        $state['progress'] = isset( $step['progress'] ) ? max( 5, (int) $step['progress'] ) : 5;
        $state['last_log'] = isset( $step['log'] ) ? (string) $step['log'] : 'Restoring…';
        update_option( self::RESTORE_OPTION_PREFIX . $job_id, $state, false );
        $this->append_settings_log( $state['last_log'], $job_id );

        if ( ! empty( $step['done'] ) ) {
            $this->rrmdir( $dir );
            delete_option( self::RESTORE_OPTION_PREFIX . $job_id );
            $this->append_settings_log( 'Restore complete.', $job_id );
            return array( 'ok' => true, 'done' => true, 'progress' => 100, 'log' => 'Restore complete.' );
        }

        return array(
            'ok' => true,
            'done' => false,
            'progress' => isset( $step['progress'] ) ? (int) $step['progress'] : 1,
            'log' => isset( $step['log'] ) ? (string) $step['log'] : 'Restoring…',
        );
    }

    public function get_restore_state( $job_id ) {
        $job_id = $this->sanitize_backup_id( $job_id );
        $state = get_option( self::RESTORE_OPTION_PREFIX . $job_id, array() );
        return is_array( $state ) ? $state : array();
    }

    public function get_backup( $backup_id ) {
        $backup_id = $this->sanitize_backup_id( $backup_id );
        $all = $this->get_backups();
        return isset( $all[ $backup_id ] ) && is_array( $all[ $backup_id ] ) ? $all[ $backup_id ] : null;
    }

    public function human_bytes( $bytes ) {
        $bytes = (int) $bytes;
        if ( $bytes <= 0 ) {
            return '0 B';
        }
        $units = array( 'B', 'KB', 'MB', 'GB', 'TB' );
        $i = (int) floor( log( $bytes, 1024 ) );
        $i = max( 0, min( $i, count( $units ) - 1 ) );
        $val = $bytes / pow( 1024, $i );
        return round( $val, 2 ) . ' ' . $units[ $i ];
    }

    public function get_download_url( $backup_id ) {
        return wp_nonce_url(
            add_query_arg(
                array(
                    'action' => 'aegisbackup_tablebacks_download',
                    'id' => (string) $this->sanitize_backup_id( $backup_id ), // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- sanitize_backup_id() validates ID.
                ),
                admin_url( 'admin-post.php' )
            ),
            'aegisbackup_tablebacks_download'
        );
    }

    public function list_backups_sorted() {
        $all = $this->get_backups();
        $items = array();
        foreach ( $all as $id => $row ) {
            if ( ! is_array( $row ) ) {
                continue;
            }
            $row['id'] = (string) ( $row['id'] ?? $id );
            $items[] = $row;
        }
        usort(
            $items,
            static function( $a, $b ) {
                return (int) ( $b['created'] ?? 0 ) <=> (int) ( $a['created'] ?? 0 );
            }
        );
        return $items;
    }

    public function list_plans_sorted() {
        $plans = $this->get_plans();
        $items = array();
        foreach ( $plans as $id => $p ) {
            if ( ! is_array( $p ) ) {
                continue;
            }
            $p['id'] = (string) ( $p['id'] ?? $id );
            $items[] = $p;
        }
        usort(
            $items,
            static function( $a, $b ) {
                return strcmp( (string) ( $a['name'] ?? '' ), (string) ( $b['name'] ?? '' ) );
            }
        );
        return $items;
    }

    public function run_plan_now( $plan_id ) {
    $plan = $this->get_plan( $plan_id );
    if ( empty( $plan ) ) {
        return array( 'ok' => false, 'message' => 'Schedule not found.' );
    }
    if ( empty( $plan['enabled'] ) ) {
        return array( 'ok' => false, 'message' => 'Schedule is disabled.' );
    }

    $db          = isset( $plan['db'] ) ? (string) $plan['db'] : '';
    $tables      = isset( $plan['tables'] ) && is_array( $plan['tables'] ) ? $plan['tables'] : array();
    $columns_map = isset( $plan['columns_map'] ) && is_array( $plan['columns_map'] ) ? $plan['columns_map'] : array();
    $plan_name   = isset( $plan['name'] ) ? (string) $plan['name'] : 'Table Backup';

    try {
        $res = $this->create_backup( $plan_name, $db, $tables, $columns_map );
    } catch ( \Throwable $e ) {
        return array( 'ok' => false, 'message' => 'Run Now failed: ' . $e->getMessage() );
    }

    if ( is_array( $res ) ) {
        if ( isset( $res['success'] ) && ! isset( $res['ok'] ) ) {
            $res['ok'] = (bool) $res['success'];
        }
        if ( ! empty( $res['ok'] ) ) {
            $plans = $this->get_plans();
            if ( isset( $plans[ $plan_id ] ) && is_array( $plans[ $plan_id ] ) ) {
                $plans[ $plan_id ]['last_run'] = (int) current_time( 'timestamp' );
                update_option( self::PLANS_OPTION, $plans, false );
            }
        }

        return array(
            'ok'      => ! empty( $res['ok'] ),
            'message' => isset( $res['message'] ) && '' !== (string) $res['message']
                ? (string) $res['message']
                : ( ! empty( $res['ok'] ) ? 'Schedule started.' : 'Failed to run schedule.' ),
        );
    }

    if ( is_object( $res ) && $res instanceof \WP_Error ) {
        return array( 'ok' => false, 'message' => $res->get_error_message() );
    }

    return array( 'ok' => false, 'message' => 'Failed to run schedule.' );
}

    public function send_download( $backup_id ) {
        $backup = $this->get_backup( $backup_id );
        if ( empty( $backup ) ) {
            return false;
        }
        $path = isset( $backup['zip'] ) ? (string) $backup['zip'] : '';
        if ( '' === $path || ! is_file( $path ) ) {
            return false;
        }

        nocache_headers();
        header( 'Content-Type: application/zip' );
        header( 'Content-Disposition: attachment; filename="' . basename( $path ) . '"' );
        header( 'Content-Length: ' . (string) filesize( $path ) );
        $fs = $this->get_filesystem();
        if ( $fs && method_exists( $fs, 'get_contents' ) ) {
            $data = $fs->get_contents( $path );
        } else {
            $data = @file_get_contents( $path );
        }
        if ( false !== $data ) {
            echo $data; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Binary download.
        }
        exit;
    }

    public function sanitize_table_identifier( $table ) {
        $table = (string) $table;
        return preg_replace( '/[^A-Za-z0-9_\$]/', '', $table );
    }

    public function sanitize_column_identifier( $col ) {
        $col = (string) $col;
        return preg_replace( '/[^A-Za-z0-9_\$]/', '', $col );
    }

    protected function filter_existing_columns( $db_name, $table, array $cols ) {
        $cols = array_values( array_filter( array_map( 'sanitize_text_field', $cols ) ) );
        if ( empty( $cols ) ) {
            return array();
        }
        $meta = $this->list_columns( $db_name, $table );
        if ( empty( $meta ) ) {
            return array();
        }
        $existing = array();
        foreach ( $meta as $r ) {
            if ( ! empty( $r['name'] ) ) {
                $existing[] = (string) $r['name'];
            }
        }
        $existing = array_flip( $existing );
        $out = array();
        foreach ( $cols as $c ) {
            if ( isset( $existing[ $c ] ) ) {
                $out[] = $c;
            }
        }
        return $out;
    }

    protected function normalize_columns_map( $db_name, array $tables, array $columns_map ) {
        $out = array();
        foreach ( $tables as $t ) {
            if ( isset( $columns_map[ $t ] ) ) {
                $out[ $t ] = $this->filter_existing_columns( $db_name, $t, (array) $columns_map[ $t ] );
            }
        }
        return $out;
    }

    protected function safe_columns_map_for_create( $db_name, array $tables, array $columns_map ) {
        $db_name = $this->sanitize_db_identifier( (string) $db_name );
        if ( '' === $db_name ) {
            global $wpdb;
            $db_name = (string) $wpdb->dbname;
        }
        return $this->normalize_columns_map( $db_name, $tables, $columns_map );
    }

    public function create_backup( $name, $db_name, array $tables, array $columns_map = array() ) {
        global $wpdb;

        $name = sanitize_text_field( (string) $name );
        if ( '' === $name ) {
            $name = 'Table Backup';
        }

        $db_name = $this->sanitize_db_identifier( (string) $db_name );
        if ( '' === $db_name ) {
            $db_name = (string) $wpdb->dbname;
        }

        $tables = array_values( array_filter( array_map( 'sanitize_text_field', $tables ) ) );
        if ( empty( $tables ) ) {
            return array( 'ok' => false, 'message' => 'No tables selected.' );
        }

        $upload = wp_upload_dir();
        $base = trailingslashit( $upload['basedir'] ) . 'aegisbackup/table-backups';
        wp_mkdir_p( $base );

        $backup_id = 'tb_' . wp_generate_password( 10, false, false );
        $ts = (int) current_time( 'timestamp' );
        $slug = 'table-backup-' . gmdate( 'Ymd_His', $ts ) . '-' . substr( $backup_id, -5 );
        $dir = trailingslashit( $base ) . $slug;
        wp_mkdir_p( $dir );

        $zip_path = trailingslashit( $dir ) . 'table-backup.zip';
        $work_dir = trailingslashit( $dir ) . 'work';
        wp_mkdir_p( $work_dir );
        $db_dir = trailingslashit( $work_dir ) . 'db/tables';
        wp_mkdir_p( $db_dir );

        $orig_db = (string) $wpdb->dbname;
        if ( $db_name !== $orig_db && method_exists( $wpdb, 'select' ) ) {
            @$wpdb->select( $db_name );
        }

        $exported = array();
        foreach ( $tables as $table ) {
            $file_safe = preg_replace( '/[^a-zA-Z0-9_\-]/', '_', (string) $table );
            $sql_path = trailingslashit( $db_dir ) . $file_safe . '.sql';
            $ok = $this->export_table_sql( $table, $sql_path, isset( $columns_map[ $table ] ) ? (array) $columns_map[ $table ] : array() );
            if ( $ok ) {
                $exported[] = array( 'table' => $table, 'file' => 'db/tables/' . $file_safe . '.sql' );
            }
        }

        if ( $orig_db !== (string) $wpdb->dbname && method_exists( $wpdb, 'select' ) ) {
            @$wpdb->select( $orig_db );
        }

        if ( empty( $exported ) ) {
            $this->rrmdir( $dir );
            return array( 'ok' => false, 'message' => 'No tables could be exported.' );
        }

        $meta = array(
            'id' => $backup_id,
            'name' => $name,
            'created' => $ts,
            'db' => $db_name,
            'tables' => $exported,
            'columns_map' => $columns_map,
        );
        @file_put_contents( trailingslashit( $work_dir ) . 'meta.json', wp_json_encode( $meta, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES ) );

        $zip = new \ZipArchive();
        $zok = $zip->open( $zip_path, \ZipArchive::CREATE | \ZipArchive::OVERWRITE );
        if ( true !== $zok ) {
            $this->rrmdir( $dir );
            return array( 'ok' => false, 'message' => 'Failed to create zip.' );
        }

        $zip->addFile( trailingslashit( $work_dir ) . 'meta.json', 'meta.json' );
        foreach ( $exported as $e ) {
            $src = trailingslashit( $work_dir ) . $e['file'];
            if ( is_file( $src ) ) {
                $zip->addFile( $src, $e['file'] );
            }
        }
        $zip->close();
        $this->rrmdir( $work_dir );
        $all = $this->get_backups();
        $all[ $backup_id ] = array(
            'id' => $backup_id,
            'name' => $name,
            'created' => $ts,
            'db' => $db_name,
            'tables' => array_map( static function( $r ) { return (string) $r['table']; }, $exported ),
            'zip' => $zip_path,
            'dir' => $dir,
            'size' => is_file( $zip_path ) ? (int) filesize( $zip_path ) : 0,
        );
        $this->save_backups( $all );

        return array( 'ok' => true, 'message' => 'Table backup created.', 'backup_id' => $backup_id );
    }

    protected function export_table_sql( $table, $path, array $columns = array() ) {
        global $wpdb;

        $table = (string) $table;
        if ( '' === $table ) {
            return false;
        }

        $safe_table = $this->sanitize_table_identifier( $table );
        if ( '' === $safe_table ) {
            return false;
        }

        $cols = array_values( array_filter( array_map( 'sanitize_text_field', $columns ) ) );
        $col_sql = '*';
        if ( ! empty( $cols ) ) {
            $safe_cols = array();
            foreach ( $cols as $c ) {
                $c = str_replace( '`', '``', $c );
                $safe_cols[] = '`' . $c . '`';
            }
            $col_sql = implode( ',', $safe_cols );
        }

        $cache_key = 'ab_tb_create_' . $table;
        $create = $this->cache_get( $cache_key );
        if ( false === $create ) {
            $table_sql = '`' . str_replace( '`', '``', $table ) . '`';
            $safe_table = $this->sanitize_table_identifier( $table );
            if ( '' === $safe_table ) {
                return false;
            }
            $create = $wpdb->get_row( $wpdb->prepare( 'SHOW CREATE TABLE %i', $safe_table ), ARRAY_N ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange
            $this->cache_set( $cache_key, $create, 300 );
        }
        if ( empty( $create ) || empty( $create[1] ) ) {
            return false;
        }

        $hdr  = "-- AegisBackup table backup: {$table}\n";
        $hdr .= '-- Generated: ' . gmdate( 'c' ) . "\n\n";
        $hdr .= 'SET FOREIGN_KEY_CHECKS=0;' . "\n\n";
        $hdr .= 'DROP TABLE IF EXISTS `' . str_replace( '`', '``', $table ) . '`;' . "\n";
        $hdr .= $create[1] . ";\n\n";
        @file_put_contents( $path, $hdr );

        $limit = 250;
        $offset = 0;
        while ( true ) {
            if ( '*' === $col_sql ) {
                $rows = $wpdb->get_results( $wpdb->prepare( 'SELECT * FROM %i LIMIT %d OFFSET %d', $safe_table, $limit, $offset ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
            } else {
                $select_cols = (array) $cols;
                $select_cols = array_values( array_filter( array_map( array( $this, 'sanitize_column_identifier' ), $select_cols ) ) );
                if ( empty( $select_cols ) ) {
                    $rows = $wpdb->get_results( $wpdb->prepare( 'SELECT * FROM %i LIMIT %d OFFSET %d', $safe_table, $limit, $offset ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                } else {
                    // Plugin Checker flags dynamic column lists inside prepared SQL.
                    // Fetch all columns, then filter in PHP when writing the dump so output remains column-scoped.
                    $selected_cols_map = array_flip( $select_cols );
                    $rows = $wpdb->get_results( $wpdb->prepare( 'SELECT * FROM %i LIMIT %d OFFSET %d', $safe_table, $limit, $offset ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                }
            }

            if ( empty( $rows ) ) {
                break;
            }
            $sql = '';
            foreach ( (array) $rows as $row ) {
                $cols_out = array();
                $vals_out = array();
                foreach ( (array) $row as $k => $v ) {
                	if ( isset( $selected_cols_map ) && ! isset( $selected_cols_map[ $k ] ) ) {
                		continue;
                	}
                	$cols_out[] = '`' . str_replace( '`', '``', (string) $k ) . '`';
                    $vals_out[] = $this->sql_escape_value( $v );
                }
                if ( empty( $cols_out ) ) {
                    continue;
                }
                $sql .= 'INSERT INTO `' . str_replace( '`', '``', $table ) . '` (' . implode( ',', $cols_out ) . ') VALUES (' . implode( ',', $vals_out ) . ");\n";
            }
            @file_put_contents( $path, $sql, FILE_APPEND );
            $offset += count( (array) $rows );
            if ( count( (array) $rows ) < $limit ) {
                break;
            }
        }

        @file_put_contents( $path, "\nSET FOREIGN_KEY_CHECKS=1;\n", FILE_APPEND );
        return is_file( $path );
    }

    protected function sql_escape_value( $v ) {
        global $wpdb;
        if ( null === $v ) {
            return 'NULL';
        }
        if ( is_bool( $v ) ) {
            return $v ? '1' : '0';
        }
        if ( is_int( $v ) || is_float( $v ) ) {
            return (string) $v;
        }
        $s = (string) $v;
        $s = $wpdb->_real_escape( $s );
        return "'" . $s . "'";
    }

    public function extract_backup_zip( $zip_path ) {
        $zip_path = (string) $zip_path;
        if ( ! is_file( $zip_path ) ) {
            return array( 'ok' => false, 'message' => 'Zip not found.' );
        }

        if ( ! class_exists( '\\ZipArchive' ) ) {
            return array( 'ok' => false, 'message' => 'ZipArchive is not available on this host (PHP Zip extension missing). Please enable it to restore table backups.' );
        }

        $upload = wp_upload_dir();
        $tmp = trailingslashit( $upload['basedir'] ) . 'aegisbackup/tmp-restore-' . wp_generate_password( 8, false, false );
        wp_mkdir_p( $tmp );

        $zip = new \ZipArchive();
        $ok = $zip->open( $zip_path );
        if ( true !== $ok ) {
            $this->rrmdir( $tmp );
            return array( 'ok' => false, 'message' => 'Failed to open zip.' );
        }
        $zip->extractTo( $tmp );
        $zip->close();

        return array( 'ok' => true, 'dir' => trailingslashit( $tmp ) );
    }

    public function get_plans() {
        $plans = get_option( self::PLANS_OPTION, array() );
        return is_array( $plans ) ? $plans : array();
    }

    public function get_plan( $plan_id ) {
        $plans = $this->get_plans();
        return isset( $plans[ $plan_id ] ) && is_array( $plans[ $plan_id ] ) ? $plans[ $plan_id ] : null;
    }

    public function save_plan( array $plan ) {
        $plans = $this->get_plans();
        if ( empty( $plan['id'] ) ) {
            $plan['id'] = 'tbp_' . wp_generate_password( 10, false, false );
        }
        $id = $this->sanitize_backup_id( $plan['id'] );

        $defaults = array(
            'id' => $id,
            'name' => 'Table Backup Plan',
            'enabled' => 1,
            'frequency' => 'daily',
            'time' => '02:00',
            'weekdays' => array(),
            'monthday' => 0,
            'db' => '',
            'tables' => array(),
            'columns_map' => array(),
        );

        $merged = array_merge( $defaults, $plan );
        $merged['tables'] = array_values( array_filter( array_map( 'sanitize_text_field', (array) $merged['tables'] ) ) );
        $cm_in = isset( $merged['columns_map'] ) && is_array( $merged['columns_map'] ) ? $merged['columns_map'] : array();
        $cm = array();
        foreach ( $cm_in as $t => $cols ) {
            $t = sanitize_text_field( (string) $t );
            if ( '' === $t ) {
                continue;
            }
            $cols = is_array( $cols ) ? $cols : array();
            $clean_cols = array();
            foreach ( $cols as $c ) {
                $c = sanitize_text_field( (string) $c );
                if ( '' !== $c ) {
                    $clean_cols[] = $c;
                }
            }
            if ( ! empty( $clean_cols ) ) {
                $cm[ $t ] = array_values( array_unique( $clean_cols ) );
            }
        }
        $merged['columns_map'] = $cm;
        $merged['db'] = sanitize_text_field( (string) $merged['db'] );
        $merged['name'] = sanitize_text_field( (string) $merged['name'] );
        $merged['frequency'] = in_array( (string) $merged['frequency'], array( 'daily', 'weekly', 'monthly' ), true ) ? (string) $merged['frequency'] : 'daily';
        $merged['time'] = preg_match( '/^\d{2}:\d{2}$/', (string) $merged['time'] ) ? (string) $merged['time'] : '02:00';
        $merged['enabled'] = ! empty( $merged['enabled'] ) ? 1 : 0;
        $w = isset( $merged['weekdays'] ) ? (array) $merged['weekdays'] : array();
        $allowed_w = array( 'mon', 'tue', 'wed', 'thu', 'fri', 'sat', 'sun' );
        $clean_w = array();
        foreach ( $w as $d ) {
            $d = $this->sanitize_backup_id( $d );
            if ( in_array( $d, $allowed_w, true ) ) {
                $clean_w[] = $d;
            }
        }
        $merged['weekdays'] = array_values( array_unique( $clean_w ) );

        $md = isset( $merged['monthday'] ) ? absint( $merged['monthday'] ) : 0;
        if ( $md < 1 || $md > 31 ) {
            $md = 0;
        }
        $merged['monthday'] = $md;

        $plans[ $id ] = $merged;
        update_option( self::PLANS_OPTION, $plans, false );

        $this->schedule_plan( $id );
        return $merged;
    }

    public function delete_plan( $plan_id ) {
        $plan_id = $this->sanitize_backup_id( $plan_id );
        $plans = $this->get_plans();
        if ( isset( $plans[ $plan_id ] ) ) {
            unset( $plans[ $plan_id ] );
            update_option( self::PLANS_OPTION, $plans, false );
        }
        $this->unschedule_plan( $plan_id );
        return true;
    }

    public function schedule_plan( $plan_id ) {
        $plan = $this->get_plan( $plan_id );
        if ( empty( $plan ) ) {
            return;
        }
        $this->unschedule_plan( $plan_id );
        if ( empty( $plan['enabled'] ) ) {
            return;
        }

        $frequency = (string) ( $plan['frequency'] ?? 'daily' );
        $time = (string) ( $plan['time'] ?? '02:00' );

        $ts = $this->next_run_timestamp( $frequency, $time );
        if ( ! $ts ) {
            return;
        }
        wp_schedule_event( $ts, 'daily', self::CRON_HOOK, array( $plan_id ) );
    }

    public function unschedule_plan( $plan_id ) {
        $plan_id = $this->sanitize_backup_id( $plan_id );
        $ts = wp_next_scheduled( self::CRON_HOOK, array( $plan_id ) );
        while ( $ts ) {
            wp_unschedule_event( $ts, self::CRON_HOOK, array( $plan_id ) );
            $ts = wp_next_scheduled( self::CRON_HOOK, array( $plan_id ) );
        }
    }

    protected function next_run_timestamp( $frequency, $time ) {
        $frequency = (string) $frequency;
        $time = (string) $time;
        if ( ! preg_match( '/^(\d{2}):(\d{2})$/', $time, $m ) ) {
            return 0;
        }
        $h = (int) $m[1];
        $min = (int) $m[2];

        $now = (int) current_time( 'timestamp' );
        $today = (int) strtotime( 'today', $now );
        $ts = $today + ( $h * HOUR_IN_SECONDS ) + ( $min * MINUTE_IN_SECONDS );
        if ( $ts <= $now ) {
            $ts += DAY_IN_SECONDS;
        }
        return $ts;
    }

    public function cron_run_plan( $plan_id ) {
        $plan = $this->get_plan( $plan_id );
        if ( empty( $plan ) || empty( $plan['enabled'] ) ) {
            return;
        }

        $freq = (string) ( $plan['frequency'] ?? 'daily' );
        $now_ts = (int) current_time( 'timestamp' );
        if ( 'weekly' === $freq ) {
            $sel = isset( $plan['weekdays'] ) ? (array) $plan['weekdays'] : array();
            if ( ! empty( $sel ) ) {
                $map = array( 'sun','mon','tue','wed','thu','fri','sat' );
                $dow = $map[ (int) gmdate( 'w', $now_ts ) ] ?? '';
                if ( '' === $dow || ! in_array( $dow, $sel, true ) ) {
                    return;
                }
            }
        } elseif ( 'monthly' === $freq ) {
            $md = isset( $plan['monthday'] ) ? absint( $plan['monthday'] ) : 0;
            if ( $md >= 1 ) {
                $today_d = (int) gmdate( 'j', $now_ts );
                $days_in_month = (int) gmdate( 't', $now_ts );
                if ( $md > $days_in_month ) {
                    $md = $days_in_month; 
                }
                if ( $today_d !== $md ) {
                    return;
                }
            }
        }
        $name = ! empty( $plan['name'] ) ? (string) $plan['name'] : 'Scheduled Table Backup';
        $db = ! empty( $plan['db'] ) ? (string) $plan['db'] : '';
        $tables = isset( $plan['tables'] ) ? (array) $plan['tables'] : array();
        $cols_map = isset( $plan['columns_map'] ) && is_array( $plan['columns_map'] ) ? (array) $plan['columns_map'] : array();
        $this->create_backup( $name, $db, $tables, $cols_map );

        if ( 'monthly' === (string) ( $plan['frequency'] ?? '' ) ) {
            $this->schedule_plan( (string) $plan_id );
        }
    }

    protected function rrmdir( $dir ) {
        $dir = (string) $dir;
        if ( '' === $dir ) {
            return;
        }

        $fs = $this->get_filesystem();
        if ( $fs ) {
            // Use WP_Filesystem recursive delete.
            $fs->rmdir( $dir, true );
            return;
        }

        // Fallback: best-effort recursive delete using WordPress helpers (avoid direct rmdir()).
        if ( ! is_dir( $dir ) ) {
            return;
        }
        $items = scandir( $dir );
        if ( ! is_array( $items ) ) {
            return;
        }
        foreach ( $items as $it ) {
            if ( '.' === $it || '..' === $it ) {
                continue;
            }
            $p = trailingslashit( $dir ) . $it;
            if ( is_dir( $p ) ) {
                $this->rrmdir( $p );
            } else {
                wp_delete_file( $p );
            }
        }

        // Final directory removal via WP_Filesystem if possible.
        $fs2 = $this->get_filesystem();
        if ( $fs2 ) {
            $fs2->rmdir( $dir, false );
        }
    }
}