<?php
namespace AegisBackup\Libs;

defined( 'ABSPATH' ) || exit;

class AB_DB_Prefix_Migrator {


    /**
     * Validate and return a safe table identifier (without backticks).
     * WordPress cannot prepare identifiers, so we strictly validate.
     *
     * @param string $name
     * @return string
     */
    private function sanitize_table_identifier( $name ) {
        $name = (string) $name;
        if ( '' === $name ) {
            return '';
        }
        // Only allow typical wpdb table identifier characters.
        if ( ! preg_match( '/^[A-Za-z0-9_]+$/', $name ) ) {
            return '';
        }
        return $name;
    }


    /**
     * Sanitize a DB prefix once (identifier-safe).
     *
     * @param string $prefix
     * @return string
     */
    private function sanitize_db_prefix( $prefix ) {
        $prefix = (string) $prefix;
        if ( '' === $prefix ) {
            return '';
        }
        // Prefixes should only contain identifier-safe characters.
        if ( ! preg_match( '/^[A-Za-z0-9_]+$/', $prefix ) ) {
            return '';
        }
        return $prefix;
    }

    /**
     * Escape a SQL identifier (table/field) with a strict allowlist.
     *
     * @param string $identifier
     * @return string Escaped identifier wrapped in backticks, or empty string on failure.
     */
    private function escape_sql_identifier( $identifier ) {
        $identifier = (string) $identifier;
        if ( '' === $identifier ) {
            return '';
        }
        if ( ! preg_match( '/^[A-Za-z0-9_]+$/', $identifier ) ) {
            return '';
        }
        // Backtick-wrap (double backticks are unnecessary given allowlist, but kept for defense-in-depth).
        return '`' . str_replace( '`', '``', $identifier ) . '`';
    }

    /**
     * Replace a {{table}} token with an escaped identifier.
     *
     * @param string $sql
     * @param string $table_identifier Identifier without backticks.
     * @return string
     */
    private function sql_with_table( $sql, $table_identifier ) {
        $escaped = $this->escape_sql_identifier( $table_identifier );
        if ( '' === $escaped ) {
            return '';
        }
        return str_replace( '{{table}}', $escaped, (string) $sql );
    }


    /**
     * Cached SHOW TABLES LIKE check.
     *
     * @param string $like
     * @return array
     */
        private function cached_show_tables_like( $like ) {
        global $wpdb;
        $like = (string) $like;
        $cache_key = 'ab_show_tables_like_' . md5( $like );
        $cached    = wp_cache_get( $cache_key, 'aegisbackup' );
        if ( false !== $cached ) {
            return (array) $cached;
        }

        $rows = $wpdb->get_col( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Schema introspection during migration (cached above).
            $wpdb->prepare( 'SHOW TABLES LIKE %s', $like )
        );
        wp_cache_set( $cache_key, (array) $rows, 'aegisbackup', 60 );
        return (array) $rows;
    }


    /**
     * Cached table existence check.
     *
     * @param string $table
     * @return bool
     */
    private function cached_table_exists( $table ) {
        global $wpdb;
        $table = (string) $table;
        $cache_key = 'ab_table_exists_' . md5( $table );
        $cached = wp_cache_get( $cache_key, 'aegisbackup' );
        if ( false !== $cached ) {
            return (bool) $cached;
        }
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Schema introspection during migration.
        $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Schema introspection during migration (cached above).
        $ok = ! empty( $exists );
        wp_cache_set( $cache_key, $ok, 'aegisbackup', 60 );
        return (bool) $ok;
    }

    public function apply_prefix_change( $prefix_current, $prefix_new, $prefix_mode = 'all' ) {
        global $wpdb;

        $prefix_current = $this->sanitize_db_prefix( (string) $prefix_current );
        $prefix_new     = $this->sanitize_db_prefix( (string) $prefix_new );
        $prefix_mode    = (string) $prefix_mode;

        if ( '' === $prefix_current || '' === $prefix_new ) {
            return array( 'success' => false, 'message' => 'Invalid database prefix provided. Aborting.' );
        }

        if ( function_exists( 'current_user_can' ) && ! current_user_can( 'manage_options' ) ) {
            return array( 'success' => false, 'message' => 'Insufficient permissions.' );
        }

        if ( function_exists( 'is_multisite' ) && is_multisite() ) {
            return array( 'success' => false, 'message' => 'Prefix migration is disabled on Multisite (safety block).' );
        }

        if ( 'all' !== $prefix_mode ) {
            return array( 'success' => false, 'message' => 'Unsafe prefix mode blocked. Only full-prefix migrations are allowed.' );
        }

        $validation_error = $this->validate_new_prefix_value( $prefix_new, $prefix_current );
        if ( $validation_error ) {
            return array( 'success' => false, 'message' => $validation_error );
        }

        $like = $wpdb->esc_like( $prefix_current ) . '%';
        $rows = $this->cached_show_tables_like( $like );

        if ( empty( $rows ) ) {
            return array( 'success' => false, 'message' => 'No tables matched the current prefix. Aborting.' );
        }

        $core_suffixes = $this->get_wp_core_table_suffixes();
        $rename_map    = array();

        foreach ( $rows as $tbl ) {
            $tbl = (string) $tbl;
            $tbl_safe = $this->sanitize_table_identifier( $tbl );
            if ( '' === $tbl_safe ) {
                continue;
            }
            $tbl = $tbl_safe;
            if ( 0 !== strpos( $tbl, $prefix_current ) ) {
                continue;
            }

            $suffix = substr( $tbl, strlen( $prefix_current ) );

            if ( 'core' === $prefix_mode ) {
                if ( ! in_array( $suffix, $core_suffixes, true ) ) {
                    continue;
                }
            }

            $rename_map[ $tbl ] = $prefix_new . $suffix;
        }

        if ( empty( $rename_map ) ) {
            return array( 'success' => false, 'message' => 'No tables qualified for renaming under the selected scope.' );
        }

        foreach ( $rename_map as $old => $new ) {
            $old = $this->sanitize_table_identifier( $old );
            $new = $this->sanitize_table_identifier( $new );
            if ( '' === $old || '' === $new ) {
                return array( 'success' => false, 'message' => 'Unsafe table name detected during prefix migration. Aborting.', 'details' => $results );
            }
            $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $new ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Schema introspection during migration.
            if ( ! empty( $exists ) ) {
                return array( 'success' => false, 'message' => 'Destination table already exists: ' . $new . '. Aborting to avoid collisions.' );
            }
        }

        $results = array(
            'renamed_tables' => array(),
            'options_rows'   => 0,
            'usermeta_rows'  => 0,
            'wpconfig'       => array( 'updated' => false, 'writable' => false, 'path' => '' ),
            'prefix_current' => $prefix_current,
            'prefix_new'     => $prefix_new,
            'prefix_mode'    => $prefix_mode,
        );

        $renamed_new_to_old = array();

        foreach ( $rename_map as $old => $new ) {
            $old = $this->sanitize_table_identifier( $old );
            $new = $this->sanitize_table_identifier( $new );
            if ( '' === $old || '' === $new ) {
                return array( 'success' => false, 'message' => 'Unsafe table name detected during prefix migration. Aborting.', 'details' => $results );
            }
            $q  = 'RENAME TABLE `' . str_replace( '`', '``', $old ) . '` TO `' . str_replace( '`', '``', $new ) . '`';
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Identifiers validated; cannot prepare table names (migration).
            $ok = $wpdb->query( $q );

            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Identifiers validated; cannot prepare table names (migration).
if ( false === $ok ) {
                foreach ( array_reverse( $renamed_new_to_old, true ) as $rolled_new => $rolled_old ) {
                    $rq = 'RENAME TABLE `' . str_replace( '`', '``', $rolled_new ) . '` TO `' . str_replace( '`', '``', $rolled_old ) . '`';
                    // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Identifiers validated; cannot prepare table names (migration).
                        $wpdb->query( $rq );
                // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Identifiers validated; cannot prepare table names (migration).
}

                return array(
                    'success' => false,
                    'message' => 'Failed renaming table ' . $old . ' -> ' . $new . '. Rollback attempted.',
                    'details' => $results,
                );
            }

            $renamed_new_to_old[ $new ] = $old;
            $results['renamed_tables'][] = array( 'from' => $old, 'to' => $new );
        }

        $new_options = $prefix_new . 'options';
        $opt_table_exists = $this->cached_table_exists( $new_options );
        if ( $opt_table_exists ) {
            $opt_like = $prefix_current . '%';
            $opt_table = $this->sanitize_table_identifier( $new_options );
            if ( '' !== $opt_table ) {
            $opt_rows = $wpdb->get_col( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- One-time migration query.
                $wpdb->prepare(
                    'SELECT option_name FROM %i WHERE option_name LIKE %s',
                    $opt_table,
                    $opt_like
                )
            );
        } else {
            $opt_rows = array();
        }

            if ( is_array( $opt_rows ) && ! empty( $opt_rows ) ) {
                $updated = 0;
                $skipped = 0;

                foreach ( $opt_rows as $old_key ) {
                    $old_key = (string) $old_key;
                    $suffix  = substr( $old_key, strlen( $prefix_current ) );
                    $new_key = $prefix_new . $suffix;

                    if ( $new_key === $old_key ) {
                        continue;
                    }

                    $exists = 0;
                    if ( '' !== $opt_table ) {
                        $exists = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- One-time migration query.
                            $wpdb->prepare(
                                'SELECT COUNT(*) FROM %i WHERE option_name = %s',
                                $opt_table,
                                $new_key
                            )
                        );
                    }

                    if ( $exists > 0 ) {
                        $wpdb->delete( $new_options, array( 'option_name' => $old_key ), array( '%s' ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cleanup during migration.
                        $skipped++;
                        continue;
                    }

                    $ok = $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Migration update.
                        $new_options,
                        array( 'option_name' => $new_key ),
                        array( 'option_name' => $old_key ),
                        array( '%s' ),
                        array( '%s' )
                    );

                    if ( false === $ok ) {
                        foreach ( array_reverse( $renamed_new_to_old, true ) as $rolled_new => $rolled_old ) {
                            $rq = 'RENAME TABLE `' . str_replace( '`', '``', $rolled_new ) . '` TO `' . str_replace( '`', '``', $rolled_old ) . '`';
                            $wpdb->query( $rq ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Identifiers validated; cannot prepare table names (migration).
                        }
                        return array( 'success' => false, 'message' => 'Failed while updating options prefix-bound keys.', 'details' => $results );
                    }

                    $updated++;
                }

                $results['options_rows'] = (int) $updated;
                if ( $skipped > 0 ) {
                    $results['options_collisions_skipped'] = (int) $skipped;
                }
            }
        }

        $new_usermeta = $prefix_new . 'usermeta';
        $um_table_exists = $this->cached_table_exists( $new_usermeta );
        if ( $um_table_exists ) {
            $um_like = $prefix_current . '%';
            $um_table = $this->sanitize_table_identifier( $new_usermeta );
            if ( '' !== $um_table ) {
            $um_rows = $wpdb->get_col( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- One-time migration query.
                $wpdb->prepare(
                    'SELECT DISTINCT meta_key FROM %i WHERE meta_key LIKE %s',
                    $um_table,
                    $um_like
                )
            );
        } else {
            $um_rows = array();
        }

            if ( is_array( $um_rows ) && ! empty( $um_rows ) ) {
                $updated = 0;
                $skipped = 0;

                foreach ( $um_rows as $old_key ) {
                    $old_key = (string) $old_key;
                    $suffix  = substr( $old_key, strlen( $prefix_current ) );
                    $new_key = $prefix_new . $suffix;

                    if ( $new_key === $old_key ) {
                        continue;
                    }

                    $exists = 0;
                    if ( '' !== $um_table ) {
                        $exists = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- One-time migration query.
                            $wpdb->prepare(
                                'SELECT COUNT(*) FROM %i WHERE meta_key = %s',
                                $um_table,
                                $new_key
                            )
                        );
                    }
                    if ( $exists > 0 ) {
                        $wpdb->delete( $new_usermeta, array( 'meta_key' => $old_key ), array( '%s' ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Cleanup during migration.
$skipped++;
                        continue;
                    }

                    $ok = $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Migration update.
                        $new_usermeta,
                        array( 'meta_key' => $new_key ), // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Migration update uses meta_key during prefix swap.
                        array( 'meta_key' => $old_key ), // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Migration update uses meta_key during prefix swap.
                        array( '%s' ),
                        array( '%s' )
                    );

                    if ( false === $ok ) {
                        foreach ( array_reverse( $renamed_new_to_old, true ) as $rolled_new => $rolled_old ) {
                            $rq = 'RENAME TABLE `' . str_replace( '`', '``', $rolled_new ) . '` TO `' . str_replace( '`', '``', $rolled_old ) . '`';
                            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Identifiers validated; cannot prepare table names (migration).
                        $wpdb->query( $rq );
                        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Identifiers validated; cannot prepare table names (migration).
}
                        return array( 'success' => false, 'message' => 'Failed while updating usermeta prefix-bound keys.', 'details' => $results );
                    }

                    $updated++;
                }

                $results['usermeta_rows'] = (int) $updated;
                if ( $skipped > 0 ) {
                    $results['usermeta_collisions_skipped'] = (int) $skipped;
                }
            }
        }

        $wpconfig = $this->try_update_wpconfig_table_prefix( $prefix_new );
        $results['wpconfig'] = $wpconfig;

        return array(
            'success' => true,
            'message' => 'Prefix change applied. IMPORTANT: ensure wp-config.php $table_prefix matches the NEW prefix.',
            'details' => $results,
            'manual_wpconfig' => "\$table_prefix = '{$prefix_new}';",
        );
    }

    public function validate_new_prefix_value( $prefix_new, $prefix_current = '' ) {
        $prefix_new     = (string) $prefix_new;
        $prefix_current = (string) $prefix_current;

        if ( '' === $prefix_new ) {
            return 'New prefix cannot be empty.';
        }

        if ( $prefix_new === $prefix_current ) {
            return 'New prefix is the same as the current prefix.';
        }

        if ( strlen( $prefix_new ) > 60 ) {
            return 'New prefix is too long.';
        }

        if ( ! preg_match( '/^[A-Za-z0-9_]+$/', $prefix_new ) ) {
            return 'New prefix contains invalid characters. Use letters, numbers, and underscore only.';
        }

        if ( '_' !== substr( $prefix_new, -1 ) ) {
            return 'New prefix should end with an underscore (_) for WordPress compatibility.';
        }

        return '';
    }

    public function try_update_wpconfig_table_prefix( $prefix_new ) {
        $prefix_new = (string) $prefix_new;

        $paths = array(
            ABSPATH . 'wp-config.php',
        );

        $parent = dirname( ABSPATH );
        if ( $parent && is_dir( $parent ) ) {
            $paths[] = trailingslashit( $parent ) . 'wp-config.php';
        }

        foreach ( $paths as $path ) {
            if ( ! file_exists( $path ) ) {
                continue;
            }

            // Use WP_Filesystem for file checks/IO to satisfy plugin checker.
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
            global $wp_filesystem;

            $writable = false;
            $contents = false;

            if ( $wp_filesystem ) {
                $writable = (bool) $wp_filesystem->is_writable( $path );
                $contents = $wp_filesystem->get_contents( $path );
            } else {
                // Fallback: WP_Filesystem should be available in wp-admin context; keep conservative fallback.
                // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_is_writable -- Fallback only.
                $writable = is_writable( $path );
                // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents -- Fallback only.
                $contents = file_get_contents( $path );
            }

            if ( false === $contents ) {
                continue;
            }

            $updated = false;
            $new_contents = preg_replace(
                "/\$table_prefix\s*=\s*['\"]([^'\"]+)['\"];\s*/",
                "\$table_prefix = '{$prefix_new}';\n",
                $contents,
                1,
                $count
            );

            if ( $count > 0 && $writable ) {
                if ( $wp_filesystem ) {
                    $ok = $wp_filesystem->put_contents( $path, $new_contents, FS_CHMOD_FILE );
                } else {
                    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Fallback only.
                    $ok = file_put_contents( $path, $new_contents );
                }
                $updated = ( false !== $ok );
            }

            return array(
                'path' => $path,
                'writable' => (bool) $writable,
                'updated' => (bool) $updated,
            );
        }

        return array( 'path' => '', 'writable' => false, 'updated' => false );
    }

    public function get_wp_core_table_suffixes() {
        return array(
            'options',
            'users',
            'usermeta',
            'posts',
            'postmeta',
            'terms',
            'termmeta',
            'term_taxonomy',
            'term_relationships',
            'comments',
            'commentmeta',
            'links',
        );
    }
}
