<?php
namespace AegisBackup\Libs;

defined( 'ABSPATH' ) || exit;

class AB_WPConfig_Writer {

    public function locate_wpconfig_path() {
        $candidates = array(
            ABSPATH . 'wp-config.php',
        );

        $above = dirname( ABSPATH );
        if ( $above && is_dir( $above ) ) {
            $candidates[] = trailingslashit( $above ) . 'wp-config.php';
        }

        foreach ( $candidates as $p ) {
            if ( is_file( $p ) ) return $p;
        }
        return '';
    }

    public function write( array $changes ) {
        $path = $this->locate_wpconfig_path();
        if ( ! $path ) {
            return array( 'success' => false, 'message' => 'wp-config.php not found.' );
        }

        $writable = wp_is_writable( $path );
        $contents = @file_get_contents( $path );
        if ( false === $contents ) {
            return array( 'success' => false, 'message' => 'Unable to read wp-config.php.' );
        }

        $backup_path = $path . '.aegisbackup.' . gmdate( 'Ymd_His' ) . '.bak';
        @file_put_contents( $backup_path, $contents );

        $new_contents = $contents;

        $map = array(
            'DB_NAME' => isset( $changes['DB_NAME'] ) ? (string) $changes['DB_NAME'] : null,
            'DB_USER' => isset( $changes['DB_USER'] ) ? (string) $changes['DB_USER'] : null,
            'DB_PASSWORD' => isset( $changes['DB_PASSWORD'] ) ? (string) $changes['DB_PASSWORD'] : null,
            'DB_HOST' => isset( $changes['DB_HOST'] ) ? (string) $changes['DB_HOST'] : null,
        );

        foreach ( $map as $const => $val ) {
            if ( null === $val || '' === $val ) continue;
            $new_contents = $this->replace_define( $new_contents, $const, $val );
        }

        if ( ! empty( $changes['table_prefix'] ) ) {
            $new_contents = $this->replace_table_prefix( $new_contents, (string) $changes['table_prefix'] );
        }

        if ( ! empty( $changes['regenerate_salts'] ) ) {
            $new_contents = $this->replace_salts( $new_contents );
        }

        $updated = false;
        if ( $new_contents !== $contents && $writable ) {
            $ok = @file_put_contents( $path, $new_contents );
            $updated = ( false !== $ok );
        }

        return array(
            'success' => true,
            'path' => $path,
            'writable' => (bool) $writable,
            'updated' => (bool) $updated,
            'backup' => $backup_path,
            'manual' => $this->manual_instructions( $changes ),
        );
    }

    protected function replace_define( $contents, $const, $value ) {
        $value = $this->php_quote( $value );

        $pattern = '/define\(\s*[\"\']' . preg_quote( $const, '/' ) . '[\"\']\s*,\s*([\"\'])(.*?)\1\s*\)\s*;?/i';
        $replacement = "define( '{$const}', {$value} );";
        if ( preg_match( $pattern, $contents ) ) {
            return preg_replace( $pattern, $replacement, $contents, 1 );
        }

        $insert = "\n" . $replacement . "\n";
        return preg_replace( '/<\?php\s*/', "<?php\n" . $replacement . "\n", $contents, 1 );
    }

    protected function replace_table_prefix( $contents, $prefix ) {
        $prefix = trim( (string) $prefix );
        $prefix_q = $this->php_quote( $prefix );

        $pattern = '/\$table_prefix\s*=\s*([\"\'])(.*?)\1\s*;/i';
        $replacement = '$table_prefix = ' . $prefix_q . ';';
        if ( preg_match( $pattern, $contents ) ) {
            return preg_replace( $pattern, $replacement, $contents, 1 );
        }

        return $contents . "\n" . $replacement . "\n";
    }

    protected function replace_salts( $contents ) {
        $salts = array(
            'AUTH_KEY',
            'SECURE_AUTH_KEY',
            'LOGGED_IN_KEY',
            'NONCE_KEY',
            'AUTH_SALT',
            'SECURE_AUTH_SALT',
            'LOGGED_IN_SALT',
            'NONCE_SALT',
        );

        foreach ( $salts as $const ) {
            $val = wp_generate_password( 64, true, true );
            $contents = $this->replace_define( $contents, $const, $val );
        }
        return $contents;
    }

    protected function php_quote( $value ) {
        $v = str_replace( "'", "\\'", (string) $value );
        return "'" . $v . "'";
    }

    protected function manual_instructions( array $changes ) {
        $lines = array();
        if ( ! empty( $changes['DB_NAME'] ) ) $lines[] = "define('DB_NAME', '" . $changes['DB_NAME'] . "');";
        if ( ! empty( $changes['DB_USER'] ) ) $lines[] = "define('DB_USER', '" . $changes['DB_USER'] . "');";
        if ( ! empty( $changes['DB_PASSWORD'] ) ) $lines[] = "define('DB_PASSWORD', '***');";
        if ( ! empty( $changes['DB_HOST'] ) ) $lines[] = "define('DB_HOST', '" . $changes['DB_HOST'] . "');";
        if ( ! empty( $changes['table_prefix'] ) ) $lines[] = "\$table_prefix = '" . $changes['table_prefix'] . "';";
        if ( ! empty( $changes['regenerate_salts'] ) ) $lines[] = '(Regenerate salts: visit WordPress.org secret-key service or let AegisBackup write them automatically.)';
        return implode( "\n", $lines );
    }
}
