<?php
namespace AegisBackup\MigrationWizard;

defined( 'ABSPATH' ) || exit;

/**
 * Installs and manages the standalone Migration Wizard runner at /aegismw/.
 *
 * This runner is intentionally NOT bootstrapped through WordPress runtime.
 * It is copied into ABSPATH/aegismw so restores can be performed without
 * WordPress loading (avoids DB overwrite crashes).
 */
class AB_MW_Installer {
	const OPTION_TOKEN   = 'aegisbackup_mw_token';
	const OPTION_ENABLED = 'aegisbackup_mw_enabled';
	const FOLDER_NAME    = 'aegismw';
	/**
	 * Get a ready WP_Filesystem instance when possible.
	 *
	 * @return \WP_Filesystem_Base|null
	 */
	private static function get_filesystem() {
		global $wp_filesystem;

		if ( $wp_filesystem instanceof \WP_Filesystem_Base ) {
			return $wp_filesystem;
		}

		require_once ABSPATH . 'wp-admin/includes/file.php';

		// Best-effort init. Local filesystem should not require credentials.
		if ( function_exists( 'WP_Filesystem' ) && WP_Filesystem() ) {
			if ( $wp_filesystem instanceof \WP_Filesystem_Base ) {
				return $wp_filesystem;
			}
		}

		return null;
	}


	public static function is_enabled() {
		return (bool) get_option( self::OPTION_ENABLED, false );
	}

	public static function set_enabled( $enabled ) {
		update_option( self::OPTION_ENABLED, $enabled ? 1 : 0, false );
	}

	public static function get_runner_path() {
		return trailingslashit( ABSPATH ) . self::FOLDER_NAME;
	}

	public static function is_installed() {
		$dst = self::get_runner_path();
		return is_dir( $dst ) && file_exists( trailingslashit( $dst ) . 'index.php' );
	}

	public static function install() {
		self::set_enabled( true );
		self::ensure_installed();
	}

	public static function uninstall() {
		self::set_enabled( false );
		$dst = self::get_runner_path();

		self::delete_runner_dir( $dst );

		// Best-effort second attempt in case of transient FS locks.
		clearstatcache();
		if ( is_dir( $dst ) ) {
			self::delete_runner_dir( $dst );
		}
	}

	public static function ensure_installed() {
		// Only install/copy the runner when explicitly enabled via the admin toggle.
		if ( ! self::is_enabled() ) {
			return;
		}
		if ( ! defined( 'AEGISBACKUP_DIR' ) || ! defined( 'ABSPATH' ) ) {
			return;
		}

		$dst = trailingslashit( ABSPATH ) . self::FOLDER_NAME;
		$src = trailingslashit( AEGISBACKUP_DIR ) . self::FOLDER_NAME;

		if ( ! is_dir( $src ) ) {
			return;
		}
		if ( ! file_exists( trailingslashit( $src ) . 'index.php' ) ) {
			return;
		}

		if ( ! is_dir( $dst ) ) {
			wp_mkdir_p( $dst );
		}

		// Copy bundled runner files.
		self::copy_dir( $src, $dst );

		// Add a minimal .htaccess to reduce accidental exposure on Apache.
		// Add a minimal .htaccess to reduce accidental exposure on Apache / LiteSpeed (avoid Options directive which can 500 on some hosts).
		$ht = "# AegisBackup Migration Wizard runner
# (Intentionally left minimal to avoid server 500s on hosts with restrictive .htaccess parsing.)
";
		if ( file_exists( trailingslashit( $dst ) . '.htaccess' ) ) {
			wp_delete_file( trailingslashit( $dst ) . '.htaccess' );
		}
		@file_put_contents( trailingslashit( $dst ) . '.htaccess', $ht );

		// Ensure report directory exists and block direct access to raw report artifacts (best-effort; Apache only).
		$report_dir = trailingslashit( $dst ) . 'report';
		if ( ! is_dir( $report_dir ) ) {
			wp_mkdir_p( $report_dir );
		}
		$report_ht = "# AegisBackup Migration Wizard report viewer
# (Intentionally left minimal to avoid server 500s on hosts with restrictive .htaccess parsing.)
";
		if ( file_exists( trailingslashit( $report_dir ) . '.htaccess' ) ) {
			wp_delete_file( trailingslashit( $report_dir ) . '.htaccess' );
		}
		@file_put_contents( trailingslashit( $report_dir ) . '.htaccess', $report_ht );

	}

	public static function regenerate_token() {
		// Access keys disabled for QA/testing.
		return '';
	}


	public static function get_token() {
		// Access keys disabled for QA/testing.
		return '';
	}


	public static function get_runner_url() {
		return home_url( '/' . self::FOLDER_NAME . '/' );
	}
	private static function delete_runner_dir( $path ) {
		if ( ! defined( 'ABSPATH' ) ) {
			return;
		}

		$path = rtrim( (string) $path, "/\\ \t\n\r\0\x0B" );
		if ( '' === $path ) {
			return;
		}

		// Only allow deleting exactly ABSPATH/aegismw
		$expected = rtrim( trailingslashit( ABSPATH ) . self::FOLDER_NAME, "/\\" );

		if ( function_exists( 'wp_normalize_path' ) ) {
			$path_norm     = rtrim( wp_normalize_path( $path ), '/' );
			$expected_norm = rtrim( wp_normalize_path( $expected ), '/' );

			if ( $path_norm !== $expected_norm ) {
				return;
			}
		} else {
			// Fallback comparison if wp_normalize_path is unavailable.
			if ( $path !== $expected ) {
				return;
			}
		}

		if ( ! is_dir( $path ) ) {
			return;
		}
		$fs = self::get_filesystem();
		if ( $fs ) {
			// Let WP_Filesystem handle recursive deletion in one call.
			$fs->delete( $path, true, 'd' );
			return;
		}

		// phpcs:disable WordPress.WP.AlternativeFunctions.file_system_operations_rmdir, WordPress.WP.AlternativeFunctions.unlink_unlink


		// Recursive delete (child-first) using SPL iterators.
		if ( class_exists( '\RecursiveDirectoryIterator' ) && class_exists( '\RecursiveIteratorIterator' ) ) {
			$dir_it = new \RecursiveDirectoryIterator(
				$path,
				\FilesystemIterator::SKIP_DOTS
			);

			$it = new \RecursiveIteratorIterator(
				$dir_it,
				\RecursiveIteratorIterator::CHILD_FIRST
			);

			foreach ( $it as $item ) {
				$full = $item->getPathname();

				if ( $item->isDir() ) {
					@rmdir( $full );
				} else {
					@unlink( $full );
				}
			}

			@rmdir( $path );
			return;
		}

		// Fallback (very old PHP): scandir recursion without the "basename must be aegismw" bug.
		$items = @scandir( $path );
		if ( is_array( $items ) ) {
			foreach ( $items as $it ) {
				if ( '.' === $it || '..' === $it ) {
					continue;
				}
				$full = $path . DIRECTORY_SEPARATOR . $it;

				if ( is_dir( $full ) ) {
					self::delete_runner_dir_fallback( $full );
					@rmdir( $full );
				} else {
					@unlink( $full );
				}
			}
		}

		@rmdir( $path );
		// phpcs:enable WordPress.WP.AlternativeFunctions.file_system_operations_rmdir, WordPress.WP.AlternativeFunctions.unlink_unlink
	}

	private static function delete_runner_dir_fallback( $dir ) {
		$fs = self::get_filesystem();
		if ( $fs ) {
			$fs->delete( $dir, true, 'd' );
			return;
		}

		// phpcs:disable WordPress.WP.AlternativeFunctions.file_system_operations_rmdir, WordPress.WP.AlternativeFunctions.unlink_unlink

		$items = @scandir( $dir );
		if ( ! is_array( $items ) ) {
			return;
		}
		foreach ( $items as $it ) {
			if ( '.' === $it || '..' === $it ) {
				continue;
			}
			$full = $dir . DIRECTORY_SEPARATOR . $it;

			if ( is_dir( $full ) ) {
				self::delete_runner_dir_fallback( $full );
				@rmdir( $full );
			} else {
				@unlink( $full );
			}
		}
		// phpcs:enable WordPress.WP.AlternativeFunctions.file_system_operations_rmdir, WordPress.WP.AlternativeFunctions.unlink_unlink
	}

	private static function copy_dir( $src, $dst ) {
		$src = rtrim( (string) $src, '/\\' );
		$dst = rtrim( (string) $dst, '/\\' );
		if ( ! is_dir( $src ) ) {
			return;
		}
		if ( ! is_dir( $dst ) ) {
			wp_mkdir_p( $dst );
		}
		$items = @scandir( $src );
		if ( ! is_array( $items ) ) {
			return;
		}
		foreach ( $items as $it ) {
			if ( '.' === $it || '..' === $it ) {
				continue;
			}
			$from = $src . DIRECTORY_SEPARATOR . $it;
			$to   = $dst . DIRECTORY_SEPARATOR . $it;
			if ( is_dir( $from ) ) {
				self::copy_dir( $from, $to );
				continue;
			}
			@copy( $from, $to );
		}
	}
}
