<?php
namespace AegisBackup\Modules;

use AegisBackup\AB_Plugin;

defined( 'ABSPATH' ) || exit;

class AB_Module_PreUpdate {
    const OPTION_KEY = 'aegisbackup_preupdate_settings';
    const LAST_SNAPSHOT_OPTION = 'aegisbackup_preupdate_last_snapshot';
    const PENDING_JOB_OPTION = 'aegisbackup_preupdate_pending_job';

    protected $plugin;

    public function __construct( AB_Plugin $plugin ) {
        $this->plugin = $plugin;

        add_filter( 'upgrader_pre_install', array( $this, 'maybe_backup_before_update' ), 10, 2 );
        add_action( 'upgrader_process_complete', array( $this, 'on_upgrade_complete' ), 10, 2 );
        add_action( 'admin_init', array( $this, 'maybe_finalize_pending' ) );
    }

    public function get_settings() {
        $s = get_option( self::OPTION_KEY, array() );
        return is_array( $s ) ? $s : array();
    }

    public function is_enabled() {
        $s = $this->get_settings();
        return ! empty( $s['enabled'] );
    }

    public function get_last_snapshot() {
        $s = get_option( self::LAST_SNAPSHOT_OPTION, array() );
        return is_array( $s ) ? $s : array();
    }

    public function maybe_backup_before_update( $reply, $hook_extra ) {
        if ( ! $this->is_enabled() ) {
            return $reply;
        }

        if ( is_wp_error( $reply ) ) {
            return $reply;
        }

        $hook_extra = is_array( $hook_extra ) ? $hook_extra : array();
        $type       = isset( $hook_extra['type'] ) ? sanitize_key( (string) $hook_extra['type'] ) : '';

        if ( ! in_array( $type, array( 'plugin', 'theme', 'core' ), true ) ) {
            return $reply;
        }

        if ( 'plugin' === $type ) {
            $self = function_exists( 'plugin_basename' ) ? plugin_basename( AEGISBACKUP_FILE ) : '';
            $p1   = ! empty( $hook_extra['plugin'] ) ? (string) $hook_extra['plugin'] : '';
            $pN   = ( ! empty( $hook_extra['plugins'] ) && is_array( $hook_extra['plugins'] ) ) ? array_map( 'strval', (array) $hook_extra['plugins'] ) : array();

            if ( $self && ( $p1 === $self || in_array( $self, $pN, true ) ) ) {
                $this->log( 'Pre-update snapshot skipped for AegisBackup self-update.' );
                return $reply;
            }
        }

        if ( defined( 'AEGISBACKUP_PREUPDATE_RUNNING' ) && AEGISBACKUP_PREUPDATE_RUNNING ) {
            return $reply;
        }
        if ( ! defined( 'AEGISBACKUP_PREUPDATE_RUNNING' ) ) {
            define( 'AEGISBACKUP_PREUPDATE_RUNNING', true );
        }

        if ( empty( $this->plugin->backup ) || ! method_exists( $this->plugin->backup, 'start_backup_job' ) ) {
            $this->log( 'Pre-update snapshot FAILED: Backup manager not available.' );
            return $reply;
        }

        $meta = $this->build_update_meta( $type, $hook_extra );

        $args = $this->build_snapshot_args( $type, $meta );
        $job  = $this->plugin->backup->start_backup_job( $args );

        if ( ! is_array( $job ) || empty( $job['job_id'] ) ) {
            $msg = is_array( $job ) && ! empty( $job['message'] ) ? (string) $job['message'] : 'Failed to start snapshot job.';
            $this->log( 'Pre-update snapshot FAILED (' . $type . '): ' . $msg );
            return $reply;
        }

        update_option(
            self::PENDING_JOB_OPTION,
            array(
                'job_id'     => (string) $job['job_id'],
                'started_at' => gmdate( 'c' ),
                'type'       => $type,
                'meta'       => $meta,
            ),
            false
        );

        $this->log( 'Pre-update snapshot job started (' . $type . '): ' . (string) $job['job_id'] );

        return $reply;
    }


    public function on_upgrade_complete( $upgrader, $hook_extra ) {
        if ( ! $this->is_enabled() ) {
            return;
        }

        $hook_extra = is_array( $hook_extra ) ? $hook_extra : array();
        $type       = isset( $hook_extra['type'] ) ? sanitize_key( (string) $hook_extra['type'] ) : '';
        if ( ! in_array( $type, array( 'plugin', 'theme', 'core' ), true ) ) {
            return;
        }

        $this->log( 'Upgrade completed (' . $type . ').' );
        $this->maybe_finalize_pending( true );
    }

public function maybe_finalize_pending( $verbose = false ) {
    if ( ! $this->is_enabled() ) {
        return;
    }

    $pending = get_option( self::PENDING_JOB_OPTION, array() );
    if ( ! is_array( $pending ) || empty( $pending['job_id'] ) ) {
        return;
    }

    if ( empty( $this->plugin->backup ) || ! method_exists( $this->plugin->backup, 'process_backup_job' ) ) {
        if ( $verbose ) {
            $this->log( 'Pre-update snapshot finalize FAILED: Backup processor not available.' );
        }
        return;
    }

    $job_id = (string) $pending['job_id'];
    $type   = ! empty( $pending['type'] ) ? sanitize_key( (string) $pending['type'] ) : 'plugin';
    $meta   = ( ! empty( $pending['meta'] ) && is_array( $pending['meta'] ) ) ? (array) $pending['meta'] : array();

    if ( function_exists( 'set_time_limit' ) ) {
        // phpcs:ignore Squiz.PHP.DiscouragedFunctions.Discouraged
        @set_time_limit( 120 );
    }

    $start       = time();
    $max_seconds = 120;
    $step        = array();

    do {
        $step = $this->plugin->backup->process_backup_job( $job_id );
        if ( ! empty( $step['done'] ) ) {
            break;
        }
    } while ( ( time() - $start ) < $max_seconds );

    if ( empty( $step['done'] ) || empty( $step['package'] ) ) {
        if ( $verbose ) {
            $this->log( 'Pre-update snapshot still running: ' . $job_id );
        }
        return;
    }

    delete_option( self::PENDING_JOB_OPTION );

    $package = (string) $step['package'];
    $this->store_last_snapshot( $package, $type, $meta );

    if ( $verbose ) {
        $this->log( 'Pre-update snapshot finalized (' . $type . '): ' . $package );
    }
}


    protected function build_update_meta( $type, array $hook_extra ) {
        $meta = array(
            'type' => $type,
            'time' => gmdate( 'c' ),
            'user' => get_current_user_id(),
        );

        if ( 'plugin' === $type ) {
            if ( ! empty( $hook_extra['plugin'] ) ) {
                $meta['plugin'] = (string) $hook_extra['plugin'];
            }
            if ( ! empty( $hook_extra['plugins'] ) && is_array( $hook_extra['plugins'] ) ) {
                $meta['plugins'] = array_values( array_map( 'strval', (array) $hook_extra['plugins'] ) );
            }
        } elseif ( 'theme' === $type ) {
            if ( ! empty( $hook_extra['theme'] ) ) {
                $meta['theme'] = (string) $hook_extra['theme'];
            }
            if ( ! empty( $hook_extra['themes'] ) && is_array( $hook_extra['themes'] ) ) {
                $meta['themes'] = array_values( array_map( 'strval', (array) $hook_extra['themes'] ) );
            }
        } elseif ( 'core' === $type ) {
            if ( ! empty( $hook_extra['action'] ) ) {
                $meta['action'] = (string) $hook_extra['action'];
            }
        }

        return $meta;
    }

protected function build_snapshot_args( $type, array $meta ) {
    $excludes = array(
        'wp-content/uploads',
        'wp-content/cache',
        'wp-content/upgrade',
        'wp-content/backups',
        'wp-content/uploads/backup',
        'wp-content/uploads/aegisbackup',
        'wp-content/plugins/aegisbackup/backups',
    );

    $args = array(
        'include_files'     => 1,
        'include_db'        => 1,
        'include_config'    => 1,
        'include_core'      => ( 'core' === $type ) ? 1 : 0,
        'include_htaccess'  => 1,
        'excludes'          => implode( "
", $excludes ),
        'db_export_mode'    => 'auto',
        'backup_type'       => 'full',
        'snapshot'          => 1,
        'package_purpose'   => 'pre_update',
        'update_context'    => $meta,
    );

    if ( 'plugin' === $type ) {
        $args['include_paths'] = array(
            'wp-content/plugins',
            'wp-content/mu-plugins',
        );
    } elseif ( 'theme' === $type ) {
        $args['include_paths'] = array(
            'wp-content/themes',
            'wp-content/mu-plugins',
        );
    }

    return $args;
}

    protected function run_preupdate_snapshot( $type, array $meta ) {
        if ( empty( $this->plugin->backup ) ) {
            return array( 'ok' => false, 'message' => 'Backup manager not available.' );
        }

        $excludes = array(
            'wp-content/uploads',
            'wp-content/cache',
            'wp-content/upgrade',
            'wp-content/backups',
            'wp-content/uploads/backup',
            'wp-content/uploads/aegisbackup',
            'wp-content/plugins/aegisbackup/backups',
        );

        $args = array(
            'include_files'     => 1,
            'include_db'        => 1,
            'include_config'    => 1,
            'include_core'      => ( 'core' === $type ) ? 1 : 0,
            'include_htaccess'  => 1,
            'excludes'          => implode( "\n", $excludes ),
            'db_export_mode'    => 'auto',
            'backup_type'       => 'full',
            'snapshot'          => 1,
            'package_purpose'   => 'pre_update',
            'update_context'    => $meta,
        );

        if ( 'plugin' === $type ) {
            $args['include_paths'] = array(
                'wp-content/plugins',
                'wp-content/mu-plugins',
            );
        } elseif ( 'theme' === $type ) {
            $args['include_paths'] = array(
                'wp-content/themes',
                'wp-content/mu-plugins',
            );
        }

        $job = $this->plugin->backup->start_backup_job( $args );
        if ( ! is_array( $job ) || empty( $job['job_id'] ) ) {
            $msg = is_array( $job ) && ! empty( $job['message'] ) ? (string) $job['message'] : 'Failed to start snapshot job.';
            return array( 'ok' => false, 'message' => $msg );
        }

        $job_id = (string) $job['job_id'];

        if ( function_exists( 'set_time_limit' ) ) {
            // phpcs:ignore Squiz.PHP.DiscouragedFunctions.Discouraged
            @set_time_limit( 120 );
        }

        $start = time();
        $max_seconds = 120;

        $step = array();
        do {
            $step = $this->plugin->backup->process_backup_job( $job_id );
            if ( ! empty( $step['done'] ) ) {
                break;
            }
        } while ( ( time() - $start ) < $max_seconds );

        if ( empty( $step['done'] ) || empty( $step['package'] ) ) {
            update_option( self::PENDING_JOB_OPTION, array(
                'job_id'     => $job_id,
                'started_at' => gmdate( 'c' ),
                'type'       => $type,
                'meta'       => $meta,
            ), false );

            return array(
                'ok'      => false,
                'message' => 'Snapshot is still building. A rollback point will appear once the snapshot completes.',
            );
        }

        delete_option( self::PENDING_JOB_OPTION );

        return array(
            'ok'      => true,
            'package' => (string) $step['package'],
        );
    }
private function store_last_snapshot( $package, $type, array $meta ) {
    $last = array(
        'package'     => (string) $package,
        'created_at'  => gmdate( 'c' ),
        'site'        => home_url(),
        'update_type' => (string) $type,
        'meta'        => $meta,
    );
    update_option( self::LAST_SNAPSHOT_OPTION, $last, false );

    $reports = get_option( 'aegisbackup_restore_reports', array() );
    if ( ! is_array( $reports ) ) {
        $reports = array();
    }

    $reports[] = array(
        'time'   => gmdate( 'c' ),
        'type'   => 'pre_update_snapshot',
        'status' => 'success',
        'message'=> 'Pre-update snapshot created: ' . basename( (string) $package ),
        'data'   => array(
            'package'     => (string) $package,
            'update_type' => (string) $type,
            'meta'        => $meta,
        ),
    );

    $reports = array_slice( $reports, -200 );
    update_option( 'aegisbackup_restore_reports', $reports, false );
}

    protected function log( $message ) {
        $logs = get_option( 'aegisbackup_logs', array() );
        if ( ! is_array( $logs ) ) {
            $logs = array();
        }
        $logs[] = '[' . gmdate( 'Y-m-d H:i:s' ) . '] ' . (string) $message;
        $logs = array_slice( $logs, -500 );
        update_option( 'aegisbackup_logs', $logs, false );
    }
}