<?php
namespace AegisBackup\Modules;

defined( 'ABSPATH' ) || exit;

require_once AEGISBACKUP_DIR . 'includes/libs/class-ab-activity-log.php';

use AegisBackup\Libs\AB_Activity_Log;

class AB_Module_Settings {

    protected function ab_get_fs() {
        global $wp_filesystem;
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        if ( ! is_object( $wp_filesystem ) ) {
            WP_Filesystem();
        }
        return is_object( $wp_filesystem ) ? $wp_filesystem : null;
    }

	public function __construct() {
		add_action( 'phpmailer_init', array( $this, 'configure_phpmailer' ) );
		add_action( 'wp_mail_failed', array( $this, 'on_wp_mail_failed' ) );
		add_action( 'wp_mail_succeeded', array( $this, 'on_wp_mail_succeeded' ) );
		add_action( 'aegisbackup_backup_completed', array( $this, 'on_backup_completed' ), 10, 2 );
		add_action( 'aegisbackup_backup_failed', array( $this, 'on_backup_failed' ), 10, 2 );
		add_action( 'aegisbackup_restore_completed', array( $this, 'on_restore_completed' ), 10, 2 );
		add_action( 'aegisbackup_restore_failed', array( $this, 'on_restore_failed' ), 10, 3 );
		add_action( 'aegisbackup_retention_daily', array( $this, 'run_retention' ) );
		if ( ! wp_next_scheduled( 'aegisbackup_retention_daily' ) ) {
			wp_schedule_event( time() + 300, 'daily', 'aegisbackup_retention_daily' );
		}
	}

	

public function on_wp_mail_failed( $wp_error ) {
	if ( ! is_object( $wp_error ) || ! method_exists( $wp_error, 'get_error_message' ) ) {
		return;
	}
	$message = (string) $wp_error->get_error_message();
	$data    = array();
	if ( method_exists( $wp_error, 'get_error_data' ) ) {
		$data = $wp_error->get_error_data();
	}

	$ctx = array();
	if ( is_array( $data ) ) {
		if ( isset( $data['to'] ) ) {
			$ctx['to'] = is_array( $data['to'] ) ? implode( ',', array_map( 'sanitize_text_field', $data['to'] ) ) : sanitize_text_field( (string) $data['to'] );
		}
		if ( isset( $data['subject'] ) ) {
			$ctx['subject'] = sanitize_text_field( (string) $data['subject'] );
		}
	}

	$entry = array(
		'time'    => time(),
		'message' => sanitize_text_field( $message ),
		'context' => $ctx,
	);

	update_option( 'aegisbackup_last_mail_error', $entry, false );
	$this->append_log( '[SMTP] wp_mail_failed: ' . ( $entry['message'] ?: 'unknown error' ) );
}

public function on_wp_mail_succeeded( $mail_data ) {
	// Keep diagnostics current: clear last failure when a mail succeeds.
	delete_option( 'aegisbackup_last_mail_error' );
}

public function configure_phpmailer( $phpmailer ) {
		$smtp = get_option( 'aegisbackup_smtp_settings', array() );
		if ( ! is_array( $smtp ) || empty( $smtp['enabled'] ) ) {
			return;
		}

		$host = isset( $smtp['host'] ) ? (string) $smtp['host'] : '';
		$port = isset( $smtp['port'] ) ? (int) $smtp['port'] : 0;
		$enc  = isset( $smtp['encryption'] ) ? (string) $smtp['encryption'] : 'none';
		$user = isset( $smtp['username'] ) ? (string) $smtp['username'] : '';
		$pass = isset( $smtp['password'] ) ? (string) $smtp['password'] : '';
		$from_name  = isset( $smtp['from_name'] ) ? (string) $smtp['from_name'] : '';
		$from_email = isset( $smtp['from_email'] ) ? (string) $smtp['from_email'] : '';

		if ( empty( $host ) ) {
			return;
		}

		$phpmailer->isSMTP();
		$phpmailer->Host = $host;
		if ( $port > 0 ) {
			$phpmailer->Port = $port;
		}
		if ( 'tls' === $enc || 'ssl' === $enc ) {
			$phpmailer->SMTPSecure = $enc;
		}
		$phpmailer->SMTPAuth = ( ! empty( $user ) );
		if ( ! empty( $user ) ) {
			$phpmailer->Username = $user;
			$phpmailer->Password = $pass;
		}

		if ( ! empty( $from_email ) && is_email( $from_email ) ) {
			$phpmailer->setFrom( $from_email, $from_name ? $from_name : null, false );
		}
	}

	public function on_backup_completed( $zip_path, $state ) {
		$zip_path = (string) $zip_path;
		if ( empty( $zip_path ) || ! is_file( $zip_path ) ) {
			$this->append_log( 'Backup completed hook fired, but package file was not found.' );
			return;
		}

		$notify = $this->get_notify_settings();
		$remote = get_option( 'aegisbackup_remote_storage', array() );
		if ( ! is_array( $remote ) ) {
			$remote = array();
		}

		$uploaded = null;
		$upload_msg = '';
		if ( ! empty( $remote['enabled'] ) ) {
			$res = $this->upload_offsite( $zip_path, $remote );
			$uploaded = ! empty( $res['success'] );
			$upload_msg = isset( $res['message'] ) ? (string) $res['message'] : '';
			$this->append_log( '[Remote Upload] ' . ( $uploaded ? 'SUCCESS' : 'FAILED' ) . ' - ' . $upload_msg );
		}

		if ( ! empty( $notify['on_backup_complete'] ) ) {
			$subject = '[AegisBackup] Backup Completed';
			$body = "Backup completed successfully.\n\n";
			$body .= 'Site: ' . home_url() . "\n";
			$body .= 'Package: ' . $zip_path . "\n";
			if ( null !== $uploaded ) {
				$body .= 'Remote Upload: ' . ( $uploaded ? 'SUCCESS' : 'FAILED' ) . "\n";
				if ( $upload_msg ) {
					$body .= 'Remote Message: ' . $upload_msg . "\n";
				}
			}
			$body .= "\nTime: " . gmdate( 'c' ) . "\n";
			wp_mail( $this->get_notify_recipient(), $subject, $body );
		}

		$this->maybe_send_webhook( 'offsite.failed', array( 'package' => $zip_path, 'message' => (string) $upload_msg, 'state' => $state ) );

		if ( null !== $uploaded && false === $uploaded && ! empty( $notify['on_offsite_failure'] ) ) {
			$subject = '[AegisBackup] Offsite Upload Failed';
			$body = "AegisBackup attempted to upload a completed package offsite but failed.\n\n";
			$body .= 'Site: ' . home_url() . "\n";
			$body .= 'Package: ' . $zip_path . "\n";
			$body .= 'Message: ' . $upload_msg . "\n\n";
			$body .= "Time: " . gmdate( 'c' ) . "\n";
			wp_mail( $this->get_notify_recipient(), $subject, $body );
		}
	}

    public function on_backup_failed( $error, $state ) {
		$notify = $this->get_notify_settings();
		$msg = 'Backup failed.';
		if ( is_array( $error ) && ! empty( $error['message'] ) ) {
			$msg .= ' ' . (string) $error['message'];
		}
		$this->append_log( $msg );
		AB_Activity_Log::add( 'backup', 'failed', $msg, array( 'error' => $error, 'state' => $state ) );

		if ( ! empty( $notify['on_backup_failed'] ) ) {
			$subject = '[' . get_bloginfo( 'name' ) . '] Backup FAILED';
			$body  = "AegisBackup reports a backup failure.\n\n";
			$body .= 'Site: ' . home_url() . "\n";
			$body .= 'Time: ' . gmdate( 'c' ) . " (UTC)\n";
			if ( is_array( $error ) ) {
				$body .= 'Error: ' . wp_json_encode( $error, JSON_PRETTY_PRINT ) . "\n";
			}
			wp_mail( $this->get_notify_recipient(), $subject, $body );
		}

		$this->maybe_send_webhook( 'backup.failed', array( 'error' => $error, 'state' => $state ) );
    }


    public function on_restore_completed( $job_id, $state ) {
		$notify = $this->get_notify_settings();

		AB_Activity_Log::add( 'restore', 'success', 'Restore completed.', array( 'job_id' => (string) $job_id ) );
		$this->save_restore_report( (string) $job_id, $state, 'success', '' );

		if ( ! empty( $notify['on_restore_complete'] ) ) {
			$subject = '[AegisBackup] Restore Completed';
			$body = "Restore completed successfully.\n\n";
			$body .= 'Site: ' . home_url() . "\n";
			$body .= 'Job: ' . (string) $job_id . "\n";
			$body .= "Time: " . gmdate( 'c' ) . "\n";
			wp_mail( $this->get_notify_recipient(), $subject, $body );
		}

		$this->maybe_send_webhook( 'restore.completed', array( 'job_id' => (string) $job_id, 'state' => $state ) );
	}

    public function on_restore_failed( $job_id, $state, $error ) {
		$notify = $this->get_notify_settings();
		$msg = 'Restore failed.';
		if ( is_array( $error ) && ! empty( $error['message'] ) ) {
			$msg .= ' ' . (string) $error['message'];
		}
		$this->append_log( $msg );
		AB_Activity_Log::add( 'restore', 'failed', $msg, array( 'job_id' => (string) $job_id, 'error' => $error ) );
		$this->save_restore_report( (string) $job_id, $state, 'failed', $msg, $error );

		if ( ! empty( $notify['on_restore_failed'] ) ) {
			$subject = '[AegisBackup] Restore FAILED';
			$body  = "AegisBackup reports a restore failure.\n\n";
			$body .= 'Site: ' . home_url() . "\n";
			$body .= 'Job: ' . (string) $job_id . "\n";
			$body .= "Time: " . gmdate( 'c' ) . " (UTC)\n";
			if ( is_array( $error ) ) {
				$body .= "\nError:\n" . wp_json_encode( $error, JSON_PRETTY_PRINT ) . "\n";
			}
			wp_mail( $this->get_notify_recipient(), $subject, $body );
		}

		$this->maybe_send_webhook( 'restore.failed', array( 'job_id' => (string) $job_id, 'state' => $state, 'error' => $error ) );
    }

private function get_notify_settings() {
		$notify = get_option( 'aegisbackup_notification_settings', array() );
		return is_array( $notify ) ? $notify : array();
	}

	private function get_notify_recipient() {
		$notify = $this->get_notify_settings();
		$to = isset( $notify['recipient'] ) ? sanitize_email( (string) $notify['recipient'] ) : '';
		if ( empty( $to ) ) {
			$to = get_option( 'admin_email' );
		}
		return $to;
	}

	private function append_log( $message ) {
		$logs = get_option( 'aegisbackup_logs', array() );
		if ( ! is_array( $logs ) ) {
			$logs = array();
		}
		$line = '[' . gmdate( 'Y-m-d H:i:s' ) . ' UTC] ' . (string) $message;
		$logs[] = $line;
		if ( count( $logs ) > 1500 ) {
			$logs = array_slice( $logs, -1200 );
		}
		update_option( 'aegisbackup_logs', $logs, false );
	}

	private function upload_offsite( $zip_path, array $remote ) {
		$provider = isset( $remote['provider'] ) ? (string) $remote['provider'] : 'generic';
		if ( 's3' === $provider ) {
			return $this->upload_s3_put( $zip_path, $remote );
		}
		return $this->upload_generic( $zip_path, $remote );
	}

	private function upload_generic( $zip_path, array $remote ) {
		$location = isset( $remote['location'] ) ? (string) $remote['location'] : '';
		$port     = isset( $remote['port'] ) ? (int) $remote['port'] : 0;
		$user     = isset( $remote['username'] ) ? (string) $remote['username'] : '';
		$pass     = isset( $remote['password'] ) ? (string) $remote['password'] : '';

		if ( empty( $location ) ) {
			return array( 'success' => false, 'message' => 'Remote location is empty.' );
		}

		$filename = basename( $zip_path );
		if ( preg_match( '#^https?://#i', $location ) ) {
			$target = trailingslashit( $location ) . rawurlencode( $filename );
			$headers = array(
				'Content-Type' => 'application/zip',
			);
			if ( '' !== $user ) {
				$headers['Authorization'] = 'Basic ' . base64_encode( $user . ':' . $pass );
			}
			$fs = $this->ab_get_fs();
			$data = $fs ? $fs->get_contents( $zip_path ) : '';
			if ( '' === $data ) {
				return array( 'success' => false, 'message' => 'Could not read package for upload.' );
			}
			$args = array(
				'timeout' => 60,
				'redirection' => 2,
				'headers' => $headers,
				'body' => $data,
				'method' => 'PUT',
			);
			$res = wp_remote_request( $target, $args );
			if ( is_wp_error( $res ) ) {
				return array( 'success' => false, 'message' => $res->get_error_message() );
			}
			$code = (int) wp_remote_retrieve_response_code( $res );
			if ( $code >= 200 && $code < 300 ) {
				return array( 'success' => true, 'message' => 'Uploaded via HTTP PUT to ' . $target );
			}
			return array( 'success' => false, 'message' => 'HTTP PUT failed (HTTP ' . $code . ').' );
		}

		if ( ! function_exists( 'ssh2_connect' ) ) {
			return array( 'success' => false, 'message' => 'SFTP requires the ssh2 PHP extension (ssh2_connect not available).' );
		}

		$host = $location;
		if ( preg_match( '#^sftp://#i', $host ) ) {
			$host = preg_replace( '#^sftp://#i', '', $host );
		}
		$host = preg_replace( '#/.*$#', '', $host );
		if ( empty( $host ) ) {
			return array( 'success' => false, 'message' => 'SFTP host could not be determined from Location.' );
		}
		if ( $port <= 0 ) {
			$port = 22;
		}

		$conn = @ssh2_connect( $host, $port );
		if ( ! $conn ) {
			return array( 'success' => false, 'message' => 'Could not connect to SFTP host.' );
		}
		if ( ! @ssh2_auth_password( $conn, $user, $pass ) ) {
			return array( 'success' => false, 'message' => 'SFTP authentication failed.' );
		}
		$sftp = @ssh2_sftp( $conn );
		if ( ! $sftp ) {
			return array( 'success' => false, 'message' => 'Could not initialize SFTP subsystem.' );
		}

		$path = '/';
		if ( false !== strpos( $location, '/' ) ) {
			$maybe_path = preg_replace( '#^[^/]+#', '', $location );
			if ( $maybe_path ) {
				$path = $maybe_path;
			}
		}
		$path = rtrim( $path, '/' ) . '/';
		$remote_file = $path . $filename;
		$stream = @fopen( 'ssh2.sftp://' . (int) $sftp . $remote_file, 'w' ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen
		if ( ! $stream ) {
			return array( 'success' => false, 'message' => 'Could not open remote file for writing via SFTP.' );
		}
		$local = @fopen( $zip_path, 'rb' ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen
		if ( ! $local ) {
			@fclose( $stream ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
			return array( 'success' => false, 'message' => 'Could not open local package for reading.' );
		}
		while ( ! feof( $local ) ) {
			$buf = fread( $local, 1024 * 1024 ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fread
			if ( false === $buf ) {
				break;
			}
			fwrite( $stream, $buf ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fwrite
		}
		fclose( $local ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
		fclose( $stream ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
		return array( 'success' => true, 'message' => 'Uploaded via SFTP to ' . $host . $remote_file );
	}

	private function upload_s3_put( $zip_path, array $remote ) {
		$bucket_url = isset( $remote['s3_bucket_url'] ) ? (string) $remote['s3_bucket_url'] : '';
		$region     = isset( $remote['s3_region'] ) ? (string) $remote['s3_region'] : '';
		$access_key = isset( $remote['s3_access_key'] ) ? (string) $remote['s3_access_key'] : '';
		$secret_key = isset( $remote['s3_secret_key'] ) ? (string) $remote['s3_secret_key'] : '';
		$json       = isset( $remote['s3_json'] ) ? (string) $remote['s3_json'] : '';
		$prefix     = isset( $remote['s3_object_path'] ) ? (string) $remote['s3_object_path'] : '';

		if ( empty( $bucket_url ) ) {
			return array( 'success' => false, 'message' => 'S3 bucket URL is empty.' );
		}
		if ( empty( $region ) ) {
			$region = 'us-east-1';
		}

		if ( ( empty( $access_key ) || empty( $secret_key ) ) && ! empty( $json ) ) {
			$decoded = json_decode( $json, true );
			if ( is_array( $decoded ) ) {
				foreach ( array( 'access_key', 'accessKey', 'accessKeyId', 'accessKeyId', 'AccessKeyId' ) as $k ) {
					if ( empty( $access_key ) && ! empty( $decoded[ $k ] ) ) {
						$access_key = (string) $decoded[ $k ];
					}
				}
				foreach ( array( 'secret_key', 'secretKey', 'secretAccessKey', 'SecretAccessKey' ) as $k ) {
					if ( empty( $secret_key ) && ! empty( $decoded[ $k ] ) ) {
						$secret_key = (string) $decoded[ $k ];
					}
				}
			}
		}
		if ( empty( $access_key ) || empty( $secret_key ) ) {
			return array( 'success' => false, 'message' => 'S3 credentials are missing (access/secret).' );
		}

		$filename = basename( $zip_path );
		$prefix = ltrim( (string) $prefix, '/' );
		if ( '' !== $prefix && '/' !== substr( $prefix, -1 ) ) {
			$prefix .= '/';
		}
		$object_key = $prefix . $filename;

		$parsed = wp_parse_url( $bucket_url );
		if ( empty( $parsed['scheme'] ) || empty( $parsed['host'] ) ) {
			return array( 'success' => false, 'message' => 'Invalid S3 bucket URL.' );
		}
		$scheme = $parsed['scheme'];
		$host   = $parsed['host'];
		$base_path = isset( $parsed['path'] ) ? rtrim( (string) $parsed['path'], '/' ) : '';
		$path = $base_path . '/' . str_replace( '%2F', '/', rawurlencode( $object_key ) );
		$path = preg_replace( '#//+#', '/', $path );
		$endpoint = $scheme . '://' . $host . $path;

		$payload_hash = hash_file( 'sha256', $zip_path );
		$amz_date = gmdate( 'Ymd\THis\Z' );
		$date = gmdate( 'Ymd' );
		$service = 's3';
		$canonical_uri = $path;
		$canonical_querystring = '';
		$canonical_headers = 'host:' . $host . "\n" . 'x-amz-content-sha256:' . $payload_hash . "\n" . 'x-amz-date:' . $amz_date . "\n";
		$signed_headers = 'host;x-amz-content-sha256;x-amz-date';
		$canonical_request = "PUT\n{$canonical_uri}\n{$canonical_querystring}\n{$canonical_headers}\n{$signed_headers}\n{$payload_hash}";
		$algorithm = 'AWS4-HMAC-SHA256';
		$credential_scope = $date . '/' . $region . '/' . $service . '/aws4_request';
		$string_to_sign = $algorithm . "\n" . $amz_date . "\n" . $credential_scope . "\n" . hash( 'sha256', $canonical_request );
		$signing_key = $this->aws4_signing_key( $secret_key, $date, $region, $service );
		$signature = hash_hmac( 'sha256', $string_to_sign, $signing_key );
		$authorization = $algorithm . ' ' .
			'Credential=' . $access_key . '/' . $credential_scope . ', ' .
			'SignedHeaders=' . $signed_headers . ', ' .
			'Signature=' . $signature;

		$headers = array(
			'Authorization' => $authorization,
			'x-amz-date' => $amz_date,
			'x-amz-content-sha256' => $payload_hash,
			'Content-Type' => 'application/zip',
		);
		$fs = $this->ab_get_fs();
		$data = $fs ? $fs->get_contents( $zip_path ) : '';
		if ( '' === $data ) {
			return array( 'success' => false, 'message' => 'Could not read package for S3 upload.' );
		}
		$res = wp_remote_request( $endpoint, array(
			'timeout' => 60,
			'redirection' => 0,
			'headers' => $headers,
			'body' => $data,
			'method' => 'PUT',
		) );
		if ( is_wp_error( $res ) ) {
			return array( 'success' => false, 'message' => $res->get_error_message() );
		}
		$code = (int) wp_remote_retrieve_response_code( $res );
		if ( $code >= 200 && $code < 300 ) {
			return array( 'success' => true, 'message' => 'Uploaded to S3 as ' . $object_key );
		}
		$resp_body = wp_remote_retrieve_body( $res );
		$resp_body = is_string( $resp_body ) ? trim( wp_strip_all_tags( $resp_body ) ) : '';
		return array( 'success' => false, 'message' => 'S3 PUT failed (HTTP ' . $code . '). ' . ( $resp_body ? $resp_body : '' ) );
	}

	private function aws4_signing_key( $secret_key, $date, $region, $service ) {
		$k_date = hash_hmac( 'sha256', $date, 'AWS4' . $secret_key, true );
		$k_region = hash_hmac( 'sha256', $region, $k_date, true );
		$k_service = hash_hmac( 'sha256', $service, $k_region, true );
		$k_signing = hash_hmac( 'sha256', 'aws4_request', $k_service, true );
		return $k_signing;
	}

	public function run_retention() {
		$ret = get_option( 'aegisbackup_local_retention', array() );
		if ( ! is_array( $ret ) || empty( $ret ) ) {
			return;
		}

		$upload = wp_upload_dir();
		$base   = isset( $upload['basedir'] ) ? (string) $upload['basedir'] : '';
		if ( ! $base ) {
			return;
		}

		$now = time();

		$contexts = array(
			'migration'     => trailingslashit( $base ) . 'aegisbackup',              // package-* dirs
			'files'         => trailingslashit( $base ) . 'aegisbackup/file-backups', // zip backups
			'db_table'      => trailingslashit( $base ) . 'aegisbackup/table-backups',
			'db_snapshots'  => trailingslashit( $base ) . 'aegisbackup/db-tools',
		);

		foreach ( $contexts as $key => $dir ) {
			if ( empty( $ret[ $key ] ) || ! is_array( $ret[ $key ] ) ) {
				continue;
			}
			$days   = isset( $ret[ $key ]['days'] ) ? absint( $ret[ $key ]['days'] ) : 0;
			$action = isset( $ret[ $key ]['action'] ) ? sanitize_key( $ret[ $key ]['action'] ) : 'purge';

			if ( $days < 1 ) { $days = 1; }
			if ( $days > 90 ) { $days = 90; }
			if ( ! in_array( $action, array( 'purge', 'offsite' ), true ) ) { $action = 'purge'; }

			$cutoff = $now - ( $days * DAY_IN_SECONDS );

			$files = $this->collect_retention_files( $key, $dir );
			foreach ( $files as $path ) {
				$mt = @filemtime( $path );
				if ( $mt && $mt > $cutoff ) {
					continue;
				}

				if ( 'offsite' === $action ) {
					$ok = $this->upload_any_to_remote( $path );
					if ( $ok ) {
						@wp_delete_file( $path );
						$this->append_log( '[RETENTION] Uploaded then deleted: ' . basename( $path ) );
					} else {
						$this->append_log( '[RETENTION] Offsite upload failed; kept local: ' . basename( $path ) );
					}
				} else {
					@wp_delete_file( $path );
					$this->append_log( '[RETENTION] Deleted: ' . basename( $path ) );
				}
			}
		}
	}

	protected function collect_retention_files( $context, $dir ) {
		if ( ! is_dir( $dir ) ) {
			return array();
		}

		$items = array();

		if ( 'migration' === $context ) {
			$glob = glob( trailingslashit( $dir ) . 'package-*/*.zip' );
			if ( is_array( $glob ) ) {
				foreach ( $glob as $p ) {
					if ( is_file( $p ) ) { $items[] = $p; }
				}
			}
			return $items;
		}

		$glob = glob( trailingslashit( $dir ) . '*.{zip,sql,tar,gz}', GLOB_BRACE );
		if ( is_array( $glob ) ) {
			foreach ( $glob as $p ) {
				if ( is_file( $p ) ) { $items[] = $p; }
			}
		}
		return $items;
	}

	protected function upload_any_to_remote( $path ) {
		$remote = get_option( 'aegisbackup_remote_storage', array() );
		if ( ! is_array( $remote ) || empty( $remote['enabled'] ) ) {
			return false;
		}

		if ( ! is_file( $path ) ) {
			return false;
		}

		return $this->upload_backup_to_remote( $path );
	}

}