<?php
namespace AegisBackup\Restore;

defined( 'ABSPATH' ) || exit;

class AB_Post_Restore_Fixer {

    /**
     * Get WP_Filesystem instance for safe file operations.
     *
     * @return \WP_Filesystem_Base|null
     */
    protected function ab_filesystem() {
        global $wp_filesystem;
        if ( $wp_filesystem instanceof \WP_Filesystem_Base ) {
            return $wp_filesystem;
        }
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        // Initialize with direct access (restore runs inside wp-admin).
        $ok = WP_Filesystem();
        if ( $ok && $wp_filesystem instanceof \WP_Filesystem_Base ) {
            return $wp_filesystem;
        }
        return null;
    }
    /**
     * Sanitize a WordPress DB table prefix (identifiers only).
     *
     * @param string $prefix
     * @return string
     */
    protected function sanitize_db_prefix( $prefix ) {
        $prefix = (string) $prefix;
        // Allow only identifier characters.
        $prefix = preg_replace( '/[^A-Za-z0-9_]/', '', $prefix );
        // Common convention: prefixes end with underscore (wp_). Ensure it when non-empty.
        if ( '' !== $prefix && '_' !== substr( $prefix, -1 ) ) {
            $prefix .= '_';
        }
        return $prefix;
    }

    /**
     * Escape an SQL identifier (table/column name) for safe direct inclusion.
     *
     * @param string $identifier
     * @return string Escaped identifier wrapped in backticks.
     */
    protected function escape_sql_identifier( $identifier ) {
        $identifier = (string) $identifier;
        // Strict allowlist: only identifier characters.
        if ( '' === $identifier || ! preg_match( '/^[A-Za-z0-9_]+$/', $identifier ) ) {
            return '``';
        }
        // Double any backticks just in case (defense in depth).
        $identifier = str_replace( '`', '``', $identifier );
        return '`' . $identifier . '`';
    }

    /**
     * Replace a {{table}} token in a query with an escaped identifier.
     *
     * @param string $sql
     * @param string $table
     * @return string
     */
    protected function sql_with_table( $sql, $table ) {
        return str_replace( '{{table}}', $this->escape_sql_identifier( $table ), (string) $sql );
    }




    protected function counts_from_state( array $state, array $counts ) {

        if ( isset( $state['upload_options_normalized_total'] ) ) {
            $counts['upload_options_normalized'] = (int) $state['upload_options_normalized_total'];
        }
        if ( isset( $state['permission_fixes_attempted_total'] ) ) {
            $counts['permission_fixes_attempted'] = (int) $state['permission_fixes_attempted_total'];
        }
        if ( isset( $state['sr_repl'] ) ) {
            $counts['serialized_replacements'] = (int) $state['sr_repl'];
        }
        if ( isset( $state['verified'] ) ) {
            $counts['attachments_verified'] = (int) $state['verified'];
        }
        if ( isset( $state['missing'] ) ) {
            $counts['attachments_missing_files'] = (int) $state['missing'];
        }
        if ( isset( $state['corrupted'] ) ) {
            $counts['attachments_corrupted_meta'] = (int) $state['corrupted'];
        }
        if ( isset( $state['regenerated'] ) ) {
            $counts['thumbnails_regenerated'] = (int) $state['regenerated'];
        }

        if ( isset( $state['zero_bytes'] ) ) {
            $counts['zero_byte_files'] = (int) $state['zero_bytes'];
        }
        if ( isset( $state['caches_cleared'] ) ) {
            $counts['caches_cleared'] = (int) $state['caches_cleared'];
        }

        return $counts;
    }

    public function run_step3( array $state, array $args ) {
        $warnings = array();
        $counts   = array(
            'upload_options_normalized'     => 0,
            'permission_fixes_attempted'    => 0,
            'attachments_verified'          => 0,
            'attachments_missing_files'     => 0,
            'attachments_corrupted_meta'    => 0,
            'thumbnails_regenerated'        => 0,
            'serialized_replacements'       => 0,
            'zero_byte_files'               => 0,
            'caches_cleared'                => 0,
        );

        if ( ! isset( $state['phase'] ) ) {
            $state['phase'] = 'init';
        }

        if ( ! isset( $state['upload_options_normalized_total'] ) ) {
            $state['upload_options_normalized_total'] = 0;
        }
        if ( ! isset( $state['permission_fixes_attempted_total'] ) ) {
            $state['permission_fixes_attempted_total'] = 0;
        }

        if ( ! isset( $state['verified'] ) ) { $state['verified'] = 0; }
        if ( ! isset( $state['missing'] ) ) { $state['missing'] = 0; }
        if ( ! isset( $state['corrupted'] ) ) { $state['corrupted'] = 0; }
        if ( ! isset( $state['regenerated'] ) ) { $state['regenerated'] = 0; }
        if ( ! isset( $state['zero_bytes'] ) ) { $state['zero_bytes'] = 0; }
        if ( ! isset( $state['caches_cleared'] ) ) { $state['caches_cleared'] = 0; }
        $old_domain = isset( $args['old_domain'] ) ? trim( (string) $args['old_domain'] ) : '';
        $new_domain = isset( $args['new_domain'] ) ? trim( (string) $args['new_domain'] ) : '';
        $new_prefix = isset( $args['new_prefix'] ) ? (string) $args['new_prefix'] : '';

        $regen_thumbs = ! empty( $args['regen_thumbs'] );
        $fix_perms    = array_key_exists( 'fix_perms', $args ) ? (bool) $args['fix_perms'] : true;
        $update_guid  = ! empty( $args['update_guid'] );

        global $wpdb;
        $prefix = $new_prefix ? preg_replace( '/[^A-Za-z0-9_]/', '', $new_prefix ) : $wpdb->prefix;

        switch ( $state['phase'] ) {
            case 'init':
                $state['phase'] = 'normalize_upload_options';
                return array(
                    'done'     => false,
                    'log'      => 'Step 3: Starting finalization (safe to run on live sites).',
                    'warnings' => $warnings,
                    'counts'   => $this->counts_from_state( $state, $counts ),
                    'state'    => $state,
                );

            case 'normalize_upload_options':
                $updated = (int) $this->normalize_upload_options( $wpdb, $prefix );
                $state['upload_options_normalized_total'] = (int) $updated;
                $counts['upload_options_normalized'] = (int) $state['upload_options_normalized_total'];
                $state['phase'] = 'serialized_replace';
                return array(
                    'done'     => false,
                    'log'      => 'Step 3: Upload options normalized.',
                    'warnings' => $warnings,
                    'counts'   => $this->counts_from_state( $state, $counts ),
                    'state'    => $state,
                );

            case 'serialized_replace':
                if ( '' !== $old_domain && '' !== $new_domain && $old_domain !== $new_domain ) {
                    $rep = $this->serialized_safe_db_replace_step( $state, $old_domain, $new_domain, $prefix, $update_guid );
                    $counts['serialized_replacements'] = isset( $rep['replacements'] ) ? (int) $rep['replacements'] : 0;
                    if ( ! empty( $rep['done'] ) ) {
                        $state['phase'] = 'verify_attachments';
                        return array(
                            'done'     => false,
                            'log'      => sprintf( 'Step 3: Serialized-safe URL replacement complete. Replacements=%d.', (int) $counts['serialized_replacements'] ),
                            'warnings' => $warnings,
                            'counts'   => $this->counts_from_state( $state, $counts ),
                            'state'    => $state,
                        );
                    }

                    return array(
                        'done'     => false,
                        'log'      => isset( $rep['log'] ) ? (string) $rep['log'] : 'Step 3: Serialized-safe URL replacement running…',
                        'warnings' => $warnings,
                        'counts'   => $this->counts_from_state( $state, $counts ),
                        'state'    => $state,
                    );
                }

                $state['phase'] = 'verify_attachments';
                return array(
                    'done'     => false,
                    'log'      => 'Step 3: URL replacement skipped (old/new domain not provided).',
                    'warnings' => $warnings,
                    'counts'   => $this->counts_from_state( $state, $counts ),
                    'state'    => $state,
                );

            case 'verify_attachments':
                $res = $this->verify_attachments_step( $state, array( 'regen_thumbs' => $regen_thumbs ), 80, 1.0, $prefix );
                $counts['attachments_verified']       = isset( $res['verified'] ) ? (int) $res['verified'] : 0;
                $counts['attachments_missing_files']  = isset( $res['missing'] ) ? (int) $res['missing'] : 0;
                $counts['attachments_corrupted_meta'] = isset( $res['corrupted'] ) ? (int) $res['corrupted'] : 0;
                $counts['thumbnails_regenerated']     = isset( $res['regenerated'] ) ? (int) $res['regenerated'] : 0;

                if ( ! empty( $res['done'] ) ) {
                    $state['phase'] = 'permissions';
                    return array(
                        'done'     => false,
                        'log'      => isset( $res['log'] ) ? (string) $res['log'] : 'Step 3: Attachment verification complete.',
                        'warnings' => $warnings,
                        'counts'   => $this->counts_from_state( $state, $counts ),
                        'state'    => $state,
                    );
                }

                return array(
                    'done'     => false,
                    'log'      => isset( $res['log'] ) ? (string) $res['log'] : 'Step 3: Verifying attachments…',
                    'warnings' => $warnings,
                    'counts'   => $this->counts_from_state( $state, $counts ),
                    'state'    => $state,
                );

            case 'permissions':
                if ( $fix_perms ) {
                    $attempts = (int) $this->permission_sanity_check( 1.0, 400 );
                    $state['permission_fixes_attempted_total'] = (int) $attempts;
                    $counts['permission_fixes_attempted'] = (int) $state['permission_fixes_attempted_total'];
                    $state['phase'] = 'cache_clear';
                    return array(
                        'done'     => false,
                        'log'      => 'Step 3: Uploads permission sanity check attempted.',
                        'warnings' => $warnings,
                        'counts'   => $this->counts_from_state( $state, $counts ),
                        'state'    => $state,
                    );
                }

                $state['phase'] = 'cache_clear';
                return array(
                    'done'     => false,
                    'log'      => 'Step 3: Permission check skipped (disabled).',
                    'warnings' => $warnings,
                    'counts'   => $this->counts_from_state( $state, $counts ),
                    'state'    => $state,
                );

            case 'cache_clear':
                $cache = $this->clear_local_caches_once( $state );
                if ( ! empty( $cache['warnings'] ) ) {
                    $warnings = array_merge( $warnings, (array) $cache['warnings'] );
                }
                $state['phase'] = 'done';
                return array(
                    'done'     => false,
                    'log'      => isset( $cache['log'] ) ? (string) $cache['log'] : 'Step 3: Cache clear complete.',
                    'warnings' => $warnings,
                    'counts'   => $this->counts_from_state( $state, $counts ),
                    'state'    => $state,
                );

            case 'done':
            default:
                return array(
                    'done'     => true,
                    'log'      => 'Step 3: Finalization completed.',
                    'warnings' => $warnings,
                    'counts'   => $this->counts_from_state( $state, $counts ),
                    'state'    => $state,
                );
        }
    }

    protected function serialized_safe_db_replace_step( array &$state, $old, $new, $prefix, $update_guid = false ) {
        global $wpdb;
        $old = (string) $old;
        $new = (string) $new;
        $prefix = $this->sanitize_db_prefix( $prefix );

        if ( ! isset( $state['sr_phase'] ) ) {
            $state['sr_phase'] = 'options';
            $state['sr_last']  = 0;
            $state['sr_repl']  = 0;
        }

        $phase = (string) $state['sr_phase'];
        $last  = (int) $state['sr_last'];

        $table = '';
        $idcol = '';
        $cols  = array();

        if ( 'options' === $phase ) {
            $table = $prefix . 'options';
            $idcol = 'option_id';
            $cols  = array( 'option_value' );
        } elseif ( 'postmeta' === $phase ) {
            $table = $prefix . 'postmeta';
            $idcol = 'meta_id';
            $cols  = array( 'meta_value' );
        } else {
            $table = $prefix . 'posts';
            $idcol = 'ID';
            $cols  = array( 'post_content', 'post_excerpt' );
            if ( $update_guid ) {
                $cols[] = 'guid';
            }
        }

        // Cache table existence checks during restore runs (prevents repeated SHOW TABLES calls).
        $cache_key = 'ab_tbl_exists_' . md5( (string) $table );
        $exists = wp_cache_get( $cache_key, 'aegisbackup' );
        if ( false === $exists ) {
            $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            wp_cache_set( $cache_key, $exists, 'aegisbackup', 300 );
        }

        if ( empty( $exists ) ) {
            if ( 'options' === $phase ) {
                $state['sr_phase'] = 'postmeta';
            } elseif ( 'postmeta' === $phase ) {
                $state['sr_phase'] = 'posts';
            } else {
                return array( 'done' => true, 'replacements' => (int) $state['sr_repl'], 'log' => 'Step 3: URL replace skipped (tables missing).' );
            }
            $state['sr_last'] = 0;
            return array( 'done' => false, 'replacements' => (int) $state['sr_repl'], 'log' => 'Step 3: URL replace continuing…' );
        }

        $limit = 80;

if ( 'options' === $phase ) {
    $rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->prepare(
            "SELECT `option_id` AS id, `option_value` FROM %i WHERE `option_id` > %d ORDER BY `option_id` ASC LIMIT %d",
            $table,
            $last,
            $limit
        ),
        ARRAY_A
    );
} elseif ( 'postmeta' === $phase ) {
    $rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->prepare(
            "SELECT `meta_id` AS id, `meta_value` FROM %i WHERE `meta_id` > %d ORDER BY `meta_id` ASC LIMIT %d",
            $table,
            $last,
            $limit
        ),
        ARRAY_A
    );
} else {
    // Always select guid; it is only updated when $update_guid is true.
    $rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->prepare(
            "SELECT `ID` AS id, `post_content`, `post_excerpt`, `guid` FROM %i WHERE `ID` > %d ORDER BY `ID` ASC LIMIT %d",
            $table,
            $last,
            $limit
        ),
        ARRAY_A
    );
}

        if ( empty( $rows ) ) {
            if ( 'options' === $phase ) {
                $state['sr_phase'] = 'postmeta';
            } elseif ( 'postmeta' === $phase ) {
                $state['sr_phase'] = 'posts';
            } else {
                return array( 'done' => true, 'replacements' => (int) $state['sr_repl'], 'log' => sprintf( 'Step 3: URL replace finished. Total replacements=%d.', (int) $state['sr_repl'] ) );
            }
            $state['sr_last'] = 0;
            return array( 'done' => false, 'replacements' => (int) $state['sr_repl'], 'log' => 'Step 3: URL replace moving to next table…' );
        }

        $did = 0;
        foreach ( $rows as $r ) {
            $id = isset( $r['id'] ) ? (int) $r['id'] : 0;
            if ( $id <= 0 ) { continue; }
            $state['sr_last'] = $id;

            foreach ( $cols as $col ) {
                if ( ! array_key_exists( $col, $r ) ) { continue; }
                $val = (string) $r[ $col ];
                if ( '' === $val || false === strpos( $val, $old ) ) {
                    continue;
                }
                $new_val = $this->serialized_safe_replace_value( $old, $new, $val );
                if ( $new_val !== $val ) {
                    $wpdb->update( $table, array( $col => $new_val ), array( $idcol => $id ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $state['sr_repl'] = (int) $state['sr_repl'] + 1;
                    $did++;
                }
            }
        }

        return array(
            'done' => false,
            'replacements' => (int) $state['sr_repl'],
            'log' => sprintf( 'Step 3: URL replace running… updated %d rows (total=%d).', (int) $did, (int) $state['sr_repl'] ),
        );
    }

    protected function serialized_safe_replace_value( $old, $new, $value ) {
        $old = (string) $old;
        $new = (string) $new;

        if ( function_exists( 'is_serialized' ) && is_serialized( $value ) ) {
            $un = @unserialize( $value );
            if ( false !== $un || 'b:0;' === $value ) {
                $fixed = $this->deep_replace( $old, $new, $un );
                return serialize( $fixed );
            }
        }

        return str_replace( $old, $new, (string) $value );
    }

    protected function deep_replace( $old, $new, $data ) {
        if ( is_string( $data ) ) {
            return str_replace( $old, $new, $data );
        }
        if ( is_array( $data ) ) {
            foreach ( $data as $k => $v ) {
                $data[ $k ] = $this->deep_replace( $old, $new, $v );
            }
            return $data;
        }
        if ( is_object( $data ) ) {
            foreach ( $data as $k => $v ) {
                $data->$k = $this->deep_replace( $old, $new, $v );
            }
            return $data;
        }
        return $data;
    }

    public function normalize_upload_options( $wpdb, $prefix ) {
        $count = 0;
        $prefix = $this->sanitize_db_prefix( $prefix );

        $opt_table = $prefix . 'options';
        // Cache table existence checks during restore runs.
        $cache_key = 'ab_tbl_exists_' . md5( (string) $opt_table );
        $exists = wp_cache_get( $cache_key, 'aegisbackup' );
        if ( false === $exists ) {
            $exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $opt_table ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            wp_cache_set( $cache_key, $exists, 'aegisbackup', 300 );
        }

        if ( empty( $exists ) ) {
            return 0;
        }

$rows = $wpdb->query( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $wpdb->prepare(
        "UPDATE %i SET option_value = %s WHERE option_name IN ('upload_path','upload_url_path') AND option_value <> %s",
        $opt_table,
        '',
        ''
    )
);
if ( false !== $rows ) {
    $count += (int) $rows;
}

$rows2 = $wpdb->query( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $wpdb->prepare(
        "UPDATE %i SET option_value = %s WHERE option_name = 'uploads_use_yearmonth_folders' AND option_value NOT IN ('0','1')",
        $opt_table,
        '1'
    )
);
if ( false !== $rows2 ) {
    $count += (int) $rows2;
}

return (int) $count;
    }

    public function verify_attachments_step( array &$state, array $opts, $batch = 80, $max_seconds = 1.0, $prefix = '' ) {
        $start = microtime( true );
        $regen = ! empty( $opts['regen_thumbs'] );

        if ( ! isset( $state['att_last_id'] ) ) {
            $state['att_last_id'] = 0;
        }
        if ( ! isset( $state['verified'] ) ) {
            $state['verified'] = 0;
        }
        if ( ! isset( $state['missing'] ) ) {
            $state['missing'] = 0;
        }
        if ( ! isset( $state['zero_bytes'] ) ) {
            $state['zero_bytes'] = 0;
        }
        if ( ! isset( $state['corrupted'] ) ) {
            $state['corrupted'] = 0;
        }
        if ( ! isset( $state['regenerated'] ) ) {
            $state['regenerated'] = 0;
        }
        if ( ! isset( $state['missing_samples'] ) || ! is_array( $state['missing_samples'] ) ) {
            $state['missing_samples'] = array();
        }
        if ( ! isset( $state['corrupted_samples'] ) || ! is_array( $state['corrupted_samples'] ) ) {
            $state['corrupted_samples'] = array();
        }

        global $wpdb;

        if ( ! function_exists( 'wp_upload_dir' ) ) {
            require_once ABSPATH . 'wp-includes/functions.php';
        }

        $upload = wp_upload_dir();
        $basedir = isset( $upload['basedir'] ) ? (string) $upload['basedir'] : '';
        if ( empty( $basedir ) || ! is_dir( $basedir ) ) {
            return array(
                'done' => true,
                'log'  => 'Step 3: Uploads directory not found. Skipping attachment verification.',
                'verified' => (int) $state['verified'],
                'missing' => (int) $state['missing'],
                'corrupted' => (int) $state['corrupted'],
                'regenerated' => (int) $state['regenerated'],
            );
        }

$pt   = (string) $wpdb->posts;
$pm   = (string) $wpdb->postmeta;
$last = (int) $state['att_last_id'];

$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $wpdb->prepare(
        "SELECT p.ID AS id, pm.meta_value AS relfile, m.meta_value AS meta
         FROM %i p
         LEFT JOIN %i pm ON (pm.post_id = p.ID AND pm.meta_key = '_wp_attached_file')
         LEFT JOIN %i m  ON (m.post_id  = p.ID AND m.meta_key  = '_wp_attachment_metadata')
         WHERE p.post_type = 'attachment' AND p.ID > %d
         ORDER BY p.ID ASC
         LIMIT %d",
        $pt,
        $pm,
        $pm,
        $last,
        (int) $batch
    ),
    ARRAY_A
);

        if ( empty( $rows ) ) {
            $log = sprintf(
                'Step 3: Attachment verification complete. Verified=%d, Missing=%d, CorruptedMeta=%d, ThumbsRegenerated=%d.',
                (int) $state['verified'],
                (int) $state['missing'],
                (int) $state['corrupted'],
                (int) $state['regenerated']
            );
            return array(
                'done' => true,
                'log'  => $log,
                'verified' => (int) $state['verified'],
                'missing' => (int) $state['missing'],
                'corrupted' => (int) $state['corrupted'],
                'regenerated' => (int) $state['regenerated'],
            );
        }

        $processed = 0;
        foreach ( $rows as $r ) {
            $id = isset( $r['id'] ) ? (int) $r['id'] : 0;
            if ( $id <= 0 ) {
                continue;
            }
            $state['att_last_id'] = $id;
            $rel = isset( $r['relfile'] ) ? (string) $r['relfile'] : '';
            $rel = ltrim( str_replace( array( '\\', chr( 0 ) ), array( '/', '' ), $rel ), '/' );
            $path = '';
            if ( '' !== $rel ) {
                if ( 0 === strpos( $rel, $basedir ) ) {
                    $path = $rel;
                } else {
                    $path = rtrim( $basedir, '/\\' ) . '/' . $rel;
                }
            }

            $size = ( '' !== $path && is_file( $path ) ) ? @filesize( $path ) : false;
            $exists = ( '' !== $path && is_file( $path ) && false !== $size );
            $zero = ( $exists && 0 === (int) $size );

            if ( ! $exists || $zero ) {
                $state['missing'] = (int) $state['missing'] + 1;
                if ( $zero ) {
                    $state['zero_bytes'] = (int) $state['zero_bytes'] + 1;
                }

                if ( count( $state['missing_samples'] ) < 25 ) {
                    $state['missing_samples'][] = array(
                        'id' => $id,
                        'rel' => $rel,
                        'path' => $path,
                        'zero' => $zero ? 1 : 0,
                    );
                }
            }

            $meta_raw = isset( $r['meta'] ) ? $r['meta'] : '';
            $meta_ok = true;
            if ( '' === $meta_raw ) {
                $meta_ok = false;
            } else {
                $meta = @maybe_unserialize( $meta_raw );
                if ( ! is_array( $meta ) ) {
                    $meta_ok = false;
                }
            }

            if ( ! $meta_ok ) {
                $state['corrupted'] = (int) $state['corrupted'] + 1;
                if ( count( $state['corrupted_samples'] ) < 25 ) {
                    $state['corrupted_samples'][] = array(
                        'id' => $id,
                        'rel' => $rel,
                    );
                }
            }

            if ( $regen && $exists ) {
                $mime = get_post_mime_type( $id );
                if ( $mime && 0 === strpos( (string) $mime, 'image/' ) ) {
                    if ( ! $meta_ok ) {
                        $file = get_attached_file( $id );
                        if ( $file && is_file( $file ) ) {
                            require_once ABSPATH . 'wp-admin/includes/image.php';
                            $new_meta = wp_generate_attachment_metadata( $id, $file );
                            if ( is_array( $new_meta ) && ! empty( $new_meta ) ) {
                                wp_update_attachment_metadata( $id, $new_meta );
                                $state['regenerated'] = (int) $state['regenerated'] + 1;
                            }
                        }
                    }
                }
            }

            $state['verified'] = (int) $state['verified'] + 1;
            $processed++;

            if ( ( microtime( true ) - $start ) >= (float) $max_seconds ) {
                break;
            }
        }

        $log = sprintf(
            'Step 3: Verifying attachments… processed %d (total verified=%d).',
            (int) $processed,
            (int) $state['verified']
        );

        return array(
            'done' => false,
            'log'  => $log,
            'verified' => (int) $state['verified'],
            'missing' => (int) $state['missing'],
            'corrupted' => (int) $state['corrupted'],
            'regenerated' => (int) $state['regenerated'],
        );
    }

    public function permission_sanity_check( $max_seconds = 1.0, $max_items = 400 ) {
        $start = microtime( true );
        $attempts = 0;

        $upload = wp_upload_dir();
        $basedir = isset( $upload['basedir'] ) ? (string) $upload['basedir'] : '';
        if ( empty( $basedir ) || ! is_dir( $basedir ) ) {
            return 0;
        }

        $fs = $this->ab_filesystem();
        if ( $fs ) {
            $fs->chmod( $basedir, 0755 );
        }
        $attempts++;

        
        $attempts++;

        try {
            $it = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator( $basedir, \FilesystemIterator::SKIP_DOTS ),
                \RecursiveIteratorIterator::SELF_FIRST
            );

            foreach ( $it as $path => $info ) {
                if ( $attempts >= (int) $max_items ) {
                    break;
                }
                if ( ( microtime( true ) - $start ) >= (float) $max_seconds ) {
                    break;
                }
                $p = (string) $path;
                if ( $info->isDir() ) {
                    if ( $fs ) {
                        $fs->chmod( $p, 0755 );
                    }
                } else {
                    if ( $fs ) {
                        $fs->chmod( $p, 0644 );
                    }
                }
                $attempts++;
            }
        } catch ( \Exception $e ) {
        }

        return (int) $attempts;
    }

    protected function clear_local_caches_once( array &$state ) {
        if ( ! empty( $state['cache_clear_ran'] ) ) {
            return array( 'log' => 'Step 3: Cache clear already ran (skipped).', 'warnings' => array() );
        }

        $state['cache_clear_ran'] = 1;

        if ( ! class_exists( '\\AegisBackup\\Restore\\AB_Cache_Clearer' ) ) {
            $path = defined( 'AEGISBACKUP_DIR' ) ? AEGISBACKUP_DIR . 'includes/restore/class-ab-cache-clearer.php' : '';
            if ( $path && file_exists( $path ) ) {
                require_once $path;
            }
        }

        if ( class_exists( '\\AegisBackup\\Restore\\AB_Cache_Clearer' ) ) {
            $c = new AB_Cache_Clearer();
            $res = $c->clear_local_caches();

            $added = isset( $res['count'] ) ? (int) $res['count'] : 0;
            $state['caches_cleared'] = isset( $state['caches_cleared'] ) ? ( (int) $state['caches_cleared'] + $added ) : $added;

            $lines = isset( $res['log_lines'] ) && is_array( $res['log_lines'] ) ? $res['log_lines'] : array();
            $msg = 'Step 3: Cleared local caches. Actions=' . (int) $added;
            if ( ! empty( $lines ) ) {
                $msg .= ' | ' . implode( ' ', array_map( 'strval', $lines ) );
            }

            return array(
                'log' => $msg,
                'warnings' => isset( $res['warnings'] ) && is_array( $res['warnings'] ) ? $res['warnings'] : array(),
            );
        }

        return array( 'log' => 'Step 3: Cache clearer not available (skipped).', 'warnings' => array() );
    }
}