<?php
namespace AegisBackup\Restore;

defined( 'ABSPATH' ) || exit;

require_once AEGISBACKUP_DIR . 'includes/restore/class-ab-db-restorer.php';
require_once AEGISBACKUP_DIR . 'includes/restore/class-ab-file-restorer.php';
require_once AEGISBACKUP_DIR . 'includes/restore/class-ab-preflight.php';
require_once AEGISBACKUP_DIR . 'includes/restore/class-ab-migration-report.php';
require_once AEGISBACKUP_DIR . 'includes/restore/class-ab-post-restore-fixer.php';
require_once AEGISBACKUP_DIR . 'includes/libs/class-ab-wpconfig-writer.php';

class AB_Restore_Manager {

	public function append_settings_log( $message, $job_id = '' ) {
		$job_id = (string) $job_id;
		$prefix = $job_id ? ('[Restore ' . $job_id . '] ') : '[Restore] ';
		$line = '[' . gmdate( 'Y-m-d H:i:s' ) . ' UTC] ' . $prefix . (string) $message;

		$logs = get_option( 'aegisbackup_logs', array() );
		if ( ! is_array( $logs ) ) {
			$logs = array();
		}
		$logs[] = $line;
		if ( count( $logs ) > 1500 ) {
			$logs = array_slice( $logs, -1200 );
		}
		update_option( 'aegisbackup_logs', $logs, false );
	}
    const JOB_OPTION_PREFIX = 'aegisbackup_restore_job_';
    const JOBS_INDEX_OPTION = 'aegisbackup_restore_jobs_index';

	protected function upsert_job_index( $job_id, array $state ) {
		$job_id = (string) $job_id;
		if ( '' === $job_id ) {
			return;
		}
		$index = get_option( self::JOBS_INDEX_OPTION, array() );
		if ( ! is_array( $index ) ) {
			$index = array();
		}

		$args = isset( $state['args'] ) && is_array( $state['args'] ) ? (array) $state['args'] : array();
		$pkg  = isset( $args['package_path'] ) ? (string) $args['package_path'] : '';
		$pkg_label = $pkg ? basename( untrailingslashit( $pkg ) ) : '';

		$index[ $job_id ] = array(
			'job_id'    => $job_id,
			'user_id'   => isset( $state['user_id'] ) ? (int) $state['user_id'] : 0,
			'type'      => isset( $state['type'] ) ? (string) $state['type'] : 'migration',
			'created'   => isset( $state['created'] ) ? (int) $state['created'] : time(),
			'updated'   => isset( $state['updated'] ) ? (int) $state['updated'] : time(),
			'last_tick' => isset( $state['last_tick'] ) ? (int) $state['last_tick'] : 0,
			'status'    => isset( $state['status'] ) ? (string) $state['status'] : '',
			'phase'     => isset( $state['phase'] ) ? (string) $state['phase'] : '',
			'progress'  => isset( $state['progress'] ) ? (int) $state['progress'] : 0,
			'last_log'  => isset( $state['last_log'] ) ? (string) $state['last_log'] : '',
			'package'   => $pkg_label,
		);

		if ( count( $index ) > 60 ) {
			uasort( $index, function( $a, $b ) {
				$au = isset( $a['updated'] ) ? (int) $a['updated'] : 0;
				$bu = isset( $b['updated'] ) ? (int) $b['updated'] : 0;
				if ( $au === $bu ) { return 0; }
				return ( $au > $bu ) ? -1 : 1;
			} );
			$index = array_slice( $index, 0, 50, true );
		}

		update_option( self::JOBS_INDEX_OPTION, $index, false );
	}

	public function delete_job_from_index( $job_id ) {
		$job_id = (string) $job_id;
		$index = get_option( self::JOBS_INDEX_OPTION, array() );
		if ( ! is_array( $index ) ) {
			return;
		}
		if ( isset( $index[ $job_id ] ) ) {
			unset( $index[ $job_id ] );
			update_option( self::JOBS_INDEX_OPTION, $index, false );
		}
	}

	protected function append_job_log( array $state, $line ) {
		$line = trim( (string) $line );
		if ( '' === $line ) {
			return $state;
		}

		if ( ! isset( $state['log'] ) || ! is_array( $state['log'] ) ) {
			$state['log'] = array();
		}

		$last = end( $state['log'] );
		if ( $last !== $line ) {
			$state['log'][] = $line;
		}

		if ( count( $state['log'] ) > 250 ) {
			$state['log'] = array_slice( $state['log'], -250 );
		}
		return $state;
	}

	protected function persist_job_state( $job_id, array $state ) {
		$state['last_tick'] = time();
		$state['updated']   = $state['last_tick'];

		if ( isset( $state['last_log'] ) ) {
			$state = $this->append_job_log( $state, (string) $state['last_log'] );
			$this->append_settings_log( (string) $state['last_log'], (string) $job_id );
		}

if ( isset( $state['status'] ) && 'error' === $state['status'] && empty( $state['failed_hook_fired'] ) ) {
	$state['failed_hook_fired'] = 1;
	$error = array(
		'code'    => isset( $state['error_code'] ) ? (string) $state['error_code'] : 'restore_error',
		'message' => isset( $state['last_log'] ) ? (string) $state['last_log'] : 'Restore failed.',
	);
	do_action( 'aegisbackup_restore_failed', (string) $job_id, $state, $error );
}

if ( ! isset( $state['activity'] ) || ! is_array( $state['activity'] ) ) {
	$state['activity'] = array();
}
if ( empty( $state['activity']['finished'] ) && isset( $state['phase'] ) && 'done' === (string) $state['phase'] ) {
	if ( ! class_exists( '\\AegisBackup\\Libs\\AB_Activity_Log' ) && defined( 'AEGISBACKUP_DIR' ) ) {
		require_once AEGISBACKUP_DIR . 'includes/libs/class-ab-activity-log.php';
	}
	if ( class_exists( '\\AegisBackup\\Libs\\AB_Activity_Log' ) ) {
		$st = isset( $state['status'] ) ? (string) $state['status'] : '';
		$ok = ( 'error' !== $st && 'failed' !== $st && 'stopped' !== $st );
		$pkg = isset( $state['args']['package_path'] ) ? basename( (string) $state['args']['package_path'] ) : '';
		\AegisBackup\Libs\AB_Activity_Log::add(
			'restore',
			$ok ? 'done' : 'failed',
			isset( $state['last_log'] ) ? (string) $state['last_log'] : ( $ok ? 'Restore completed.' : 'Restore failed.' ),
			array( 'job_id' => (string) $job_id, 'package' => $pkg, 'progress' => isset( $state['progress'] ) ? (int) $state['progress'] : 0 )
		);
	}
	$state['activity']['finished'] = time();
}

update_option( self::JOB_OPTION_PREFIX . $job_id, $state, false );
		$this->upsert_job_index( $job_id, $state );
		return $state;
	}

	protected function cleanup_extracted_work_dir( array $state ) {
		$dir = isset( $state['tmp_dir'] ) ? (string) $state['tmp_dir'] : '';
		if ( '' === $dir || ! is_dir( $dir ) ) {
			return $state;
		}

		$uploads = wp_upload_dir();
		$base = trailingslashit( (string) $uploads['basedir'] ) . 'aegisbackup/';
		$dir_n = wp_normalize_path( $dir );
		$base_n = wp_normalize_path( $base );

		if ( 0 !== strpos( $dir_n, $base_n ) ) {
			return $state;
		}

		if ( false === strpos( basename( $dir_n ), 'tmp-' ) && false === strpos( $dir_n, '/restore/' ) ) {
			return $state;
		}

		$this->rmdir_recursive( $dir );
		$state['cleanup'] = array( 'work_deleted' => true, 'when' => time() );
		unset( $state['tmp_dir'] );
		return $state;
	}

	protected function rmdir_recursive( $dir ) {
    $dir = (string) $dir;
    if ( '' === $dir || ! is_dir( $dir ) ) {
        return;
    }

    if ( ! function_exists( 'WP_Filesystem' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    global $wp_filesystem;
    if ( empty( $wp_filesystem ) ) {
        WP_Filesystem();
    }

    if ( is_object( $wp_filesystem ) && method_exists( $wp_filesystem, 'delete' ) ) {
        // delete() handles both files and directories recursively when $recursive = true.
        $wp_filesystem->delete( $dir, true );
        return;
    }

    // Fallback: best-effort delete files only (avoid direct rmdir/unlink).
    $items = @scandir( $dir );
    if ( ! is_array( $items ) ) {
        return;
    }
    foreach ( $items as $item ) {
        if ( '.' === $item || '..' === $item ) {
            continue;
        }
        $path = $dir . DIRECTORY_SEPARATOR . $item;
        if ( is_dir( $path ) ) {
            $this->rmdir_recursive( $path );
        } else {
            wp_delete_file( $path );
        }
    }
}

    protected function get_target_wpdb( array $args ) {
        global $wpdb;

        $mode = isset( $args['mode'] ) ? (string) $args['mode'] : 'existing';
        if ( 'newdb' !== $mode ) {
            return array( 'wpdb' => $wpdb, 'used_newdb' => false, 'error' => '' );
        }

        $db_name = isset( $args['db_name'] ) ? (string) $args['db_name'] : '';
        $db_user = isset( $args['db_user'] ) ? (string) $args['db_user'] : '';
        $db_pass = isset( $args['db_pass'] ) ? (string) $args['db_pass'] : '';
        $db_host = isset( $args['db_host'] ) ? (string) $args['db_host'] : 'localhost';

        if ( '' === $db_name || '' === $db_user ) {
            return array( 'wpdb' => $wpdb, 'used_newdb' => false, 'error' => 'Missing DB credentials.' );
        }

        $newdb = new \wpdb( $db_user, $db_pass, $db_name, $db_host );
        $newdb->hide_errors();
        $ok = $newdb->check_connection();
        if ( ! $ok ) {
            return array( 'wpdb' => $wpdb, 'used_newdb' => false, 'error' => 'Failed to connect to the provided database.' );
        }

        $r = $newdb->get_var( 'SELECT 1' );
        if ( null === $r ) {
            return array( 'wpdb' => $wpdb, 'used_newdb' => false, 'error' => 'Connected but failed to query the provided database.' );
        }

        return array( 'wpdb' => $newdb, 'used_newdb' => true, 'error' => '' );
    }

    protected function swap_global_wpdb( $temp ) {
        global $wpdb;
        $orig = $wpdb;
        $wpdb = $temp;
        return $orig;
    }


    public function start_restore_job( array $args ) {
        $job_id = 'abr_' . wp_generate_password( 12, false, false );

        $state = array(
            'job_id' => $job_id,
            'status' => 'running',
            'user_id' => function_exists( 'get_current_user_id' ) ? (int) get_current_user_id() : 0,
            'type' => 'migration',
            'created' => time(),
            'progress' => 0,
            'phase' => 'validate',
            'args' => array(
                'package_path' => isset( $args['package_path'] ) ? (string) $args['package_path'] : '',
                'mode' => isset( $args['mode'] ) ? sanitize_key( (string) $args['mode'] ) : 'existing',
                'new_prefix' => isset( $args['new_prefix'] ) ? (string) $args['new_prefix'] : '',
                'old_domain' => isset( $args['old_domain'] ) ? (string) $args['old_domain'] : '',
                'new_domain' => isset( $args['new_domain'] ) ? (string) $args['new_domain'] : '',
				'src_root' => isset( $args['src_root'] ) ? (string) $args['src_root'] : '',
				'dst_root' => isset( $args['dst_root'] ) ? (string) $args['dst_root'] : '',
				'paths'    => isset( $args['paths'] ) && is_array( $args['paths'] ) ? (array) $args['paths'] : array(),
                'wpconfig_update' => ! empty( $args['wpconfig_update'] ) ? 1 : 0,
                'db_name' => isset( $args['db_name'] ) ? (string) $args['db_name'] : '',
                'db_user' => isset( $args['db_user'] ) ? (string) $args['db_user'] : '',
                'db_pass' => isset( $args['db_pass'] ) ? (string) $args['db_pass'] : '',
                'db_host' => isset( $args['db_host'] ) ? (string) $args['db_host'] : '',
                'wpconfig_regen_salts' => ! empty( $args['wpconfig_regen_salts'] ) ? 1 : 0,
                'run_step3_auto' => ! empty( $args['run_step3_auto'] ) ? 1 : 0,
            ),
            'tmp_dir' => '',
            'manifest' => array(),
            'db_prepare' => array(),
            'db_import' => array(),
            'checksum' => array(),
            'files_restore' => array(),
            'domain' => array(),
            'last_log' => '',
        );

if ( ! isset( $state['activity'] ) || ! is_array( $state['activity'] ) ) {
    $state['activity'] = array();
}
if ( empty( $state['activity']['started'] ) ) {
    if ( ! class_exists( '\\AegisBackup\\Libs\\AB_Activity_Log' ) && defined( 'AEGISBACKUP_DIR' ) ) {
        require_once AEGISBACKUP_DIR . 'includes/libs/class-ab-activity-log.php';
    }
    if ( class_exists( '\\AegisBackup\\Libs\\AB_Activity_Log' ) ) {
        $pkg = isset( $state['args']['package_path'] ) ? basename( (string) $state['args']['package_path'] ) : '';
        \AegisBackup\Libs\AB_Activity_Log::add(
            'restore',
            'started',
            'Restore job started.',
            array( 'job_id' => $job_id, 'package' => $pkg )
        );
    }
    $state['activity']['started'] = time();
}

$state = $this->persist_job_state( $job_id, $state );

        return array( 'job_id' => $job_id );
    }

    public function process_restore_job( $job_id ) {
        $state = get_option( self::JOB_OPTION_PREFIX . $job_id, array() );
        if ( empty( $state ) || empty( $state['job_id'] ) ) {
            return array( 'done' => true, 'progress' => 100, 'log' => 'Restore job not found.' );
        }

		if ( ! empty( $state['stop_requested'] ) ) {
			$state['status'] = 'stopped';
			$state['phase'] = 'done';
			$state['last_log'] = 'Restore stopped by user.';
			$state = $this->cleanup_extracted_work_dir( $state );
			$state = $this->persist_job_state( $job_id, $state );
			return array( 'done' => true, 'progress' => (int) $state['progress'], 'log' => (string) $state['last_log'] );
		}

        $args = isset( $state['args'] ) ? (array) $state['args'] : array();
        $pkg_dir = (string) $args['package_path'];

        $restorer = new AB_DB_Restorer();

        if ( 'validate' === $state['phase'] ) {
            if ( empty( $state['validate'] ) || ! is_array( $state['validate'] ) ) {
                $state['validate'] = array();
            }
            if ( empty( $state['validate']['stage'] ) ) {
                $state['validate']['stage'] = 'init';
            }

            if ( empty( $pkg_dir ) || ( ! is_dir( $pkg_dir ) && ! ( is_file( $pkg_dir ) && preg_match( '/\.zip$/i', $pkg_dir ) ) ) ) {
                $state['status'] = 'error';
                $state['phase'] = 'done';
                $state['progress'] = 100;
                $state['last_log'] = 'Invalid package path.';
				$state = $this->cleanup_extracted_work_dir( $state );
                $state = $this->persist_job_state( $job_id, $state );
                return array( 'done' => true, 'progress' => 100, 'log' => $state['last_log'] );
            }

            if ( is_file( $pkg_dir ) ) {
                $zip = $pkg_dir;
            } else {
                $zip = trailingslashit( $pkg_dir ) . 'AegisBackup-' . basename( $pkg_dir ) . '.zip';
                if ( ! file_exists( $zip ) ) {
                    $zip = trailingslashit( $pkg_dir ) . 'package.zip';
                }
            }
            $work = is_dir( $pkg_dir ) ? ( trailingslashit( $pkg_dir ) . 'work' ) : '';

            if ( is_dir( $work ) ) {
                $state['tmp_dir'] = $work;
                $state['validate']['stage'] = 'manifest';
            }

            if ( 'init' === (string) $state['validate']['stage'] ) {
                $state['validate']['zip'] = $zip;

                $tmp = trailingslashit( wp_upload_dir()['basedir'] ) . 'aegisbackup/tmp-' . $job_id;
                wp_mkdir_p( $tmp );
                $state['tmp_dir'] = $tmp;

                if ( ! file_exists( $zip ) ) {
                    $state['status'] = 'error';
                    $state['phase'] = 'done';
                    $state['progress'] = 100;
                    $state['last_log'] = 'Package ZIP not found.';
				$state = $this->cleanup_extracted_work_dir( $state );
                    $state = $this->persist_job_state( $job_id, $state );
                    return array( 'done' => true, 'progress' => 100, 'log' => $state['last_log'] );
                }
                if ( ! class_exists( 'ZipArchive' ) ) {
                    $state['status'] = 'error';
                    $state['phase'] = 'done';
                    $state['progress'] = 100;
                    $state['last_log'] = 'ZipArchive is not available on this server.';
				$state = $this->cleanup_extracted_work_dir( $state );
                    $state = $this->persist_job_state( $job_id, $state );
                    return array( 'done' => true, 'progress' => 100, 'log' => $state['last_log'] );
                }

                $z = new \ZipArchive();
                $open = $z->open( $zip );
                if ( true !== $open ) {
                    $state['status'] = 'error';
                    $state['phase'] = 'done';
                    $state['progress'] = 100;
                    $state['last_log'] = 'Failed to open ZIP archive (code: ' . (int) $open . ').';
				$state = $this->cleanup_extracted_work_dir( $state );
                    $state = $this->persist_job_state( $job_id, $state );
                    return array( 'done' => true, 'progress' => 100, 'log' => $state['last_log'] );
                }

                $state['validate']['total'] = (int) $z->numFiles;
                $state['validate']['i'] = 0;
                $state['validate']['stage'] = 'extract';
                $state['last_log'] = 'Extracting package… 0/' . (int) $state['validate']['total'];
                $state['progress'] = 1;
                $z->close();

                $state = $this->persist_job_state( $job_id, $state );
                return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => $state['last_log'] );
            }

            if ( 'extract' === (string) $state['validate']['stage'] ) {
                $zip = isset( $state['validate']['zip'] ) ? (string) $state['validate']['zip'] : $zip;
                $tmp = isset( $state['tmp_dir'] ) ? (string) $state['tmp_dir'] : '';
                $i   = isset( $state['validate']['i'] ) ? (int) $state['validate']['i'] : 0;
                $tot = isset( $state['validate']['total'] ) ? (int) $state['validate']['total'] : 0;

                if ( '' === $tmp || ! is_dir( $tmp ) ) {
                    $state['status'] = 'error';
                    $state['phase'] = 'done';
                    $state['progress'] = 100;
                    $state['last_log'] = 'Extraction directory missing.';
				$state = $this->cleanup_extracted_work_dir( $state );
                    $state = $this->persist_job_state( $job_id, $state );
                    return array( 'done' => true, 'progress' => 100, 'log' => $state['last_log'] );
                }

                $z = new \ZipArchive();
                $open = $z->open( $zip );
                if ( true !== $open ) {
                    $state['status'] = 'error';
                    $state['phase'] = 'done';
                    $state['progress'] = 100;
                    $state['last_log'] = 'Failed to open ZIP archive during extraction (code: ' . (int) $open . ').';
				$state = $this->cleanup_extracted_work_dir( $state );
                    $state = $this->persist_job_state( $job_id, $state );
                    return array( 'done' => true, 'progress' => 100, 'log' => $state['last_log'] );
                }

                $chunk = array();
                $max_names = 200;
                $end = min( $tot, $i + $max_names );
                for ( $k = $i; $k < $end; $k++ ) {
                    $name = $z->getNameIndex( $k );
                    if ( false !== $name && '' !== $name ) {
                        $chunk[] = $name;
                    }
                }
                if ( ! empty( $chunk ) ) {
                    $z->extractTo( $tmp, $chunk );
                }
                $z->close();

                $i = $end;
                $state['validate']['i'] = $i;
                $pct = 0;
                if ( $tot > 0 ) {
                    $pct = (int) floor( ( $i / $tot ) * 100 );
                    $pct = max( 0, min( 100, $pct ) );
                }

                $state['progress'] = max( 1, min( 5, (int) floor( ( $pct / 100 ) * 5 ) ) );
                $state['last_log'] = 'Extracting package… ' . $i . '/' . $tot . ' (' . $pct . '%)';

                if ( $tot > 0 && $i >= $tot ) {
                    $state['validate']['stage'] = 'manifest';
                    $state['last_log'] = 'Extraction complete. Reading manifest.';
                    $state['progress'] = 5;
                }

                $state = $this->persist_job_state( $job_id, $state );
                return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => $state['last_log'] );
            }

            if ( 'manifest' === (string) $state['validate']['stage'] ) {
                if ( empty( $state['tmp_dir'] ) || ! is_dir( $state['tmp_dir'] ) ) {
                    $state['status'] = 'error';
                    $state['phase'] = 'done';
                    $state['progress'] = 100;
                    $state['last_log'] = 'Failed to extract or locate package work directory.';
				$state = $this->cleanup_extracted_work_dir( $state );
                    $state = $this->persist_job_state( $job_id, $state );
                    return array( 'done' => true, 'progress' => 100, 'log' => $state['last_log'] );
                }

                $manifest_path = trailingslashit( $state['tmp_dir'] ) . 'manifest.json';
                $manifest = array();
                if ( file_exists( $manifest_path ) ) {
                    $raw = file_get_contents( $manifest_path );
                    $manifest = json_decode( (string) $raw, true );
                    if ( ! is_array( $manifest ) ) {
                        $manifest = array();
                    }
                }

                $state['manifest'] = $manifest;
                $state['phase'] = 'preflight';
                $state['progress'] = 5;
                $state['last_log'] = 'Package validated. Running pre-flight checks.';
                if ( isset( $state['report'] ) && is_array( $state['report'] ) ) { \AegisBackup\Restore\AB_Migration_Report::add_step( $state['report'], 'validate', $state['last_log'] ); }

                $state = $this->persist_job_state( $job_id, $state );
                return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => $state['last_log'] );
            }
        }

        
        
        if ( 'preflight' === $state['phase'] ) {
            $tmp_dir = isset( $state['tmp_dir'] ) ? (string) $state['tmp_dir'] : '';
            $pf = new AB_Preflight();
            $expected = 0;
            if ( ! empty( $state['manifest']['tables'] ) && is_array( $state['manifest']['tables'] ) ) {
                foreach ( $state['manifest']['tables'] as $t ) {
                    if ( isset( $t['data_length'] ) ) $expected += (int) $t['data_length'];
                }
            }
            $pf_res = $pf->run_checks( array( 'expected_bytes' => $expected ) );
            $state['preflight'] = $pf_res;

            $hard_fail = false;
            if ( isset( $pf_res['results'] ) && is_array( $pf_res['results'] ) ) {
                foreach ( $pf_res['results'] as $r ) {
                    if ( isset( $r['level'] ) && 'fail' === $r['level'] ) { $hard_fail = true; break; }
                }
            }

            if ( isset( $state['report'] ) && is_array( $state['report'] ) ) {
                \AegisBackup\Restore\AB_Migration_Report::add_step( $state['report'], 'preflight', $hard_fail ? 'Pre-flight failed (hard failures).' : 'Pre-flight passed (warnings possible).' );
                if ( $hard_fail ) { \AegisBackup\Restore\AB_Migration_Report::add_error( $state['report'], 'Pre-flight checks reported one or more hard failures.' ); }
            }

            if ( $hard_fail ) {
                $state['status'] = 'error';
                $state['phase'] = 'done';
                $state['progress'] = 100;
                $state['last_log'] = 'Pre-flight checks failed. Fix the failed items and retry.';
                $state = $this->persist_job_state( $job_id, $state );
                return array( 'done' => true, 'progress' => 100, 'log' => $state['last_log'], 'preflight' => $pf_res );
            }

            $state['phase'] = 'verify';
            $state['progress'] = 10;
            $state['last_log'] = 'Pre-flight checks passed. Verifying integrity.';
            $state = $this->persist_job_state( $job_id, $state );
            return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => $state['last_log'], 'preflight' => $pf_res );
        }

if ( 'verify' === $state['phase'] ) {
            $step = $restorer->verify_checksums_step( (string) $state['tmp_dir'], $state['checksum'], 250, 1.0 );
            $state['progress'] = (int) ( 5 + ( $step['progress'] * 0.10 ) ); // 5..15
            $state['last_log'] = $step['log'];

            if ( ! empty( $step['done'] ) ) {
                if ( ! empty( $step['bad'] ) ) {
                    $state['last_log'] .= ' (Warning: checksum mismatches detected.)';
                }
                $state['phase'] = 'db_prepare';
                $state['progress'] = 15;
                $state['last_log'] = 'Integrity verification complete. Preparing target database (drop existing tables).';
            }

            $state = $this->persist_job_state( $job_id, $state );
            return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => $state['last_log'] );
        }

if ( 'db_prepare' === $state['phase'] ) {
            global $wpdb;

            $args = isset( $state['args'] ) ? (array) $state['args'] : array();
            $conn = $this->get_target_wpdb( $args );
            if ( ! empty( $conn['error'] ) ) {
                $state['status'] = 'error';
                $state['phase'] = 'done';
                $state['progress'] = 100;
                $state['last_log'] = (string) $conn['error'];
                $state = $this->persist_job_state( $job_id, $state );
                return array( 'done' => true, 'progress' => 100, 'log' => $state['last_log'] );
            }

            $target_wpdb = $conn['wpdb'];
            $current_prefix = isset( $wpdb->prefix ) ? (string) $wpdb->prefix : '';
            $backup_prefix = '';
            if ( ! empty( $state['manifest']['db_prefix'] ) ) {
                $backup_prefix = (string) $state['manifest']['db_prefix'];
            }
            if ( '' === $backup_prefix ) {
                $backup_prefix = $restorer->detect_backup_prefix( (string) $state['tmp_dir'] );
            }
            if ( '' === $backup_prefix ) {
                $backup_prefix = $current_prefix;
            }

            $mode = isset( $args['mode'] ) ? (string) $args['mode'] : 'existing';
            $target_prefix = $current_prefix;

            if ( 'newdb' === $mode && ! empty( $args['new_prefix'] ) ) {
                $target_prefix = (string) $args['new_prefix'];
            }

            $state['old_prefix'] = $backup_prefix;
            $state['target_prefix'] = $target_prefix;
            $orig = $this->swap_global_wpdb( $target_wpdb );
            $drop = $restorer->drop_tables_with_prefix_step( $target_prefix, $state['db_prepare'], 50 );
            $this->swap_global_wpdb( $orig );
            $state['progress'] = (int) ( 15 + ( $drop['progress'] * 0.10 ) ); // 15..25
            $state['last_log'] = $drop['log'];

            if ( ! empty( $drop['done'] ) ) {
                $state['phase'] = 'db_import';
                $state['progress'] = 25;
                $state['last_log'] = 'Database prepared. Starting DB import.';
            }

            $state = $this->persist_job_state( $job_id, $state );
            return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => $state['last_log'] );
        }

if ( 'db_import' === $state['phase'] ) {
                       $args = isset( $state['args'] ) ? (array) $state['args'] : array();
            $conn = $this->get_target_wpdb( $args );
            if ( ! empty( $conn['error'] ) ) {
                $state['status'] = 'error';
                $state['phase'] = 'done';
                $state['progress'] = 100;
                $state['last_log'] = (string) $conn['error'];
                $state = $this->persist_job_state( $job_id, $state );
                return array( 'done' => true, 'progress' => 100, 'log' => $state['last_log'] );
            }

            $old_prefix = isset( $state['old_prefix'] ) ? (string) $state['old_prefix'] : '';
            $target_prefix = isset( $state['target_prefix'] ) ? (string) $state['target_prefix'] : '';

            $orig = $this->swap_global_wpdb( $conn['wpdb'] );
            $step = $restorer->import_db_step( (string) $state['tmp_dir'], $state['db_import'], 60, 1.0, $old_prefix, $target_prefix );
            $this->swap_global_wpdb( $orig );
            $state['progress'] = (int) ( 5 + ( $step['progress'] * 0.55 ) ); // 5..60
            $state['last_log'] = $step['log'];

            if ( ! empty( $step['done'] ) ) {
                $state['phase'] = 'prefix';
                $state['progress'] = 60;
                $state['last_log'] = 'DB import complete. Next: prefix/domain transforms.';
            }

            $state = $this->persist_job_state( $job_id, $state );
            return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => $state['last_log'] );
        }

        if ( 'prefix' === $state['phase'] ) {
            $args = isset( $state['args'] ) ? (array) $state['args'] : array();
            $conn = $this->get_target_wpdb( $args );
            if ( ! empty( $conn['error'] ) ) {
                $state['status'] = 'error';
                $state['phase'] = 'done';
                $state['progress'] = 100;
                $state['last_log'] = (string) $conn['error'];
                $state = $this->persist_job_state( $job_id, $state );
                return array( 'done' => true, 'progress' => 100, 'log' => $state['last_log'] );
            }

            $old_prefix = isset( $state['old_prefix'] ) ? (string) $state['old_prefix'] : '';
            $new_prefix = isset( $state['target_prefix'] ) ? (string) $state['target_prefix'] : '';

            if ( '' === $new_prefix ) {
                global $wpdb;
                $new_prefix = isset( $wpdb->prefix ) ? (string) $wpdb->prefix : '';
            }

            if ( '' !== $old_prefix && '' !== $new_prefix && $old_prefix !== $new_prefix ) {
                $orig = $this->swap_global_wpdb( $conn['wpdb'] );
                $ok = $restorer->apply_prefix_migration( $old_prefix, $new_prefix );
                $this->swap_global_wpdb( $orig );

                $state['last_log'] = $ok ? ( 'Prefix migration applied: ' . $old_prefix . ' → ' . $new_prefix ) : 'Prefix migration failed (best-effort).';
            } else {
                $state['last_log'] = 'Prefix unchanged.';
            }

            $state['phase'] = 'domain';
            $state['progress'] = 75;
            $state = $this->persist_job_state( $job_id, $state );
            return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => $state['last_log'] );
        }

if ( 'domain' === $state['phase'] ) {
            $args = isset( $state['args'] ) ? (array) $state['args'] : array();
            $conn = $this->get_target_wpdb( $args );
            if ( ! empty( $conn['error'] ) ) {
                $state['status'] = 'error';
                $state['phase'] = 'done';
                $state['progress'] = 100;
                $state['last_log'] = (string) $conn['error'];
                $state = $this->persist_job_state( $job_id, $state );
                return array( 'done' => true, 'progress' => 100, 'log' => $state['last_log'] );
            }
            $orig = $this->swap_global_wpdb( $conn['wpdb'] );
            global $wpdb;

            $old = isset( $args['old_domain'] ) ? (string) $args['old_domain'] : '';
            $new = isset( $args['new_domain'] ) ? (string) $args['new_domain'] : '';

            if ( '' === $old && ! empty( $state['manifest']['home_url'] ) ) {
                $old = (string) $state['manifest']['home_url'];
            }
            if ( '' === $new ) {
                $new = home_url();
            }

            
            $prefix = isset( $state['target_prefix'] ) ? (string) $state['target_prefix'] : $wpdb->prefix;

            $step = $restorer->domain_replace_step( $old, $new, $prefix, $state['domain'] );
            $state['progress'] = (int) ( 75 + ( $step['progress'] * 0.20 ) ); // 75..95
            $state['last_log'] = $step['log'];

            if ( ! empty( $step['done'] ) ) {
                $state['phase'] = 'files';
                $state['progress'] = 95;
                $state['last_log'] = 'Transforms complete. Cleaning up…';
            }

            $state = $this->persist_job_state( $job_id, $state );
            $this->swap_global_wpdb( $orig );
            return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => $state['last_log'] );
        }

        
        if ( 'files' === $state['phase'] ) {
            $file_restorer = new AB_File_Restorer();
            $step = $file_restorer->restore_files_step( (string) $state['tmp_dir'], $state['files_restore'], 200, 1.0 );
            $state['progress'] = (int) ( 80 + ( $step['progress'] * 0.15 ) ); // 80..95
            $state['last_log'] = $step['log'];

            if ( ! empty( $step['done'] ) ) {
                $state['phase'] = 'wpconfig';
                $state['progress'] = 95;
                $state['last_log'] = 'File restore complete. Next: wp-config rewrite (optional).';
                if ( isset( $state['report'] ) && is_array( $state['report'] ) ) { \AegisBackup\Restore\AB_Migration_Report::add_step( $state['report'], 'files', $state['last_log'] ); }

            }

            $state = $this->persist_job_state( $job_id, $state );
            return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => $state['last_log'] );
        }


        if ( 'wpconfig' === $state['phase'] ) {
            $do_wpconfig = ! empty( $args['wpconfig_update'] );
            $res = array( 'success' => true, 'skipped' => true );
            if ( $do_wpconfig ) {
                $writer = new \AegisBackup\Libs\AB_WPConfig_Writer();
                $changes = array();
                if ( isset( $args['db_name'] ) && '' !== (string) $args['db_name'] ) $changes['DB_NAME'] = (string) $args['db_name'];
                if ( isset( $args['db_user'] ) && '' !== (string) $args['db_user'] ) $changes['DB_USER'] = (string) $args['db_user'];
                if ( isset( $args['db_pass'] ) && '' !== (string) $args['db_pass'] ) $changes['DB_PASSWORD'] = (string) $args['db_pass'];
                if ( isset( $args['db_host'] ) && '' !== (string) $args['db_host'] ) $changes['DB_HOST'] = (string) $args['db_host'];

                if ( isset( $args['new_prefix'] ) && '' !== (string) $args['new_prefix'] ) {
                    $changes['table_prefix'] = (string) $args['new_prefix'];
                } elseif ( isset( $args['old_prefix'] ) && '' !== (string) $args['old_prefix'] ) {
                    $changes['table_prefix'] = (string) $args['old_prefix'];
                }

                $changes['regenerate_salts'] = ! empty( $args['wpconfig_regen_salts'] );

                $res = $writer->write( $changes );
                $state['wpconfig_result'] = $res;

                if ( isset( $state['report'] ) && is_array( $state['report'] ) ) {
                    $state['report']['wpconfig'] = $res;
                    \AegisBackup\Restore\AB_Migration_Report::add_step( $state['report'], 'wpconfig', ( ! empty( $res['updated'] ) ) ? 'wp-config.php updated.' : 'wp-config.php not updated (not writable or no changes).' );
                    if ( ! empty( $res['writable'] ) && empty( $res['updated'] ) ) {
                        \AegisBackup\Restore\AB_Migration_Report::add_warning( $state['report'], 'wp-config.php was writable but did not change (no matching patterns found). Review manual instructions.' );
                    }
                    if ( empty( $res['writable'] ) ) {
                        \AegisBackup\Restore\AB_Migration_Report::add_warning( $state['report'], 'wp-config.php is not writable. You may need to apply changes manually.' );
                    }
                }

                $state['last_log'] = ! empty( $res['updated'] ) ? 'wp-config.php updated.' : 'wp-config.php unchanged (skipped or not writable).';
            } else {
                $state['last_log'] = 'wp-config rewrite skipped.';
            }

            if ( ! empty( $args['run_step3_auto'] ) ) {
                $state['phase'] = 'step3';
                $state['progress'] = 96;
            } else {
                $state['phase'] = 'cleanup';
                $state['progress'] = 97;
            }
            $state = $this->persist_job_state( $job_id, $state );
            return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => $state['last_log'], 'wpconfig' => $res );
        }

        if ( 'step3' === $state['phase'] ) {
            $target = $this->get_target_wpdb( $args );
            if ( ! empty( $target['error'] ) ) {
                if ( isset( $state['report'] ) && is_array( $state['report'] ) ) {
                    \AegisBackup\Restore\AB_Migration_Report::add_warning( $state['report'], 'Step 3 skipped: ' . (string) $target['error'] );
                }
                $state['last_log'] = 'Step 3 skipped: could not connect to target DB.';
                $state['phase'] = 'cleanup';
                $state['progress'] = 97;
                $state = $this->persist_job_state( $job_id, $state );
                return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => $state['last_log'] );
            }

            $orig = $this->swap_global_wpdb( $target['wpdb'] );

            $fixer = new AB_Post_Restore_Fixer();
            $s3 = isset( $state['step3'] ) && is_array( $state['step3'] ) ? (array) $state['step3'] : array();
            $s3res = $fixer->run_step3( $s3, $args );
            $state['step3'] = $s3res['state'];

            if ( isset( $state['report'] ) && is_array( $state['report'] ) ) {
                $c = isset( $s3res['counts'] ) && is_array( $s3res['counts'] ) ? (array) $s3res['counts'] : array();
                foreach ( $c as $k => $v ) {
                    if ( isset( $state['report']['counts'][ $k ] ) ) {
                        $state['report']['counts'][ $k ] = (int) $v;
                    }
                }
                if ( ! empty( $s3res['message'] ) ) {
                    \AegisBackup\Restore\AB_Migration_Report::add_step( $state['report'], 'step3', (string) $s3res['message'] );
                }
                if ( ! empty( $s3res['warnings'] ) && is_array( $s3res['warnings'] ) ) {
                    foreach ( $s3res['warnings'] as $w ) {
                        \AegisBackup\Restore\AB_Migration_Report::add_warning( $state['report'], (string) $w );
                    }
                }
            }

            $this->swap_global_wpdb( $orig );

            if ( ! empty( $s3res['done'] ) ) {
                $state['last_log'] = 'Step 3 complete: attachments + uploads normalized.';
                $state['phase'] = 'cleanup';
                $state['progress'] = 97;
            } else {
                $state['last_log'] = ! empty( $s3res['log'] ) ? (string) $s3res['log'] : 'Step 3 running...';
                $state['progress'] = 96;
            }

            $state = $this->persist_job_state( $job_id, $state );
            return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => (string) $state['last_log'] );
        }

        if ( 'cleanup' === $state['phase'] ) {
            $restorer->post_restore_cleanup();

            if ( isset( $state['report'] ) && is_array( $state['report'] ) ) {
                if ( isset( $state['db_import'] ) && is_array( $state['db_import'] ) ) {
                    $state['report']['counts']['db_statements_executed'] = isset( $state['db_import']['statements_done'] ) ? (int) $state['db_import']['statements_done'] : (int) $state['report']['counts']['db_statements_executed'];
                    $state['report']['counts']['db_tables_imported'] = isset( $state['db_import']['tables_done'] ) ? (int) $state['db_import']['tables_done'] : (int) $state['report']['counts']['db_tables_imported'];
                }

                if ( isset( $state['prefix_result'] ) && is_array( $state['prefix_result'] ) && ! empty( $state['prefix_result']['details'] ) ) {
                    $d = (array) $state['prefix_result']['details'];
                    if ( isset( $d['renamed_tables'] ) && is_array( $d['renamed_tables'] ) ) {
                        $state['report']['counts']['prefix_renamed_tables'] = count( $d['renamed_tables'] );
                    }
                    if ( isset( $d['options_rows'] ) ) $state['report']['counts']['prefix_updated_options'] = (int) $d['options_rows'];
                    if ( isset( $d['usermeta_rows'] ) ) $state['report']['counts']['prefix_updated_usermeta'] = (int) $d['usermeta_rows'];
                }

                if ( isset( $state['domain_state'] ) && is_array( $state['domain_state'] ) ) {
                    $state['report']['counts']['domain_rows_updated'] = isset( $state['domain_state']['updated'] ) ? (int) $state['domain_state']['updated'] : (int) $state['report']['counts']['domain_rows_updated'];
                    $state['report']['counts']['domain_replacements'] = isset( $state['domain_state']['replacements'] ) ? (int) $state['domain_state']['replacements'] : (int) $state['report']['counts']['domain_replacements'];
                }

                if ( isset( $state['files_restore'] ) && is_array( $state['files_restore'] ) ) {
                    $state['report']['counts']['files_restored'] = isset( $state['files_restore']['restored'] ) ? (int) $state['files_restore']['restored'] : (int) $state['report']['counts']['files_restored'];
                    $state['report']['counts']['bytes_restored'] = isset( $state['files_restore']['bytes'] ) ? (int) $state['files_restore']['bytes'] : (int) $state['report']['counts']['bytes_restored'];

                    if ( isset( $state['files_restore']['failed_writes'] ) ) {
                        $state['report']['counts']['failed_writes'] = (int) $state['files_restore']['failed_writes'];
                    }
                    if ( isset( $state['files_restore']['hash_mismatch'] ) ) {
                        $state['report']['counts']['hash_mismatch'] = (int) $state['files_restore']['hash_mismatch'];
                    }

                    if ( ! isset( $state['report']['samples'] ) || ! is_array( $state['report']['samples'] ) ) {
                        $state['report']['samples'] = array();
                    }
                    if ( isset( $state['files_restore']['failed_samples'] ) && is_array( $state['files_restore']['failed_samples'] ) ) {
                        $state['report']['samples']['failed_writes'] = array_slice( (array) $state['files_restore']['failed_samples'], 0, 25 );
                    }
                    if ( isset( $state['files_restore']['hash_mismatch_samples'] ) && is_array( $state['files_restore']['hash_mismatch_samples'] ) ) {
                        $state['report']['samples']['hash_mismatch'] = array_slice( (array) $state['files_restore']['hash_mismatch_samples'], 0, 25 );
                    }
                }

                $state['report']['package'] = isset( $args['package_path'] ) ? (string) $args['package_path'] : '';
                \AegisBackup\Restore\AB_Migration_Report::finish( $state['report'] );

                update_option( 'aegisbackup_last_migration_report', $state['report'], false );
            }

            $state['phase'] = 'done';
            $state['progress'] = 100;
            $state['status'] = 'done';
			$state['last_log'] = 'Restore completed.';
			$state = $this->cleanup_extracted_work_dir( $state );
			do_action( 'aegisbackup_restore_completed', $job_id, $state );

            $state = $this->persist_job_state( $job_id, $state );
            return array( 'done' => true, 'progress' => 100, 'log' => $state['last_log'] );
        }

        return array( 'done' => ( 'done' === $state['phase'] ), 'progress' => (int) $state['progress'], 'log' => (string) $state['last_log'] );
    }
}
