<?php

if (!defined('ABSPATH')) { exit; }

if (!class_exists('Aegisify_Bundled_Plugins_Page')) {

if (defined('WP_DEBUG') && WP_DEBUG && defined('AEGISIFY_DEBUG') && AEGISIFY_DEBUG) {
    error_log('[Aegisify Plugins] Loaded bundled plugins page from: ' . __FILE__);
}

class Aegisify_Bundled_Plugins_Page {
	private static $booted = false;

	private static $config = array(
		'parent_slug' => 'aegisify-dashboard',

		'menu_slug'   => 'aegisify-bundled-plugins',

		'page_title'  => 'Aegisify Plugins',
		'menu_title'  => 'Aegisify Plugins',

		'view_cap'    => 'manage_options',

		'products' => array(
			'aegisshield' => array(
				'plugin_file' => 'aegisshield-security/aegisshield-security.php',
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2026/02/aegisshield-security-7.1.16.zip',
				'title'       => 'AegisShield Free & PRO',
				'desc'        => 'Full WordPress security suite with Pro Upgrade: hardening, monitoring, malware checks, integrity tools, and protection layers built for real admins.',
				'image'       => '../../assets/aegisshield.png', // set your image URL if you want
			),
			'aegiswaf' => array(
				'plugin_file' => 'aegiswaf/aegiswaf.php',
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2026/01/aegiswaf-1.8.17-submit-now.zip',
				'title'       => 'AegisWAF',
				'desc'        => 'Web Application Firewall for WordPress, DDOS Layer7, Bot Control, API Shield and Logs Attack Story. Advanced features upgrades',
				'image'       => '../../assets/aegiswaf.png',
			),
			('aegis' . 'seo') => array(
				'plugin_file' => ('aegis' . 'seo/' . 'aegis' . 'seo.php'),
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2026/02/aegisseo-1.6.14.zip',
				'title'       => 'AegisSEO',
				'desc'        => 'SEO OPs Center, Global SEO Automation, Social Meta, Schema, Advanced SiteMAPS, Issues & Fixes, Evidence, Linking Assistance and FREE advanced features.',
				'image'       => ('../../assets/' . 'aegis' . 'seo.png'),
			),
			'aegisbackup' => array(
				'plugin_file' => 'aegisbackup/aegisbackup.php',
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2026/02/aegisbackup-1.1.25.zip',
				'title'       => 'AegisBackup & DR',
				'desc'        => 'AegisBackup is a professional WordPress backup, migration, and disaster-recovery plugin engineered for stability and safe restores without risky background jobs',
				'image'       => '../../assets/aegisbackup.png',
			),
			'aegisspam' => array(
				'plugin_file' => 'aegisspam/aegisspam.php',
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2026/02/aegisspam-1.1.11.zip',
				'title'       => 'AegisSpam',
				'desc'        => 'Local-first anti-spam protection for WordPress comments, registrations, and Contact Forms with transparent scoring, logs, allow/deny lists, firewall, and local Machine learning.',
				'image'       => '../../assets/aegisspam.png', // set your image URL if you want
			),
			'aegislink' => array(
				'plugin_file' => 'AegisLink/aegislink.php',
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2026/02/aegislink-1.1.19.zip',
				'title'       => 'AegisLink',
				'desc'        => 'Free Smart internal linking + linking assistant by Aegisify. Create keyword URL smart links and generate context-aware internal link suggestions.',
				'image'       => '../../assets/aegislink.png', // set your image URL if you want
			),
			'aegissitemap' => array(
				'plugin_file' => 'aegissitemap/aegissitemap.php',
				'zip_url'     => 'https://aegisify.com/wp-content/uploads/2026/01/aegissitemap-1.2.15-submit-now.zip',
				'title'       => 'AegisSiteMap',
				'desc'        => 'Free XML sitemap engine by Aegisify: fast, cache-friendly sitemap index, HTML sitemap, images, and optional video/news sitemaps.',
				'image'       => '../../assets/aegissitemap.png', // set your image URL if you want
			),
			'aegisticket' => array(
				'plugin_file' => 'aegisticket/support-ticket-system.php',
				'zip_url'     => '	https://aegisify.com/wp-content/uploads/2026/02/aegisticket-1.8.4.zip',
				'title'       => 'AegisTicket Free',
				'desc'        => 'Free Support ticketing plugin built for ops: CPT-based tickets, admin workflows, and customer-friendly submission + tracking. NOTE: Use at your own risks.',
				'image'       => '../../assets/aegisticket.png', // set your image URL if you want
			),
		),
	);
	
	public static function init(array $override = array()) : void {
		if (!is_admin()) { return; }

		if (!did_action('init')) {
			add_action('init', function() use ($override) {
				self::init($override);
			}, 1);
			return;
		}

		if (self::$booted) { return; }
		self::$booted = true;

		$override_products = array();
		if (isset($override['products']) && is_array($override['products'])) {
			$override_products = $override['products'];
			unset($override['products']);
		}

		self::$config = array_merge(self::$config, $override);

		if (!empty($override_products)) {
			self::$config['products'] = array_merge((array) self::$config['products'], $override_products);
		}

		$products = (array) self::$config['products'];

		if (isset($products['aegislink']) && is_array($products['aegislink'])) {
			if (defined('AEGISLINK_BASENAME') && AEGISLINK_BASENAME) {
				$products['aegislink']['plugin_file'] = AEGISLINK_BASENAME;
			}
		}
		
		if (isset($products[('aegis' . 'seo')]) && is_array($products[('aegis' . 'seo')])) {
			if (defined('AEGISSEO_BASENAME') && AEGISSEO_BASENAME) {
				$products[('aegis' . 'seo')]['plugin_file'] = AEGISSEO_BASENAME;
			}
		}

		self::$config['products'] = $products;

		add_action('admin_post_aegisify_products_install', array(__CLASS__, 'handle_install'));
	}

	private static function maybe_handle_inline_install() : void {
		if (empty($_GET['aegisify_install'])) {
			return;
		}

		if (!current_user_can('install_plugins')) {
			wp_die('You do not have permission to install plugins.');
		}

		$key = sanitize_key((string) $_GET['aegisify_install']);
		check_admin_referer('aegisify_inline_install_' . $key);

		$products = self::get_products();
		if (empty($products[$key])) {
			wp_die('Invalid product.');
		}

		$p = $products[$key];
		$plugin_file = !empty($p['plugin_file']) ? (string) $p['plugin_file'] : '';
		if ($plugin_file && self::is_installed($plugin_file)) {
			self::redirect_back('Plugin is already installed.', 'updated');
		}

		$download = !empty($p['zip_url']) ? esc_url_raw((string) $p['zip_url']) : '';
		if (!$download || !preg_match('#^https?://#i', $download)) {
			self::redirect_back('Install failed: Missing or invalid zip_url for this product.', 'error');
		}

		$check = self::validate_zip_url($download);
		if (empty($check['ok'])) {
			self::redirect_back('Install failed: ' . $check['error'], 'error');
		}

		require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/plugin.php';

		$req_uri = isset( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
		$url = remove_query_arg( array( 'aegisify_notice', 'aegisify_type', '_wpnonce' ), $req_uri );

		$creds = request_filesystem_credentials($url);
		if ($creds === false) {
			exit;
		}
		if (!WP_Filesystem($creds)) {
			request_filesystem_credentials($url, '', true);
			exit;
		}

		$skin = new Automatic_Upgrader_Skin();
		$upgrader = new Plugin_Upgrader($skin);

		$result = $upgrader->install($download);

		if (is_wp_error($result)) {
			self::redirect_back('Install failed: ' . $result->get_error_message(), 'error');
		}

		if (!$result) {
			self::redirect_back('Install failed: WordPress could not install the package (permissions / invalid zip).', 'error');
		}

		self::redirect_back('Installed successfully.', 'updated');
	}

	public static function add_submenu() : void {
		$parent = (string) self::$config['parent_slug'];
		$slug   = (string) self::$config['menu_slug'];

		add_submenu_page(
			$parent,
			(string) self::$config['page_title'],
			(string) self::$config['menu_title'],
			(string) self::$config['view_cap'],
			$slug,
			array(__CLASS__, 'render_page')
		);
	}

	private static function get_products() : array {
		$products = (array) self::$config['products'];

		if (has_filter('aegisify_bundled_plugins_products')) {
			$products = (array) apply_filters('aegisify_bundled_plugins_products', $products, self::$config);
		}

		return $products;
	}

	private static function resolve_asset_url(string $img) : string {
		$img = trim($img);
		if ($img === '') { return ''; }
		if (preg_match('#^(https?:)?//#i', $img) || stripos($img, 'data:') === 0) {
			return $img;
		}

		if (strpos($img, '..') !== false) {
			$img = basename($img);
		}

		$img = ltrim($img, '/');
		if (stripos($img, 'assets/') === 0) {
			$img = substr($img, strlen('assets/'));
		}

		$base = plugin_dir_url(__FILE__) . '../../assets/';
		return $base . $img;
	}
	
	private static function resolve_plugin_file(string $plugin_file): string {
		$plugin_file = ltrim($plugin_file, '/');
		if (!$plugin_file) { return ''; }

		if (!did_action('init')) {
			return $plugin_file;
		}

		if (file_exists(WP_PLUGIN_DIR . '/' . $plugin_file)) {
			return $plugin_file;
		}

		if (!function_exists('get_plugins')) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$target = strtolower($plugin_file);
		foreach (array_keys(get_plugins()) as $pf) {
			if (strtolower($pf) === $target) {
				return $pf; 
			}
		}

		return $plugin_file;
	}

	private static function is_installed(string $plugin_file) : bool {
		$plugin_file = self::resolve_plugin_file($plugin_file);
		if (!$plugin_file) { return false; }
		return file_exists(WP_PLUGIN_DIR . '/' . $plugin_file);
	}
	
	private static function is_active_plugin(string $plugin_file) : bool {
		$plugin_file = self::resolve_plugin_file($plugin_file);
		if (!$plugin_file) { return false; }

		if (!function_exists('is_plugin_active')) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		return is_plugin_active($plugin_file);
	}

	private static function validate_zip_url(string $url) : array {

		$cache_key = 'aegisify_zipcheck_' . md5($url);
		$cached = get_transient($cache_key);
		if (is_array($cached) && isset($cached['ok'])) {
			return $cached;
		}

		$head = wp_remote_head($url, array(
			'timeout'     => 10,
			'redirection' => 5,
		));

		if (is_wp_error($head)) {
			$out = array('ok' => false, 'error' => 'Zip URL request failed: ' . $head->get_error_message());
			set_transient($cache_key, $out, DAY_IN_SECONDS);
			return $out;
		}

		$code = (int) wp_remote_retrieve_response_code($head);
		if ($code < 200 || $code >= 400) {
			$out = array('ok' => false, 'error' => 'Zip URL returned HTTP ' . $code . ' (blocked or redirected).');
			set_transient($cache_key, $out, DAY_IN_SECONDS);
			return $out;
		}

		$content_type = (string) wp_remote_retrieve_header($head, 'content-type');
		if ($content_type && stripos($content_type, 'text/html') !== false) {
			$out = array('ok' => false, 'error' => 'Zip URL returned HTML instead of a ZIP file.');
			set_transient($cache_key, $out, DAY_IN_SECONDS);
			return $out;
		}

		$get = wp_remote_get($url, array(
			'timeout'     => 10,
			'redirection' => 5,
			'headers'     => array(
				'Range' => 'bytes=0-1',
			),
			'limit_response_size' => 2,
		));

		if (is_wp_error($get)) {
			$out = array('ok' => false, 'error' => 'Zip URL download failed: ' . $get->get_error_message());
			set_transient($cache_key, $out, DAY_IN_SECONDS);
			return $out;
		}

		$get_code = (int) wp_remote_retrieve_response_code($get);
		if ($get_code !== 200 && $get_code !== 206) {
			$out = array('ok' => false, 'error' => 'Zip URL byte-check returned HTTP ' . $get_code . '.');
			set_transient($cache_key, $out, DAY_IN_SECONDS);
			return $out;
		}

		$body = (string) wp_remote_retrieve_body($get);
		if (strlen($body) < 2 || substr($body, 0, 2) !== 'PK') {
			$out = array('ok' => false, 'error' => 'Downloaded file is not a valid ZIP archive.');
			set_transient($cache_key, $out, DAY_IN_SECONDS);
			return $out;
		}

		$out = array('ok' => true);
		set_transient($cache_key, $out, DAY_IN_SECONDS);
		return $out;
	}

	private static function redirect_back(string $notice = '', string $type = 'updated') : void {
		$parent = (string) self::$config['parent_slug'];
		$slug   = (string) self::$config['menu_slug'];

		$url = admin_url('admin.php?page=' . rawurlencode($slug));

		if ($notice) {
			$url = add_query_arg(array(
				'aegisify_notice' => rawurlencode($notice),
				'aegisify_type'   => rawurlencode($type),
				'_wpnonce'        => wp_create_nonce( 'aegisify_notice' ),
			), $url);
		}

		wp_safe_redirect($url);
		exit;
	}
	
	private static function get_wp_activate_url(string $plugin_file) : string {
		$plugin_file = ltrim($plugin_file, '/');

		$url = admin_url('plugins.php?action=activate&plugin=' . rawurlencode($plugin_file));

		return wp_nonce_url($url, 'activate-plugin_' . $plugin_file);
	}



	private static function get_wp_deactivate_url(string $plugin_file) : string {
		$plugin_file = ltrim($plugin_file, '/');
		$url = admin_url('plugins.php?action=deactivate&plugin=' . rawurlencode($plugin_file));
		return wp_nonce_url($url, 'deactivate-plugin_' . $plugin_file);
	}

	private static function get_wp_update_url(string $plugin_file) : string {
		$plugin_file = ltrim($plugin_file, '/');
		$url = admin_url('update.php?action=upgrade-plugin&plugin=' . rawurlencode($plugin_file));
		return wp_nonce_url($url, 'upgrade-plugin_' . $plugin_file);
	}

	private static function has_update(string $plugin_file) : bool {
		$plugin_file = ltrim($plugin_file, '/');
		$updates = get_site_transient('update_plugins');
		return (is_object($updates) && isset($updates->response) && isset($updates->response[$plugin_file]));
	}


	public static function handle_install() : void {
		if (!current_user_can('install_plugins')) {
			self::redirect_back('You do not have permission to install plugins.', 'error');
		}
		check_admin_referer('aegisify_products_install');

		$key = isset($_POST['product_key']) ? sanitize_key((string) $_POST['product_key']) : '';
		$products = self::get_products();

		if (!$key || !isset($products[$key])) {
			self::redirect_back('Invalid product.', 'error');
		}
		
		$p = $products[$key];

		$download = isset($p['zip_url']) ? esc_url_raw((string) $p['zip_url']) : '';
		if (!$download || !preg_match('#^https?://#i', $download)) {
			self::redirect_back('Install failed: Missing or invalid zip_url for this product.', 'error');
		}

		$check = self::validate_zip_url($download);
		if (empty($check['ok'])) {
			self::add_notice('Install failed: ' . $check['error'], 'error');
			return;
		}

		require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/misc.php';
		require_once ABSPATH . 'wp-admin/includes/plugin.php';

		$skin = new Automatic_Upgrader_Skin();
		$upgrader = new Plugin_Upgrader($skin);
		$result = $upgrader->install($download);

		if (is_wp_error($result)) {
			self::redirect_back('Install failed: ' . $result->get_error_message(), 'error');
		}

		if (!$result) {
			$err = (isset($skin->result) && is_wp_error($skin->result))
				? $skin->result->get_error_message()
				: 'Unknown installer failure.';
			self::redirect_back('Install failed: ' . $err, 'error');
		}

		self::redirect_back('Installed successfully.', 'updated');
	}
	
	public static function render_page() : void {
		self::maybe_handle_inline_install();
		if (!current_user_can((string) self::$config['view_cap'])) {
			wp_die(esc_html__('Sorry, you are not allowed to access this page.', 'aegisify'));
		}

		$notice = '';
		$type   = '';
		$nonce  = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';
		if ( $nonce && wp_verify_nonce( $nonce, 'aegisify_notice' ) ) {
			$notice = isset( $_GET['aegisify_notice'] ) ? sanitize_text_field( wp_unslash( $_GET['aegisify_notice'] ) ) : '';
			$type   = isset( $_GET['aegisify_type'] ) ? sanitize_key( wp_unslash( $_GET['aegisify_type'] ) ) : '';
		}

		$products = self::get_products();
		?>
		<div class="wrap">
			<h1 style="margin-bottom:8px;"><?php echo esc_html(self::$config['page_title']); ?></h1>
			<div class="aegisify-banner" style="margin: 12px 0 20px 0;">
				<img
					src="<?php echo esc_url( plugin_dir_url( __FILE__ ) . '../../assets/all.png' ); ?>"
					alt="<?php echo esc_attr__( 'Aegisify Plugin Suite', 'aegisify' ); ?>"
					style="
						width:70%;
						max-width:800px;
						align: center;
						height:auto;
						display:block;
						border-radius:8px;
						box-shadow:0 2px 6px rgba(0,0,0,0.08);
					"
				/>
			</div>			
			<p style="max-width: 980px;">
			    <h1 style="margin-bottom:8px;"><?php echo esc_html('Everything Your WordPress Site Needs One Platform. One Price.'); ?></h1>
				<?php echo esc_html__('Welcome to Aegisify, and thank you for choosing our WordPress products. Every Aegisify plugin is carefully engineered, thoroughly tested, and built with a deep commitment to delivering real value to the WordPress community. We believe powerful tools should be accessible to everyone, which is why each product includes meaningful, high-impact features that comes free out of the box!  No forced upgrades, no hidden restrictions, and no pressure to go Pro. You’ll continue to receive updates, improvements, and critical bug fixes whether you upgrade or not. Pro features are available for those who want advanced capabilities, but the choice is always yours. We’re grateful for your trust and proud to support your WordPress site with tools designed to protect, optimize, and grow your business today and into the future.', 'aegisify'); ?>
				<h3 style="margin-bottom:8px;"><?php echo esc_html('Built for site owners, freelancers, and agencies who want stability'); ?></h3></p>
				<div style="margin-bottom: 8px; font-size: 14px;">
					<strong><?php echo esc_html__('Aegisify:', 'aegisify'); ?></strong>
					<a href="https://aegisify.com" target="_blank" rel="noopener noreferrer">https://aegisify.com</a>
				</div>

				<?php
					$quicklinks = array(
						array(
							'label' => __('Register Account', 'aegisify'),
							'url'   => 'https://aegisify.com/register/',
							'icon'  => 'dashicons-id',
						),
						array(
							'label' => __('Buy Now', 'aegisify'),
							'url'   => 'https://aegisify.com/profile/',
							'icon'  => 'dashicons-cart',
						),
						array(
							'label' => __('Open a Ticket', 'aegisify'),
							'url'   => 'https://aegisify.com/profile/',
							'icon'  => 'dashicons-sos',
						),
						array(
							'label' => __('Report a Bug', 'aegisify'),
							'url'   => 'https://aegisify.com/report-a-bug/',
							'icon'  => 'dashicons-dismiss',
						),
						array(
							'label' => __('Donate', 'aegisify'),
							'url'   => 'https://www.paypal.com/ncp/payment/8ZU8J9GTCALCU',
							'icon'  => 'dashicons-heart',
						),
					);
				?>

				<div style="display:flex; flex-wrap:wrap; gap:10px;">
					<?php foreach ($quicklinks as $q): ?>
						<a
							href="<?php echo esc_url($q['url']); ?>"
							target="_blank"
							rel="noopener noreferrer"
							style="
								display:inline-flex;
								align-items:center;
								gap:6px;
								padding:8px 10px;
								border:1px solid #dcdcde;
								border-radius:8px;
								background:#fff;
								text-decoration:none;
								font-size:13px;
							"
						>
							<span class="dashicons <?php echo esc_attr($q['icon']); ?>" style="font-size:18px; width:18px; height:18px;"></span>
							<span><?php echo esc_html($q['label']); ?></span>
						</a>
					<?php endforeach; ?>
				</div>

			</p>

			<?php if ($notice): ?>
				<div class="<?php echo esc_attr(($type === 'error') ? 'notice notice-error' : 'notice notice-success'); ?> is-dismissible">
					<p><?php echo esc_html($notice); ?></p>
				</div>
			<?php endif; ?>

			<div class="notice notice-info" style="max-width:980px;">
				<p>
					<?php echo esc_html__('Tip: WordPress may request filesystem credentials (FTP) depending on server permissions. That is normal for plugin installs on some hosts.', 'aegisify'); ?>
				</p>
			</div>

			<style>
				.aegisify-grid{
					display:grid;
					grid-template-columns: repeat(4, minmax(0, 1fr));
					gap: 16px;
					max-width: 1200px;
				}
				@media (max-width: 1200px){
					.aegisify-grid{ grid-template-columns: repeat(3, minmax(0,1fr)); }
				}
				@media (max-width: 900px){
					.aegisify-grid{ grid-template-columns: repeat(2, minmax(0,1fr)); }
				}
				@media (max-width: 520px){
					.aegisify-grid{ grid-template-columns: 1fr; }
				}
				.aegisify-card{
					background:#fff;
					border:1px solid #e5e5e5;
					border-radius:8px;
					overflow:hidden;
					box-shadow: 0 1px 2px rgba(0,0,0,0.04);
					display:flex;
					flex-direction:column;
					min-height: 280px;
				}
				.aegisify-card__top{
					position:relative;
					padding: 18px 18px 10px;
					border-bottom: 1px solid #f0f0f0;
					display:flex;
					align-items:center;
					justify-content:center;
					min-height: 120px;
				}
				.aegisify-card__badge{
					position:absolute;
					top:10px;
					left:10px;
					background:#2271b1;
					color:#fff;
					font-size: 11px;
					padding: 4px 8px;
					border-radius: 4px;
					letter-spacing: .3px;
					text-transform: uppercase;
				}
				.aegisify-card__img{
					max-width: 78%;
					max-height: 78px;
					width:auto;
					height:auto;
					object-fit:contain;
				}
				.aegisify-card__body{
					padding: 14px 16px 0;
					flex:1;
				}
				.aegisify-card__title{
					margin: 0 0 8px;
					font-size: 14px;
					font-weight: 700;
				}
				.aegisify-card__desc{
					margin: 0 0 14px;
					color:#4b5563;
					font-size: 12.5px;
					line-height: 1.45;
					min-height: 54px;
				}
				.aegisify-card__footer{
					padding: 12px 16px 16px;
					display:flex;
					align-items:center;
					justify-content:space-between;
					gap: 10px;
				}
				.aegisify-status{
					font-size: 12px;
					color:#111827;
				}
				.aegisify-status strong{ font-weight:700; }
				.aegisify-actions form{ display:inline; }
			</style>

			<div class="aegisify-grid">
				<?php foreach ($products as $k => $p): ?>
					<?php
						$k = (string) $k;
						$title = isset($p['title']) ? (string) $p['title'] : $k;
						$desc  = isset($p['desc']) ? (string) $p['desc'] : '';
						$img_raw = isset($p['image']) ? (string) $p['image'] : '';
						$img     = self::resolve_asset_url($img_raw);
						$plugin_file = isset($p['plugin_file']) ? (string) $p['plugin_file'] : '';

						$installed = ($plugin_file ? self::is_installed($plugin_file) : false);
						$active = ($installed && $plugin_file) ? self::is_active_plugin($plugin_file) : false;
						$update_available = ($installed && $plugin_file) ? self::has_update($plugin_file) : false;
					?>
					<div class="aegisify-card">
						<div class="aegisify-card__top">
							<div class="aegisify-card__badge"><?php echo esc_html($installed ? 'Installed' : 'Available'); ?></div>
							<?php if ($img): ?>
								<img class="aegisify-card__img" src="<?php echo esc_url($img); ?>" alt="<?php echo esc_attr($title); ?>">
							<?php else: ?>
								<div style="font-weight:800;font-size:18px;opacity:.25;"><?php echo esc_html($title); ?></div>
							<?php endif; ?>
						</div>

						<div class="aegisify-card__body">
							<div class="aegisify-card__title"><?php echo esc_html($title); ?></div>
							<div class="aegisify-card__desc"><?php echo esc_html($desc); ?></div>
						</div>

						<div class="aegisify-card__footer">
							<div class="aegisify-status">
								<strong><?php echo esc_html($installed ? 'Status:' : 'Status:'); ?></strong>
								<?php echo esc_html(!$installed ? ' Not Installed' : ($active ? ' Active' : ' Installed')); ?>
							</div>

							<div class="aegisify-actions">
								<?php if ($installed && $update_available && current_user_can('update_plugins')): ?>
									<?php $update_url = self::get_wp_update_url($plugin_file); ?>
									<a href="<?php echo esc_url($update_url); ?>" class="button button-primary">
										<?php echo esc_html__('Update Plugin', 'aegisify'); ?>
									</a>
								<?php endif; ?>

								<?php if (!$installed): ?>
									<?php
										$install_url = add_query_arg(
											array('aegisify_install' => $k),
											menu_page_url((string) self::$config['menu_slug'], false)
										);
										$install_url = wp_nonce_url($install_url, 'aegisify_inline_install_' . $k);
									?>
									<a href="<?php echo esc_url($install_url); ?>" class="button button-primary">
										<?php echo esc_html__('Install Plugin', 'aegisify'); ?>
									</a>

								<?php elseif (!$active): ?>
									<?php $activate_url = self::get_wp_activate_url($plugin_file); ?>
									<a href="<?php echo esc_url($activate_url); ?>" class="button button-secondary">
										<?php echo esc_html__('Activate Plugin', 'aegisify'); ?>
									</a>

								<?php else: ?>
									<?php $deactivate_url = self::get_wp_deactivate_url($plugin_file); ?>
									<a href="<?php echo esc_url($deactivate_url); ?>" class="button button-secondary">
										<?php echo esc_html__('Deactivate Plugin', 'aegisify'); ?>
									</a>
								<?php endif; ?>
							</div>
						</div>
					</div>
				<?php endforeach; ?>
			</div>

		</div>
		<?php
	}
}

} 
