<?php
if (!defined('ABSPATH')) exit;

class Aegisify_Modules {

    const OPTION_ENABLED = 'aegisify_enabled_modules'; 
    const OPTION_INSTALLED = 'aegisify_installed_modules'; 

    private static $modules = array();

    public static function register(Aegisify_Module_Interface $module): void {
        $slug = sanitize_key($module->get_slug());
        self::$modules[$slug] = $module;
    }

    public static function all(): array {
        return self::$modules;
    }

    public static function is_enabled(string $slug): bool {
        $slug = sanitize_key($slug);
        $enabled = get_option(self::OPTION_ENABLED, array());
        return !empty($enabled[$slug]);
    }

    public static function enable(string $slug): bool {
        $slug = sanitize_key($slug);

        if (!empty(self::$modules[$slug])) {
            if (!self::$modules[$slug]->is_licensed()) return false;
        }

        $enabled = get_option(self::OPTION_ENABLED, array());
        $enabled[$slug] = 1;
        update_option(self::OPTION_ENABLED, $enabled, false);

        if (!empty(self::$modules[$slug])) {
            self::$modules[$slug]->on_enable();
        }

        return true;
    }

    public static function disable(string $slug): bool {
        $slug = sanitize_key($slug);

        $enabled = get_option(self::OPTION_ENABLED, array());
        unset($enabled[$slug]);
        update_option(self::OPTION_ENABLED, $enabled, false);

        if (!empty(self::$modules[$slug])) {
            self::$modules[$slug]->on_disable();
        }

        return true;
    }

    public static function boot_enabled_modules(): void {
        foreach (self::$modules as $slug => $module) {
            if (self::is_enabled($slug) && $module->is_licensed()) {
                $module->boot_if_enabled();
            }
        }
    }

    public static function is_registered(string $slug): bool {
        $slug = sanitize_key($slug);
        return !empty(self::$modules[$slug]);
    }

    public static function get_default_catalog(): array {
        return array(
            'aegisshield' => array(
                'name' => 'AegisShield',
                'desc' => 'Security suite: hardening, file integrity, malware defense.',
            ),
            'aegiswaf' => array(
                'name' => 'AegisWAF',
                'desc' => 'Web Application Firewall: block attacks, rate limit, rulesets.',
            ),
            ('aegis' . 'seo') => array(
                'name' => 'AegisSEO',
                'desc' => 'SEO automation: metadata, schema, indexing controls.',
            ),
            'aegisspam' => array(
                'name' => 'AegisSPAM',
                'desc' => 'Anti-spam engine: forms, comments, bots, abuse control.',
            ),
            'aegislink' => array(
                'name' => 'AegisLINK',
                'desc' => 'Link manager: short links, redirects, tracking-ready structure.',
            ),
            'aegisbackup' => array(
                'name' => 'AegisBackup',
                'desc' => 'Backup & DR: snapshots, restores, migration wizard.',
            ),
            'aegissitemap' => array(
                'name' => 'AegisSiteMAP',
                'desc' => 'Sitemaps & robots: indexing control and crawl management.',
            ),
            'aegisticket' => array(
                'name' => 'AegisTicket',
                'desc' => 'Support ticketing: lightweight helpdesk built for WP admins.',
            ),
        );
    }
}
