<?php

/**
 * Plugin Name: AegisSEO
 * Plugin URI: https://aegisify.com/aegisseo/
 * Description: AegisSEO is an advanced WordPress SEO optimization plugin that helps improve search visibility through structured data, Search Console integration, automated analysis, and performance-focused SEO tools.
 * Version: 1.6.13
 * Requires at least: 6.8
 * Requires PHP: 8.2
 * Author:      Aegisify, LLC
 * Author URI:  https://aegisify.com/
 * Text Domain: aegisseo
 * License: GPLv2 or later
 */

if (!defined('ABSPATH')) {
    exit;
}

define('AEGISSEO_VERSION', '1.6.13');
define('AEGISSEO_FILE', __FILE__);
define('AEGISSEO_DIR', plugin_dir_path(__FILE__));

require_once AEGISSEO_DIR . 'includes/class-aegisseo-plugin.php';
require_once AEGISSEO_DIR . 'includes/log/class-aegisseo-logger.php';

define('AEGISSEO_URL', plugin_dir_url(__FILE__));
define('AEGISSEO_BASENAME', plugin_basename(__FILE__));

if (!defined('AEGISSEO_WPORG_BUILD')) {
    define('AEGISSEO_WPORG_BUILD', true);
}

define('AEGISSEO_SLUG', 'aegisseo');
define('AEGISSEO_CLM_BASE', 'https://aegisify.com');
define('AEGISSEO_CLM_FREE_REG_ENDPOINT', AEGISSEO_CLM_BASE . '/wp-json/chue-license/v1/aegisseo/free-register');

define('AEGISSEO_CLM_ACTIVATE_ENDPOINT',   AEGISSEO_CLM_BASE . '/wp-json/chue-license/v1/aegisseo/activate');
define('AEGISSEO_CLM_DEACTIVATE_ENDPOINT', AEGISSEO_CLM_BASE . '/wp-json/chue-license/v1/aegisseo/deactivate');
define('AEGISSEO_CLM_STATUS_ENDPOINT',     AEGISSEO_CLM_BASE . '/wp-json/chue-license/v1/aegisseo/status');		
																			 

/**
 * Activation / deactivation hooks must be registered at load time (not inside plugins_loaded),
 * otherwise WordPress will not call them during plugin activation and rewrite rules won't flush.
 */
function aegisseo_activate() : void {
    // Load only what we need for activation (avoid depending on plugins_loaded/boot()).
    require_once AEGISSEO_DIR . 'includes/utils/class-aegisseo-options.php';
    require_once AEGISSEO_DIR . 'includes/seo/class-aegisseo-installer.php';

    $opts = new \AegisSEO\Utils\Options();
    $opts->ensure_defaults();

    \AegisSEO\SEO\Installer::install();

    set_transient('aegisseo_do_wizard', '1', 60);

    if (!wp_next_scheduled('aegisseo_daily_maintenance')) {
        wp_schedule_event(time() + 300, 'daily', 'aegisseo_daily_maintenance');
    }

    if (!wp_next_scheduled('aegisseo_gsc_snapshot_runner')) {
        wp_schedule_event(time() + 600, 'daily', 'aegisseo_gsc_snapshot_runner');
    }

    if (!wp_next_scheduled('aegisseo_alerts_runner')) {
        wp_schedule_event(time() + 900, 'daily', 'aegisseo_alerts_runner');
    }

    flush_rewrite_rules();
}

function aegisseo_deactivate() : void {
    set_transient('aegisseo_do_wizard', '1', 60);

    wp_clear_scheduled_hook('aegisseo_daily_maintenance');
    wp_clear_scheduled_hook('aegisseo_gsc_snapshot_runner');
    wp_clear_scheduled_hook('aegisseo_alerts_runner');

    flush_rewrite_rules();
}

register_activation_hook(AEGISSEO_FILE, 'aegisseo_activate');
register_deactivation_hook(AEGISSEO_FILE, 'aegisseo_deactivate');

/**
 * Disable WordPress core sitemaps when AegisSEO sitemaps are enabled.
 *
 * Running multiple sitemap providers (core + plugin) can cause Search Console validation
 * and duplicate discovery issues. Major SEO plugins disable core sitemaps for this reason.
 *
 * This is conditional: if AegisSEO sitemaps are disabled in settings, core sitemaps remain enabled.
 */
function aegisseo_maybe_disable_wp_core_sitemaps( $enabled ) {
    // Allow developers/site owners to override via filter.
    $force = apply_filters( 'aegisseo_disable_wp_core_sitemap', null );
    if ( null !== $force ) {
        // If filter returns true, disable core sitemaps. If false, keep default behavior.
        if ( true === $force ) {
            return false;
        }
        return $enabled;
    }

    // If our sitemap feature is enabled, disable core sitemaps.
    if ( defined( 'AEGISSEO_DIR' ) ) {
        require_once AEGISSEO_DIR . 'includes/utils/class-aegisseo-options.php';
        $opts = new \AegisSEO\Utils\Options();
        $aegis_enabled = (int) $opts->get( 'sitemap_enabled', 1 );
        if ( 1 === $aegis_enabled ) {
            return false;
        }
    }

    return $enabled;
}
add_filter( 'wp_sitemaps_enabled', 'aegisseo_maybe_disable_wp_core_sitemaps', 0 );


function aegisseo_update_log( string $msg, array $ctx = array(), string $level = 'INFO' ) : void {
    if (class_exists('\AegisSEO\Log\Logger')) {
        switch (strtoupper($level)) {
            case 'ERROR':
                \AegisSEO\Log\Logger::error($msg, $ctx);
                break;
            case 'WARN':
            case 'WARNING':
                \AegisSEO\Log\Logger::warn($msg, $ctx);
                break;
            case 'DEBUG':
                \AegisSEO\Log\Logger::debug($msg, $ctx);
                break;
            default:
                \AegisSEO\Log\Logger::info($msg, $ctx);
        }
        return;
    }

    // Fallback (should be rare)
    // No logger available; do not write to error_log in production.

}

function aegisseo_clm_register_free_install(bool $force = false) : array {

    // Prevent spamming endpoint (manual click can override with $force=true)
    if (!$force && get_transient('aegisseo_clm_free_reg_sent')) {
        return array(
            'success' => false,
            'status'  => 'backoff',
            'message' => 'Backoff active (already attempted recently).',
        );
    }

    $site_url = home_url();
    $domain   = wp_parse_url($site_url, PHP_URL_HOST);

    $payload = array(
        'product'        => AEGISSEO_SLUG,
        'email'          => (string) get_option('admin_email'),
        'site_url'       => (string) $site_url,
        'domain'         => (string) $domain,
        'plugin_version' => defined('AEGISSEO_VERSION') ? (string) AEGISSEO_VERSION : 'unknown',
        'wp_version'     => (string) get_bloginfo('version'),
        'php_version'    => (string) PHP_VERSION,
    );

    $args = array(
        'timeout' => 15,
        'headers' => array('Content-Type' => 'application/json'),
        'body'    => wp_json_encode($payload),
    );

    $resp = wp_remote_post(AEGISSEO_CLM_FREE_REG_ENDPOINT, $args);

    if (is_wp_error($resp)) {
        aegisseo_update_log('CLM free-register failed (WP_Error).', array(
            'endpoint' => AEGISSEO_CLM_FREE_REG_ENDPOINT,
            'error'    => $resp->get_error_message(),
        ));

        return array(
            'success' => false,
            'status'  => 'wp_error',
            'message' => $resp->get_error_message(),
        );
    }

    $code = (int) wp_remote_retrieve_response_code($resp);
    $body = (string) wp_remote_retrieve_body($resp);

    $backoff = ($code >= 500) ? HOUR_IN_SECONDS : DAY_IN_SECONDS;
    set_transient('aegisseo_clm_free_reg_sent', 1, $backoff);

    aegisseo_update_log('CLM free-register response.', array(
        'endpoint'  => AEGISSEO_CLM_FREE_REG_ENDPOINT,
        'http_code' => $code,
        'body'      => $body,
        'backoff_s' => $backoff,
    ));

    if ($code >= 200 && $code < 300) {
        return array(
            'success' => true,
            'status'  => 'ok',
            'message' => 'Registered',
        );
    }

    return array(
        'success' => false,
        'status'  => 'http_' . $code,
        'message' => 'Non-2xx response from CLM',
        'body'    => $body,
    );
}

function aegisseo() {
    return \AegisSEO\Plugin::instance();
}

	add_action('plugins_loaded', function() {
		\AegisSEO\Log\Logger::init(12);
		aegisseo()->boot();

	}, 5);