(function($){
    'use strict';

    function getGutenbergContent() {
        try {
            if (window.wp && wp.data && wp.data.select) {
                var ed = wp.data.select('core/editor');
                if (ed) {
                    var content = ed.getEditedPostContent ? ed.getEditedPostContent() : '';
                    var title = ed.getEditedPostAttribute ? ed.getEditedPostAttribute('title') : '';
                    return { content: content || '', title: title || '' };
                }
            }
        } catch(e) {}
        return { content: '', title: '' };
    }

    function getClassicContent() {
        var c = '';
        if (typeof window.tinyMCE !== 'undefined' && window.tinyMCE.activeEditor && !window.tinyMCE.activeEditor.isHidden()) {
            c = window.tinyMCE.activeEditor.getContent() || '';
        } else if ($('#content').length) {
            c = $('#content').val() || '';
        }
        return c;
    }

    function truncate(str, n) {
        str = (str || '').toString().trim();
        if (str.length <= n) return str;
        return str.substring(0, n - 1).trim() + '…';
    }

    function pickFirstNonEmpty() {
        for (var i=0;i<arguments.length;i++) {
            var v = (arguments[i] || '').toString().trim();
            if (v !== '') return v;
        }
        return '';
    }

	function computeDescFallback(html) {
		html = (html || '').toString();

		// Strip shortcodes like [fusion_builder_container ...] so preview isn't junk
		html = html.replace(/\[[^\]]+\]/g, ' ');
		// Strip HTML comments (Gutenberg / builder noise)
		html = html.replace(/<!--[\s\S]*?-->/g, ' ');

		var text = $('<div>').html(html).text().replace(/\s+/g,' ').trim();
		return truncate(text, 155);
	}

    function updatePreviews($box) {
        var postId = parseInt($box.data('post-id') || 0, 10);
        var permalink = ($box.data('permalink') || '').toString();

        var focus = $('input[name="aegisseo_focus_phrase"]').val() || '';
        var seoTitle = $('input[name="aegisseo_title"]').val() || '';
        var seoDesc  = $('textarea[name="aegisseo_description"]').val() || '';

        var ogTitle = $('input[name="aegisseo_og_title"]').val() || '';
        var ogDesc  = $('textarea[name="aegisseo_og_description"]').val() || '';
        var ogImg   = $('input[name="aegisseo_og_image"]').val() || '';

        var twTitle = $('input[name="aegisseo_twitter_title"]').val() || '';
        var twDesc  = $('textarea[name="aegisseo_twitter_description"]').val() || '';
        var twImg   = $('input[name="aegisseo_twitter_image"]').val() || '';

        var gb = getGutenbergContent();
        var content = gb.content || getClassicContent();
        var postTitle = pickFirstNonEmpty(gb.title, $('input#title').val(), document.title);

        var finalTitle = pickFirstNonEmpty(seoTitle, postTitle);
        var finalDesc  = pickFirstNonEmpty(seoDesc, computeDescFallback(content));

        // SERP
        $box.find('.aegisseo-serp-title').text(truncate(finalTitle, 70));
        $box.find('.aegisseo-serp-url').text((permalink || '').replace(/^https?:\/\//,''));
        $box.find('.aegisseo-serp-desc').text(truncate(finalDesc, 160));

        // Social (OG default)
        var site = (permalink || '').replace(/^https?:\/\//,'').split('/')[0] || window.location.host;
        var social = $box.find('.aegisseo-social-preview').attr('data-social') || 'og';

        var sTitle = (social === 'twitter') ? pickFirstNonEmpty(twTitle, ogTitle, finalTitle) : pickFirstNonEmpty(ogTitle, finalTitle);
        var sDesc  = (social === 'twitter') ? pickFirstNonEmpty(twDesc, ogDesc, finalDesc) : pickFirstNonEmpty(ogDesc, finalDesc);
        var sImg   = (social === 'twitter') ? pickFirstNonEmpty(twImg, ogImg) : ogImg;

        $box.find('.aegisseo-social-site').text(site);
        $box.find('.aegisseo-social-title').text(truncate(sTitle, 90));
        $box.find('.aegisseo-social-desc').text(truncate(sDesc, 140));
        if (sImg) {
            $box.find('.aegisseo-social-img').html('<img src="'+encodeURI(sImg)+'" alt="" />');
        } else {
            $box.find('.aegisseo-social-img').html('<div class="aegisseo-social-img-placeholder">No image set</div>');
        }

        return { post_id: postId, focus: focus, title: finalTitle, description: finalDesc, content: content };
    }

    var debounceTimer = null;
    function requestAnalysis($box) {
        if (!window.AegisSEOAdmin || !AegisSEOAdmin.ajax_url) return;

        var payload = updatePreviews($box);

        clearTimeout(debounceTimer);
        debounceTimer = setTimeout(function(){
            $.post(AegisSEOAdmin.ajax_url, {
                action: 'aegisseo_analyze_content',
                nonce: AegisSEOAdmin.nonce,
                post_id: payload.post_id,
                focus: payload.focus,
                title: payload.title,
                description: payload.description,
                content: payload.content
            }).done(function(resp){
                if (!resp || !resp.success || !resp.data) return;

                var score = parseInt(resp.data.score || 0, 10);
                var actions = resp.data.actions || [];

                var $badge = $box.find('.aegisseo-score-badge');
                var $label = $box.find('.aegisseo-score-label');
                var $list  = $box.find('.aegisseo-actions-list');

                $badge.text(score);
                $badge.removeClass('is-bad is-ok is-good');
                if (score < 50) $badge.addClass('is-bad');
                else if (score < 80) $badge.addClass('is-ok');
                else $badge.addClass('is-good');

                $label.text(score < 50 ? 'Needs work' : (score < 80 ? 'Good progress' : 'Strong'));

                $list.empty();
                if (!actions.length) {
                    $list.append('<li><strong>Nice.</strong> Nothing urgent detected.</li>');
                    return;
                }
                actions.forEach(function(a){
                    var pri = parseInt(a.priority || 3, 10);
                    var priLabel = (pri === 1 ? 'High' : (pri === 2 ? 'Med' : 'Low'));
                    var hint = (a.hint || '').toString();
                    $list.append(
                        '<li><span class="aegisseo-pri pri-'+pri+'">'+priLabel+'</span> ' +
                        '<strong>'+escapeHtml(a.label || '')+'</strong> ' +
                        (a.detail ? '<span class="aegisseo-detail">('+escapeHtml(a.detail)+')</span>' : '') +
                        (hint ? '<div class="aegisseo-hint">'+escapeHtml(hint)+'</div>' : '') +
                        '</li>'
                    );
                });
            });
        }, 450);
    }

    function escapeHtml(s) {
        return (s || '').toString()
            .replace(/&/g,'&amp;')
            .replace(/</g,'&lt;')
            .replace(/>/g,'&gt;')
            .replace(/"/g,'&quot;')
            .replace(/'/g,'&#039;');
    }

    function initMetabox() {
        var $box = $('.aegisseo-metabox-phaseA');
        if (!$box.length) return;

        // toggle SERP device
        $box.on('click', '.aegisseo-preview-toggle [data-device]', function(){
            var d = $(this).data('device');
            $box.find('.aegisseo-preview-toggle [data-device]').removeClass('is-active');
            $(this).addClass('is-active');
            $box.find('.aegisseo-serp-preview').attr('data-device', d);
        });

        // toggle social
        $box.on('click', '.aegisseo-preview-toggle [data-social]', function(){
            var s = $(this).data('social');
            $box.find('.aegisseo-preview-toggle [data-social]').removeClass('is-active');
            $(this).addClass('is-active');
            $box.find('.aegisseo-social-preview').attr('data-social', s);
            requestAnalysis($box);
        });

        // watch inputs
        $(document).on('input change', 'input[name^="aegisseo_"], textarea[name^="aegisseo_"]', function(){
            requestAnalysis($box);
        });

        // watch editor updates (Gutenberg)
        if (window.wp && wp.data && wp.data.subscribe) {
            var last = '';
            wp.data.subscribe(function(){
                var gb = getGutenbergContent();
                var now = (gb.title || '') + '|' + (gb.content || '').substring(0, 200);
                if (now !== last) {
                    last = now;
                    requestAnalysis($box);
                }
            });
        } else {
            // Classic editor
            setInterval(function(){
                requestAnalysis($box);
            }, 2500);
        }

        requestAnalysis($box);
    }


    function initRecentEventsBulkRollback(){
        var $form = $('#aegisseo-bulk-rollback-form');
        if (!$form.length) { return; }

        var $selectAll = $('#aegisseo-evt-select-all');
        var $boxes = $form.find('input.aegisseo-evt-cb');
        var $btn = $('#aegisseo-bulk-rollback-btn');

        function sync(){
            var checked = $boxes.filter(':checked').length;
            $btn.prop('disabled', checked === 0);
            if ($selectAll.length) {
                $selectAll.prop('checked', checked > 0 && checked === $boxes.length);
                $selectAll.prop('indeterminate', checked > 0 && checked < $boxes.length);
            }
        }

        $selectAll.on('change', function(){
            var on = $(this).is(':checked');
            $boxes.prop('checked', on);
            sync();
        });

        $form.on('change', 'input.aegisseo-evt-cb', function(){
            sync();
        });

        sync();
    }


    $(function(){
        initMetabox();
        initRecentEventsBulkRollback();
    });


    // Bulk Manager: inline AJAX save (no full page reload)
    $(document).on('submit', '#aegisseo-bulk-form', function(e){
        e.preventDefault();
        var $form = $(this);
        var $btn = $('#aegisseo-bulk-save-btn');
        var $saving = $('#aegisseo-bulk-saving');
        var $notice = $('#aegisseo-bulk-notice');

        var nonce = $form.data('aegisseo-ajax-nonce') || '';
        if (!nonce) {
            // fallback: full submit
            this.submit();
            return;
        }

        $btn.prop('disabled', true);
        $saving.show();
        $notice.empty();

        var data = $form.serializeArray();
        data.push({name:'action', value:'aegisseo_bulk_save_ajax'});
        data.push({name:'nonce', value: nonce});

        $.post((window.AegisSEOAdmin && AegisSEOAdmin.ajax_url) ? AegisSEOAdmin.ajax_url : ajaxurl, data)
            .done(function(resp){
                if (resp && resp.success) {
                    var upd = resp.data && resp.data.updated ? resp.data.updated : 0;
                    var msg = 'Saved. Updated: ' + upd;
                    $notice.html('<div class="notice notice-success is-dismissible"><p>' + msg + '</p></div>');
                } else {
                    var em = (resp && resp.data && resp.data.message) ? resp.data.message : 'Save failed.';
                    $notice.html('<div class="notice notice-error"><p>' + em + '</p></div>');
                }
            })
            .fail(function(xhr){
                var em = 'Save failed.';
                if (xhr && xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                    em = xhr.responseJSON.data.message;
                }
                $notice.html('<div class="notice notice-error"><p>' + em + '</p></div>');
            })
            .always(function(){
                $btn.prop('disabled', false);
                $saving.hide();
            });
    });




    // Per-field generator buttons in metabox (SEO Title, Meta, Canonical, OG, Twitter)
    $(document).on('click', '.aegisseo-generate-btn', function(e){
        e.preventDefault();

        var $btn = $(this);
        var postId = parseInt($btn.attr('data-post-id'), 10) || 0;
        var field  = ($btn.attr('data-field') || '').toString();

        if (!postId || !field) { return; }

        var ajaxUrl = (window.AegisSEOAdmin && AegisSEOAdmin.ajax_url) ? AegisSEOAdmin.ajax_url : (typeof ajaxurl !== 'undefined' ? ajaxurl : '');
        var nonce   = (window.AegisSEOAdmin && AegisSEOAdmin.nonce) ? AegisSEOAdmin.nonce : '';

        if (!ajaxUrl) { alert('AegisSEO: ajax URL missing'); return; }

        var oldText = $btn.text();
        $btn.prop('disabled', true).text('Generating...');

        $.post(ajaxUrl, {
            action: 'aegisseo_generate_field',
            nonce: nonce,
            post_id: postId,
            field: field
        }).done(function(resp){
            if (resp && resp.success && resp.data && typeof resp.data.value !== 'undefined') {
                var val = resp.data.value;

                // Map field -> input/textarea
                var selectorMap = {
                    'title': 'input[name="aegisseo_title"]',
                    'description': 'textarea[name="aegisseo_description"]',
                    'canonical': 'input[name="aegisseo_canonical"]',
                    'og_title': 'input[name="aegisseo_og_title"]',
                    'og_description': 'textarea[name="aegisseo_og_description"]',
                    'twitter_title': 'input[name="aegisseo_twitter_title"]',
                    'twitter_description': 'textarea[name="aegisseo_twitter_description"]'
                };

                var sel = selectorMap[field] ? selectorMap[field] : null;
                if (sel) {
                    var $el = $(sel);
                    if ($el.length) {
                        $el.val(val).trigger('change');
                    }
                }

                // Visible confirmation
                alert('Generated for: ' + field);
            } else {
                var msg = (resp && resp.data && resp.data.message) ? resp.data.message : 'Generation failed.';
                alert(msg);
            }
        }).fail(function(){
            alert('Generation failed (request error).');
        }).always(function(){
            $btn.prop('disabled', false).text(oldText);
        });
    });


})(jQuery);
