<?php
namespace AegisSEO\Admin;

use AegisSEO\SEO\Score;
use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Admin_Ajax {

    private $options;

    private $score;

    public function __construct(Options $options, Score $score) {
        $this->options = $options;
        $this->score   = $score;

        add_action('wp_ajax_aegisseo_analyze_content', array($this, 'analyze_content'));
        add_action('wp_ajax_aegisseo_bulk_save_ajax', array($this, 'bulk_save_ajax'));
        add_action('wp_ajax_aegisseo_generate_field', array($this, 'generate_field'));
    }

    public function analyze_content() {
        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => 'Forbidden'), 403);
        }
        check_ajax_referer('aegisseo_analyze', 'nonce');

        $post_id = isset($_POST['post_id']) ? (int) $_POST['post_id'] : 0;

        $focus = isset($_POST['focus']) ? sanitize_text_field(wp_unslash($_POST['focus'])) : '';
        $title = isset($_POST['title']) ? sanitize_text_field(wp_unslash($_POST['title'])) : '';
        $desc  = isset($_POST['description']) ? sanitize_textarea_field(wp_unslash($_POST['description'])) : '';
        $content = isset( $_POST['content'] ) ? wp_kses_post( wp_unslash( (string) $_POST['content'] ) ) : '';

        $result = $this->score->analyze_raw($post_id, $content, $title, $desc, $focus);

        wp_send_json_success($result);
    }
    public function bulk_save_ajax() {
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Forbidden'), 403);
        }
        check_ajax_referer('aegisseo_bulk_save_ajax', 'nonce');

        $rows = array();
        // Read and sanitize request payload early (PluginCheck: ValidatedSanitizedInput).
        $rows_input = filter_input( INPUT_POST, 'rows', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY );
        if ( null === $rows_input ) {
            $rows_input = filter_input( INPUT_POST, 'rows', FILTER_UNSAFE_RAW );
        }
        if ( null !== $rows_input ) {
            $rows_input = wp_unslash( $rows_input );
            if ( is_array( $rows_input ) ) {
                foreach ( $rows_input as $k => $v ) {
                    if ( ! is_array( $v ) ) {
                        continue;
                    }
                    $k = absint( $k );
                    if ( $k <= 0 ) {
                        continue;
                    }
                    $rows[ $k ] = array(
                        'title'       => isset( $v['title'] ) ? sanitize_text_field( wp_unslash( (string) $v['title'] ) ) : '',
                        'description' => isset( $v['description'] ) ? sanitize_textarea_field( wp_unslash( (string) $v['description'] ) ) : '',
                        'canonical'   => isset( $v['canonical'] ) ? esc_url_raw( wp_unslash( (string) $v['canonical'] ) ) : '',
                        'noindex'     => ! empty( $v['noindex'] ) ? 1 : 0,
                        'nofollow'    => ! empty( $v['nofollow'] ) ? 1 : 0,
                    );
                }
            }
        }

        foreach ($rows as $post_id => $data) {
            $post_id = (int) $post_id;
            if ($post_id <= 0) { $skipped++; continue; }
            if (!current_user_can('edit_post', $post_id)) { $skipped++; continue; }

            $title = isset($data['title']) ? sanitize_text_field(wp_unslash($data['title'])) : '';
            $desc  = isset($data['description']) ? sanitize_textarea_field(wp_unslash($data['description'])) : '';
            $canon = isset($data['canonical']) ? esc_url_raw((string) wp_unslash($data['canonical'])) : '';
            $noindex = !empty($data['noindex']) ? 1 : 0;
            $nofollow= !empty($data['nofollow']) ? 1 : 0;

            update_post_meta($post_id, '_aegisseo_title', $title);
            update_post_meta($post_id, '_aegisseo_description', $desc);
            update_post_meta($post_id, '_aegisseo_canonical', $canon);
            update_post_meta($post_id, '_aegisseo_noindex', $noindex);
            update_post_meta($post_id, '_aegisseo_nofollow', $nofollow);

            $updated++;
        }

        wp_send_json_success(array(
            'updated' => (int) $updated,
            'skipped' => (int) $skipped,
            'errors'  => $errors,
        ));
    

    }
    /**
     * Generate a single field value for a given post (metabox Generate buttons).
     * Returns generated text only; does not save automatically.
     */
    public function generate_field() {
        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => 'Forbidden'), 403);
        }

        // Reuse existing nonce so we don't need new localized data.
        $nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( (string) $_POST['nonce'] ) ) : '';
        if (!$nonce || !wp_verify_nonce($nonce, 'aegisseo_analyze')) {
            wp_send_json_error(array('message' => 'Bad nonce'), 400);
        }

        $post_id = isset($_POST['post_id']) ? (int) $_POST['post_id'] : 0;
        $field   = isset($_POST['field']) ? sanitize_key((string) wp_unslash($_POST['field'])) : '';

        if ($post_id <= 0 || $field === '') {
            wp_send_json_error(array('message' => 'Missing parameters'), 400);
        }

        if (!current_user_can('edit_post', $post_id)) {
            wp_send_json_error(array('message' => 'Forbidden'), 403);
        }

        $post = get_post($post_id);
        if (!$post) {
            wp_send_json_error(array('message' => 'Post not found'), 404);
        }

        $min_desc = (int) apply_filters('aegisseo_meta_description_min', 420);
        if ($min_desc < 180) { $min_desc = 180; }
        $max_desc = (int) apply_filters('aegisseo_meta_description_max', 920);
        if ($max_desc < $min_desc) { $max_desc = $min_desc; }

        $site_name = function_exists('get_bloginfo') ? (string) get_bloginfo('name') : '';

        $gen_title = function() use ($post, $site_name) {
            $t = (string) get_the_title($post);
            $t = trim(preg_replace('/\s+/', ' ', $t));
            if ($t === '') { $t = 'Untitled'; }

            // Light SEO suffix (keeps it professional; avoids over-long titles)
            if ($site_name !== '') {
                $candidate = $t . ' | ' . $site_name;
                if (function_exists('mb_strlen')) {
                    if (mb_strlen($candidate) <= 70) { $t = $candidate; }
                } else {
                    if (strlen($candidate) <= 70) { $t = $candidate; }
                }
            }
            return $t;
        };

        $gen_description = function() use ($post, $min_desc, $max_desc) {
            // Prefer excerpt if present and meaningful
            $excerpt = (string) $post->post_excerpt;
            $excerpt = trim(preg_replace('/\s+/', ' ', wp_strip_all_tags($excerpt)));
            if ($excerpt !== '') {
                $len = function_exists('mb_strlen') ? mb_strlen($excerpt) : strlen($excerpt);
                if ($len >= 80) {
                    // Expand excerpt with content sentences if needed
                    $base = $excerpt;
                } else {
                    $base = '';
                }
            } else {
                $base = '';
            }

            // Render content similarly to Fix Pack
            $rendered = (string) $post->post_content;

            $GLOBALS['post'] = $post;
            setup_postdata($post);

            if (function_exists('has_blocks') && has_blocks($rendered) && function_exists('do_blocks')) {
                $rendered = do_blocks($rendered);
            }

            $rendered = apply_filters( 'the_content', $rendered ); // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Core WP filter.
            wp_reset_postdata();

            $rendered = preg_replace('/\[[^\]]+\]/', ' ', (string) $rendered);
            $rendered = wp_strip_all_tags((string) $rendered);
            $rendered = html_entity_decode((string) $rendered, ENT_QUOTES, 'UTF-8');
            $rendered = preg_replace('/\s+/', ' ', (string) $rendered);
            $rendered = trim((string) $rendered);

            if ($base === '') {
                $plain_src = $rendered;
            } else {
                $plain_src = $base . ' ' . $rendered;
            }

            $plain_src = trim(preg_replace('/\s+/', ' ', (string) $plain_src));
            if ($plain_src === '') { return ''; }

            // Break into sentences and build a paragraph-like description
            $sentences = preg_split('/(?<=[\.\!\?])\s+/', $plain_src, -1, PREG_SPLIT_NO_EMPTY);
            if (!is_array($sentences) || empty($sentences)) { $sentences = array($plain_src); }

            $out = '';
            foreach ($sentences as $s) {
                $s = trim((string) $s);
                if ($s === '') { continue; }

                // Skip obvious boilerplate
                $low = strtolower($s);
                if (strpos($low, 'welcome') === 0) { continue; }

                $candidate = trim($out . ' ' . $s);
                $len = function_exists('mb_strlen') ? mb_strlen($candidate) : strlen($candidate);
                if ($len > $max_desc) { break; }
                $out = $candidate;

                $len_out = function_exists('mb_strlen') ? mb_strlen($out) : strlen($out);
                if ($len_out >= $min_desc) { break; }
            }

            $out = trim(preg_replace('/\s+/', ' ', (string) $out));
            if ($out === '') { $out = mb_substr($plain_src, 0, $max_desc); }

            // Ensure punctuation doesn't end mid-word
            if (function_exists('mb_strlen') && mb_strlen($out) > $max_desc) {
                $out = mb_substr($out, 0, $max_desc);
            } elseif (strlen($out) > $max_desc) {
                $out = substr($out, 0, $max_desc);
            }

            return trim($out);
        };

        $value = '';

        switch ($field) {
            case 'title':
                $value = $gen_title();
                break;
            case 'description':
                $value = $gen_description();
                break;
            case 'canonical':
                $value = (string) get_permalink($post_id);
                break;
            case 'og_title':
                $value = (string) get_post_meta($post_id, '_aegisseo_og_title', true);
                $value = trim((string) $value);
                if ($value === '') { $value = $gen_title(); }
                break;
            case 'og_description':
                $value = (string) get_post_meta($post_id, '_aegisseo_og_description', true);
                $value = trim((string) $value);
                if ($value === '') { $value = $gen_description(); }
                break;
            case 'twitter_title':
                $value = (string) get_post_meta($post_id, '_aegisseo_twitter_title', true);
                $value = trim((string) $value);
                if ($value === '') { $value = $gen_title(); }
                break;
            case 'twitter_description':
                $value = (string) get_post_meta($post_id, '_aegisseo_twitter_description', true);
                $value = trim((string) $value);
                if ($value === '') { $value = $gen_description(); }
                break;
            default:
                wp_send_json_error(array('message' => 'Unsupported field'), 400);
        }

        $value = is_string($value) ? $value : '';
        $value = trim(preg_replace('/\s+/', ' ', $value));

        wp_send_json_success(array(
            'field' => $field,
            'value' => $value,
        ));
    }


}
