<?php
namespace AegisSEO\Admin;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Admin_Bulk {

    private $options;

    public function __construct(Options $options) {
        $this->options = $options;

        add_action('admin_post_aegisseo_bulk_save', array($this, 'save'));
        add_action('admin_post_aegisseo_bulk_export', array($this, 'export_csv'));
        add_action('admin_post_aegisseo_bulk_import', array($this, 'import_csv'));
    }

    private function cap() {
        return current_user_can('manage_options');
    }

    /**
     * Build a CSV line without using fopen()/fputcsv() so WP.org checks pass.
     *
     * @param array $fields
     * @return string
     */
    private function csv_line(array $fields) {
        $out = array();
        foreach ($fields as $field) {
            $field = (string) $field;
            $field = str_replace("\"", "\"\"", $field);
            $out[] = '"' . $field . '"';
        }
        return implode(',', $out) . "\r\n";
    }

    public function save() {
        if (!$this->cap()) { wp_die('Forbidden'); }
        check_admin_referer('aegisseo_bulk_save');

        $rows = array();
        // Read and sanitize request payload early (PluginCheck: ValidatedSanitizedInput).
        $rows_input = filter_input( INPUT_POST, 'rows', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY );
        if ( null === $rows_input ) {
            $rows_input = filter_input( INPUT_POST, 'rows', FILTER_UNSAFE_RAW );
        }
        if ( null !== $rows_input ) {
            $rows_input = wp_unslash( $rows_input );
            if ( is_array( $rows_input ) ) {
                foreach ( $rows_input as $k => $v ) {
                    if ( ! is_array( $v ) ) {
                        continue;
                    }
                    $k = absint( $k );
                    if ( $k <= 0 ) {
                        continue;
                    }
                    $rows[ $k ] = array(
                        'title'       => isset( $v['title'] ) ? sanitize_text_field( wp_unslash( (string) $v['title'] ) ) : '',
                        'description' => isset( $v['description'] ) ? sanitize_textarea_field( wp_unslash( (string) $v['description'] ) ) : '',
                        'canonical'   => isset( $v['canonical'] ) ? esc_url_raw( wp_unslash( (string) $v['canonical'] ) ) : '',
                        'noindex'     => ! empty( $v['noindex'] ) ? 1 : 0,
                        'nofollow'    => ! empty( $v['nofollow'] ) ? 1 : 0,
                    );
                }
            }
        }

        foreach ($rows as $post_id => $data) {
            $post_id = (int) $post_id;
            if ($post_id <= 0) continue;
            if (!current_user_can('edit_post', $post_id)) continue;

            $title = isset($data['title']) ? sanitize_text_field( (string) $data['title'] ) : '';
            $desc  = isset($data['description']) ? sanitize_textarea_field( (string) $data['description'] ) : '';
            $canon = isset($data['canonical']) ? esc_url_raw( (string) $data['canonical'] ) : '';
            $noindex = !empty($data['noindex']) ? 1 : 0;
            $nofollow= !empty($data['nofollow']) ? 1 : 0;

            update_post_meta($post_id, '_aegisseo_title', $title);
            update_post_meta($post_id, '_aegisseo_description', $desc);
            update_post_meta($post_id, '_aegisseo_canonical', $canon);
            update_post_meta($post_id, '_aegisseo_noindex', $noindex);
            update_post_meta($post_id, '_aegisseo_nofollow', $nofollow);
        }

        wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=bulk&saved=1'));
        exit;
    }

    public function export_csv() {
        if (!$this->cap()) { wp_die('Forbidden'); }
        check_admin_referer('aegisseo_bulk_export');

        $post_type = isset($_GET['post_type']) ? sanitize_key(wp_unslash($_GET['post_type'])) : 'any';

        $args = array(
            'post_type'      => ($post_type === 'any' ? $this->options->get_scope_post_types() : $post_type),
            'post_status'    => array('publish','draft','pending','future','private'),
            'posts_per_page' => 2000,
            'fields'         => 'ids',
        );
        $ids = get_posts($args);

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=aegisseo-bulk-export.csv');

        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- CSV export output.
        echo $this->csv_line(array('post_id','post_type','post_title','seo_title','meta_description','canonical','noindex','nofollow'));

        foreach ($ids as $id) {
            $row = array(
                $id,
                get_post_type($id),
                get_the_title($id),
                (string) get_post_meta($id, '_aegisseo_title', true),
                (string) get_post_meta($id, '_aegisseo_description', true),
                (string) get_post_meta($id, '_aegisseo_canonical', true),
                (int) get_post_meta($id, '_aegisseo_noindex', true),
                (int) get_post_meta($id, '_aegisseo_nofollow', true),
            );
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- CSV export output.
            echo $this->csv_line($row);
        }

        exit;
    }

    public function import_csv() {
        if (!$this->cap()) { wp_die('Forbidden'); }
        check_admin_referer('aegisseo_bulk_import');

        if (empty($_FILES['csv_file']['tmp_name'])) {
            wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=bulk&import=0'));
            exit;
        }
        $tmp_name = isset($_FILES['csv_file']['tmp_name']) ? sanitize_text_field(wp_unslash($_FILES['csv_file']['tmp_name'])) : '';
        if (empty($tmp_name) || !is_uploaded_file($tmp_name)) {
            wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=bulk&import=0'));
            exit;
        }

        require_once ABSPATH . 'wp-admin/includes/file.php';
        global $wp_filesystem;
        if (!function_exists('WP_Filesystem')) {
            wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=bulk&import=0'));
            exit;
        }
        if (!WP_Filesystem()) {
            wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=bulk&import=0'));
            exit;
        }

        $csv = $wp_filesystem ? $wp_filesystem->get_contents($tmp_name) : false;
        if (false === $csv || '' === $csv) {
            wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=bulk&import=0'));
            exit;
        }

        $lines = preg_split("/\r\n|\n|\r/", trim($csv));
        if (empty($lines) || !is_array($lines)) {
            wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=bulk&import=0'));
            exit;
        }

        $header = str_getcsv(array_shift($lines));
        $count  = 0;

        foreach ($lines as $line) {
            if ('' === trim($line)) { continue; }
            $row = str_getcsv($line);
            $map = array();
            foreach ($header as $i => $k) {
                $map[$k] = isset($row[$i]) ? $row[$i] : '';
            }
            $post_id = isset($map['post_id']) ? (int) $map['post_id'] : 0;
            if ($post_id <= 0) continue;
            if (!current_user_can('edit_post', $post_id)) continue;

            update_post_meta($post_id, '_aegisseo_title', sanitize_text_field($map['seo_title'] ?? ''));
            update_post_meta($post_id, '_aegisseo_description', sanitize_textarea_field($map['meta_description'] ?? ''));
            update_post_meta($post_id, '_aegisseo_canonical', esc_url_raw($map['canonical'] ?? ''));
            update_post_meta($post_id, '_aegisseo_noindex', !empty($map['noindex']) ? 1 : 0);
            update_post_meta($post_id, '_aegisseo_nofollow', !empty($map['nofollow']) ? 1 : 0);
            $count++;
        }
        wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=bulk&import=' . (int)$count));
        exit;
    }
}
