<?php
namespace AegisSEO\Admin;

use AegisSEO\Plugin;

if (!defined('ABSPATH')) { exit; }

class Admin_Issues {

    public function __construct() {
        add_action('admin_post_aegisseo_apply_fix', array($this, 'apply_fix'));
        add_action('admin_post_aegisseo_rollback_event', array($this, 'rollback_event'));
        add_action('admin_post_aegisseo_bulk_rollback_events', array($this, 'bulk_rollback_events'));
		add_action('admin_post_aegisseo_schema_approve', array($this, 'approve_schema'));
        add_action('admin_post_aegisseo_apply_links', array($this, 'apply_links'));
		add_action('admin_post_aegisseo_send_test_alert', array($this, 'send_test_alert'));
        add_action('admin_post_aegisseo_redirect_approve', array($this, 'approve_redirect'));
        add_action('admin_post_aegisseo_redirect_dismiss', array($this, 'dismiss_redirect'));

        add_action('admin_post_aegisseo_apply_fix_pack', array($this, 'apply_fix_pack'));
        add_action('admin_post_aegisseo_rollback_fix_pack', array($this, 'rollback_fix_pack'));
    }


    private function sanitize_db_prefix( $prefix ) {
        return preg_replace( '/[^A-Za-z0-9_]/', '', (string) $prefix );
    }

    private function escape_sql_identifier( $identifier ) {
        $identifier = preg_replace( '/[^A-Za-z0-9_]/', '', (string) $identifier );
        if ( $identifier === '' ) {
            return '``';
        }
        return '`' . $identifier . '`';
    }

    private function sql_with_table( $sql, $table ) {
        return str_replace( '{{table}}', $this->escape_sql_identifier( $table ), $sql );
    }

	
	public function send_test_alert() {
		if (!current_user_can('manage_options')) { wp_die('Forbidden'); }
		check_admin_referer('aegisseo_send_test_alert');

		$plugin = \AegisSEO\Plugin::instance();
		if (isset($plugin->alerts) && $plugin->alerts) {
			$plugin->alerts->send_test();
		}

		wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=monitor&test_alert=1'));
		exit;
	}
	
    public function apply_fix() {
        if (!current_user_can('manage_options')) { wp_die('Forbidden'); }
        check_admin_referer('aegisseo_apply_fix');

        $post_id  = isset( $_POST['post_id'] ) ? absint( wp_unslash( $_POST['post_id'] ) ) : 0;
        $fix_type = isset( $_POST['fix_type'] ) ? sanitize_key( wp_unslash( $_POST['fix_type'] ) ) : '';

        if ($post_id <= 0 || empty($fix_type)) {
            wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=issues&error=1'));
            exit;
        }

        if (!current_user_can('edit_post', $post_id)) {
            wp_die('Forbidden');
        }

        $applied = 0;

        if ($fix_type === 'gen_meta_desc') {
            $old = get_post_meta($post_id, '_aegisseo_description', true);

            $post = get_post($post_id);
            $raw = '';
            if ($post) {
                $raw = (string) $post->post_excerpt;
                if ($raw === '') {
                    $raw = (string) $post->post_content;
                }
            }

            $raw = preg_replace('/<!--\s*\/?wp:.*?-->/', ' ', $raw);
            $raw = strip_shortcodes($raw);
            $raw = preg_replace('/\[[^\]]+\]/', ' ', $raw);
            $raw = wp_strip_all_tags($raw, true);
            $raw = html_entity_decode($raw, ENT_QUOTES, get_bloginfo('charset'));
            $raw = preg_replace('/\s+/u', ' ', $raw);
            $raw = trim($raw);

            $new = mb_substr($raw, 0, 155);
            $new = trim($new);


            if ($new !== '') {
                update_post_meta($post_id, '_aegisseo_description', $new);
                $applied = 1;

                $plugin = Plugin::instance();
                if (isset($plugin->events) && $plugin->events) {
                    $plugin->events->log_event('fix_applied', 'post', $post_id, '_aegisseo_description', $old, $new);
                }
            }
        }

        $redir = admin_url('admin.php?page=aegisseo&tab=issues');
        $redir = add_query_arg(array('applied' => $applied, 'fix' => $fix_type, 'post_id' => $post_id), $redir);
        wp_safe_redirect($redir);
        exit;
    }

	public function rollback_event() {
		if (!current_user_can('manage_options')) { wp_die('Forbidden'); }
		check_admin_referer('aegisseo_rollback_event');

		$event_id = isset( $_POST['event_id'] ) ? absint( wp_unslash( $_POST['event_id'] ) ) : 0;

		$plugin = Plugin::instance();
		if (!$event_id || !isset($plugin->events) || !$plugin->events) {
			wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=issues&rolled=0'));
			exit;
		}

		$event = $plugin->events->get_event($event_id);
		if (!$event || empty($event['object_type']) || empty($event['object_id']) || empty($event['meta_key'])) {
			wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=issues&rolled=0'));
			exit;
		}

		$etype    = isset($event['event_type']) ? (string) $event['event_type'] : '';
		$otype    = (string) $event['object_type'];
		$post_id  = (int) $event['object_id'];
		$meta_key = (string) $event['meta_key'];

		$allowed_keys = array(
			'_aegisseo_title',
			'_aegisseo_description',
			'_aegisseo_canonical',
			'_aegisseo_robots',
			'_aegisseo_focus_phrase',
			'_aegisseo_schema_mode',
			'_aegisseo_schema_type',
			'_aegisseo_schema_data',
			'_aegisseo_og_title',
			'_aegisseo_og_description',
			'_aegisseo_og_image',
			'_aegisseo_twitter_title',
			'_aegisseo_twitter_description',
			'_aegisseo_twitter_image',
		);

		$allowed_types = array('meta_update', 'meta_add', 'meta_delete', 'fixpack_generate_meta_descriptions');

		if ($otype !== 'post' || $post_id <= 0 || !in_array($etype, $allowed_types, true) || !in_array($meta_key, $allowed_keys, true)) {
			wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=issues&rolled=0'));
			exit;
		}

		$old = isset($event['old_value']) ? $event['old_value'] : '';
		$new = isset($event['new_value']) ? $event['new_value'] : '';

		if ($old === null || $old === '') {
			delete_post_meta($post_id, $meta_key);
		} else {
			update_post_meta($post_id, $meta_key, $old);
		}

		if (isset($plugin->events) && $plugin->events) {
			$plugin->events->log_event('rollback', 'post', $post_id, $meta_key, $new, $old);
		}

		$redir = admin_url('admin.php?page=aegisseo&tab=issues');
		$redir = add_query_arg(array('rolled' => 1, 'event_id' => $event_id, 'post_id' => $post_id), $redir);
		wp_safe_redirect($redir);
		exit;
	}

	public function bulk_rollback_events() {
		if (!current_user_can('manage_options')) { wp_die('Forbidden'); }
		check_admin_referer('aegisseo_bulk_rollback_events');

		$ids = isset( $_POST['event_ids'] ) ? array_map( 'sanitize_text_field', (array) wp_unslash( $_POST['event_ids'] ) ) : array();
		$ids = array_map( 'absint', $ids );
		$ids = array_values( array_filter( $ids ) );
$ok = 0;
		$fail = 0;

		$plugin = Plugin::instance();
		if (empty($ids) || !isset($plugin->events) || !$plugin->events) {
			wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=issues&bulk_rolled=0'));
			exit;
		}

		if (count($ids) > 200) {
			$ids = array_slice($ids, 0, 200);
		}

		$allowed_keys = array(
			'_aegisseo_title',
			'_aegisseo_description',
			'_aegisseo_canonical',
			'_aegisseo_robots',
			'_aegisseo_focus_phrase',
			'_aegisseo_schema_mode',
			'_aegisseo_schema_type',
			'_aegisseo_schema_data',
			'_aegisseo_og_title',
			'_aegisseo_og_description',
			'_aegisseo_og_image',
			'_aegisseo_twitter_title',
			'_aegisseo_twitter_description',
			'_aegisseo_twitter_image',
		);

		$allowed_types = array('meta_update', 'meta_add', 'meta_delete', 'fixpack_generate_meta_descriptions');

		foreach ($ids as $event_id) {
			$event = $plugin->events->get_event($event_id);
			if (!$event || empty($event['object_type']) || empty($event['object_id']) || empty($event['meta_key'])) {
				$fail++;
				continue;
			}

			$etype    = isset($event['event_type']) ? (string) $event['event_type'] : '';
			$otype    = (string) $event['object_type'];
			$post_id  = (int) $event['object_id'];
			$meta_key = (string) $event['meta_key'];

			if ($otype !== 'post' || $post_id <= 0) { $fail++; continue; }
			if (!in_array($etype, $allowed_types, true)) { $fail++; continue; }
			if (!in_array($meta_key, $allowed_keys, true)) { $fail++; continue; }

			$old = isset($event['old_value']) ? $event['old_value'] : '';
			$new = isset($event['new_value']) ? $event['new_value'] : '';

			if ((string)$old === (string)$new) { $fail++; continue; }

			if ($old === null || $old === '') {
				delete_post_meta($post_id, $meta_key);
			} else {
				update_post_meta($post_id, $meta_key, $old);
			}

			if (isset($plugin->events) && $plugin->events) {
				$plugin->events->log_event('rollback', 'post', $post_id, $meta_key, $new, $old);
			}

			$ok++;
		}

		$redir = admin_url('admin.php?page=aegisseo&tab=issues');
		$redir = add_query_arg(array('bulk_rolled' => 1, 'ok' => $ok, 'fail' => $fail), $redir);
		wp_safe_redirect($redir);
		exit;
	}

public function approve_schema() {
    if (!current_user_can('manage_options')) { wp_die('Forbidden'); }
    check_admin_referer('aegisseo_schema_approve');

    $post_id = isset($_POST['post_id']) ? absint($_POST['post_id']) : 0;
    $schema_type = isset($_POST['schema_type']) ? sanitize_text_field(wp_unslash($_POST['schema_type'])) : '';
    if ($post_id <= 0 || $schema_type === '') { wp_die('Invalid'); }

    $allowed = array('FAQPage','HowTo','WebPage','Article');
    if (!in_array($schema_type, $allowed, true)) { wp_die('Invalid schema type'); }

    $old = (string) get_post_meta($post_id, '_aegisseo_schema_type', true);
    update_post_meta($post_id, '_aegisseo_schema_type', $schema_type);
    update_post_meta($post_id, '_aegisseo_schema_mode', 'locked');

    $plugin = aegisseo();
    if (isset($plugin->events) && $plugin->events) {
        $plugin->events->log_event('schema_approved', 'post', $post_id, '_aegisseo_schema_type', $old, $schema_type);
    }

    $redir = admin_url('admin.php?page=aegisseo&tab=issues');
    $redir = add_query_arg(array('schema_approved' => 1, 'post_id' => $post_id), $redir);
    wp_safe_redirect($redir);
    exit;
}

public function apply_links() {
    if (!current_user_can('manage_options')) { wp_die('Forbidden'); }
    check_admin_referer('aegisseo_apply_links');

    $post_id = isset($_POST['post_id']) ? absint($_POST['post_id']) : 0;
    if ($post_id <= 0) { wp_die('Invalid'); }

    $plugin = aegisseo();
    if (empty($plugin->autopilot)) { wp_die('Autopilot not available'); }

    $suggestions = $plugin->autopilot->suggest_internal_links($post_id, 3);
    $res = $plugin->autopilot->apply_internal_links($post_id, $suggestions, 3);

    $applied = 0;
    if (!empty($res['success']) && !empty($res['new_content'])) {
        $p = get_post($post_id);
        if ($p) {
			$old_content = get_post_field('post_content', $post_id);

			wp_update_post(array(
				'ID' => $post_id,
				'post_content' => $res['new_content'],
			));

			$applied = (int) $res['changed_count'];

			if (isset($plugin->events) && $plugin->events) {
				$plugin->events->log_event(
					'links_applied',
					'post',
					$post_id,
					'post_content',
					$old_content,
					$res['new_content']
				);
			}
        }
    }

    $redir = admin_url('admin.php?page=aegisseo&tab=issues');
    $redir = add_query_arg(array('links_applied' => $applied, 'post_id' => $post_id), $redir);
    wp_safe_redirect($redir);
    exit;
}

public function approve_redirect() {
    if (!current_user_can('manage_options')) { wp_die('Forbidden'); }
    check_admin_referer('aegisseo_redirect_approve');

    $id = isset( $_POST['id'] ) ? absint( wp_unslash( $_POST['id'] ) ) : 0;
    if ($id <= 0) { wp_die('Invalid'); }

    global $wpdb;
$t = $this->sanitize_db_prefix( $wpdb->prefix ) . 'aegisseo_redirects';

if ( ! $wpdb->has_cap( 'identifier_placeholders' ) ) {
    // WP < 6.2 lacks %i support. This plugin requires WP with identifier placeholders for safe table binding.
    wp_die( 'Unsupported WordPress version.' );
}
$r = $wpdb->get_row( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $wpdb->prepare( 'SELECT * FROM %i WHERE id=%d', $t, $id ),
    ARRAY_A
); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
if (!$r) { wp_die('Not found'); }
    if (empty($r['is_suggestion']) || (int)$r['is_suggestion'] !== 1) { wp_die('Not a suggestion'); }

    $wpdb->update($t, array('enabled'=>1, 'suggestion_status'=>'approved', 'updated_at'=>gmdate('Y-m-d H:i:s')), array('id'=>$id), array('%d','%s','%s'), array('%d')); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

    $plugin = aegisseo();
    if (isset($plugin->events) && $plugin->events && !empty($r['source_post_id'])) {
        $plugin->events->log_event('redirect_approved', 'post', (int)$r['source_post_id'], 'redirect', $r['source'], $r['target']);
    }

    $redir = admin_url('admin.php?page=aegisseo&tab=issues');
    $redir = add_query_arg(array('redirect_approved' => 1), $redir);
    wp_safe_redirect($redir);
    exit;
}

public function dismiss_redirect() {
    if (!current_user_can('manage_options')) { wp_die('Forbidden'); }
    check_admin_referer('aegisseo_redirect_dismiss');

    $id = isset( $_POST['id'] ) ? absint( wp_unslash( $_POST['id'] ) ) : 0;
    if ($id <= 0) { wp_die('Invalid'); }

    global $wpdb;
$t = $this->sanitize_db_prefix( $wpdb->prefix ) . 'aegisseo_redirects';

if ( ! $wpdb->has_cap( 'identifier_placeholders' ) ) {
    // WP < 6.2 lacks %i support. This plugin requires WP with identifier placeholders for safe table binding.
    wp_die( 'Unsupported WordPress version.' );
}
$r = $wpdb->get_row( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $wpdb->prepare( 'SELECT * FROM %i WHERE id=%d', $t, $id ),
    ARRAY_A
); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
if (!$r) { wp_die('Not found'); }
    if (empty($r['is_suggestion']) || (int)$r['is_suggestion'] !== 1) { wp_die('Not a suggestion'); }

    $wpdb->update($t, array('suggestion_status'=>'dismissed', 'updated_at'=>gmdate('Y-m-d H:i:s')), array('id'=>$id), array('%s','%s'), array('%d')); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

    $redir = admin_url('admin.php?page=aegisseo&tab=issues');
    $redir = add_query_arg(array('redirect_dismissed' => 1), $redir);
    wp_safe_redirect($redir);
    exit;
}

    public function apply_fix_pack() {
        if (!current_user_can('manage_options')) { wp_die('Forbidden'); }
        check_admin_referer('aegisseo_fix_pack');

        $pack = isset( $_POST['pack'] ) ? sanitize_key( wp_unslash( $_POST['pack'] ) ) : '';
        if ($pack === '') {
            wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=issues'));
            exit;
        }

        $plugin = function_exists('aegisseo') ? aegisseo() : null;
        if ($plugin && isset($plugin->options) && (int) $plugin->options->get('fixpacks_enabled', 1) !== 1) {
            wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=issues&pack=0'));
            exit;
        }

        $pack_id = wp_generate_uuid4();
        $event_ids = array();
        $changed = 0;
        $flagged = 0;

        if ($pack === 'duplicate_titles') {
            $posts = get_posts(array(
                'post_type' => Plugin::instance()->options->get_scope_post_types(),
                'post_status' => 'publish',
                'numberposts' => 500,
                'orderby' => 'date',
                'order' => 'DESC',
                'fields' => 'ids',
            ));

            $map = array();
            foreach ((array)$posts as $pid) {
                $t = get_post_meta($pid, '_aegisseo_title', true);
                if ($t === '') {
                    $p = get_post($pid);
                    $t = $p ? (string)$p->post_title : '';
                }
                $t = trim(wp_strip_all_tags((string)$t));
                if ($t === '') { continue; }
                $key = strtolower($t);
                if (!isset($map[$key])) { $map[$key] = array(); }
                $map[$key][] = (int)$pid;
            }

            foreach ($map as $key => $ids) {
                if (count($ids) < 2) { continue; }

                $base_title = '';
                $first = (int)$ids[0];
                $base_title = get_post_meta($first, '_aegisseo_title', true);
                if ($base_title === '') {
                    $p = get_post($first);
                    $base_title = $p ? (string)$p->post_title : '';
                }
                $base_title = trim(wp_strip_all_tags((string)$base_title));
                if ($base_title === '') { continue; }

                foreach (array_slice($ids, 1) as $pid) {
                    $old = get_post_meta((int)$pid, '_aegisseo_title', true);
                    $new = $base_title . ' (' . (int)$pid . ')';

                    if ((string)$old === (string)$new) { continue; }

                    update_post_meta((int)$pid, '_aegisseo_title', $new);
                    $changed++;

                    if ($plugin && isset($plugin->events) && $plugin->events) {
                        $plugin->events->log_event('fixpack_duplicate_titles', 'post', (int)$pid, '_aegisseo_title', $old, $new);
                        global $wpdb;
                        $eid = (int) $wpdb->insert_id;
                        if ($eid > 0) { $event_ids[] = $eid; }
                    }
                }
            }
        } elseif ($pack === 'missing_og_image') {
            $opts = ($plugin && isset($plugin->options)) ? $plugin->options : new \AegisSEO\Utils\Options();
            $default = (string) $opts->get('social_default_image', '');
            if ($default === '') {
                $icon = function_exists('get_site_icon_url') ? (string) get_site_icon_url(512) : '';
                if ($icon) { $default = $icon; }
            }

            $posts = get_posts(array(
                'post_type' => Plugin::instance()->options->get_scope_post_types(),
                'post_status' => 'publish',
                'numberposts' => 500,
                'orderby' => 'date',
                'order' => 'DESC',
                'fields' => 'ids',
            ));

            foreach ((array)$posts as $pid) {
                $pid = (int)$pid;
                $og = (string) get_post_meta($pid, '_aegisseo_og_image', true);
                if ($og !== '') { continue; }
                $thumb = get_post_thumbnail_id($pid);
                if ($thumb) { continue; } 

                if ($default === '') { continue; } 

                update_post_meta($pid, '_aegisseo_og_image', $default);
                $changed++;

                if ($plugin && isset($plugin->events) && $plugin->events) {
                    $plugin->events->log_event('fixpack_missing_og_image', 'post', $pid, '_aegisseo_og_image', '', $default);
                    global $wpdb;
                    $eid = (int) $wpdb->insert_id;
                    if ($eid > 0) { $event_ids[] = $eid; }
                }
            }
        
        } elseif ($pack === 'generate_meta_descriptions') {

			$posts = get_posts(array(
				'post_type'   => 'any',
				'post_status' => 'publish',
				'numberposts' => -1,
				'fields'      => 'ids',
				'meta_query'  => array( // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
					'relation' => 'OR',
					array(
						'key'     => '_aegisseo_description',
						'value'   => 'fusion_builder_',
						'compare' => 'LIKE',
					),
					array(
						'key'     => '_aegisseo_description',
						'value'   => '[',
						'compare' => 'LIKE',
					),
					array(
						'key'     => '_aegisseo_description',
						'compare' => 'NOT EXISTS',
					),
					array(
						'key'     => '_aegisseo_description',
						'value'   => '',
						'compare' => '=',
					),
				),
			));

            foreach ((array)$posts as $pid) {
                $pid = (int)$pid;
                if ($pid <= 0) { continue; }

				$old = (string) get_post_meta($pid, '_aegisseo_description', true);
				$old_trim = trim($old);

				$is_junk = false;
				if ($old_trim !== '') {
					if ($old_trim[0] === '[') { $is_junk = true; }
					if (stripos($old_trim, 'fusion_builder_') !== false) { $is_junk = true; }
					if (stripos($old_trim, 'fusion_') !== false && stripos($old_trim, '[') !== false) { $is_junk = true; }
					if (preg_match('/\[[a-z0-9_]+\s+[^\]]+\]/i', $old_trim)) { $is_junk = true; }
				}

				$min = (int) apply_filters('aegisseo_meta_description_min', 420);
				if ($min < 180) { $min = 180; }

				$old_len = function_exists('mb_strlen') ? mb_strlen($old_trim) : strlen($old_trim);
				if ($old_trim !== '' && !$is_junk && $old_len >= $min) { continue; }

                $title = (string) get_the_title($pid);
                $focus = (string) get_post_meta($pid, '_aegisseo_focus_phrase', true);

                $raw = (string) get_post_field('post_excerpt', $pid);

                if ($raw === '') {
				$content = (string) get_post_field('post_content', $pid);
				if ($content === '') { continue; }

				$post_obj = get_post($pid);
				if (!$post_obj) { continue; }

				$rendered = (string) $post_obj->post_content;

				$GLOBALS['post'] = $post_obj;
				setup_postdata($post_obj);

				if (function_exists('has_blocks') && has_blocks($rendered) && function_exists('do_blocks')) {
					$rendered = do_blocks($rendered);
				}

				$rendered = apply_filters('the_content', $rendered); // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound

				wp_reset_postdata();

				$rendered = preg_replace('/\[[^\]]+\]/', ' ', (string)$rendered);
				$rendered = preg_replace('/<!--(.|\s)*?-->/', ' ', (string)$rendered);
				$rendered = preg_replace('#<script\b[^>]*>.*?</script>#is', ' ', (string)$rendered);
				$rendered = preg_replace('#<style\b[^>]*>.*?</style>#is', ' ', (string)$rendered);

				$plain = wp_strip_all_tags((string)$rendered, false);
				$plain = html_entity_decode($plain, ENT_QUOTES, get_bloginfo('charset'));
				$plain = preg_replace('/[\x00-\x1F\x7F]+/u', ' ', $plain);
				$plain = preg_replace("/[ \t]+/", ' ', $plain);
				$plain = preg_replace("/\r\n|\r/", "\n", $plain);
				$plain = preg_replace("/\n{3,}/", "\n\n", $plain);
				$plain = trim($plain);

				if ($plain === '') { continue; }

				$content = $plain;


				$content = html_entity_decode($content, ENT_QUOTES, get_bloginfo('charset'));
				$content = preg_replace('/[\x00-\x1F\x7F]+/u', ' ', $content);

				$content = preg_replace("/[ \t]+/", ' ', $content);
				$content = preg_replace("/\n{3,}/", "\n\n", $content);
				$content = trim((string)$content);


                    if ($content === '') { continue; }

                    $paras = preg_split('/\s{2,}|\s*\n\s*\n\s*/', $content);
                    if (!is_array($paras) || empty($paras)) { $paras = array($content); }

                    $bad_starts = array(
                        'welcome', 'hello', 'hi ', 'thanks', 'thank you', 'read more', 'learn more',
                        'click here', 'subscribe', 'sign up', 'log in', 'login', 'register',
                        'cookie', 'cookies', 'privacy policy', 'terms', 'copyright',
                        'share this', 'share on', 'follow us', 'newsletter',
                    );

                    $pick = '';
                    foreach ($paras as $p) {
                        $p = trim((string)$p);
                        if ($p === '') { continue; }

                        if (function_exists('mb_strlen')) {
                            if (mb_strlen($p) < 40) { continue; }
                        } else {
                            if (strlen($p) < 40) { continue; }
                        }

                        $low = function_exists('mb_strtolower') ? mb_strtolower($p) : strtolower($p);
                        $is_bad = false;
                        foreach ($bad_starts as $bs) {
                            if ($bs === '') { continue; }

                            if (strpos($low, $bs) === 0) { $is_bad = true; break; }
                        }
                        if ($is_bad) { continue; }

                        if (substr_count($p, '|') >= 3 || substr_count($p, '»') >= 2) { continue; }

                        $pick = $p;
                        break;
                    }

                    if ($pick === '') {

                        $sentences = preg_split('/(?<=[\.!\?])\s+/', $content);
                        if (!is_array($sentences) || empty($sentences)) { $sentences = array($content); }

                        $title_words = preg_split('/\W+/u', function_exists('mb_strtolower') ? mb_strtolower($title) : strtolower($title));
                        $focus_words = preg_split('/\W+/u', function_exists('mb_strtolower') ? mb_strtolower($focus) : strtolower($focus));

                        $keywords = array();
                        foreach ((array)$title_words as $w) { $w = trim($w); if ($w !== '' && (function_exists('mb_strlen') ? mb_strlen($w) >= 3 : strlen($w) >= 3)) $keywords[$w] = true; }
                        foreach ((array)$focus_words as $w) { $w = trim($w); if ($w !== '' && (function_exists('mb_strlen') ? mb_strlen($w) >= 3 : strlen($w) >= 3)) $keywords[$w] = true; }
                        $keywords = array_keys($keywords);

                        $best = '';
                        $best_score = -1;

                        $max_check = min(8, count($sentences)); // keep it lightweight
                        for ($i = 0; $i < $max_check; $i++) {
                            $s = trim((string)$sentences[$i]);
                            if ($s === '') { continue; }

                            if (function_exists('mb_strlen')) {
                                if (mb_strlen($s) < 30) { continue; }
                            } else {
                                if (strlen($s) < 30) { continue; }
                            }

                            $low = function_exists('mb_strtolower') ? mb_strtolower($s) : strtolower($s);

                            $is_bad = false;
                            foreach ($bad_starts as $bs) {
                                if ($bs === '') { continue; }
                                if (strpos($low, $bs) === 0) { $is_bad = true; break; }
                            }
                            if ($is_bad) { continue; }

                            $score = 0;
                            foreach ($keywords as $kw) {
                                if ($kw === '') continue;
                                if (strpos($low, $kw) !== false) { $score += 2; }
                            }

                            $score += max(0, 4 - $i);

                            if ($score > $best_score) {
                                $best_score = $score;
                                $best = $s;
                            }
                        }

                        $pick = ($best !== '') ? $best : (string)$sentences[0];
                    }

                    $raw = $pick;
                }

                
                $plain_src = (string) $raw;
                $plain_src = preg_replace('/\s+/', ' ', $plain_src);
                $plain_src = trim((string)$plain_src);

                $max = (int) apply_filters('aegisseo_meta_description_max', 920);
				$min = (int) apply_filters('aegisseo_meta_description_min', 420);
				if ($min < 180) { $min = 180; } 
				if ($max < $min) { $max = $min; }

                $sentences = preg_split('/(?<=[\.\!\?])\s+/', $plain_src, -1, PREG_SPLIT_NO_EMPTY);
                if (!is_array($sentences) || empty($sentences)) { $sentences = array($plain_src); }

                $candidate = '';
                foreach ($sentences as $s) {
                    $s = trim((string)$s);
                    if ($s === '') { continue; }

                    $next = ($candidate === '') ? $s : ($candidate . ' ' . $s);

                    $len = function_exists('mb_strlen') ? mb_strlen($next) : strlen($next);
                    if ($len > $max) {
                        break;
                    }
                    $candidate = $next;
                }

                $plain = ($candidate !== '') ? $candidate : $plain_src;

                // If still too long (single long sentence), cut at a word boundary.
                if (function_exists('mb_strlen') && function_exists('mb_substr')) {
                    if (mb_strlen($plain) > $max) {
                        $cut = mb_substr($plain, 0, $max);
                        $sp = mb_strrpos($cut, ' ');
                        if ($sp !== false && $sp > 80) { $cut = mb_substr($cut, 0, $sp); }
                        $plain = rtrim($cut) . '…';
                    }
                } else {
                    if (strlen($plain) > $max) {
                        $cut = substr($plain, 0, $max);
                        $sp = strrpos($cut, ' ');
                        if ($sp !== false && $sp > 80) { $cut = substr($cut, 0, $sp); }
                        $plain = rtrim($cut) . '...';
                    }
                }


                $new = $plain;
                if ($new === '' || $new === $old) { continue; }

                update_post_meta($pid, '_aegisseo_description', $new);
                $changed++;

                if ($plugin && isset($plugin->events) && $plugin->events) {
                    $plugin->events->log_event('fixpack_generate_meta_descriptions', 'post', $pid, '_aegisseo_description', $old, $new);
                    global $wpdb;
                    $eid = (int) $wpdb->insert_id;
                    if ($eid) { $event_ids[] = $eid; }
                }
            }

} elseif ($pack === 'orphan_pages') {
            $pages = get_posts(array(
                'post_type' => 'page',
                'post_status' => 'publish',
                'numberposts' => 200,
                'orderby' => 'date',
                'order' => 'DESC',
                'fields' => 'ids',
            ));

            global $wpdb;
            foreach ((array)$pages as $pid) {
                $pid = (int)$pid;
                $url = get_permalink($pid);
                if (!$url) { continue; }
                $path = wp_parse_url($url, PHP_URL_PATH);
                if (!$path) { continue; }

                $like1 = '%' . $wpdb->esc_like($url) . '%';
                $like2 = '%' . $wpdb->esc_like($path) . '%';
                $cnt = (int) $wpdb->get_var($wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    "SELECT COUNT(ID) FROM {$wpdb->posts} WHERE post_status='publish' AND post_type IN ('post','page') AND post_content LIKE %s",
                    $like2
                ));
                if ($cnt > 0) { continue; }

                $old = (string) get_post_meta($pid, '_aegisseo_orphan', true);
                if ($old === '1') { continue; }
                update_post_meta($pid, '_aegisseo_orphan', '1');
                $flagged++;

                if ($plugin && isset($plugin->events) && $plugin->events) {
                    $plugin->events->log_event('fixpack_orphan_flagged', 'post', $pid, '_aegisseo_orphan', $old, '1');
                    global $wpdb;
                    $eid = (int) $wpdb->insert_id;
                    if ($eid > 0) { $event_ids[] = $eid; }
                }
            }
        }

        $key = 'aegisseo_fixpacks_log';
        $log = get_option($key, array());
        if (!is_array($log)) { $log = array(); }
        $log[$pack_id] = array(
            'pack' => $pack,
            'created_at' => time(),
            'event_ids' => array_values(array_unique(array_map('intval', (array)$event_ids))),
            'changed' => (int)$changed,
            'flagged' => (int)$flagged,
        );
        if (count($log) > 25) {
            uasort($log, function($a,$b){ return (int)$b['created_at'] - (int)$a['created_at']; });
            $log = array_slice($log, 0, 25, true);
        }
        update_option($key, $log, false);

        $qs = 'admin.php?page=aegisseo&tab=issues&pack=1&pack_id=' . urlencode($pack_id) . '&pack_name=' . urlencode($pack) . '&changed=' . (int)$changed . '&flagged=' . (int)$flagged;
        wp_safe_redirect(admin_url($qs));
        exit;
    }

    public function rollback_fix_pack() {
        if (!current_user_can('manage_options')) { wp_die('Forbidden'); }
        check_admin_referer('aegisseo_fix_pack_rollback');

        $pack_id = isset($_GET['pack_id']) ? sanitize_text_field(wp_unslash($_GET['pack_id'])) : '';
        if ($pack_id === '') {
            wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=issues'));
            exit;
        }

        $log = get_option('aegisseo_fixpacks_log', array());
        if (!is_array($log) || empty($log[$pack_id]) || empty($log[$pack_id]['event_ids'])) {
            wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=issues&rolled_pack=0'));
            exit;
        }

        $event_ids = array_map('intval', (array)$log[$pack_id]['event_ids']);
        $rolled = 0;

        $plugin = function_exists('aegisseo') ? aegisseo() : null;
        global $wpdb;
        $table = $this->sanitize_db_prefix( $wpdb->prefix ) . 'aegisseo_events';

        foreach ($event_ids as $eid) {
        if ( ! $wpdb->has_cap( 'identifier_placeholders' ) ) {
        // WP < 6.2 lacks %i support. This plugin requires WP with identifier placeholders for safe table binding.
        continue;
    }
    $row = $wpdb->get_row( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->prepare( 'SELECT * FROM %i WHERE id=%d', $table, $eid ),
        ARRAY_A
    ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
if (!$row) { continue; }

            $post_id = isset($row['object_id']) ? (int)$row['object_id'] : 0;
            $meta_key = isset($row['meta_key']) ? (string)$row['meta_key'] : '';
            $old = isset($row['old_value']) ? (string)$row['old_value'] : '';
            $new = isset($row['new_value']) ? (string)$row['new_value'] : '';

            if ($post_id <= 0 || $meta_key === '') { continue; }

            if (!in_array($meta_key, array('_aegisseo_title','_aegisseo_og_image','_aegisseo_orphan'), true)) { continue; }

            if ($old === '' || $old === null) {
                delete_post_meta($post_id, $meta_key);
            } else {
                update_post_meta($post_id, $meta_key, $old);
            }

            if ($plugin && isset($plugin->events) && $plugin->events) {
                $plugin->events->log_event('fixpack_rollback', 'post', $post_id, $meta_key, $new, $old);
            }

            $rolled++;
        }

        $log[$pack_id]['rolled_at'] = time();
        $log[$pack_id]['rolled'] = (int)$rolled;
        update_option('aegisseo_fixpacks_log', $log, false);

        wp_safe_redirect(admin_url('admin.php?page=aegisseo&tab=issues&rolled_pack=1&pack_id=' . urlencode($pack_id) . '&rolled=' . (int)$rolled));
        exit;
    }

}