<?php
namespace AegisSEO\Admin;

use AegisSEO\Utils\Options;
use AegisSEO\Utils\Sanitize;

if (!defined('ABSPATH')) { exit; }

class Admin_Metabox {

    private function role_allows() {
        $opts = $this->options->get_all();
        $allowed = $opts['allowed_roles'] ?? array('administrator'=>1);
        $user = wp_get_current_user();
        if (!$user || empty($user->roles)) { return current_user_can('manage_options'); }
        foreach ($user->roles as $r) { if (!empty($allowed[$r])) return true; }
        return false;
    }

    private $options;

    public function __construct(Options $options) {
        $this->options = $options;

        add_action('add_meta_boxes', array($this, 'register'));
        add_action('save_post', array($this, 'save'), 10, 2);
        add_action('admin_enqueue_scripts', array($this, 'enqueue_editor_assets'));
		add_action('wp_ajax_aegisseo_metabox_quicksave', array($this, 'ajax_quicksave'));
    }

    public function register() {
        if (!$this->role_allows()) { return; }

        $post_types = (array) $this->options->get_scope_post_types();
        foreach ($post_types as $pt) {
            add_meta_box(
                'aegisseo_metabox',
                __('aegisseo', 'aegisseo'),
                array($this, 'render'),
                $pt,
                'normal',
                'high'
            );
        }
    }

    
    /**
     * Enqueue admin assets on the post editor where the AegisSEO metabox appears.
     * Needed for per-field Generate buttons to work.
     */
    public function enqueue_editor_assets($hook) {
        // Only on editor screens.
        if ($hook !== 'post.php' && $hook !== 'post-new.php') { return; }
        if (!$this->role_allows()) { return; }

        $screen = function_exists('get_current_screen') ? get_current_screen() : null;
        if (!$screen || empty($screen->post_type)) { return; }

        // Respect AegisSEO scope post types.
        $post_types = (array) $this->options->get_scope_post_types();
        if (!in_array($screen->post_type, $post_types, true)) { return; }

        wp_enqueue_style('aegisseo-admin', AEGISSEO_URL . 'assets/css/admin.css', array('dashicons'), AEGISSEO_VERSION);
        wp_enqueue_script('aegisseo-admin', AEGISSEO_URL . 'assets/js/admin.js', array('jquery'), AEGISSEO_VERSION, true);

        // Provide ajax_url + nonce for editor actions.
        wp_localize_script('aegisseo-admin', 'AegisSEOAdmin', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('aegisseo_analyze'),
			'save_nonce' => wp_create_nonce('aegisseo_metabox_quicksave'),
        ));

        // Per-field Generate buttons (metabox)
        wp_enqueue_script('aegisseo-generate', AEGISSEO_URL . 'assets/js/admin-generate.js', array('jquery'), AEGISSEO_VERSION, true);
        wp_localize_script('aegisseo-generate', 'AegisSEOGenerate', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('aegisseo_analyze'),
        ));

    }

public function render($post) {
        wp_nonce_field('aegisseo_metabox_save', 'aegisseo_nonce');

        // Ensure metabox assets load on editor screens (Generate buttons + live previews).
        // Some sites only enqueue these on AegisSEO admin pages; this guarantees the metabox works.
        if (!wp_script_is('aegisseo-admin', 'enqueued') && !wp_script_is('aegisseo-admin', 'done')) {
            wp_enqueue_style('aegisseo-admin', AEGISSEO_URL . 'assets/css/admin.css', array('dashicons'), AEGISSEO_VERSION);
            wp_enqueue_script('aegisseo-admin', AEGISSEO_URL . 'assets/js/admin.js', array('jquery'), AEGISSEO_VERSION, true);

            wp_localize_script('aegisseo-admin', 'AegisSEOAdmin', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce'    => wp_create_nonce('aegisseo_analyze'),
				'save_nonce' => wp_create_nonce('aegisseo_metabox_quicksave'),
            ));

        // Per-field Generate buttons (metabox)
        wp_enqueue_script('aegisseo-generate', AEGISSEO_URL . 'assets/js/admin-generate.js', array('jquery'), AEGISSEO_VERSION, true);
        wp_localize_script('aegisseo-generate', 'AegisSEOGenerate', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('aegisseo_analyze'),
        ));

        }

        $pid = (int) $post->ID;


        $title = get_post_meta($post->ID, '_aegisseo_title', true);
		$desc  = (string) get_post_meta($post->ID, '_aegisseo_description', true);

		// If stored description is builder/shortcode junk, do NOT show it in the UI.
		// Treat it as empty so AegisSEO can auto-generate a real meta description.
		$desc_trim = trim($desc);
		$desc_is_junk = false;
		if ($desc_trim !== '') {
			if ($desc_trim[0] === '[') { $desc_is_junk = true; }
			if (stripos($desc_trim, 'fusion_builder_') !== false) { $desc_is_junk = true; }
			if (preg_match('/\[[a-z0-9_]+\s+[^\]]+\]/i', $desc_trim)) { $desc_is_junk = true; }
		}
		if ($desc_is_junk) {
			$desc = '';
		}
        $canon = get_post_meta($post->ID, '_aegisseo_canonical', true);
        $noindex = (int) get_post_meta($post->ID, '_aegisseo_noindex', true);
        $nofollow= (int) get_post_meta($post->ID, '_aegisseo_nofollow', true);

        $focus = get_post_meta($post->ID, '_aegisseo_focus_phrase', true);

        $schema_type = get_post_meta($post->ID, '_aegisseo_schema_type', true);
        $schema_data_raw = get_post_meta($post->ID, '_aegisseo_schema_data', true);
        $schema_data = array();
        if (is_string($schema_data_raw) && $schema_data_raw !== '') {
            $tmp = json_decode($schema_data_raw, true);
            if (is_array($tmp)) { $schema_data = $tmp; }
        }


        echo '<p><label><strong>' . esc_html__('Focus Phrase', 'aegisseo') . '</strong></label><br />';
        echo '<input type="text" class="widefat" name="aegisseo_focus_phrase" value="' . esc_attr($focus) . '" placeholder="' . esc_attr__('Optional focus phrase for Aegis Score', 'aegisseo') . '" /></p>';
$templates = array();
if (!empty(aegisseo()->schema) && method_exists(aegisseo()->schema, 'get_schema_templates')) {
    $templates = aegisseo()->schema->get_schema_templates();
}

echo '<hr style="margin:14px 0;" />';
echo '<h4 style="margin:0 0 8px 0;">' . esc_html__('Schema', 'aegisseo') . '</h4>';

echo '<p><label><strong>' . esc_html__('Schema Type (Override)', 'aegisseo') . '</strong></label><br />';
echo '<select name="aegisseo_schema_type" id="aegisseo_schema_type" style="min-width:260px;">';
echo '<option value="">' . esc_html__('(Use global post-type default)', 'aegisseo') . '</option>';
foreach ((array)$templates as $k => $label) {
    echo '<option value="' . esc_attr($k) . '"' . selected((string)$schema_type, (string)$k, false) . '>' . esc_html($label) . '</option>';
}
echo '</select>';
echo '<span class="description" style="margin-left:8px;">' . esc_html__('Only set this if you want to override the post-type default.', 'aegisseo') . '</span>';
echo '</p>';

echo '<div id="aegisseo_schema_fields" style="padding:10px; border:1px solid #ddd; border-radius:6px; background:#fff;">';

echo '<div class="aegisseo-schema-group" data-schema="Product" style="display:none;">';
echo '<p><label><strong>' . esc_html__('Price', 'aegisseo') . '</strong></label><br/><input type="text" class="widefat" name="aegisseo_schema_data[price]" value="' . esc_attr($schema_data['price'] ?? '') . '" placeholder="19.99" /></p>';
echo '<p><label><strong>' . esc_html__('Currency', 'aegisseo') . '</strong></label><br/><input type="text" class="widefat" name="aegisseo_schema_data[currency]" value="' . esc_attr($schema_data['currency'] ?? '') . '" placeholder="USD" /></p>';
echo '<p><label><strong>' . esc_html__('Availability', 'aegisseo') . '</strong></label><br/><input type="text" class="widefat" name="aegisseo_schema_data[availability]" value="' . esc_attr($schema_data['availability'] ?? '') . '" placeholder="InStock" /></p>';
echo '<p><label><strong>' . esc_html__('Brand (optional)', 'aegisseo') . '</strong></label><br/><input type="text" class="widefat" name="aegisseo_schema_data[brand]" value="' . esc_attr($schema_data['brand'] ?? '') . '" /></p>';
echo '<p><label><strong>' . esc_html__('SKU (optional)', 'aegisseo') . '</strong></label><br/><input type="text" class="widefat" name="aegisseo_schema_data[sku]" value="' . esc_attr($schema_data['sku'] ?? '') . '" /></p>';
echo '</div>';

echo '<div class="aegisseo-schema-group" data-schema="LocalBusiness" style="display:none;">';
echo '<p><label><strong>' . esc_html__('Phone', 'aegisseo') . '</strong></label><br/><input type="text" class="widefat" name="aegisseo_schema_data[telephone]" value="' . esc_attr($schema_data['telephone'] ?? '') . '" /></p>';
echo '<p><label><strong>' . esc_html__('Address', 'aegisseo') . '</strong></label><br/><input type="text" class="widefat" name="aegisseo_schema_data[address]" value="' . esc_attr($schema_data['address'] ?? '') . '" /></p>';
echo '<p><label><strong>' . esc_html__('Geo Latitude (optional)', 'aegisseo') . '</strong></label><br/><input type="text" class="widefat" name="aegisseo_schema_data[geo_lat]" value="' . esc_attr($schema_data['geo_lat'] ?? '') . '" /></p>';
echo '<p><label><strong>' . esc_html__('Geo Longitude (optional)', 'aegisseo') . '</strong></label><br/><input type="text" class="widefat" name="aegisseo_schema_data[geo_lng]" value="' . esc_attr($schema_data['geo_lng'] ?? '') . '" /></p>';
echo '</div>';

echo '<div class="aegisseo-schema-group" data-schema="Event" style="display:none;">';
echo '<p><label><strong>' . esc_html__('Start Date (ISO 8601)', 'aegisseo') . '</strong></label><br/><input type="text" class="widefat" name="aegisseo_schema_data[startDate]" value="' . esc_attr($schema_data['startDate'] ?? '') . '" placeholder="2025-12-31T19:00:00-08:00" /></p>';
echo '<p><label><strong>' . esc_html__('End Date (ISO 8601, optional)', 'aegisseo') . '</strong></label><br/><input type="text" class="widefat" name="aegisseo_schema_data[endDate]" value="' . esc_attr($schema_data['endDate'] ?? '') . '" /></p>';
echo '<p><label><strong>' . esc_html__('Location Name', 'aegisseo') . '</strong></label><br/><input type="text" class="widefat" name="aegisseo_schema_data[location_name]" value="' . esc_attr($schema_data['location_name'] ?? '') . '" /></p>';
echo '<p><label><strong>' . esc_html__('Location Address', 'aegisseo') . '</strong></label><br/><input type="text" class="widefat" name="aegisseo_schema_data[location_address]" value="' . esc_attr($schema_data['location_address'] ?? '') . '" /></p>';
echo '</div>';

echo '<p class="description" style="margin:8px 0 0 0;">' . esc_html__('FAQPage and HowTo are detected from content automatically. Use Q: ... A: ... for FAQs, and numbered steps for HowTo.', 'aegisseo') . '</p>';
echo '</div>';

echo "<script>
(function(){
    function updateSchemaGroups(){
        var sel=document.getElementById('aegisseo_schema_type');
        if(!sel) return;
        var type=sel.value||'';
        var groups=document.querySelectorAll('#aegisseo_schema_fields .aegisseo-schema-group');
        for(var i=0;i<groups.length;i++){
            var g=groups[i];
            g.style.display = (type && g.getAttribute('data-schema')===type) ? 'block' : 'none';
        }
    }
    document.addEventListener('DOMContentLoaded', updateSchemaGroups);
    var sel=document.getElementById('aegisseo_schema_type');
    if(sel){ sel.addEventListener('change', updateSchemaGroups); }
    updateSchemaGroups();
})();
</script>";

        echo '<p><label><strong>' . esc_html__('SEO Title', 'aegisseo') . '</strong><button type="button" class="button button-small aegisseo-generate-btn" data-post-id="' . (int) $pid . '" data-field="title" style="float:right;">' . esc_html__('Generate', 'aegisseo') . '</button></label><br />';
        echo '<input type="text" class="widefat" name="aegisseo_title" value="' . esc_attr($title) . '" placeholder="' . esc_attr__('Leave blank to use templates', 'aegisseo') . '" /></p>';

        echo '<p><label><strong>' . esc_html__('Meta Description', 'aegisseo') . '</strong><button type="button" class="button button-small aegisseo-generate-btn" data-post-id="' . (int) $pid . '" data-field="description" style="float:right;">' . esc_html__('Generate', 'aegisseo') . '</button></label><br />';
        echo '<textarea class="widefat" rows="3" name="aegisseo_description" placeholder="' . esc_attr__('Leave blank to use templates', 'aegisseo') . '">' . esc_textarea($desc) . '</textarea></p>';

        echo '<p><label><strong>' . esc_html__('Canonical URL', 'aegisseo') . '</strong><button type="button" class="button button-small aegisseo-generate-btn" data-post-id="' . (int) $pid . '" data-field="canonical" style="float:right;">' . esc_html__('Generate', 'aegisseo') . '</button></label><br />';
        echo '<input type="url" class="widefat" name="aegisseo_canonical" value="' . esc_attr($canon) . '" placeholder="' . esc_attr__('Leave blank to use default permalink', 'aegisseo') . '" /></p>';

        echo '<p><label><input type="checkbox" name="aegisseo_noindex" value="1" ' . checked(1, $noindex, false) . ' /> ' . esc_html__('Noindex', 'aegisseo') . '</label> ';
        echo '<label style="margin-left:12px;"><input type="checkbox" name="aegisseo_nofollow" value="1" ' . checked(1, $nofollow, false) . ' /> ' . esc_html__('Nofollow', 'aegisseo') . '</label></p>';

        echo '<p class="description">' . esc_html__('Phase 1: Titles, meta description, canonical, robots, plus social overrides.', 'aegisseo') . '</p>';

        echo '<hr style="margin:16px 0;" />';
        echo '<h4 style="margin:0 0 8px 0;">' . esc_html__('Social Overrides (Phase 1)', 'aegisseo') . '</h4>';

        $og_title = get_post_meta($post->ID, '_aegisseo_og_title', true);
        $og_desc  = get_post_meta($post->ID, '_aegisseo_og_description', true);
        $og_img   = get_post_meta($post->ID, '_aegisseo_og_image', true);
        $tw_title = get_post_meta($post->ID, '_aegisseo_twitter_title', true);
        $tw_desc  = get_post_meta($post->ID, '_aegisseo_twitter_description', true);
        $tw_img   = get_post_meta($post->ID, '_aegisseo_twitter_image', true);

        echo '<p><label><strong>' . esc_html__('Open Graph Title', 'aegisseo') . '</strong><button type="button" class="button button-small aegisseo-generate-btn" data-post-id="' . (int) $pid . '" data-field="og_title" style="float:right;">' . esc_html__('Generate', 'aegisseo') . '</button></label><br />';
        echo '<input type="text" class="widefat" name="aegisseo_og_title" value="' . esc_attr($og_title) . '" placeholder="' . esc_attr__('Leave blank to use SEO Title', 'aegisseo') . '" /></p>';

        echo '<p><label><strong>' . esc_html__('Open Graph Description', 'aegisseo') . '</strong><button type="button" class="button button-small aegisseo-generate-btn" data-post-id="' . (int) $pid . '" data-field="og_description" style="float:right;">' . esc_html__('Generate', 'aegisseo') . '</button></label><br />';
        echo '<textarea class="widefat" rows="2" name="aegisseo_og_description" placeholder="' . esc_attr__('Leave blank to use Meta Description', 'aegisseo') . '">' . esc_textarea($og_desc) . '</textarea></p>';

        echo '<p><label><strong>' . esc_html__('Open Graph Image URL', 'aegisseo') . '</strong></label><br />';
        echo '<input type="url" class="widefat" name="aegisseo_og_image" value="' . esc_attr($og_img) . '" placeholder="' . esc_attr__('Leave blank to use featured image', 'aegisseo') . '" /></p>';

        echo '<p><label><strong>' . esc_html__('Twitter Title', 'aegisseo') . '</strong><button type="button" class="button button-small aegisseo-generate-btn" data-post-id="' . (int) $pid . '" data-field="twitter_title" style="float:right;">' . esc_html__('Generate', 'aegisseo') . '</button></label><br />';
        echo '<input type="text" class="widefat" name="aegisseo_twitter_title" value="' . esc_attr($tw_title) . '" placeholder="' . esc_attr__('Leave blank to use SEO Title', 'aegisseo') . '" /></p>';

        echo '<p><label><strong>' . esc_html__('Twitter Description', 'aegisseo') . '</strong><button type="button" class="button button-small aegisseo-generate-btn" data-post-id="' . (int) $pid . '" data-field="twitter_description" style="float:right;">' . esc_html__('Generate', 'aegisseo') . '</button></label><br />';
        echo '<textarea class="widefat" rows="2" name="aegisseo_twitter_description" placeholder="' . esc_attr__('Leave blank to use Meta Description', 'aegisseo') . '">' . esc_textarea($tw_desc) . '</textarea></p>';

        echo '<p><label><strong>' . esc_html__('Twitter Image URL', 'aegisseo') . '</strong></label><br />';
        echo '<input type="url" class="widefat" name="aegisseo_twitter_image" value="' . esc_attr($tw_img) . '" placeholder="' . esc_attr__('Leave blank to use featured image', 'aegisseo') . '" /></p>';

        echo '<p class="description">' . esc_html__('Tip: Use absolute URLs (https://...).', 'aegisseo') . '</p>';

        $permalink = get_permalink($post->ID);
        echo '<div class="aegisseo-metabox-phaseA" data-post-id="' . (int)$post->ID . '" data-permalink="' . esc_attr($permalink) . '">';

        echo '<div class="aegisseo-mb-grid">';
            echo '<div class="aegisseo-mb-col">';
                echo '<h4 style="margin:8px 0;">' . esc_html__('Live SERP Preview', 'aegisseo') . '</h4>';
                echo '<div class="aegisseo-preview-toggle"><button type="button" class="button button-small is-active" data-device="desktop">' . esc_html__('Desktop', 'aegisseo') . '</button> <button type="button" class="button button-small" data-device="mobile">' . esc_html__('Mobile', 'aegisseo') . '</button></div>';
                echo '<div class="aegisseo-serp-preview" data-device="desktop">';
                    echo '<div class="aegisseo-serp-title"></div>';
                    echo '<div class="aegisseo-serp-url"></div>';
                    echo '<div class="aegisseo-serp-desc"></div>';
                echo '</div>';
            echo '</div>';

            echo '<div class="aegisseo-mb-col">';
                echo '<h4 style="margin:8px 0;">' . esc_html__('Live Social Preview', 'aegisseo') . '</h4>';
                echo '<div class="aegisseo-preview-toggle"><button type="button" class="button button-small is-active" data-social="og">' . esc_html__('Open Graph', 'aegisseo') . '</button> <button type="button" class="button button-small" data-social="twitter">' . esc_html__('Twitter', 'aegisseo') . '</button></div>';
                echo '<div class="aegisseo-social-preview" data-social="og">';
                    echo '<div class="aegisseo-social-img"></div>';
                    echo '<div class="aegisseo-social-body"><div class="aegisseo-social-site"></div><div class="aegisseo-social-title"></div><div class="aegisseo-social-desc"></div></div>';
                echo '</div>';
            echo '</div>';
        echo '</div>';

        echo '<div class="aegisseo-actionable">';
            echo '<h4 style="margin:12px 0 6px 0;">' . esc_html__('Actionable SEO Score', 'aegisseo') . '</h4>';
            echo '<div class="aegisseo-score-row"><span class="aegisseo-score-badge">0</span> <span class="aegisseo-score-label">' . esc_html__('Updating…', 'aegisseo') . '</span></div>';
            echo '<ol class="aegisseo-actions-list"></ol>';
            echo '<p class="description" style="margin-top:8px;">' . esc_html__('This score updates live from your draft content (does not save anything).', 'aegisseo') . '</p>';
        echo '</div>';
		// Metabox Save button (saves only AegisSEO meta fields via AJAX; does NOT update the post content/status)
		echo '<div class="aegisseo-metabox-save" style="text-align:right; margin-top:12px;">';
			echo '<button type="button" class="button button-primary" id="aegisseo-metabox-save" data-post-id="' . (int) $pid . '">' . esc_html__('Save', 'aegisseo') . '</button>';
			echo '<span id="aegisseo-metabox-save-status" style="margin-left:10px;"></span>';
		echo '</div>';

		echo "<script>
		jQuery(function($){
			var \$btn = $('#aegisseo-metabox-save');
			if(!\$btn.length) return;

			\$btn.on('click', function(e){
				e.preventDefault();

				var pid = parseInt(\$btn.attr('data-post-id') || '0', 10);
				if(!pid) return;

				if(typeof window.AegisSEOAdmin === 'undefined' || !AegisSEOAdmin.ajax_url || !AegisSEOAdmin.save_nonce){
					alert('AegisSEO save is not available (missing ajax_url/save_nonce).');
					return;
				}

				var \$box = $('#aegisseo_metabox');
				var \$status = $('#aegisseo-metabox-save-status');

				\$status.text('" . esc_js(__('Saving…', 'aegisseo')) . "');
				\$btn.prop('disabled', true);

				// Serialize only fields inside this metabox
				var data = {
					action: 'aegisseo_metabox_quicksave',
					nonce: AegisSEOAdmin.save_nonce,
					post_id: pid
				};

				// Include all inputs/textareas/selects inside the metabox
				var arr = \$box.find('input, textarea, select').serializeArray();
				for(var i=0;i<arr.length;i++){
					data[arr[i].name] = arr[i].value;
				}

				// Ensure unchecked boxes are sent as 0
				data['aegisseo_noindex']  = \$box.find('input[name=\"aegisseo_noindex\"]').is(':checked') ? 1 : 0;
				data['aegisseo_nofollow'] = \$box.find('input[name=\"aegisseo_nofollow\"]').is(':checked') ? 1 : 0;

				$.post(AegisSEOAdmin.ajax_url, data)
					.done(function(resp){
						if(resp && resp.success){
							\$status.text('" . esc_js(__('Saved.', 'aegisseo')) . "');
						} else {
							var msg = (resp && resp.data && resp.data.message) ? resp.data.message : 'Save failed.';
							\$status.text(msg);
						}
					})
					.fail(function(){
						\$status.text('Save failed.');
					})
					.always(function(){
						\$btn.prop('disabled', false);
						setTimeout(function(){ \$status.text(''); }, 2500);
					});
			});
		});
		</script>";
        echo '</div>';
    }

	public function ajax_quicksave() {
		// Nonce
		$p = wp_unslash($_POST);
		$nonce = isset($p['nonce']) ? sanitize_text_field($p['nonce']) : '';
		if (!wp_verify_nonce($nonce, 'aegisseo_metabox_quicksave')) {
			wp_send_json_error(array('message' => 'Invalid nonce.'));
		}
		$post_id = isset($p['post_id']) ? absint($p['post_id']) : 0;
		if ($post_id <= 0) {
			wp_send_json_error(array('message' => 'Missing post_id.'));
		}

		if (!current_user_can('edit_post', $post_id)) {
			wp_send_json_error(array('message' => 'Insufficient permissions.'));
		}

		$pt = get_post_type($post_id);
		$pto = get_post_type_object($pt);
		if (!$pto || empty($pto->public)) {
			wp_send_json_error(array('message' => 'Invalid post type.'));
		}

		// Same sanitization + meta keys as save()
		$title = Sanitize::text($p['aegisseo_title'] ?? '', 300);
		$desc  = Sanitize::text($p['aegisseo_description'] ?? '', 1000);

		// Never store shortcode/builder markup as a "meta description".
		$desc_trim = trim((string)$desc);
		$desc_is_junk = false;
		if ($desc_trim !== '') {
			if ($desc_trim[0] === '[') { $desc_is_junk = true; }
			if (stripos($desc_trim, 'fusion_builder_') !== false) { $desc_is_junk = true; }
			if (preg_match('/\[[a-z0-9_]+\s+[^\]]+\]/i', $desc_trim)) { $desc_is_junk = true; }
		}
		if ($desc_is_junk) {
			$desc = '';
		}

		$focus = Sanitize::text($p['aegisseo_focus_phrase'] ?? '', 120);
		$canon = Sanitize::url($p['aegisseo_canonical'] ?? '', 500);
		$noindex = Sanitize::bool_int($p['aegisseo_noindex'] ?? 0);
		$nofollow= Sanitize::bool_int($p['aegisseo_nofollow'] ?? 0);

		$og_title = Sanitize::text($p['aegisseo_og_title'] ?? '', 300);
		$og_desc  = Sanitize::text($p['aegisseo_og_description'] ?? '', 1000);
		$og_img   = Sanitize::url($p['aegisseo_og_image'] ?? '', 500);
		$tw_title = Sanitize::text($p['aegisseo_twitter_title'] ?? '', 300);
		$tw_desc  = Sanitize::text($p['aegisseo_twitter_description'] ?? '', 1000);
		$tw_img   = Sanitize::url($p['aegisseo_twitter_image'] ?? '', 500);

		if ($title !== '') update_post_meta($post_id, '_aegisseo_title', $title);
		else delete_post_meta($post_id, '_aegisseo_title');

		if ($desc !== '') update_post_meta($post_id, '_aegisseo_description', $desc);
		else delete_post_meta($post_id, '_aegisseo_description');

		if (!empty($focus)) update_post_meta($post_id, '_aegisseo_focus_phrase', $focus);
		else delete_post_meta($post_id, '_aegisseo_focus_phrase');

		if ($canon !== '') update_post_meta($post_id, '_aegisseo_canonical', $canon);
		else delete_post_meta($post_id, '_aegisseo_canonical');

		if ($noindex) update_post_meta($post_id, '_aegisseo_noindex', 1);
		else delete_post_meta($post_id, '_aegisseo_noindex');

		if ($nofollow) update_post_meta($post_id, '_aegisseo_nofollow', 1);
		else delete_post_meta($post_id, '_aegisseo_nofollow');

		if ($og_title !== '') update_post_meta($post_id, '_aegisseo_og_title', $og_title);
		else delete_post_meta($post_id, '_aegisseo_og_title');

		if ($og_desc !== '') update_post_meta($post_id, '_aegisseo_og_description', $og_desc);
		else delete_post_meta($post_id, '_aegisseo_og_description');

		if ($og_img !== '') update_post_meta($post_id, '_aegisseo_og_image', $og_img);
		else delete_post_meta($post_id, '_aegisseo_og_image');

		if ($tw_title !== '') update_post_meta($post_id, '_aegisseo_twitter_title', $tw_title);
		else delete_post_meta($post_id, '_aegisseo_twitter_title');

		if ($tw_desc !== '') update_post_meta($post_id, '_aegisseo_twitter_description', $tw_desc);
		else delete_post_meta($post_id, '_aegisseo_twitter_description');

		if ($tw_img !== '') update_post_meta($post_id, '_aegisseo_twitter_image', $tw_img);
		else delete_post_meta($post_id, '_aegisseo_twitter_image');

		$schema_type = Sanitize::text($p['aegisseo_schema_type'] ?? '', 50);
		if ($schema_type !== '') update_post_meta($post_id, '_aegisseo_schema_type', $schema_type);
		else delete_post_meta($post_id, '_aegisseo_schema_type');

		$schema_data = $p['aegisseo_schema_data'] ?? array();
		$schema_out = array();
		if (is_array($schema_data)) {
			foreach ($schema_data as $k => $v) {
				$k = sanitize_key($k);
				$schema_out[$k] = Sanitize::text($v, 300);
			}
		}

		$has = false;
		foreach ($schema_out as $v) { if (trim((string)$v) !== '') { $has = true; break; } }
		if ($has) update_post_meta($post_id, '_aegisseo_schema_data', wp_json_encode($schema_out));
		else delete_post_meta($post_id, '_aegisseo_schema_data');

		wp_send_json_success(array('message' => 'Saved.'));
	}

    public function save($post_id, $post) {
        $nonce = isset($p['aegisseo_nonce']) ? sanitize_text_field(wp_unslash($p['aegisseo_nonce'])) : '';
        if ($nonce === '' || !wp_verify_nonce($nonce, 'aegisseo_metabox_save')) {
            return;
        }
        $p = wp_unslash($_POST);
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) { return; }
        if (!current_user_can('edit_post', $post_id)) { return; }

        $pt = get_post_type($post_id);
        $pto = get_post_type_object($pt);
        if (!$pto || empty($pto->public)) { return; }

        $title = Sanitize::text($p['aegisseo_title'] ?? '', 300);
        $desc  = Sanitize::text($p['aegisseo_description'] ?? '', 1000);
		// Never store shortcode/builder markup as a "meta description".
		// If it looks like Avada/Fusion or generic shortcode noise, treat it as empty.
		$desc_trim = trim((string)$desc);
		$desc_is_junk = false;
		if ($desc_trim !== '') {
			if ($desc_trim[0] === '[') { $desc_is_junk = true; }
			if (stripos($desc_trim, 'fusion_builder_') !== false) { $desc_is_junk = true; }
			if (preg_match('/\[[a-z0-9_]+\s+[^\]]+\]/i', $desc_trim)) { $desc_is_junk = true; }
		}
		if ($desc_is_junk) {
			$desc = '';
		}
		$focus = Sanitize::text($p['aegisseo_focus_phrase'] ?? '', 120);
        $canon = Sanitize::url($p['aegisseo_canonical'] ?? '', 500);
        $noindex = Sanitize::bool_int($p['aegisseo_noindex'] ?? 0);
        $nofollow= Sanitize::bool_int($p['aegisseo_nofollow'] ?? 0);

        $og_title = Sanitize::text($p['aegisseo_og_title'] ?? '', 300);
        $og_desc  = Sanitize::text($p['aegisseo_og_description'] ?? '', 1000);
        $og_img   = Sanitize::url($p['aegisseo_og_image'] ?? '', 500);
        $tw_title = Sanitize::text($p['aegisseo_twitter_title'] ?? '', 300);
        $tw_desc  = Sanitize::text($p['aegisseo_twitter_description'] ?? '', 1000);
        $tw_img   = Sanitize::url($p['aegisseo_twitter_image'] ?? '', 500);

        if ($title !== '') update_post_meta($post_id, '_aegisseo_title', $title);
        else delete_post_meta($post_id, '_aegisseo_title');

		if ($desc !== '') {
			update_post_meta($post_id, '_aegisseo_description', $desc);
		} else {
			delete_post_meta($post_id, '_aegisseo_description');
		}

		if (!empty($focus)) {
			update_post_meta($post_id, '_aegisseo_focus_phrase', $focus);
		} else {
			delete_post_meta($post_id, '_aegisseo_focus_phrase');
		}

        if ($canon !== '') update_post_meta($post_id, '_aegisseo_canonical', $canon);
        else delete_post_meta($post_id, '_aegisseo_canonical');

        if ($noindex) update_post_meta($post_id, '_aegisseo_noindex', 1);
        else delete_post_meta($post_id, '_aegisseo_noindex');

        if ($nofollow) update_post_meta($post_id, '_aegisseo_nofollow', 1);
        else delete_post_meta($post_id, '_aegisseo_nofollow');

        if ($og_title !== '') update_post_meta($post_id, '_aegisseo_og_title', $og_title);
        else delete_post_meta($post_id, '_aegisseo_og_title');

        if ($og_desc !== '') update_post_meta($post_id, '_aegisseo_og_description', $og_desc);
        else delete_post_meta($post_id, '_aegisseo_og_description');

        if ($og_img !== '') update_post_meta($post_id, '_aegisseo_og_image', $og_img);
        else delete_post_meta($post_id, '_aegisseo_og_image');

        if ($tw_title !== '') update_post_meta($post_id, '_aegisseo_twitter_title', $tw_title);
        else delete_post_meta($post_id, '_aegisseo_twitter_title');

        if ($tw_desc !== '') update_post_meta($post_id, '_aegisseo_twitter_description', $tw_desc);
        else delete_post_meta($post_id, '_aegisseo_twitter_description');

        if ($tw_img !== '') update_post_meta($post_id, '_aegisseo_twitter_image', $tw_img);
        else delete_post_meta($post_id, '_aegisseo_twitter_image');

$schema_type = Sanitize::text($p['aegisseo_schema_type'] ?? '', 50);
if ($schema_type !== '') update_post_meta($post_id, '_aegisseo_schema_type', $schema_type);
else delete_post_meta($post_id, '_aegisseo_schema_type');

$schema_data = $p['aegisseo_schema_data'] ?? null;
$schema_out = array();
if (is_array($schema_data)) {
    foreach ($schema_data as $k => $v) {
        $k = sanitize_key($k);
        $schema_out[$k] = Sanitize::text($v, 300);
    }
}

$has = false;
foreach ($schema_out as $v) { if (trim((string)$v) !== '') { $has = true; break; } }
if ($has) update_post_meta($post_id, '_aegisseo_schema_data', wp_json_encode($schema_out));
else delete_post_meta($post_id, '_aegisseo_schema_data');

    }
}
