<?php
namespace AegisSEO\Admin;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Admin_Ops {

    private $options;

    public function __construct(Options $options) {
        $this->options = $options;
    }

    /**
     * Sanitize DB prefix to an identifier-safe string (letters, numbers, underscore only).
     *
     * @param string $prefix Raw prefix (e.g., $wpdb->prefix).
     * @return string Safe prefix.
     */
    private function sanitize_db_prefix($prefix) {
        $prefix = (string) $prefix;
        return preg_replace('/[^A-Za-z0-9_]/', '', $prefix);
    }

    /**
     * Escape a SQL identifier (table/field name) using a strict allowlist and backticks.
     *
     * @param string $identifier Identifier without backticks.
     * @return string Escaped identifier wrapped in backticks, or empty string if invalid.
     */
    private function escape_sql_identifier($identifier) {
        $identifier = (string) $identifier;
        if ($identifier === '' || !preg_match('/^[A-Za-z0-9_]+$/', $identifier)) {
            return '';
        }
        return '`' . $identifier . '`';
    }

    /**
     * Replace a {{table}} token in a SQL string with a safely escaped table identifier.
     *
     * @param string $sql SQL containing a {{table}} token.
     * @param string $table_identifier Raw table identifier (without backticks).
     * @return string SQL with {{table}} replaced by escaped identifier.
     */
    private function sql_with_table($sql, $table_identifier) {
        $escaped = $this->escape_sql_identifier($table_identifier);
        if ($escaped === '') {
            return $sql;
        }
        return str_replace('{{table}}', $escaped, $sql);
    }


    public function get_dashboard_payload() {
        $payload = array(
            'now_gmt' => gmdate('c'),
            'gsc_connected' => false,
            'overview' => array(),
            'timeseries_28' => array(),
            'winners_losers' => array(),
            'queries' => array(),
            'clusters' => array(),
            'content_gaps' => array(),
            'cwv_warnings' => $this->get_cwv_warnings(),
            'notes' => array(),
            'security' => array(),
        );

        $opts = $this->options->get_all();
        $property = (string) ($opts['gsc_property'] ?? '');

        $has_gsc = !empty($opts['gsc_client_id']) && !empty($opts['gsc_client_secret']) && !empty($opts['gsc_refresh_token']) && !empty($property);
        if ($has_gsc) {
            $payload['gsc_connected'] = true;
            $payload['overview'] = array(
                '7'  => $this->gsc_site_summary($property, 7),
                '28' => $this->gsc_site_summary($property, 28),
                '90' => $this->gsc_site_summary($property, 90),
            );
            $payload['winners_losers'] = array(
                '7'  => $this->gsc_winners_losers($property, 7),
                '28' => $this->gsc_winners_losers($property, 28),
                '90' => $this->gsc_winners_losers($property, 90),
            );
            $payload['timeseries_28'] = $this->gsc_timeseries($property, 28);
            $q = $this->gsc_top_queries($property, 28, 500);
            $payload['queries'] = $q;
            $payload['clusters'] = $this->cluster_queries($q);
            $payload['content_gaps'] = $this->detect_content_gaps($q);
        } else {
            $payload['notes'][] = 'GSC is not connected. Showing snapshot-based metrics where available.';
            $payload['overview'] = array(
                '7'  => $this->snapshots_site_summary(7),
                '28' => $this->snapshots_site_summary(28),
                '90' => $this->snapshots_site_summary(90),
            );
            $payload['winners_losers'] = array(
                '7'  => $this->snapshots_winners_losers(7),
                '28' => $this->snapshots_winners_losers(28),
                '90' => $this->snapshots_winners_losers(90),
            );
            $payload['timeseries_28'] = $this->snapshots_timeseries(28);
        }

        $payload['security'] = $this->security_summary();

        return $payload;
    }

    
    private function security_summary() {
        global $wpdb;
        $table_identifier = $this->sanitize_db_prefix($wpdb->prefix) . 'aegisseo_events';
        $since = gmdate('Y-m-d H:i:s', time() - (7 * DAY_IN_SECONDS));

        $cache_key = 'security_summary_' . md5($table_identifier . '|' . $since);
        $cached = wp_cache_get($cache_key, 'aegisseo');
        if (false !== $cached) {
            return $cached;
        }

        $rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->prepare(
                'SELECT event_type, COUNT(*) AS c FROM %i WHERE event_type LIKE %s AND created_at >= %s GROUP BY event_type ORDER BY c DESC',
                $table_identifier,
                'security_%',
                $since
            ),
            ARRAY_A
        );

        $out = array(
            'since_gmt' => $since,
            'counts' => array(),
            'total' => 0,
        );

        foreach ((array)$rows as $r) {
            $t = (string)($r['event_type'] ?? '');
            $c = (int)($r['c'] ?? 0);
            if ($t === '' || $c <= 0) { continue; }
            $out['counts'][$t] = $c;
            $out['total'] += $c;
        }

        wp_cache_set($cache_key, $out, 'aegisseo', 5 * MINUTE_IN_SECONDS);
        return $out;
    }

    private function gsc_access_token() {
        $opts = $this->options->get_all();
        if (empty($opts['gsc_client_id']) || empty($opts['gsc_client_secret']) || empty($opts['gsc_refresh_token'])) {
            return new \WP_Error('aegisseo_ops', 'Missing GSC credentials.');
        }

		$resp = wp_remote_post('https://oauth2.googleapis.com/token', array(
			'timeout' => 20,
			'headers' => array('Content-Type'=>'application/x-www-form-urlencoded'),
			'body' => array(
				'client_id' => (string) $opts['gsc_client_id'],
				'client_secret' => (string) $opts['gsc_client_secret'],
				'refresh_token' => (string) $opts['gsc_refresh_token'],
				'grant_type' => 'refresh_token',
			),
		));

		if (is_wp_error($resp)) {
			\AegisSEO\Log\Logger::error('GSC token refresh wp_error', array(
				'error' => $resp->get_error_message(),
			));
			return $resp;
		}

		$code = (int) wp_remote_retrieve_response_code($resp);
		$body = (string) wp_remote_retrieve_body($resp);

		if ($code < 200 || $code >= 300) {
			\AegisSEO\Log\Logger::warn('GSC token refresh non-2xx', array(
				'http_code' => $code,
				'body'      => $body,
			));
			return new \WP_Error('aegisseo_ops', 'Failed to refresh GSC access token.');
		}

		$data = json_decode($body, true);
		if (!is_array($data)) {
			\AegisSEO\Log\Logger::error('GSC token refresh JSON decode failed', array(
				'http_code' => $code,
				'body'      => $body,
			));
			return new \WP_Error('aegisseo_ops', 'Failed to parse token response.');
		}

		if (empty($data['access_token'])) {
			\AegisSEO\Log\Logger::error('GSC token refresh missing access_token', array(
				'http_code' => $code,
				'decoded'   => $data,
			));
			return new \WP_Error('aegisseo_ops', 'Failed to refresh GSC access token.');
		}

		return (string) $data['access_token'];

    }

    private function gsc_query($property, $body) {
        $token = $this->gsc_access_token();
        if (is_wp_error($token)) return $token;

        $property = rtrim((string) $property, '/') . '/';
        $endpoint = 'https://www.googleapis.com/webmasters/v3/sites/' . rawurlencode($property) . 'searchAnalytics/query';

        $resp = wp_remote_post($endpoint, array(
            'timeout' => 20,
            'headers' => array(
                'Authorization' => 'Bearer ' . $token,
                'Content-Type'  => 'application/json',
            ),
            'body' => wp_json_encode($body),
        ));
        if (is_wp_error($resp)) return $resp;

        $code = (int) wp_remote_retrieve_response_code($resp);
        $raw  = (string) wp_remote_retrieve_body($resp);
        if ($code < 200 || $code >= 300) {
            return new \WP_Error('aegisseo_ops', 'GSC query failed: HTTP ' . $code);
        }
        $data = json_decode($raw, true);
        if (!is_array($data)) return new \WP_Error('aegisseo_ops', 'Invalid GSC response.');
        return $data;
    }

    private function date_range($days, $offset_days = 0) {
        $days = max(1, (int) $days);
        $offset_days = (int) $offset_days;

        $end_ts = strtotime('-' . $offset_days . ' days');
        $start_ts = strtotime('-' . ($offset_days + $days) . ' days');
        $start = gmdate('Y-m-d', $start_ts);
        $end   = gmdate('Y-m-d', $end_ts);
        return array($start, $end);
    }

    private function gsc_site_summary($property, $days) {
        list($start, $end) = $this->date_range($days, 0);
        $body = array(
            'startDate' => $start,
            'endDate'   => $end,
            'dimensions' => array(),
            'rowLimit' => 1,
        );
        $data = $this->gsc_query($property, $body);
        if (is_wp_error($data)) {
            return array('error' => $data->get_error_message(), 'clicks'=>0,'impressions'=>0,'ctr'=>0,'position'=>0);
        }
        $row = !empty($data['rows'][0]) ? (array) $data['rows'][0] : array();
        $clicks = (float) ($row['clicks'] ?? 0);
        $impr   = (float) ($row['impressions'] ?? 0);
        $ctr    = (float) ($row['ctr'] ?? 0);
        $pos    = (float) ($row['position'] ?? 0);
        return array(
            'start' => $start,
            'end'   => $end,
            'clicks' => round($clicks, 2),
            'impressions' => round($impr, 2),
            'ctr' => round($ctr * 100, 2),
            'position' => round($pos, 2),
        );
    }

    private function gsc_winners_losers($property, $days) {
        list($start1, $end1) = $this->date_range($days, 0);
        list($start0, $end0) = $this->date_range($days, $days);

        $rows1 = $this->gsc_pages($property, $start1, $end1, 250);
        $rows0 = $this->gsc_pages($property, $start0, $end0, 250);

        if (is_wp_error($rows1) || is_wp_error($rows0)) {
            $msg = is_wp_error($rows1) ? $rows1->get_error_message() : (is_wp_error($rows0) ? $rows0->get_error_message() : 'Unknown error');
            return array('error' => $msg, 'winners'=>array(), 'losers'=>array());
        }

        $m1 = $this->index_by_key($rows1, 'page');
        $m0 = $this->index_by_key($rows0, 'page');
        $deltas = array();
        foreach ($m1 as $page => $r1) {
            $r0 = $m0[$page] ?? array('clicks'=>0,'impressions'=>0,'ctr'=>0,'position'=>0);
            $dc = (float) $r1['clicks'] - (float) $r0['clicks'];
            $di = (float) $r1['impressions'] - (float) $r0['impressions'];
            $deltas[] = array(
                'page' => $page,
                'clicks' => round((float)$r1['clicks'],2),
                'impressions' => round((float)$r1['impressions'],2),
                'ctr' => round((float)$r1['ctr'] * 100, 2),
                'position' => round((float)$r1['position'],2),
                'delta_clicks' => round($dc,2),
                'delta_impressions' => round($di,2),
            );
        }

        usort($deltas, function($a,$b){ return ($b['delta_clicks'] <=> $a['delta_clicks']); });
        $winners = array_slice($deltas, 0, 10);
        $losers  = array_slice(array_reverse($deltas), 0, 10);

        return array(
            'range' => array('current'=>array($start1,$end1),'previous'=>array($start0,$end0)),
            'winners' => $winners,
            'losers'  => $losers,
        );
    }

    private function gsc_pages($property, $start, $end, $limit) {
        $body = array(
            'startDate' => $start,
            'endDate'   => $end,
            'dimensions' => array('page'),
            'rowLimit' => (int) $limit,
        );
        $data = $this->gsc_query($property, $body);
        if (is_wp_error($data)) return $data;

        $rows = array();
        foreach ((array)($data['rows'] ?? array()) as $r) {
            $page = isset($r['keys'][0]) ? (string) $r['keys'][0] : '';
            if ($page === '') continue;
            $rows[] = array(
                'page' => $page,
                'clicks' => (float)($r['clicks'] ?? 0),
                'impressions' => (float)($r['impressions'] ?? 0),
                'ctr' => (float)($r['ctr'] ?? 0),
                'position' => (float)($r['position'] ?? 0),
            );
        }
        return $rows;
    }

    private function gsc_top_queries($property, $days, $limit) {
        list($start, $end) = $this->date_range($days, 0);
        $body = array(
            'startDate' => $start,
            'endDate'   => $end,
            'dimensions' => array('query'),
            'rowLimit' => (int) $limit,
        );
        $data = $this->gsc_query($property, $body);
        if (is_wp_error($data)) {
            return array();
        }
        $rows = array();
        foreach ((array)($data['rows'] ?? array()) as $r) {
            $q = isset($r['keys'][0]) ? (string) $r['keys'][0] : '';
            if ($q === '') continue;
            $rows[] = array(
                'query' => $q,
                'clicks' => round((float)($r['clicks'] ?? 0),2),
                'impressions' => round((float)($r['impressions'] ?? 0),2),
                'ctr' => round((float)($r['ctr'] ?? 0) * 100, 2),
                'position' => round((float)($r['position'] ?? 0),2),
            );
        }
        return $rows;
    }

    private function gsc_timeseries($property, $days) {
        list($start, $end) = $this->date_range($days, 0);
        $body = array(
            'startDate' => $start,
            'endDate'   => $end,
            'dimensions' => array('date'),
            'rowLimit' => 1000,
        );
        $data = $this->gsc_query($property, $body);
        if (is_wp_error($data)) return array();

        $out = array();
        foreach ((array)($data['rows'] ?? array()) as $r) {
            $d = isset($r['keys'][0]) ? (string) $r['keys'][0] : '';
            if ($d === '') continue;
            $clicks = (float)($r['clicks'] ?? 0);
            $impr = (float)($r['impressions'] ?? 0);
            $out[] = array(
                'date' => $d,
                'clicks' => round($clicks, 2),
                'impressions' => round($impr, 2),
            );
        }
        return $out;
    }

    private function index_by_key($rows, $key) {
        $out = array();
        foreach ((array)$rows as $r) {
            if (!isset($r[$key])) continue;
            $out[(string)$r[$key]] = $r;
        }
        return $out;
    }

    private function normalize_tokens($s) {
        $s = strtolower((string) $s);
        $s = preg_replace('/[^a-z0-9\s]+/i', ' ', $s);
        $parts = preg_split('/\s+/', trim($s));
        $stop = array('the','and','or','to','for','of','in','on','at','a','an','is','are','with','near','me','my');
        $tokens = array();
        foreach ($parts as $p) {
            if ($p === '' || strlen($p) < 2) continue;
            if (in_array($p, $stop, true)) continue;
            $tokens[] = $p;
        }
        return $tokens;
    }

    private function cluster_queries($queries) {
        $clusters = array();
        foreach ((array)$queries as $r) {
            $q = (string) ($r['query'] ?? '');
            $tokens = $this->normalize_tokens($q);
            if (empty($tokens)) continue;
            $topic = $tokens[0];
            if (!isset($clusters[$topic])) {
                $clusters[$topic] = array('topic'=>$topic,'impressions'=>0,'clicks'=>0,'queries'=>array());
            }
            $clusters[$topic]['impressions'] += (float) ($r['impressions'] ?? 0);
            $clusters[$topic]['clicks'] += (float) ($r['clicks'] ?? 0);
            if (count($clusters[$topic]['queries']) < 10) {
                $clusters[$topic]['queries'][] = $q;
            }
        }

        $out = array_values($clusters);
        usort($out, function($a,$b){ return ($b['impressions'] <=> $a['impressions']); });
        $out = array_slice($out, 0, 12);

        foreach ($out as &$c) {
            $c['impressions'] = round((float)$c['impressions'], 2);
            $c['clicks'] = round((float)$c['clicks'], 2);
            $c['ctr'] = $c['impressions'] > 0 ? round(($c['clicks'] / $c['impressions']) * 100, 2) : 0;
        }

        return $out;
    }

    private function detect_content_gaps($queries) {
        global $wpdb;

        $cache_key = 'content_gaps_posts_v1';
        $posts = wp_cache_get($cache_key, 'aegisseo');
        if (false === $posts) {
            $posts = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    "SELECT ID, post_title, post_name FROM {$wpdb->posts} WHERE 1=%d AND post_status=%s AND post_type NOT IN ('revision','nav_menu_item') ORDER BY post_modified_gmt DESC LIMIT 2000",
                    1,
                    'publish'
                ),
                ARRAY_A
            );
            wp_cache_set($cache_key, $posts, 'aegisseo', 5 * MINUTE_IN_SECONDS);
        }
        $hay = '';
        foreach ((array)$posts as $p) {
            $hay .= ' ' . strtolower((string)($p['post_title'] ?? '')) . ' ' . strtolower((string)($p['post_name'] ?? ''));
        }

        $gaps = array();
        foreach ((array)$queries as $r) {
            $q = (string) ($r['query'] ?? '');
            $impr = (float) ($r['impressions'] ?? 0);
            $ctr  = (float) ($r['ctr'] ?? 0);

            if ($impr < 50) continue;
            if ($ctr > 1.0) continue; 

            $tokens = $this->normalize_tokens($q);
            if (empty($tokens)) continue;

            $hits = 0;
            foreach (array_slice($tokens, 0, 5) as $t) {
                if (strpos($hay, ' ' . $t) !== false) { $hits++; }
                if ($hits >= 2) break;
            }
            if ($hits >= 2) continue;

            $gaps[] = array(
                'query' => $q,
                'impressions' => round($impr,2),
                'clicks' => round((float)($r['clicks'] ?? 0),2),
                'ctr' => round($ctr,2),
                'position' => round((float)($r['position'] ?? 0),2),
                'hint' => $hits === 0 ? 'Not found in titles/slugs' : 'Partially covered',
            );
            if (count($gaps) >= 12) break;
        }

        return $gaps;
    }

    private function snapshots_site_summary($days) {
        global $wpdb;
        $t_identifier = $this->sanitize_db_prefix($wpdb->prefix) . 'aegisseo_gsc_snapshots';
        $days = max(1, min(120, (int)$days));
        $start = gmdate('Y-m-d', strtotime('-' . ($days-1) . ' days'));

        $cache_key = 'snapshots_site_summary_' . md5($t_identifier . '|' . $start);
        $row = wp_cache_get($cache_key, 'aegisseo');
        if (false === $row) {
            $row = $wpdb->get_row( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    'SELECT SUM(clicks) AS clicks, SUM(impressions) AS impressions, AVG(position) AS position FROM %i WHERE snap_date >= %s',
                    $t_identifier,
                    $start
                ),
                ARRAY_A
            );
            wp_cache_set($cache_key, $row, 'aegisseo', 5 * MINUTE_IN_SECONDS);
        }

        $clicks = (float) ($row['clicks'] ?? 0);
        $impr   = (float) ($row['impressions'] ?? 0);
        $pos    = (float) ($row['position'] ?? 0);
        $ctr    = $impr > 0 ? ($clicks / $impr) * 100 : 0;

        return array(
            'start' => $start,
            'end'   => gmdate('Y-m-d'),
            'clicks' => round($clicks,2),
            'impressions' => round($impr,2),
            'ctr' => round($ctr,2),
            'position' => round($pos,2),
        );
    }

    private function snapshots_winners_losers($days) {
        global $wpdb;
        $t_identifier = $this->sanitize_db_prefix($wpdb->prefix) . 'aegisseo_gsc_snapshots';
        $days = max(1, min(120, (int)$days));

        $start1 = gmdate('Y-m-d', strtotime('-' . ($days-1) . ' days'));
        $end1   = gmdate('Y-m-d');
        $start0 = gmdate('Y-m-d', strtotime('-' . (2*$days-1) . ' days'));
        $end0   = gmdate('Y-m-d', strtotime('-' . $days . ' days'));

        $cache_key_1 = 'snapshots_wl_rows1_' . md5($t_identifier . '|' . $start1 . '|' . $end1);
        $rows1 = wp_cache_get($cache_key_1, 'aegisseo');
        if (false === $rows1) {
            $rows1 = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    'SELECT page_url, SUM(clicks) AS clicks, SUM(impressions) AS impressions FROM %i WHERE snap_date >= %s AND snap_date <= %s GROUP BY page_url ORDER BY clicks DESC LIMIT 250',
                    $t_identifier,
                    $start1,
                    $end1
                ),
                ARRAY_A
            );
            wp_cache_set($cache_key_1, $rows1, 'aegisseo', 5 * MINUTE_IN_SECONDS);
        }

        $cache_key_0 = 'snapshots_wl_rows0_' . md5($t_identifier . '|' . $start0 . '|' . $end0);
        $rows0 = wp_cache_get($cache_key_0, 'aegisseo');
        if (false === $rows0) {
            $rows0 = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    'SELECT page_url, SUM(clicks) AS clicks, SUM(impressions) AS impressions FROM %i WHERE snap_date >= %s AND snap_date <= %s GROUP BY page_url ORDER BY clicks DESC LIMIT 250',
                    $t_identifier,
                    $start0,
                    $end0
                ),
                ARRAY_A
            );
            wp_cache_set($cache_key_0, $rows0, 'aegisseo', 5 * MINUTE_IN_SECONDS);
        }

        $m1 = array();
        foreach ($rows1 as $r) { $m1[(string)$r['page_url']] = $r; }
        $m0 = array();
        foreach ($rows0 as $r) { $m0[(string)$r['page_url']] = $r; }

        $deltas = array();
        foreach ($m1 as $page => $r1) {
            $r0 = $m0[$page] ?? array('clicks'=>0,'impressions'=>0);
            $dc = (float)$r1['clicks'] - (float)$r0['clicks'];
            $di = (float)$r1['impressions'] - (float)$r0['impressions'];
            $deltas[] = array(
                'page' => $page,
                'clicks' => round((float)$r1['clicks'],2),
                'impressions' => round((float)$r1['impressions'],2),
                'delta_clicks' => round($dc,2),
                'delta_impressions' => round($di,2),
            );
        }

        usort($deltas, function($a,$b){ return ($b['delta_clicks'] <=> $a['delta_clicks']); });
        $winners = array_slice($deltas, 0, 10);
        $losers  = array_slice(array_reverse($deltas), 0, 10);

        return array(
            'range' => array('current'=>array($start1,$end1),'previous'=>array($start0,$end0)),
            'winners' => $winners,
            'losers'  => $losers,
        );
    }

    private function snapshots_timeseries($days) {
        global $wpdb;
        $t_identifier = $this->sanitize_db_prefix($wpdb->prefix) . 'aegisseo_gsc_snapshots';
        $days = max(1, min(120, (int)$days));
        $start = gmdate('Y-m-d', strtotime('-' . ($days-1) . ' days'));

        $cache_key = 'snapshots_timeseries_' . md5($t_identifier . '|' . $start);
        $rows = wp_cache_get($cache_key, 'aegisseo');
        if (false === $rows) {
            $rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    'SELECT snap_date AS date, SUM(clicks) AS clicks, SUM(impressions) AS impressions FROM %i WHERE snap_date >= %s GROUP BY snap_date ORDER BY snap_date ASC',
                    $t_identifier,
                    $start
                ),
                ARRAY_A
            );
            wp_cache_set($cache_key, $rows, 'aegisseo', 5 * MINUTE_IN_SECONDS);
        }

        $out = array();
        foreach ((array)$rows as $r) {
            $out[] = array(
                'date' => (string)($r['date'] ?? ''),
                'clicks' => round((float)($r['clicks'] ?? 0), 2),
                'impressions' => round((float)($r['impressions'] ?? 0), 2),
            );
        }
        return $out;
    }



    private function get_cwv_warnings() {
        $warnings = array();

        $has_cache = false;
        $cache_indicators = array(
            'WP_ROCKET_VERSION',
            'LSCWP_V',
            'W3TC',
            'WP_CACHE',
        );
        foreach ($cache_indicators as $c) {
            if (defined($c)) { $has_cache = true; break; }
        }
        if (!$has_cache) {
            $active = (array) get_option('active_plugins', array());
            $needle = array('wp-rocket','litespeed-cache','w3-total-cache','wp-super-cache','cache-enabler','wp-optimize');
            foreach ($active as $p) {
                foreach ($needle as $n) {
                    if (strpos($p, $n) !== false) { $has_cache = true; break 2; }
                }
            }
        }
        if (!$has_cache) {
            $warnings[] = array(
                'level' => 'warning',
                'title' => 'No page caching detected',
                'detail' => 'Core Web Vitals often improve significantly with page caching. Consider enabling a caching plugin (or your host\'s built-in cache) and then re-test CWV in Search Console.',
            );
        }

        if (!wp_using_ext_object_cache()) {
            $warnings[] = array(
                'level' => 'info',
                'title' => 'No persistent object cache detected',
                'detail' => 'If your site has many visitors or heavy queries, Redis/Memcached object caching can help reduce backend latency (TTFB) which impacts CWV.',
            );
        }

        global $wpdb;
        $row = $wpdb->get_row("SELECT SUM(LENGTH(option_value)) AS bytes FROM {$wpdb->options} WHERE autoload='yes'", ARRAY_A); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $bytes = (int) ($row['bytes'] ?? 0);
        if ($bytes > 1024 * 1024) {
            $warnings[] = array(
                'level' => 'warning',
                'title' => 'Large autoloaded options',
                'detail' => 'Your autoloaded options total is about ' . size_format($bytes) . '. Large autoload can slow every request and affect CWV. Consider cleaning transients/options or using AegisShield DB Tools if available.',
            );
        }

        if (has_filter('wp_lazy_loading_enabled') && apply_filters('wp_lazy_loading_enabled', true, 'img') === false) { // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
            $warnings[] = array(
                'level' => 'info',
                'title' => 'Image lazy-loading appears disabled',
                'detail' => 'Lazy-loading helps LCP/CLS when used correctly. If disabled globally, consider enabling it and ensuring images have width/height attributes.',
            );
        }

        return $warnings;
    }
}
