<?php
namespace AegisSEO\Admin;

use AegisSEO\Utils\Options;
use AegisSEO\Utils\Sanitize;

if (!defined('ABSPATH')) { exit; }

class Admin_Settings {

    private $options;

    public function __construct(Options $options) {
        $this->options = $options;
        add_action('admin_init', array($this, 'register_settings'));
    }

    public function register_settings() {
        register_setting('aegisseo_settings_group', Options::OPTION_KEY, array($this, 'sanitize'));

        add_settings_section('aegisseo_section_general', __('General', 'aegisseo'), '__return_false', 'aegisseo_global');
        add_settings_field('separator', __('Title Separator', 'aegisseo'), array($this, 'field_separator'), 'aegisseo_global', 'aegisseo_section_general');

        add_settings_section('aegisseo_section_titles', __('Titles & Meta', 'aegisseo'), '__return_false', 'aegisseo_global');
        // Content scope
        add_settings_section('aegisseo_section_scope', __('Content Scope', 'aegisseo'), '__return_false', 'aegisseo_global');
        add_settings_field('scope_mode', __('Content Scope Mode', 'aegisseo'), array($this, 'field_select'), 'aegisseo_global', 'aegisseo_section_scope', array('key' => 'scope_mode', 'options' => array('smart' => __('Smart (recommended)', 'aegisseo'), 'include_only' => __('Include only (manual)', 'aegisseo'))));
        add_settings_field('scope_include', __('Include Post Types', 'aegisseo'), array($this, 'field_post_type_list'), 'aegisseo_global', 'aegisseo_section_scope', array('key' => 'scope_include', 'placeholder' => 'post,page,product,attachment'));
        add_settings_field('scope_exclude', __('Exclude Post Types', 'aegisseo'), array($this, 'field_post_type_list'), 'aegisseo_global', 'aegisseo_section_scope', array('key' => 'scope_exclude', 'placeholder' => 'as_scan,as_event'));

        add_settings_field('title_home', __('Home Title Template', 'aegisseo'), array($this, 'field_text'), 'aegisseo_global', 'aegisseo_section_titles', array('key' => 'title_home'));
        add_settings_field('meta_desc_home', __('Home Meta Description Template', 'aegisseo'), array($this, 'field_textarea'), 'aegisseo_global', 'aegisseo_section_titles', array('key' => 'meta_desc_home'));
        add_settings_field('title_single', __('Post Title Template', 'aegisseo'), array($this, 'field_text'), 'aegisseo_global', 'aegisseo_section_titles', array('key' => 'title_single'));
        add_settings_field('meta_desc_single', __('Post Meta Description Template', 'aegisseo'), array($this, 'field_textarea'), 'aegisseo_global', 'aegisseo_section_titles', array('key' => 'meta_desc_single'));
        add_settings_field('title_page', __('Page Title Template', 'aegisseo'), array($this, 'field_text'), 'aegisseo_global', 'aegisseo_section_titles', array('key' => 'title_page'));
        add_settings_field('meta_desc_page', __('Page Meta Description Template', 'aegisseo'), array($this, 'field_textarea'), 'aegisseo_global', 'aegisseo_section_titles', array('key' => 'meta_desc_page'));
        add_settings_field('title_archive', __('Archive Title Template', 'aegisseo'), array($this, 'field_text'), 'aegisseo_global', 'aegisseo_section_titles', array('key' => 'title_archive'));

        add_settings_section('aegisseo_section_social', __('Social Meta', 'aegisseo'), '__return_false', 'aegisseo_social');
        add_settings_field('og_enabled', __('Enable Open Graph', 'aegisseo'), array($this, 'field_checkbox'), 'aegisseo_social', 'aegisseo_section_social', array('key' => 'og_enabled'));
        add_settings_field('twitter_enabled', __('Enable Twitter Cards', 'aegisseo'), array($this, 'field_checkbox'), 'aegisseo_social', 'aegisseo_section_social', array('key' => 'twitter_enabled'));
        add_settings_field('twitter_card', __('Twitter Card Type', 'aegisseo'), array($this, 'field_select'), 'aegisseo_social', 'aegisseo_section_social', array('key' => 'twitter_card', 'options' => array('summary'=>'summary','summary_large_image'=>'summary_large_image')));

        add_settings_field('twitter_site', __('Twitter @site', 'aegisseo'), array($this, 'field_text'), 'aegisseo_social', 'aegisseo_section_social', array('key' => 'twitter_site'));
        add_settings_field('twitter_creator', __('Twitter @creator', 'aegisseo'), array($this, 'field_text'), 'aegisseo_social', 'aegisseo_section_social', array('key' => 'twitter_creator'));
        add_settings_field('facebook_app_id', __('Facebook App ID', 'aegisseo'), array($this, 'field_text'), 'aegisseo_social', 'aegisseo_section_social', array('key' => 'facebook_app_id'));
        add_settings_field('facebook_admins', __('Facebook Admins (comma-separated)', 'aegisseo'), array($this, 'field_text'), 'aegisseo_social', 'aegisseo_section_social', array('key' => 'facebook_admins'));
        add_settings_field('social_default_image', __('Default Social Image URL', 'aegisseo'), array($this, 'field_text'), 'aegisseo_social', 'aegisseo_section_social', array('key' => 'social_default_image'));
        add_settings_field('social_profiles', __('Social Profile URLs', 'aegisseo'), array($this, 'field_social_profiles'), 'aegisseo_social', 'aegisseo_section_social');
        add_settings_field('social_enable_article_meta', __('Add Article Meta (posts)', 'aegisseo'), array($this, 'field_checkbox'), 'aegisseo_social', 'aegisseo_section_social', array('key' => 'social_enable_article_meta'));

		add_settings_section('aegisseo_section_schema', __('Schema', 'aegisseo'), '__return_false', 'aegisseo_schema');
        
        add_settings_field('schema_post_type_map', __('Post Type → Schema Default', 'aegisseo'), array($this, 'field_schema_post_type_map'), 'aegisseo_schema', 'aegisseo_section_schema');
add_settings_field('site_type', __('Publisher Type', 'aegisseo'), array($this, 'field_select'), 'aegisseo_schema', 'aegisseo_section_schema', array('key' => 'site_type', 'options' => array('organization'=>__('Organization','aegisseo'),'person'=>__('Person','aegisseo'))));
        add_settings_field('org_name', __('Organization Name', 'aegisseo'), array($this, 'field_text'), 'aegisseo_schema', 'aegisseo_section_schema', array('key' => 'org_name'));
        add_settings_field('person_name', __('Person Name', 'aegisseo'), array($this, 'field_text'), 'aegisseo_schema', 'aegisseo_section_schema', array('key' => 'person_name'));
        add_settings_field('logo_id', __('Logo Attachment ID', 'aegisseo'), array($this, 'field_number'), 'aegisseo_schema', 'aegisseo_section_schema', array('key' => 'logo_id'));

        add_settings_section('aegisseo_section_sitemap', __('Sitemap', 'aegisseo'), '__return_false', 'aegisseo_sitemap');
        add_settings_field('sitemap_enabled', __('Enable Sitemap', 'aegisseo'), array($this, 'field_checkbox'), 'aegisseo_sitemap', 'aegisseo_section_sitemap', array('key' => 'sitemap_enabled'));
        add_settings_field('sitemap_info', __('Sitemap URL', 'aegisseo'), array($this, 'field_sitemap_info'), 'aegisseo_sitemap', 'aegisseo_section_sitemap');

        add_settings_field('sitemap_index_enabled', __('Sitemap Index', 'aegisseo'), array($this, 'field_checkbox'), 'aegisseo_sitemap', 'aegisseo_section_sitemap', array('key' => 'sitemap_index_enabled', 'label' => __('Enable /sitemap_index.xml', 'aegisseo')));
        add_settings_field('sitemap_page_size', __('Max URLs per child sitemap', 'aegisseo'), array($this, 'field_number'), 'aegisseo_sitemap', 'aegisseo_section_sitemap', array('key' => 'sitemap_page_size', 'min' => 500, 'max' => 5000, 'step' => 100));
        add_settings_field('sitemap_include_images', __('Include Images', 'aegisseo'), array($this, 'field_checkbox'), 'aegisseo_sitemap', 'aegisseo_section_sitemap', array('key' => 'sitemap_include_images', 'label' => __('Include featured images in XML', 'aegisseo')));
        add_settings_field('sitemap_include_attached_images', __('Include Attached Images', 'aegisseo'), array($this, 'field_checkbox'), 'aegisseo_sitemap', 'aegisseo_section_sitemap', array('key' => 'sitemap_include_attached_images', 'label' => __('Also include attached images (slower on very large sites)', 'aegisseo')));

        add_settings_field('sitemap_video_enabled', __('Video Sitemap', 'aegisseo'), array($this, 'field_checkbox'), 'aegisseo_sitemap', 'aegisseo_section_sitemap', array('key' => 'sitemap_video_enabled', 'label' => __('Enable /video-sitemap.xml', 'aegisseo')));
        add_settings_field('sitemap_news_enabled', __('News Sitemap', 'aegisseo'), array($this, 'field_checkbox'), 'aegisseo_sitemap', 'aegisseo_section_sitemap', array('key' => 'sitemap_news_enabled', 'label' => __('Enable /news-sitemap.xml (last 48 hours)', 'aegisseo')));
        add_settings_field('sitemap_html_enabled', __('HTML Sitemap', 'aegisseo'), array($this, 'field_checkbox'), 'aegisseo_sitemap', 'aegisseo_section_sitemap', array('key' => 'sitemap_html_enabled', 'label' => __('Enable /sitemap/ (HTML)', 'aegisseo')));
        add_settings_field('sitemap_last_ping_status', __('Last Ping Status', 'aegisseo'), array($this, 'field_sitemap_last_ping'), 'aegisseo_sitemap', 'aegisseo_section_sitemap');

        add_settings_field('ping_search_engines', __('Ping Search Engines on Publish', 'aegisseo'), array($this, 'field_checkbox'), 'aegisseo_sitemap', 'aegisseo_section_sitemap', array('key' => 'ping_search_engines'));
        add_settings_field('sitemap_post_types', __('Include Post Types', 'aegisseo'), array($this, 'field_sitemap_post_types'), 'aegisseo_sitemap', 'aegisseo_section_sitemap');
        add_settings_field('sitemap_taxonomies', __('Include Taxonomies', 'aegisseo'), array($this, 'field_sitemap_taxonomies'), 'aegisseo_sitemap', 'aegisseo_section_sitemap');
        add_settings_field('sitemap_robots_add', __('Add Sitemap to robots.txt', 'aegisseo'), array($this, 'field_checkbox'), 'aegisseo_sitemap', 'aegisseo_section_sitemap', array('key' => 'sitemap_robots_add'));


        add_settings_section('aegisseo_section_tools', __('Safe Defaults', 'aegisseo'), '__return_false', 'aegisseo_tools');
        add_settings_field('safe_defaults_enabled', __('Enable Safe Defaults', 'aegisseo'), array($this, 'field_checkbox'), 'aegisseo_tools', 'aegisseo_section_tools', array('key' => 'safe_defaults_enabled'));
        add_settings_field('focus_phrase_enabled', __('Enable Focus Phrase Field', 'aegisseo'), array($this, 'field_checkbox'), 'aegisseo_tools', 'aegisseo_section_tools', array('key' => 'focus_phrase_enabled'));
		add_settings_section(
			'aegisseo_section_injection',
			__('Custom Injections', 'aegisseo'),
			array($this, 'section_injection_desc'),
			'aegisseo_injection'
		);

		add_settings_field(
			'injection_enabled',
			__('Enable Custom Injections', 'aegisseo'),
			array($this, 'field_checkbox'),
			'aegisseo_injection',
			'aegisseo_section_injection',
			array('key' => 'injection_enabled')
		);

		add_settings_field(
			'verify_google',
			__('Google Site Verification (meta content)', 'aegisseo'),
			array($this, 'field_text'),
			'aegisseo_injection',
			'aegisseo_section_injection',
			array('key' => 'verify_google')
		);

		add_settings_field(
			'verify_bing',
			__('Bing Site Verification (meta content)', 'aegisseo'),
			array($this, 'field_text'),
			'aegisseo_injection',
			'aegisseo_section_injection',
			array('key' => 'verify_bing')
		);

		add_settings_field(
			'custom_schema_sitewide',
			__('Site-wide JSON-LD (extra schema)', 'aegisseo'),
			array($this, 'field_textarea'),
			'aegisseo_injection',
			'aegisseo_section_injection',
			array('key' => 'custom_schema_sitewide', 'rows' => 10)
		);

		add_settings_field(
			'inject_head_html',
			__('Head Injection (advanced)', 'aegisseo'),
			array($this, 'field_textarea'),
			'aegisseo_injection',
			'aegisseo_section_injection',
			array('key' => 'inject_head_html', 'rows' => 8)
		);

		add_settings_field(
			'inject_footer_html',
			__('Footer Injection (advanced)', 'aegisseo'),
			array($this, 'field_textarea'),
			'aegisseo_injection',
			'aegisseo_section_injection',
			array('key' => 'inject_footer_html', 'rows' => 8)
		);
    }

    public function sanitize($input) {
        $out = $this->options->get_all();

        $out['separator'] = Sanitize::text($input['separator'] ?? $out['separator'], 5);

        // Content scope
        if (isset($input['scope_mode'])) {
            $mode = Sanitize::text($input['scope_mode'], 20);
            $out['scope_mode'] = in_array($mode, array('smart','include_only'), true) ? $mode : 'smart';
        }
        foreach (array('scope_include','scope_exclude') as $k) {
            if (isset($input[$k])) {
                $raw = is_string($input[$k]) ? $input[$k] : '';
                $raw = str_replace(array("\r","\n","\t"), ',', $raw);
                $parts = array_filter(array_map('trim', explode(',', $raw)));
                $clean = array();
                foreach ($parts as $pt) {
                    $pt = sanitize_key($pt);
                    if ($pt !== '') $clean[] = $pt;
                }
                $out[$k] = array_values(array_unique($clean));
            }
        }

        foreach (array('title_home','title_single','title_page','title_archive') as $k) {
            if (isset($input[$k])) $out[$k] = Sanitize::text($input[$k], 300);
        }
        foreach (array('meta_desc_home','meta_desc_single','meta_desc_page') as $k) {
            if (isset($input[$k])) $out[$k] = Sanitize::text($input[$k], 500);
        }

        foreach (array('og_enabled','twitter_enabled','sitemap_enabled','safe_defaults_enabled','focus_phrase_enabled','local_enabled','social_enable_article_meta','ping_search_engines','sitemap_robots_add','sitemap_video_enabled','sitemap_news_enabled','sitemap_html_enabled') as $k) {
            if (isset($input[$k])) $out[$k] = Sanitize::bool_int($input[$k]);
        }

        foreach (array('sitemap_index_enabled','sitemap_include_images','sitemap_include_attached_images') as $k) {
            if (isset($input[$k])) $out[$k] = Sanitize::bool_int($input[$k]);
        }

		if (isset($input['sitemap_page_size'])) {
			$out['sitemap_page_size'] = $this->int_range($input['sitemap_page_size'], 500, 5000, (int) ($out['sitemap_page_size'] ?? 2000));
		}

		if (isset($input['sitemap_plugins_present'])) {
			$plugins = array();

			if (isset($input['sitemap_plugins']) && is_array($input['sitemap_plugins'])) {
				foreach ($input['sitemap_plugins'] as $p) {
					$p = is_string($p) ? trim($p) : '';
					if ($p === '') continue;

					// safety
					if (strpos($p, '..') !== false) continue;

					$plugins[] = Sanitize::text($p, 300);
				}
			}

			$plugins = array_values(array_unique($plugins));
			$out['sitemap_plugins'] = $plugins; // will become [] when none are submitted
		}

        if (isset($input['sitemap_folder_exclusions'])) {
            $raw = is_string($input['sitemap_folder_exclusions']) ? $input['sitemap_folder_exclusions'] : '';
            $raw = sanitize_textarea_field($raw);

            $lines = preg_split('/\r\n|\r|\n/', $raw);
            $out_lines = array();

            foreach ($lines as $line) {
                $line = trim($line);
                if ($line === '') continue;

                $line = preg_replace('#^https?://[^/]+#i', '', $line);

                if ($line[0] !== '/') $line = '/' . $line;

                $line = rtrim($line, '/');

                if (strpos($line, '..') !== false) continue;

                $out_lines[] = $line;
            }

            $out_lines = array_values(array_unique($out_lines));
            $out['sitemap_folder_exclusions'] = implode("\n", $out_lines);
        }

        $prev_opts = $this->options->get_all();
        $prev_sitemap = (int) ($prev_opts['sitemap_enabled'] ?? 1);
        $new_sitemap  = (int) ($out['sitemap_enabled'] ?? $prev_sitemap);
        if ($prev_sitemap !== $new_sitemap) {
            set_transient('aegisseo_flush_rewrite', '1', 120);
        }


        if (isset($input['twitter_card'])) $out['twitter_card'] = Sanitize::text($input['twitter_card'], 50);

        foreach (array('twitter_site','twitter_creator') as $k) {
            if (isset($input[$k])) {
                // allow @handle or plain handle
                $v = Sanitize::text($input[$k], 50);
                $v = trim($v);
                if ($v !== '' && $v[0] !== '@') { $v = '@' . ltrim($v, '@'); }
                $out[$k] = $v;
            }
        }

        if (isset($input['facebook_app_id'])) $out['facebook_app_id'] = Sanitize::text($input['facebook_app_id'], 80);
        if (isset($input['facebook_admins'])) $out['facebook_admins'] = Sanitize::text($input['facebook_admins'], 300);
        if (isset($input['social_default_image'])) $out['social_default_image'] = Sanitize::url($input['social_default_image'], 500);

        if (isset($input['social_profiles']) && is_array($input['social_profiles'])) {
            $sp = $out['social_profiles'] ?? array();
            foreach ($input['social_profiles'] as $k => $v) {
                $k = sanitize_key($k);
                $sp[$k] = Sanitize::url($v, 500);
            }
            $out['social_profiles'] = $sp;
        }

        if (isset($input['sitemap_post_types']) && is_array($input['sitemap_post_types'])) {
            $pt = array();
            foreach ($input['sitemap_post_types'] as $k => $v) {
                $pt[sanitize_key($k)] = (int) (!empty($v));
            }
            $out['sitemap_post_types'] = $pt;
        }

        if (isset($input['sitemap_taxonomies']) && is_array($input['sitemap_taxonomies'])) {
            $tx = array();
            foreach ($input['sitemap_taxonomies'] as $k => $v) {
                $tx[sanitize_key($k)] = (int) (!empty($v));
            }
            $out['sitemap_taxonomies'] = $tx;
        }

        if (isset($input['site_type'])) {
            $st = Sanitize::text($input['site_type'], 20);
            $out['site_type'] = in_array($st, array('organization','person'), true) ? $st : $out['site_type'];
        }
        if (isset($input['org_name'])) $out['org_name'] = Sanitize::text($input['org_name'], 150);
        if (isset($input['person_name'])) $out['person_name'] = Sanitize::text($input['person_name'], 150);
        if (isset($input['logo_id'])) $out['logo_id'] = (int) $input['logo_id'];

        foreach (array('gsc_client_id','gsc_client_secret','gsc_property','gsc_refresh_token') as $k) {
            if (isset($input[$k])) $out[$k] = Sanitize::text($input[$k], 500);
        }
        foreach (array('local_business_type','local_name','local_phone','local_email','local_street','local_city','local_state','local_zip','local_country','local_lat','local_lng') as $k) {
            if (isset($input[$k])) $out[$k] = Sanitize::text($input[$k], 200);
        }

        if (isset($input['allowed_roles']) && is_array($input['allowed_roles'])) {
            $ar = array();
            foreach ($input['allowed_roles'] as $rk => $rv) {
                $rk = sanitize_key($rk);
                $ar[$rk] = (int) (!empty($rv));
            }
            $ar['administrator'] = 1;
            $out['allowed_roles'] = $ar;
        }

        if (isset($input['schema_post_type_map']) && is_array($input['schema_post_type_map'])) {
    $map = array();
    foreach ($input['schema_post_type_map'] as $pt => $type) {
        $pt = sanitize_key($pt);
        $type = Sanitize::text($type, 50);
        $map[$pt] = $type;
    }
    $out['schema_post_type_map'] = $map;
}
	if (isset($input['alerts_enabled'])) {
		$out['alerts_enabled'] = Sanitize::bool_int($input['alerts_enabled']);
	}

	if (isset($input['alerts_emails'])) {
		$emails_raw = (string) wp_unslash($input['alerts_emails']);
		$emails_raw = str_replace(array("\r\n","\r"), "\n", $emails_raw);
		$parts = preg_split('/[\n,]+/', $emails_raw);
		$clean = array();

		foreach ($parts as $e) {
			$e = trim($e);
			if ($e === '') continue;
			$v = sanitize_email($e);
			if ($v && is_email($v)) $clean[] = $v;
		}

		$out['alerts_emails'] = implode(", ", array_unique($clean));
	}

	if (isset($input['injection_enabled'])) {
		$out['injection_enabled'] = Sanitize::bool_int($input['injection_enabled']);
	}

	foreach (array('verify_google','verify_bing') as $k) {
		if (isset($input[$k])) {
			$out[$k] = Sanitize::text($input[$k], 200);
		}
	}

	if (isset($input['custom_schema_sitewide'])) {
		$raw = is_string($input['custom_schema_sitewide']) ? $input['custom_schema_sitewide'] : '';
		$raw = trim($raw);

		if ($raw === '') {
			$out['custom_schema_sitewide'] = '';
		} else {
			$decoded = json_decode($raw, true);

			if (json_last_error() === JSON_ERROR_NONE && (is_array($decoded) || is_object($decoded))) {
				$out['custom_schema_sitewide'] = wp_json_encode($decoded);
			} else {
				add_settings_error(
					'aegisseo_settings',
					'aegisseo_custom_schema_sitewide_invalid',
					__('Custom schema JSON is invalid. Please paste valid JSON.', 'aegisseo'),
					'error'
				);
			}
		}
	}

	foreach (array('inject_head_html','inject_footer_html') as $k) {
		if (!isset($input[$k])) { continue; }

		$raw = is_string($input[$k]) ? $input[$k] : '';
		$raw = str_replace("\0", '', $raw);

		$raw = str_replace(array('<?', '?>'), '', $raw);

		if (strlen($raw) > 20000) {
			$raw = substr($raw, 0, 20000);
		}

		if (current_user_can('unfiltered_html')) {
			$out[$k] = $raw;
		} else {
			$out[$k] = wp_kses_post($raw);
		}
	}
	
	if (isset($input['alerts_drop_clicks_pct'])) {
		$out['alerts_drop_clicks_pct'] = $this->int_range($input['alerts_drop_clicks_pct'], 5, 95, $out['alerts_drop_clicks_pct']);
	}
	if (isset($input['alerts_drop_impressions_pct'])) {
		$out['alerts_drop_impressions_pct'] = $this->int_range($input['alerts_drop_impressions_pct'], 5, 95, $out['alerts_drop_impressions_pct']);
	}
	if (isset($input['alerts_drop_ctr_pct'])) {
		$out['alerts_drop_ctr_pct'] = $this->int_range($input['alerts_drop_ctr_pct'], 5, 95, $out['alerts_drop_ctr_pct']);
	}
	if (isset($input['alerts_position_worse'])) {
		$pos = is_numeric($input['alerts_position_worse']) ? (float) $input['alerts_position_worse'] : (float) $out['alerts_position_worse'];
		if ($pos < 0.5) $pos = 0.5;
		if ($pos > 20) $pos = 20;
		$out['alerts_position_worse'] = $pos;
	}

	return $out;
    }

	private function int_range($value, $min, $max, $fallback) {
		if ($value === '' || $value === null) return (int) $fallback;

		// Normalize
		$n = is_numeric($value) ? (int) $value : (int) preg_replace('/[^0-9\-]/', '', (string) $value);

		if ($n < (int) $min) $n = (int) $min;
		if ($n > (int) $max) $n = (int) $max;

		return $n;
	}

    
    public function field_sitemap_last_ping() {
        $lp = $this->options->get('sitemap_last_ping', array());
        if (!is_array($lp) || empty($lp['time'])) {
            echo '<p><em>' . esc_html__('No ping has been recorded yet.', 'aegisseo') . '</em></p>';
            return;
        }

        $ok = !empty($lp['ok']);
        $time = (int) $lp['time'];
        $sitemap = !empty($lp['sitemap']) ? (string) $lp['sitemap'] : '';
        $results = !empty($lp['results']) && is_array($lp['results']) ? $lp['results'] : array();

        echo '<div style="padding:10px 12px;border:1px solid #ccd0d4;border-radius:6px;max-width:700px;background:#fff;">';
        echo '<p style="margin:0 0 6px 0;"><strong>' . esc_html($ok ? __('Success', 'aegisseo') : __('Failed', 'aegisseo')) . '</strong></p>';
        echo '<p style="margin:0 0 6px 0;">' . esc_html__('Time:', 'aegisseo') . ' ' . esc_html(gmdate('Y-m-d H:i:s', $time) . ' UTC') . '</p>';
        if ($sitemap !== '') {
            echo '<p style="margin:0 0 6px 0;">' . esc_html__('Sitemap:', 'aegisseo') . ' <code>' . esc_html($sitemap) . '</code></p>';
        }
        if (!empty($results)) {
            echo '<p style="margin:0;">' . esc_html__('Targets:', 'aegisseo') . ' ';
            $parts = array();
            foreach ($results as $k => $code) {
                $parts[] = $k . ': ' . (string) $code;
            }
            echo esc_html(implode(' | ', $parts));
            echo '</p>';
        }
        echo '</div>';
    }

	private function get_value($key) {
        $opts = $this->options->get_all();
        return $opts[$key] ?? '';
    }

    public function field_separator() {
        $val = $this->get_value('separator');
        echo '<input type="text" name="' . esc_attr(Options::OPTION_KEY) . '[separator]" value="' . esc_attr($val) . '" class="regular-text" maxlength="5" />';
        echo '<p class="description">' . esc_html__('Use %%sep%% in templates.', 'aegisseo') . '</p>';
    }

    public function field_text($args) {
        $key = $args['key'];
        $val = $this->get_value($key);
        echo '<input type="text" name="' . esc_attr(Options::OPTION_KEY) . '[' . esc_attr($key) . ']" value="' . esc_attr($val) . '" class="regular-text" />';
        echo '<p class="description">' . esc_html__('Variables: %%title%%, %%sitename%%, %%tagline%%, %%sep%%, %%excerpt%%, %%category%%', 'aegisseo') . '</p>';
    }

	public function field_textarea($args) {
		$key  = $args['key'];
		$val  = $this->get_value($key);
		$rows = isset($args['rows']) ? max(2, (int) $args['rows']) : 3;

		echo '<textarea name="' . esc_attr(Options::OPTION_KEY) . '[' . esc_attr($key) . ']" rows="' . esc_attr($rows) . '" class="large-text">' . esc_textarea($val) . '</textarea>';
	}

	public function section_injection_desc() {
		echo '<p class="description">' . esc_html__('AegisSEO can inject verification meta tags, add site-wide JSON-LD schema, and optionally output custom head/footer snippets. Only administrators should use HTML injection.', 'aegisseo') . '</p>';

		echo '<div style="margin-top:10px;padding:12px;border:1px solid #dcdcde;border-radius:10px;background:#fff;">';
		echo '<strong>' . esc_html__('Enterprise / Google-safe approach', 'aegisseo') . '</strong><br />';
		echo esc_html__('Preferred: use JSON-LD field (validated JSON) for schema. Use head/footer injection only when necessary (e.g., pixels, custom tags).', 'aegisseo');
		echo '</div>';
	}
    
    public function field_post_type_list($args) {
        $key = $args['key'];
        $val = $this->get_value($key);
        if (is_array($val)) {
            $val = implode(', ', array_filter(array_map('strval', $val)));
        }
        $val = (string) $val;
        $placeholder = isset($args['placeholder']) ? (string) $args['placeholder'] : '';
        echo '<textarea name="' . esc_attr(Options::OPTION_KEY) . '[' . esc_attr($key) . ']" rows="2" class="large-text" placeholder="' . esc_attr($placeholder) . '">' . esc_textarea($val) . '</textarea>';
    }

	public function field_checkbox($args) {
        $key = $args['key'];
        $val = (int) $this->get_value($key);
        echo '<label><input type="checkbox" name="' . esc_attr(Options::OPTION_KEY) . '[' . esc_attr($key) . ']" value="1" ' . checked(1, $val, false) . ' /> ' . esc_html__('Enabled', 'aegisseo') . '</label>';
    }

    public function field_select($args) {
        $key = $args['key'];
        $val = esc_attr($this->get_value($key));
        $options = $args['options'] ?? array();
        echo '<select name="' . esc_attr(Options::OPTION_KEY) . '[' . esc_attr($key) . ']">';
        foreach ($options as $k => $label) {
            echo '<option value="' . esc_attr($k) . '" ' . selected($k, $val, false) . '>' . esc_html($label) . '</option>';
        }
        echo '</select>';
    }

    public function field_number($args) {
        $key = $args['key'];
        $val = (int) $this->get_value($key);
        echo '<input type="number" class="small-text" name="' . esc_attr(Options::OPTION_KEY) . '[' . esc_attr($key) . ']" value="' . esc_attr($val) . '" />';
    }

    public function field_social_profiles() {
        $profiles = (array) $this->get_value('social_profiles');
        if (!is_array($profiles)) { $profiles = array(); }

        $fields = array(
            'facebook'  => 'Facebook',
            'twitter'   => 'Twitter / X',
            'instagram' => 'Instagram',
            'linkedin'  => 'LinkedIn',
            'youtube'   => 'YouTube',
            'tiktok'    => 'TikTok',
            'pinterest' => 'Pinterest',
            'threads'   => 'Threads',
        );

        echo '<div class="aegisseo-grid" style="max-width:720px;">';
        foreach ($fields as $k => $label) {
            $val = isset($profiles[$k]) ? (string) $profiles[$k] : '';
            echo '<p style="margin:0 0 10px 0;">';
            echo '<label><strong>' . esc_html($label) . '</strong></label><br />';
            echo '<input type="url" class="regular-text" name="' . esc_attr(Options::OPTION_KEY) . '[social_profiles][' . esc_attr($k) . ']" value="' . esc_attr($val) . '" placeholder="https://..." />';
            echo '</p>';
        }
        echo '</div>';
        echo '<p class="description">' . esc_html__('Used for Schema “sameAs” and for quick site-wide consistency.', 'aegisseo') . '</p>';
    }

    public function field_sitemap_info() {
        $use_index = (int) $this->options->get('sitemap_index_enabled', 1) === 1;
        $url_index = home_url('/sitemap_index.xml');
        $url_legacy = home_url('/sitemap.xml');

        echo '<code>' . esc_html($use_index ? $url_index : $url_legacy) . '</code>';
        if ($use_index) {
            echo '<p class="description">' . esc_html__('Sitemap index is enabled. Child sitemaps are listed inside /sitemap_index.xml.', 'aegisseo') . '</p>';
        }
        echo '<p class="description">' . esc_html__('After enabling sitemaps, go to Settings → Permalinks and click “Save Changes” once to ensure rewrite rules are flushed.', 'aegisseo') . '</p>';

        // Conflict diagnostics (display-only). These checks do not change behavior.
        if (!function_exists('is_plugin_active')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $conflicts = array();
        // WordPress core sitemaps.
        if (function_exists('wp_sitemaps_get_server')) {
            $core_enabled = (bool) apply_filters( 'wp_sitemaps_enabled', true ); // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Core WP filter.
            if ($core_enabled) {
                $conflicts[] = __('WordPress core sitemap is enabled (wp-sitemap.xml). Consider using only one sitemap provider to avoid duplicates.', 'aegisseo');
            }
        }

        // Common SEO plugins that also generate sitemaps.
        $known = array(
            'wordpress-seo/wp-seo.php' => 'Yoast SEO',
            'rank-math/rank-math.php' => 'Rank Math',
            'all-in-one-seo-pack/all_in_one_seo_pack.php' => 'All in One SEO',
            'aioseo/aioseo.php' => 'All in One SEO',
            'autodescription/autodescription.php' => 'The SEO Framework',
            'seo-by-rank-math/rank-math.php' => 'Rank Math',
            'wp-seopress/seopress.php' => 'SEOPress',
        );

        foreach ($known as $file => $name) {
            if (is_plugin_active($file)) {
                $conflicts[] = sprintf(
                    /* translators: %s: plugin name */
                    __('Another SEO plugin that may generate sitemaps is active: %s. Consider disabling its sitemap feature to avoid duplicates.', 'aegisseo'),
                    $name
                );
            }
        }

        if (!empty($conflicts)) {
            echo '<div style="margin-top:10px; padding:10px; border-left:4px solid #dba617; background:#fff8e5;">';
            echo '<strong>' . esc_html__('Sitemap conflict check:', 'aegisseo') . '</strong>';
            echo '<ul style="margin:8px 0 0 18px;">';
            foreach ($conflicts as $msg) {
                echo '<li>' . esc_html($msg) . '</li>';
            }
            echo '</ul>';
            echo '</div>';
        }
    }

    public function field_sitemap_post_types() {
        $current = (array) $this->get_value('sitemap_post_types');
        $post_types = get_post_types(array('public' => true), 'objects');

        echo '<fieldset>';
        foreach ($post_types as $pt => $obj) {
            if ($pt === 'attachment') { continue; }
            $checked = !empty($current[$pt]) ? 1 : 0;
            echo '<label style="display:block;margin:4px 0;">';
            echo '<input type="checkbox" name="' . esc_attr(Options::OPTION_KEY) . '[sitemap_post_types][' . esc_attr($pt) . ']" value="1" ' . checked(1, $checked, false) . ' /> ';
			$label = $obj->labels->singular_name ?? $pt;
			$label = preg_replace('/^\s*Avada\s+/i', '', (string) $label);

			if ($pt === 'slide') {
				$label = __('Slides', 'aegisseo');
			} elseif ($pt === 'avada_portfolio') {
				$label = __('Portfolio', 'aegisseo');
			} elseif ($pt === 'avada_faq') {
				$label = __('FAQ', 'aegisseo');
			}

			$ui_slug = preg_replace('/^avada_/', '', (string) $pt);

			echo esc_html($label);
			echo ' <span class="description">(' . esc_html($ui_slug) . ')</span>';
            echo '</label>';
        }
        echo '</fieldset>';
        echo '<p class="description">' . esc_html__('Only published items are included. Noindex pages are excluded automatically.', 'aegisseo') . '</p>';
    }

    public function field_sitemap_taxonomies() {
        $current = (array) $this->get_value('sitemap_taxonomies');
        $taxes = get_taxonomies(array('public' => true), 'objects');

        echo '<fieldset>';
        foreach ($taxes as $tax => $obj) {
            $checked = !empty($current[$tax]) ? 1 : 0;
            echo '<label style="display:block;margin:4px 0;">';
            echo '<input type="checkbox" name="' . esc_attr(Options::OPTION_KEY) . '[sitemap_taxonomies][' . esc_attr($tax) . ']" value="1" ' . checked(1, $checked, false) . ' /> ';
			$label = $obj->labels->singular_name ?? $tax;

			/**
			 * UI-only cleanup: remove third-party branding from labels
			 * and hide `avada_` prefix in the displayed slug (do NOT change the real slug).
			 */
			$label = preg_replace('/^\s*Avada\s+/i', '', (string) $label);

			// Hard override for the exact taxonomy in your screenshot
			if ($tax === 'slide-page') {
				$label = __('Sliders', 'aegisseo');
			}

			// UI-only slug shown in parentheses
			$ui_slug = preg_replace('/^avada_/', '', (string) $tax);

			echo esc_html($label);
			echo ' <span class="description">(' . esc_html($ui_slug) . ')</span>';
            echo '</label>';
        }
        echo '</fieldset>';
        echo '<p class="description">' . esc_html__('Only non-empty terms are included.', 'aegisseo') . '</p>';
    }


public function field_schema_post_type_map() {
    $opts = $this->options->get_all();
    $map = $opts['schema_post_type_map'] ?? array();
    if (!is_array($map)) { $map = array(); }

    $templates = array();
    if (!empty(aegisseo()->schema) && method_exists(aegisseo()->schema, 'get_schema_templates')) {
        $templates = aegisseo()->schema->get_schema_templates();
    }

    $pts = get_post_types(array('public' => true), 'objects');
    echo '<div style="max-width:880px;">';
    echo '<p class="description">' . esc_html__('Choose a default Schema type per post type. Per-post overrides (in the editor) take priority.', 'aegisseo') . '</p>';
    echo '<table class="widefat striped" style="max-width:880px;">';
    echo '<thead><tr><th style="width:220px;">' . esc_html__('Post Type', 'aegisseo') . '</th><th>' . esc_html__('Default Schema Type', 'aegisseo') . '</th></tr></thead><tbody>';

    foreach ((array)$pts as $pt => $obj) {
        $label = !empty($obj->labels->singular_name) ? $obj->labels->singular_name : $pt;
        $cur = isset($map[$pt]) ? (string)$map[$pt] : '';
        echo '<tr>';
        echo '<td><strong>' . esc_html($label) . '</strong><br/><code>' . esc_html($pt) . '</code></td>';
        echo '<td><select name="' . esc_attr(Options::OPTION_KEY) . '[schema_post_type_map][' . esc_attr($pt) . ']" style="min-width:260px;">';
        echo '<option value="">' . esc_html__('(Auto)', 'aegisseo') . '</option>';
        foreach ((array)$templates as $k => $v) {
            echo '<option value="' . esc_attr($k) . '"' . selected($cur, (string)$k, false) . '>' . esc_html($v) . '</option>';
        }
        echo '</select></td>';
        echo '</tr>';
    }

    echo '</tbody></table></div>';
}

}