<?php
namespace AegisSEO\SEO;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Monitor_404 {

    private $options;

    public function __construct(Options $options) {
        $this->options = $options;

        add_action('template_redirect', array($this, 'maybe_log_404'), 1);
        add_action('aegisseo_daily_maintenance', array($this, 'daily_prune'));
    }


    /**
     * Sanitize DB prefix to identifier-safe characters.
     *
     * @param string $prefix DB prefix.
     * @return string
     */
    private function sanitize_db_prefix( $prefix ) {
        return preg_replace( '/[^A-Za-z0-9_]/', '', (string) $prefix );
    }

    /**
     * Escape a SQL identifier (table/column) using a strict allowlist and backticks.
     *
     * @param string $identifier Identifier.
     * @return string Escaped identifier with backticks.
     */
    private function escape_sql_identifier( $identifier ) {
        $identifier = preg_replace( '/[^A-Za-z0-9_]/', '', (string) $identifier );
        if ( $identifier === '' ) {
            return '``';
        }
        return '`' . $identifier . '`';
    }

    /**
     * Replace {{table}} token with an escaped identifier.
     *
     * @param string $sql SQL containing {{table}} token.
     * @param string $table Table name.
     * @return string
     */
    private function sql_with_table( $sql, $table ) {
        return str_replace( '{{table}}', $this->escape_sql_identifier( $table ), $sql );
    }

    private function is_enabled() {
        return ((int)$this->options->get('monitor_404_enabled', 1) === 1);
    }

    public function maybe_log_404() {
        if (!$this->is_enabled()) { return; }
        if (!is_404()) { return; }
        if (is_admin()) { return; }
        if (defined('DOING_AJAX') && DOING_AJAX) { return; }

        $request_uri = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
        $request_uri = is_string($request_uri) ? trim($request_uri) : '';
        $request_uri = sanitize_text_field($request_uri);
        if ($request_uri === '') { return; }

        $lower = strtolower($request_uri);
        $skip = array('/favicon.ico', '/robots.txt');
        foreach ($skip as $s) {
            if ($lower === $s) { return; }
        }
        if (preg_match('#\.(css|js|map|jpg|jpeg|png|gif|webp|svg|ico|woff|woff2|ttf|eot)$#i', $lower)) {
            return;
        }

        $ref = isset($_SERVER['HTTP_REFERER']) ? esc_url_raw(wp_unslash($_SERVER['HTTP_REFERER'])) : '';
        $ua  = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '';
        $ip  = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '';

        $this->log($request_uri, $ref, $ua, $ip);
    }

    public function log($url, $referrer = '', $user_agent = '', $ip = '') {
        global $wpdb;

        $t404 = $this->sanitize_db_prefix( $wpdb->prefix ) . 'aegisseo_404';

        $url = (string) $url;
        $url = mb_substr($url, 0, 2000);

        $now = current_time('mysql');
        $hash = md5($url);

        $row = $wpdb->get_row( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->prepare( 'SELECT id, hits FROM %i WHERE url_hash = %s', $t404, $hash ),
            ARRAY_A
        );

        if ($row) {
            $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $t404,
                array(
                    'hits' => (int)$row['hits'] + 1,
                    'last_seen' => $now,
                    'referrer' => $referrer ? mb_substr((string)$referrer, 0, 2000) : null,
                    'user_agent' => $user_agent ? mb_substr((string)$user_agent, 0, 2000) : null,
                    'ip' => $ip ? mb_substr((string)$ip, 0, 45) : null,
                ),
                array('id' => (int)$row['id']),
                array('%d','%s','%s','%s','%s'),
                array('%d')
            );
        } else {
            $wpdb->insert( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
                $t404,
                array(
                    'url' => $url,
                    'url_hash' => $hash,
                    'referrer' => $referrer ? mb_substr((string)$referrer, 0, 2000) : null,
                    'user_agent' => $user_agent ? mb_substr((string)$user_agent, 0, 2000) : null,
                    'ip' => $ip ? mb_substr((string)$ip, 0, 45) : null,
                    'hits' => 1,
                    'first_seen' => $now,
                    'last_seen' => $now,
                ),
                array('%s','%s','%s','%s','%s','%d','%s','%s')
            );

            $max = (int) $this->options->get('monitor_404_max_rows', 5000);
            if ($max > 0) {
                $count = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $wpdb->prepare( 'SELECT COUNT(*) FROM %i WHERE 1=%d', $t404, 1 )
                );
                if ($count > $max) {
                    $excess = $count - $max;
                    $wpdb->query( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->prepare( 'DELETE FROM %i ORDER BY last_seen ASC LIMIT %d', $t404, $excess )
                    );
                }
            }
        }
    }

    public function daily_prune() {
        if (!$this->is_enabled()) { return; }
        $this->prune();
    }

    public function prune() {
        global $wpdb;
        $t404 = $this->sanitize_db_prefix( $wpdb->prefix ) . 'aegisseo_404';

        $days = absint( $this->options->get( 'monitor_404_retention_days', 30 ) );
        if ($days <= 0) { return 0; }

        $cutoff = gmdate('Y-m-d H:i:s', time() - ($days * DAY_IN_SECONDS));
        return $wpdb->query( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->prepare( 'DELETE FROM %i WHERE last_seen < %s', $t404, $cutoff )
        );
    }

    public function clear_all() {
        global $wpdb;
        $t404 = $this->sanitize_db_prefix( $wpdb->prefix ) . 'aegisseo_404';
        return $wpdb->query(
            $wpdb->prepare( 'TRUNCATE TABLE %i', $t404 )
        ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    }

    public function export_csv() {
        if (!current_user_can('manage_options')) { wp_die('Forbidden'); }

        global $wpdb;
        $t404 = $this->sanitize_db_prefix( $wpdb->prefix ) . 'aegisseo_404';

        $rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->prepare( 'SELECT url, hits, first_seen, last_seen, referrer, user_agent, ip FROM %i WHERE 1=%d ORDER BY last_seen DESC LIMIT 5000', $t404, 1 ),
            ARRAY_A
        );

        nocache_headers();
        header('Content-Type: text/csv; charset=UTF-8');
        header('Content-Disposition: attachment; filename="aegisseo-404-monitor.csv"');

        $out = fopen('php://output', 'w');
        fputcsv($out, array('url','hits','first_seen','last_seen','referrer','user_agent','ip'));
        foreach ($rows as $r) {
            fputcsv($out, $r);
        }
        if (is_resource($out)) {
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing php://output stream.
            fclose($out);
        }
        exit;
    }
}
