<?php
namespace AegisSEO\SEO;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Alerts {

    private $options;

    public function __construct(Options $options) {
        $this->options = $options;
        add_action('aegisseo_alerts_runner', array($this, 'run_daily'));
    }


    private function sanitize_db_prefix( $prefix ) {
        return preg_replace( '/[^A-Za-z0-9_]/', '', (string) $prefix );
    }

    private function escape_sql_identifier( $identifier ) {
        $identifier = preg_replace( '/[^A-Za-z0-9_]/', '', (string) $identifier );
        if ( $identifier === '' ) {
            return '``';
        }
        return '`' . $identifier . '`';
    }

    private function sql_with_table( $sql, $table ) {
        return str_replace( '{{table}}', $this->escape_sql_identifier( $table ), $sql );
    }


    private function get_recipients($opts) {
        $raw = isset($opts['alerts_emails']) ? (string)$opts['alerts_emails'] : '';
        $parts = preg_split('/[\s,]+/', $raw);
        $out = array();

        foreach ($parts as $e) {
            $e = trim($e);
            if ($e === '') continue;
            $v = sanitize_email($e);
            if ($v && is_email($v)) $out[] = $v;
        }

        if (empty($out)) {
            $admin = get_option('admin_email');
            if ($admin && is_email($admin)) $out[] = $admin;
        }

        return array_unique($out);
    }

    public function send_test() {
        $opts = $this->options->get_all();
        $to = $this->get_recipients($opts);

        $subject = '[AegisSEO] Test Alert Email';
        $body = "This is a test alert from AegisSEO.\n\nIf you received this, alerts email delivery is working.";
        wp_mail($to, $subject, $body);
    }

    public function run_daily() {
        global $wpdb;

        $opts = $this->options->get_all();
        if (empty($opts['alerts_enabled'])) return;
        if (empty($opts['gsc_refresh_token']) || empty($opts['gsc_property'])) return;

        // Throttle: max once per day
        $today = gmdate('Y-m-d');
        $last = isset($opts['alerts_last_sent_ymd']) ? (string)$opts['alerts_last_sent_ymd'] : '';
        if ($last === $today) return;

        $drop_clicks_pct      = (int) ($opts['alerts_drop_clicks_pct'] ?? 35);
        $drop_impr_pct        = (int) ($opts['alerts_drop_impressions_pct'] ?? 35);
        $drop_ctr_pct         = (int) ($opts['alerts_drop_ctr_pct'] ?? 25);
        $pos_worse            = (float) ($opts['alerts_position_worse'] ?? 2.0);

        $table = $this->sanitize_db_prefix( $wpdb->prefix ) . 'aegisseo_gsc_snapshots';

        // Defensive validation for table identifier (prevents identifier injection).
        if ( preg_match( '/^[A-Za-z0-9_]+$/', $table ) !== 1 ) {
            return;
        }


        // Compare last 7 days vs prior 7 days (14 day window)
        $start_prev = gmdate('Y-m-d', strtotime('-14 days'));
        $start_curr = gmdate('Y-m-d', strtotime('-7 days'));

        // Get rows in last 14 days for all tracked posts (cached to reduce DB load)
        $cache_key = 'aegisseo_alerts_rows_' . md5($t . '|' . $start_prev);
        $rows      = wp_cache_get($cache_key, 'aegisseo');

        if (false === $rows) {
            $rows = $wpdb->get_results( $wpdb->prepare( 'SELECT object_id, snap_date, clicks, impressions, ctr, position FROM %i WHERE snap_date >= %s ORDER BY object_id ASC, snap_date ASC', $table, $start_prev ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            // Cache for 10 minutes (safe if object cache is unavailable).
            wp_cache_set($cache_key, $rows, 'aegisseo', 10 * MINUTE_IN_SECONDS);
        }

        if (empty($rows)) return;

        $by = array();
        foreach ($rows as $r) {
            $pid = (int) $r['object_id'];
            if ($pid <= 0) continue;
            if (!isset($by[$pid])) $by[$pid] = array();
            $by[$pid][] = $r;
        }

        $alerts = array();

        foreach ($by as $pid => $series) {
            $prev = array('clicks'=>0,'impr'=>0,'ctr'=>0,'pos'=>0,'n'=>0);
            $curr = array('clicks'=>0,'impr'=>0,'ctr'=>0,'pos'=>0,'n'=>0);

            foreach ($series as $r) {
                $d = (string) $r['snap_date'];
                $bucket = ($d >= $start_curr) ? 'curr' : 'prev';

                if ($bucket === 'prev') {
                    $prev['clicks'] += (float)$r['clicks'];
                    $prev['impr']   += (float)$r['impressions'];
                    $prev['ctr']    += (float)$r['ctr'];
                    $prev['pos']    += (float)$r['position'];
                    $prev['n']++;
                } else {
                    $curr['clicks'] += (float)$r['clicks'];
                    $curr['impr']   += (float)$r['impressions'];
                    $curr['ctr']    += (float)$r['ctr'];
                    $curr['pos']    += (float)$r['position'];
                    $curr['n']++;
                }
            }

            if ($prev['n'] < 3 || $curr['n'] < 3) continue; // avoid noise

            $prev_clicks = $prev['clicks'] / $prev['n'];
            $curr_clicks = $curr['clicks'] / $curr['n'];

            $prev_impr = $prev['impr'] / $prev['n'];
            $curr_impr = $curr['impr'] / $curr['n'];

            $prev_ctr = $prev['ctr'] / $prev['n'];
            $curr_ctr = $curr['ctr'] / $curr['n'];

            $prev_pos = $prev['pos'] / $prev['n'];
            $curr_pos = $curr['pos'] / $curr['n'];

            $clicks_drop = ($prev_clicks > 0) ? (100.0 * (($prev_clicks - $curr_clicks) / $prev_clicks)) : 0;
            $impr_drop   = ($prev_impr > 0)   ? (100.0 * (($prev_impr - $curr_impr) / $prev_impr)) : 0;
            $ctr_drop    = ($prev_ctr > 0)    ? (100.0 * (($prev_ctr - $curr_ctr) / $prev_ctr)) : 0;
            $pos_worsened = ($curr_pos - $prev_pos);

            $trigger =
                ($clicks_drop >= $drop_clicks_pct) ||
                ($impr_drop   >= $drop_impr_pct) ||
                ($ctr_drop    >= $drop_ctr_pct) ||
                ($pos_worsened >= $pos_worse);

            if (!$trigger) continue;

            $alerts[] = array(
                'post_id' => $pid,
                'title'   => get_the_title($pid),
                'edit'    => admin_url('post.php?post=' . $pid . '&action=edit'),
                'clicks_drop' => round($clicks_drop, 1),
                'impr_drop'   => round($impr_drop, 1),
                'ctr_drop'    => round($ctr_drop, 1),
                'pos_worse'   => round($pos_worsened, 2),
            );
        }

        if (empty($alerts)) {
            // still mark last sent to avoid hammering
            $opts['alerts_last_sent_ymd'] = $today;
            update_option(Options::OPTION_KEY, $opts, false);
            return;
        }

        $to = $this->get_recipients($opts);
        $subject = '[AegisSEO] Performance Drop Alert (GSC evidence)';

        $lines = array();
        $lines[] = "AegisSEO detected performance drops comparing last 7 days vs previous 7 days.";
        $lines[] = "";
        foreach ($alerts as $a) {
            $lines[] = "- {$a['title']} (Post #{$a['post_id']})";
            $lines[] = "  Edit: {$a['edit']}";
            $lines[] = "  Drops: Clicks {$a['clicks_drop']}% | Impr {$a['impr_drop']}% | CTR {$a['ctr_drop']}% | Position worse +{$a['pos_worse']}";
            $lines[] = "";
        }

        wp_mail($to, $subject, implode("\n", $lines));

        $opts['alerts_last_sent_ymd'] = $today;
        update_option(Options::OPTION_KEY, $opts, false);
    }
}