<?php
namespace AegisSEO\SEO;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Autopilot {

    private $options;

    public function __construct(Options $options) {
        $this->options = $options;

        add_action('post_updated', array($this, 'maybe_suggest_redirect_on_slug_change'), 10, 3);
    }


    private function sanitize_db_prefix( $prefix ) {
        return preg_replace( '/[^A-Za-z0-9_]/', '', (string) $prefix );
    }

    private function escape_sql_identifier( $identifier ) {
        $identifier = preg_replace( '/[^A-Za-z0-9_]/', '', (string) $identifier );
        if ( $identifier === '' ) {
            return '``';
        }
        return '`' . $identifier . '`';
    }

    private function sql_with_table( $sql, $table ) {
        return str_replace( '{{table}}', $this->escape_sql_identifier( $table ), $sql );
    }


    public function maybe_suggest_redirect_on_slug_change($post_id, $post_after, $post_before) {
        if (!is_object($post_after) || !is_object($post_before)) { return; }
        if (!(int) $this->options->get('autopilot_enabled', 1)) { return; }
        if (!(int) $this->options->get('autopilot_redirect_suggestions', 1)) { return; }
        if ((int)$post_id <= 0) { return; }

        $pt = get_post_type($post_id);
        if (!$pt) { return; }
        $pto = get_post_type_object($pt);
        if (!$pto || empty($pto->public)) { return; }

        if ($post_after->post_status !== 'publish') { return; }

        $old_slug = (string) $post_before->post_name;
        $new_slug = (string) $post_after->post_name;
        if ($old_slug === '' || $new_slug === '' || $old_slug === $new_slug) { return; }

        $old_url = get_permalink($post_id);
        $new_url = get_permalink($post_id);

        if (!$old_url || !$new_url) { return; }

        $old_path = wp_parse_url($old_url, PHP_URL_PATH);
        $new_path = wp_parse_url($new_url, PHP_URL_PATH);
        if (!$old_path || !$new_path) { return; }

        $old_path_guess = $this->swap_last_path_segment($new_path, $old_slug);

        $limit = (int) $this->options->get('autopilot_daily_redirect_limit', 25);
        if ($limit > 0) {
            global $wpdb;
            $table = $this->sanitize_db_prefix( $wpdb->prefix ) . 'aegisseo_redirects';
            $today = gmdate('Y-m-d');
            $cnt = (int) $wpdb->get_var( $wpdb->prepare( 'SELECT COUNT(*) FROM %i WHERE suggestion_status = %s AND created_at >= %s', $table, 'pending', $today . ' 00:00:00' ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            if ($cnt >= $limit) { return; }
        }
        if ($this->has_existing_redirect_suggestion($post_id, $old_path_guess, $new_path)) {
            return;
        }

        $data = array(
            'source' => $old_path_guess,
            'target' => $new_path,
            'match_type' => 'exact',
            'status_code' => 301,
            'enabled' => 0,
            'is_suggestion' => 1,
            'suggestion_status' => 'pending',
            'source_post_id' => (int)$post_id,
            'reason' => 'Slug changed',
        );

        $res = aegisseo()->redirects->upsert($data);
        if (is_wp_error($res)) { return; }

        if (!empty(aegisseo()->events) && method_exists(aegisseo()->events, 'log_event')) {
            aegisseo()->events->log_event('redirect_suggested', 'post', (int)$post_id, 'redirect_suggestion', $old_path_guess, $new_path);
        }
    }

        private function is_internal_url($url) {
        $url = (string)$url;
        if ($url === '') return false;
        $h = wp_parse_url(home_url('/'));
        $u = wp_parse_url($url);
        if (empty($u['host'])) {
            // Relative URL.
            return true;
        }
        return (!empty($h['host']) && strtolower($u['host']) === strtolower($h['host']));
    }

private function swap_last_path_segment($path, $slug) {
        $path = '/' . ltrim((string)$path, '/');
        $parts = explode('/', trim($path, '/'));
        if (empty($parts)) { return '/' . trim($slug, '/') . '/'; }
        $parts[count($parts)-1] = trim($slug, '/');
        return '/' . implode('/', $parts) . '/';
    }

    private function has_existing_redirect_suggestion($post_id, $source, $target) {
        $list = aegisseo()->redirects->list(array('limit'=>200, 'offset'=>0));
        foreach ((array)$list as $r) {
            if (!isset($r['is_suggestion']) || (int)$r['is_suggestion'] !== 1) { continue; }
            if (!isset($r['source_post_id']) || (int)$r['source_post_id'] !== (int)$post_id) { continue; }
            if ((string)$r['source'] === (string)$source && (string)$r['target'] === (string)$target && (string)$r['suggestion_status'] === 'pending') {
                return true;
            }
        }
        return false;
    }

    public function suggest_schema($post_id) {
        $post = get_post($post_id);
        if (!$post || $post->post_status !== 'publish') { return array(); }
        if (!(int) $this->options->get('autopilot_enabled', 1)) { return; }
        if (!(int) $this->options->get('autopilot_schema_suggestions', 1)) { return array(); }

        $content = (string) $post->post_content;
        $why = array();
        $type = '';

        $faq_hits = 0;
        if (strpos($content, 'faq-block') !== false || strpos($content, 'FAQ') !== false) {
            $faq_hits++;
            $why[] = 'Detected FAQ-like content (FAQ block/markup).';
        }
        if (preg_match_all('/\bQ\s*[:\-]\s*(.+?)\b\s*\bA\s*[:\-]\s*(.+?)(?:\n|\r|$)/is', $content, $m)) {
            if (count($m[0]) >= 2) {
                $faq_hits += 2;
                $why[] = 'Detected multiple Q/A pairs (Q: ... A: ...).';
            }
        }

        $howto_hits = 0;
        if (preg_match('/\bStep\s+1\b/i', $content) && preg_match('/\bStep\s+2\b/i', $content)) {
            $howto_hits += 2;
            $why[] = 'Detected step-based instructions (Step 1, Step 2...).';
        }
        if (preg_match('/<ol\b/i', $content) && preg_match('/<li\b/i', $content)) {
            $howto_hits++;
            $why[] = 'Detected an ordered list that may represent steps.';
        }

        if ($faq_hits >= 2) {
            $type = 'FAQPage';
        } elseif ($howto_hits >= 2) {
            $type = 'HowTo';
        }

        if ($type === '') { return array(); }

        $confidence = min(0.95, 0.55 + 0.15 * max($faq_hits, $howto_hits));
        return array('type'=>$type, 'why'=>$why, 'confidence'=>$confidence);
    }

    public function suggest_internal_links($post_id, $max = 3) {
        $post = get_post($post_id);
        if (!$post || $post->post_status !== 'publish') { return array(); }
        if (!(int) $this->options->get('autopilot_enabled', 1)) { return; }
        if (!(int) $this->options->get('autopilot_link_suggestions', 1)) { return array(); }
        $max_opt = (int) $this->options->get('autopilot_max_internal_links', 3);
        if ($max_opt > 0) { $max = $max_opt; }

        $content = (string) $post->post_content;
        $raw = aegisseo()->linking->suggest_links($post_id, 8);
        if (empty($raw)) return array();

        $out = array();
        $used_urls = array();
        foreach ($raw as $r) {
            if (count($out) >= (int)$max) break;
            $url = isset($r['url']) ? (string)$r['url'] : '';
            $title = isset($r['title']) ? (string)$r['title'] : '';
            $tid = isset($r['ID']) ? (int)$r['ID'] : 0;
            if ($tid <= 0 || $url === '' || $title === '') continue;

            if ($tid === (int)$post_id) continue;
            if (isset($used_urls[$url])) continue;

            $anchor = $this->choose_anchor($content, $title);
            if ($anchor === '') continue;

            $used_urls[$url] = 1;
            $out[] = array(
                'target_id' => $tid,
                'url' => $url,
                'title' => $title,
                'anchor' => $anchor,
            );
        }
        return $out;
    }

    private function choose_anchor($content, $title) {
        $title = trim(wp_strip_all_tags((string)$title));
        $content_plain = wp_strip_all_tags((string)$content);

        if ($title !== '' && stripos($content_plain, $title) !== false) {
            return $title;
        }

        $words = preg_split('/\s+/', $title, -1, PREG_SPLIT_NO_EMPTY);
        if (empty($words)) return '';

        for ($n=4; $n>=2; $n--) {
            if (count($words) < $n) continue;
            $phrase = implode(' ', array_slice($words, 0, $n));
            if ($phrase !== '' && stripos($content_plain, $phrase) !== false) return $phrase;
        }

        foreach ($words as $w) {
            $w = trim($w);
            if (strlen($w) < 5) continue;
            if (stripos($content_plain, $w) !== false) return $w;
        }
        return '';
    }

    public function apply_internal_links($post_id, $suggestions, $max = 3) {
        $post = get_post($post_id);
        if (!$post) return array('success'=>false,'changed_count'=>0,'new_content'=>'','errors'=>array('Post not found.'));

        $content = (string)$post->post_content;
        $changed = 0;
        $errors = array();

        if (empty($suggestions)) return array('success'=>false,'changed_count'=>0,'new_content'=>$content,'errors'=>array('No suggestions.'));

        $html = '<div>' . $content . '</div>';
        libxml_use_internal_errors(true);
        $dom = new \DOMDocument('1.0', 'UTF-8');
        $dom->loadHTML('<?xml encoding="utf-8" ?>' . $html, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
        libxml_clear_errors();

        $xpath = new \DOMXPath($dom);

        foreach ((array)$suggestions as $s) {
        if (!(int) $this->options->get('autopilot_enabled', 1)) { return; }
        if (!(int) $this->options->get('autopilot_link_suggestions', 1)) { return array('ok'=>false,'message'=>'Autopilot disabled'); }
            if ($changed >= (int)$max) break;
            $url = isset($s['url']) ? (string)$s['url'] : '';
            $anchor = isset($s['anchor']) ? (string)$s['anchor'] : '';
            if ($url === '' || $anchor === '') continue;
            $existing = $xpath->query("//a[@href='" . htmlspecialchars($url, ENT_QUOTES) . "']");
            if ($existing && $existing->length > 0) continue;

            if (!$this->is_internal_url($url)) continue;

            $all_links = $xpath->query("//a");
            if ($all_links && $all_links->length > 0) {
                foreach ($all_links as $lnk) {
                    $t = (string) $lnk->textContent;
                    if ($t !== '' && stripos($t, $anchor) !== false) {
                        continue 2;
                    }
                }
            }

            $nodes = $xpath->query("//text()[not(ancestor::a) and not(ancestor::script) and not(ancestor::style) and not(ancestor::pre) and not(ancestor::code) and not(ancestor::textarea) and not(ancestor::button) and not(ancestor::nav) and not(ancestor::header) and not(ancestor::footer) and not(ancestor::h1) and not(ancestor::h2) and not(ancestor::h3) and not(ancestor::h4) and not(ancestor::h5) and not(ancestor::h6)]");
            $inserted = false;
            foreach ($nodes as $node) {
                $txt = $node->nodeValue;
                if (strpos($txt, '[') !== false && strpos($txt, ']') !== false) { continue; }
                if (stripos($txt, $anchor) === false) continue;

                $pos = stripos($txt, $anchor);
                $before = substr($txt, 0, $pos);
                $match  = substr($txt, $pos, strlen($anchor));
                $after  = substr($txt, $pos + strlen($anchor));

                $frag = $dom->createDocumentFragment();
                if ($before !== '') $frag->appendChild($dom->createTextNode($before));

                $a = $dom->createElement('a', $match);
                $a->setAttribute('href', esc_url_raw($url));
                $a->setAttribute('rel', 'internal');
                $frag->appendChild($a);

                if ($after !== '') $frag->appendChild($dom->createTextNode($after));

                $node->parentNode->replaceChild($frag, $node);
                $inserted = true;
                break;
            }

            if ($inserted) {
                $changed++;
            } else {
                $errors[] = 'Could not place anchor: ' . $anchor;
            }
        }

        $div = $dom->getElementsByTagName('div')->item(0);
        $new_content = '';
        if ($div) {
            foreach ($div->childNodes as $child) {
                $new_content .= $dom->saveHTML($child);
            }
        } else {
            $new_content = $content;
        }

        return array('success'=>($changed>0), 'changed_count'=>$changed, 'new_content'=>$new_content, 'errors'=>$errors);
    }
}