<?php
namespace AegisSEO\SEO;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Breadcrumbs {

    private $options;
    private $schema;

    public function __construct(Options $options, Schema $schema) {
        $this->options = $options;
        $this->schema  = $schema;

        add_shortcode('aegisseo_breadcrumbs', array($this, 'shortcode'));
        add_filter('aegisseo_schema_graph', array($this, 'add_breadcrumb_schema'), 20, 2);
    }

    private function enabled() {
        return ((int)$this->options->get('breadcrumbs_enabled', 0) === 1);
    }

    public function shortcode($atts = array()) {
        if (!$this->enabled()) { return ''; }
        $atts = shortcode_atts(array('class' => 'aegisseo-breadcrumbs'), $atts, 'aegisseo_breadcrumbs');
        return $this->render($atts['class']);
    }

    public function render($class = 'aegisseo-breadcrumbs') {
        if (!$this->enabled()) { return ''; }

        $items = $this->get_items();
        if (empty($items)) { return ''; }

        $sep = esc_html($this->options->get('breadcrumbs_sep', '›'));

        $out = '<nav class="' . esc_attr($class) . '" aria-label="Breadcrumbs">';
        $out .= '<span class="aegisseo-bc">';
        $i = 0;
        foreach ($items as $item) {
            $i++;
            if ($i > 1) {
                $out .= ' <span class="aegisseo-bc-sep">' . $sep . '</span> ';
            }
            if (!empty($item['url']) && $i < count($items)) {
                $out .= '<a href="' . esc_url($item['url']) . '">' . esc_html($item['label']) . '</a>';
            } else {
                $out .= '<span class="aegisseo-bc-current">' . esc_html($item['label']) . '</span>';
            }
        }
        $out .= '</span></nav>';

        return $out;
    }

    private function get_items() {
        $items = array();
        $items[] = array('label' => get_bloginfo('name'), 'url' => home_url('/'));

        if (is_front_page()) {
            return $items;
        }

        if (is_home()) {
            $items[] = array('label' => get_the_title(get_option('page_for_posts')), 'url' => '');
            return $items;
        }

        if (is_singular()) {
            $post_id = get_queried_object_id();
            $post_type = get_post_type($post_id);

            if ($post_type === 'post') {
                $cats = get_the_category($post_id);
                if (!empty($cats) && !is_wp_error($cats)) {
                    // Use primary-ish: first category.
                    $cat = $cats[0];
                    $items[] = array('label' => $cat->name, 'url' => get_category_link($cat->term_id));
                }
            }

            $items[] = array('label' => get_the_title($post_id), 'url' => '');
            return $items;
        }

        if (is_category() || is_tag() || is_tax()) {
            $term = get_queried_object();
            if ($term && !is_wp_error($term)) {
                $items[] = array('label' => single_term_title('', false), 'url' => '');
            }
            return $items;
        }

        if (is_search()) {
            /* translators: %s: search query. */
            $items[] = array('label' => sprintf(__('Search results for: %s', 'aegisseo'), get_search_query()), 'url' => '');
            return $items;
        }

        if (is_404()) {
            $items[] = array('label' => __('404 Not Found', 'aegisseo'), 'url' => '');
            return $items;
        }

        return $items;
    }

    public function add_breadcrumb_schema($graph, $context) {
        if (!$this->enabled()) { return $graph; }
        if (is_admin() || is_feed()) { return $graph; }

        $items = $this->get_items();
        if (count($items) < 2) { return $graph; }

        $list = array();
        $pos = 0;
        foreach ($items as $it) {
            $pos++;
            $url = !empty($it['url']) ? $it['url'] : (is_singular() ? get_permalink(get_queried_object_id()) : home_url(add_query_arg(array(), $GLOBALS['wp']->request)));
            $list[] = array(
                '@type' => 'ListItem',
                'position' => $pos,
                'name' => $it['label'],
                'item' => $url,
            );
        }

        $graph[] = array(
            '@type' => 'BreadcrumbList',
            '@id'   => trailingslashit(home_url('/')) . '#breadcrumb',
            'itemListElement' => $list,
        );

        return $graph;
    }
}