<?php
namespace AegisSEO\SEO;

if (!defined('ABSPATH')) { exit; }

class Events {

    private $tracked_meta_keys = array(
        '_aegisseo_title',
        '_aegisseo_description',
        '_aegisseo_canonical',
        '_aegisseo_robots',
        '_aegisseo_focus_phrase',
        '_aegisseo_schema_mode',
        '_aegisseo_schema_type',
        '_aegisseo_schema_data',
        '_aegisseo_noindex',
        '_aegisseo_nofollow',
        '_aegisseo_og_title',
        '_aegisseo_og_description',
        '_aegisseo_og_image',
        '_aegisseo_twitter_title',
        '_aegisseo_twitter_description',
        '_aegisseo_twitter_image',
    );

    public function __construct() {
        add_filter('add_post_metadata', array($this, 'capture_add_post_meta'), 10, 5);
        add_filter('update_post_metadata', array($this, 'capture_update_post_meta'), 10, 5);
        add_filter('delete_post_metadata', array($this, 'capture_delete_post_meta'), 10, 5);
        add_action('post_updated', array($this, 'capture_post_updated'), 10, 3);
    }

    private function is_tracked_meta($meta_key) {
        return in_array($meta_key, $this->tracked_meta_keys, true);
    }

	private function is_allowed_post_object($post_id) {
		$post_id = (int) $post_id;
		if ($post_id <= 0) { return false; }

		$pt = get_post_type($post_id);
		if (!$pt) { return false; }

		// Ignore internal/system types + any AegisSEO internal post types.
		$deny = array(
			'revision',
			'nav_menu_item',
			'custom_css',
			'customize_changeset',
			'oembed_cache',
			'user_request',
			'wp_block',
			'wp_template',
			'wp_template_part',
			'wp_navigation',
			'wp_global_styles',
			'attachment',
		);
		if (in_array($pt, $deny, true)) { return false; }
		if (stripos($pt, 'aegisseo') === 0) { return false; }

		// Prefer AegisSEO scope post types (your intended content types).
		$scope = array();
		if (function_exists('aegisseo')) {
			$plugin = aegisseo();
			if ($plugin && isset($plugin->options) && method_exists($plugin->options, 'get_scope_post_types')) {
				$scope = (array) $plugin->options->get_scope_post_types();
			}
		}

		if (!empty($scope)) {
			return in_array($pt, $scope, true);
		}

		// Fallback: allow public post types only if scope not available.
		$public = get_post_types(array('public' => true), 'names');
		if (!is_array($public) || empty($public)) { return false; }

		return in_array($pt, $public, true);
	}

	private function get_allowed_event_post_types_sql_in() {
		static $cached = null;
		if (null !== $cached) { return $cached; }
		// Prefer AegisSEO scope post types so event list only shows real content.
		$scope = array();
		if (function_exists('aegisseo')) {
			$plugin = aegisseo();
			if ($plugin && isset($plugin->options) && method_exists($plugin->options, 'get_scope_post_types')) {
				$scope = (array) $plugin->options->get_scope_post_types();
			}
		}
		if (!empty($scope)) {
			$cached = array_values(array_filter(array_map('sanitize_key', $scope)));
			if (empty($cached)) { $cached = array('post', 'page'); }
			return $cached;
		}

		$public = get_post_types(array('public' => true), 'names');
		if (!is_array($public)) { $public = array(); }

		$deny = array(
			'revision',
			'nav_menu_item',
			'custom_css',
			'customize_changeset',
			'oembed_cache',
			'user_request',
			'wp_block',
			'wp_template',
			'wp_template_part',
			'wp_navigation',
			'wp_global_styles',
			'attachment',
		);

		$allowed = array();
		foreach ($public as $pt) {
			if (!is_string($pt) || $pt === '') { continue; }
			if (in_array($pt, $deny, true)) { continue; }
			if (stripos($pt, 'aegisseo') === 0) { continue; }
			$allowed[] = $pt;
		}

		if (empty($allowed)) { $allowed = array('post','page'); }

		$cached = $allowed;
		return $cached;
	}

	public function capture_add_post_meta($check, $object_id, $meta_key, $meta_value, $unique) {
		if (!$this->is_allowed_post_object((int) $object_id)) { return $check; }
		if (!$this->is_tracked_meta($meta_key)) { return $check; }
		$this->log_event('meta_add', 'post', (int) $object_id, $meta_key, null, $meta_value);
		return $check;
	}

	public function capture_update_post_meta($check, $object_id, $meta_key, $meta_value, $prev_value) {
		if (!$this->is_allowed_post_object((int) $object_id)) { return $check; }
		if (!$this->is_tracked_meta($meta_key)) { return $check; }
        $old = get_post_meta((int) $object_id, $meta_key, true);
        $this->log_event('meta_update', 'post', (int) $object_id, $meta_key, $old, $meta_value);
        return $check;
    }

    public function capture_delete_post_meta($check, $object_id, $meta_key, $meta_value, $delete_all) {
        if (!$this->is_tracked_meta($meta_key)) { return $check; }
        $old = get_post_meta((int) $object_id, $meta_key, true);
        $this->log_event('meta_delete', 'post', (int) $object_id, $meta_key, $old, null);
        return $check;
    }

    public function log_event($event_type, $object_type, $object_id = 0, $meta_key = null, $old_value = null, $new_value = null, $user_id = null) {
        global $wpdb;

        $events_table = $wpdb->prefix . 'aegisseo_events';
        $posts_table  = $wpdb->posts;
if (null === $user_id) {
            $user_id = get_current_user_id();
        }

        $data = array(
            'event_type'   => (string) $event_type,
            'object_type'  => (string) $object_type,
            'object_id'    => (int) $object_id,
            'meta_key'     => $meta_key ? (string) $meta_key : null, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
            'old_value'    => is_scalar($old_value) || null === $old_value ? $old_value : wp_json_encode($old_value),
            'new_value'    => is_scalar($new_value) || null === $new_value ? $new_value : wp_json_encode($new_value),
            'user_id'      => (int) $user_id,
            'created_at'   => current_time('mysql'),
        );

        $formats = array('%s','%s','%d','%s','%s','%s','%d','%s');

        if (null === $data['meta_key']) {
            $formats[3] = '%s';
            $data['meta_key'] = null; // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
        }

        $wpdb->insert($table, $data, $formats); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    }

public function capture_post_updated($post_id, $post_after, $post_before) {
	if (!$this->is_allowed_post_object((int) $post_id)) { return; }
    if (!is_object($post_after) || !is_object($post_before)) { return; }
    $post_id = (int) $post_id;
    if ($post_id <= 0) { return; }

    $pt = get_post_type($post_id);
    if (!$pt) { return; }
    $pto = get_post_type_object($pt);
    if (!$pto || empty($pto->public)) { return; }

    $old_title = (string) $post_before->post_title;
    $new_title = (string) $post_after->post_title;
    if ($old_title !== $new_title) {
        $this->log_event('title_changed', 'post', $post_id, 'post_title', $old_title, $new_title);
    }

    $old_slug = (string) $post_before->post_name;
    $new_slug = (string) $post_after->post_name;
    if ($old_slug !== '' && $new_slug !== '' && $old_slug !== $new_slug) {
        $this->log_event('slug_changed', 'post', $post_id, 'post_slug', $old_slug, $new_slug);
    }

    $old_content = (string) $post_before->post_content;
    $new_content = (string) $post_after->post_content;

    $old_text = trim(wp_strip_all_tags($old_content));
    $new_text = trim(wp_strip_all_tags($new_content));

    $old_words = $old_text !== '' ? str_word_count($old_text) : 0;
    $new_words = $new_text !== '' ? str_word_count($new_text) : 0;

    $old_h = 0; $new_h = 0;
    if ($old_content !== '') { $old_h = preg_match_all('/<h[1-6][^>]*>/i', $old_content); }
    if ($new_content !== '') { $new_h = preg_match_all('/<h[1-6][^>]*>/i', $new_content); }

    if ($old_words !== $new_words || $old_h !== $new_h) {
        $summary = sprintf('Words: %d→%d; Headings: %d→%d', (int)$old_words, (int)$new_words, (int)$old_h, (int)$new_h);
        $this->log_event('onpage_changed', 'post', $post_id, 'onpage_diff', null, $summary);
    }
}


	public function get_recent( $limit = 50, $object_type = null, $object_id = null, $content_only = false ) {
		global $wpdb;

		if ( ! $wpdb->has_cap( 'identifier_placeholders' ) ) {
			wp_die( esc_html__( 'Unsupported WordPress version.', 'aegisseo' ) );
		}

		$limit = max( 1, absint( $limit ) );

		$events_table = $wpdb->prefix . 'aegisseo_events';
		$events_table = preg_replace( '/[^a-zA-Z0-9_]/', '', $events_table );
		$posts_table  = $wpdb->posts;
		$posts_table  = preg_replace( '/[^a-zA-Z0-9_]/', '', $posts_table );

		$object_type_sanitized = null;
		if ( null !== $object_type ) {
			$object_type_sanitized = sanitize_key( $object_type );
			if ( '' === $object_type_sanitized ) {
				$object_type_sanitized = null;
			}
		}

		$object_id_int = null;
		if ( null !== $object_id ) {
			$object_id_int = absint( $object_id );
			if ( 0 === $object_id_int ) {
				$object_id_int = null;
			}
		}

		$cache_parts = array(
			'limit'        => $limit,
			'object_type'  => $object_type_sanitized,
			'object_id'    => $object_id_int,
			'content_only' => (bool) $content_only,
		);

		// When content_only is applied (and no explicit object filters), include allowed types in cache key.
		if ( null === $object_type_sanitized && null === $object_id_int && $content_only ) {
			$allowed_types              = $this->get_allowed_event_post_types();
			$allowed_types              = array_values( array_filter( array_map( 'sanitize_key', (array) $allowed_types ) ) );
			$cache_parts['allowed_csv'] = implode( ',', $allowed_types );
		}

		$cache_key = 'aegisseo_events_recent_' . md5( wp_json_encode( $cache_parts ) );
		$cached    = wp_cache_get( $cache_key, 'aegisseo' );
		if ( false !== $cached ) {
			return $cached;
		}

		// Fetch rows.
		if ( null !== $object_type_sanitized && null !== $object_id_int ) {
			$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$wpdb->prepare(
					'SELECT * FROM %i WHERE object_type = %s AND object_id = %d ORDER BY created_at DESC LIMIT %d',
					$events_table,
					$object_type_sanitized,
					$object_id_int,
					$limit
				),
				ARRAY_A
			);
		} elseif ( null !== $object_type_sanitized ) {
			$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$wpdb->prepare(
					'SELECT * FROM %i WHERE object_type = %s ORDER BY created_at DESC LIMIT %d',
					$events_table,
					$object_type_sanitized,
					$limit
				),
				ARRAY_A
			);
		} elseif ( null !== $object_id_int ) {
			$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$wpdb->prepare(
					'SELECT * FROM %i WHERE object_id = %d ORDER BY created_at DESC LIMIT %d',
					$events_table,
					$object_id_int,
					$limit
				),
				ARRAY_A
			);
		} else {
			if ( $content_only ) {
				$allowed_types = $this->get_allowed_event_post_types();
				$allowed_types = array_values( array_filter( array_map( 'sanitize_key', (array) $allowed_types ) ) );
				$allowed_csv   = implode( ',', $allowed_types );

				$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
					$wpdb->prepare(
						'SELECT * FROM %i WHERE ( object_type <> %s OR object_id IN ( SELECT ID FROM %i WHERE FIND_IN_SET( post_type, %s ) ) ) ORDER BY created_at DESC LIMIT %d',
						$events_table,
						'post',
						$posts_table,
						$allowed_csv,
						$limit
					),
					ARRAY_A
				);
			} else {
				$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
					$wpdb->prepare(
						'SELECT * FROM %i ORDER BY created_at DESC LIMIT %d',
						$events_table,
						$limit
					),
					ARRAY_A
				);
			}
		}

		// Count total (mirrors the same filter logic above).
		if ( null !== $object_type_sanitized && null !== $object_id_int ) {
			$total = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$wpdb->prepare(
					'SELECT COUNT(*) FROM %i WHERE object_type = %s AND object_id = %d',
					$events_table,
					$object_type_sanitized,
					$object_id_int
				)
			);
		} elseif ( null !== $object_type_sanitized ) {
			$total = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$wpdb->prepare(
					'SELECT COUNT(*) FROM %i WHERE object_type = %s',
					$events_table,
					$object_type_sanitized
				)
			);
		} elseif ( null !== $object_id_int ) {
			$total = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$wpdb->prepare(
					'SELECT COUNT(*) FROM %i WHERE object_id = %d',
					$events_table,
					$object_id_int
				)
			);
		} else {
			if ( $content_only ) {
				$allowed_types = $this->get_allowed_event_post_types();
				$allowed_types = array_values( array_filter( array_map( 'sanitize_key', (array) $allowed_types ) ) );
				$allowed_csv   = implode( ',', $allowed_types );

				$total = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
					$wpdb->prepare(
						'SELECT COUNT(*) FROM %i WHERE ( object_type <> %s OR object_id IN ( SELECT ID FROM %i WHERE FIND_IN_SET( post_type, %s ) ) )',
						$events_table,
						'post',
						$posts_table,
						$allowed_csv
					)
				);
			} else {
				$total = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
					$wpdb->prepare(
						'SELECT COUNT(*) FROM %i',
						$events_table
					)
				);
			}
		}

		$data = array(
			'total' => $total,
			'items' => $rows,
		);

		wp_cache_set( $cache_key, $data, 'aegisseo', 60 );
		return $data;
	}


	public function count_all( $object_type = null, $object_id = null, $allowed_post_types = null ) {
		global $wpdb;

		if ( ! $wpdb->has_cap( 'identifier_placeholders' ) ) {
			wp_die( esc_html__( 'Unsupported WordPress version.', 'aegisseo' ) );
		}

		$events_table = $wpdb->prefix . 'aegisseo_events';
		$posts_table  = $wpdb->posts;

		// Hard-sanitize table identifiers.
		$events_table = preg_replace( '/[^A-Za-z0-9_]/', '', (string) $events_table );
		$posts_table  = preg_replace( '/[^A-Za-z0-9_]/', '', (string) $posts_table );

		$object_type_sanitized = null;
		if ( null !== $object_type && '' !== (string) $object_type ) {
			$object_type_sanitized = sanitize_key( (string) $object_type );
			if ( '' === $object_type_sanitized ) {
				$object_type_sanitized = null;
			}
		}

		$object_id_int = null;
		if ( null !== $object_id ) {
			$object_id_int = absint( $object_id );
			if ( 0 === $object_id_int ) {
				$object_id_int = null;
			}
		}

		$allowed_csv = '';
		$allowed     = array();
		if ( is_array( $allowed_post_types ) ) {
			$allowed = array_values( array_filter( array_map( 'sanitize_key', $allowed_post_types ) ) );
		}
		if ( ! empty( $allowed ) ) {
			$allowed_csv = implode( ',', $allowed );
		}

		$cache_key = 'aegisseo_events_count_' . md5(
			wp_json_encode(
				array(
					'object_type'  => $object_type_sanitized,
					'object_id'    => $object_id_int,
					'allowed_csv'  => $allowed_csv,
					'content_only' => (bool) apply_filters( 'aegisseo_events_content_only', true ),
				)
			)
		);

		$cached = wp_cache_get( $cache_key, 'aegisseo' );
		if ( false !== $cached ) {
			return (int) $cached;
		}

		if ( null !== $object_type_sanitized && null !== $object_id_int ) {
			$count = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$wpdb->prepare(
					'SELECT COUNT(*) FROM %i WHERE object_type = %s AND object_id = %d',
					$events_table,
					$object_type_sanitized,
					$object_id_int
				)
			);
		} elseif ( null !== $object_type_sanitized ) {
			$count = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$wpdb->prepare(
					'SELECT COUNT(*) FROM %i WHERE object_type = %s',
					$events_table,
					$object_type_sanitized
				)
			);
		} elseif ( null !== $object_id_int ) {
			$count = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$wpdb->prepare(
					'SELECT COUNT(*) FROM %i WHERE object_id = %d',
					$events_table,
					$object_id_int
				)
			);
		} else {
			if ( '' === $allowed_csv && (bool) apply_filters( 'aegisseo_events_content_only', true ) ) {
				$allowed_pts = $this->get_allowed_event_post_types_sql_in();
				$allowed_csv = implode( ',', array_values( array_filter( array_map( 'sanitize_key', (array) $allowed_pts ) ) ) );
			}

			if ( '' !== $allowed_csv ) {
				$count = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
					$wpdb->prepare(
						'SELECT COUNT(*) FROM %i WHERE ( object_type <> %s OR object_id IN ( SELECT ID FROM %i WHERE FIND_IN_SET( post_type, %s ) ) )',
						$events_table,
						'post',
						$posts_table,
						$allowed_csv
					)
				);
			} else {
				$count = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
					$wpdb->prepare(
						'SELECT COUNT(*) FROM %i',
						$events_table
					)
				);
			}
		}

		wp_cache_set( $cache_key, $count, 'aegisseo', 60 );

		return $count;
	}


	public function get_recent_paged( $per_page = 50, $page = 1, $object_type = null, $object_id = null, $content_only = false ) {
		global $wpdb;

		if ( ! $wpdb->has_cap( 'identifier_placeholders' ) ) {
			wp_die( esc_html__( 'Unsupported WordPress version.', 'aegisseo' ) );
		}

		$per_page = max( 1, absint( $per_page ) );
		$page     = max( 1, absint( $page ) );
		$offset   = ( $page - 1 ) * $per_page;

		$events_table = $wpdb->prefix . 'aegisseo_events';
		$events_table = preg_replace( '/[^a-zA-Z0-9_]/', '', $events_table );
		$posts_table  = $wpdb->posts;
		$posts_table  = preg_replace( '/[^a-zA-Z0-9_]/', '', $posts_table );

		$object_type_sanitized = null;
		if ( null !== $object_type ) {
			$object_type_sanitized = sanitize_key( $object_type );
			if ( '' === $object_type_sanitized ) {
				$object_type_sanitized = null;
			}
		}

		$object_id_int = null;
		if ( null !== $object_id ) {
			$object_id_int = absint( $object_id );
			if ( 0 === $object_id_int ) {
				$object_id_int = null;
			}
		}
		$cache_key = 'aegisseo_events_paged_' . md5(

			wp_json_encode(

				array(

					'per_page'     => $per_page,

					'page'         => $page,

					'object_type'  => $object_type_sanitized,

					'object_id'    => $object_id_int,

					'content_only' => (bool) $content_only,

				)

			)

		);

		$cached = wp_cache_get( $cache_key, 'aegisseo' );

		if ( false !== $cached ) {

			return (array) $cached;

		}




		if ( null !== $object_type_sanitized && null !== $object_id_int ) {
			$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$wpdb->prepare(
					'SELECT * FROM %i WHERE object_type = %s AND object_id = %d ORDER BY created_at DESC LIMIT %d OFFSET %d',
					$events_table,
					$object_type_sanitized,
					$object_id_int,
					$per_page,
					$offset
				),
				ARRAY_A
			);
		} elseif ( null !== $object_type_sanitized ) {
			$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$wpdb->prepare(
					'SELECT * FROM %i WHERE object_type = %s ORDER BY created_at DESC LIMIT %d OFFSET %d',
					$events_table,
					$object_type_sanitized,
					$per_page,
					$offset
				),
				ARRAY_A
			);
		} elseif ( null !== $object_id_int ) {
			$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$wpdb->prepare(
					'SELECT * FROM %i WHERE object_id = %d ORDER BY created_at DESC LIMIT %d OFFSET %d',
					$events_table,
					$object_id_int,
					$per_page,
					$offset
				),
				ARRAY_A
			);
		} else {
			if ( $content_only ) {
				$allowed_types = $this->get_allowed_event_post_types();
				$allowed_types = array_values( array_filter( array_map( 'sanitize_key', (array) $allowed_types ) ) );
				$allowed_csv   = implode( ',', $allowed_types );

				$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
					$wpdb->prepare(
						'SELECT * FROM %i WHERE ( object_type <> %s OR object_id IN ( SELECT ID FROM %i WHERE FIND_IN_SET( post_type, %s ) ) ) ORDER BY created_at DESC LIMIT %d OFFSET %d',
						$events_table,
						'post',
						$posts_table,
						$allowed_csv,
						$per_page,
						$offset
					),
					ARRAY_A
				);
			} else {
				$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
					$wpdb->prepare(
						'SELECT * FROM %i ORDER BY created_at DESC LIMIT %d OFFSET %d',
						$events_table,
						$per_page,
						$offset
					),
					ARRAY_A
				);
			}
		}

		wp_cache_set( $cache_key, $rows, 'aegisseo', 60 );

		return (array) $rows;
	}


	public function get_event( $id ) {
		global $wpdb;

		if ( ! $wpdb->has_cap( 'identifier_placeholders' ) ) {
			wp_die( esc_html__( 'Unsupported WordPress version.', 'aegisseo' ) );
		}

		$id = absint( $id );
		if ( $id < 1 ) {
			return null;
		}
		$cache_key = 'aegisseo_event_' . (string) $id;

		$cached    = wp_cache_get( $cache_key, 'aegisseo' );

		if ( false !== $cached ) {

			return $cached ? (array) $cached : null;

		}




		$events_table = $wpdb->prefix . 'aegisseo_events';
		$events_table = preg_replace( '/[^a-zA-Z0-9_]/', '', $events_table );

		$row = $wpdb->get_row( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
			$wpdb->prepare(
				'SELECT * FROM %i WHERE id = %d LIMIT 1',
				$events_table,
				$id
			),
			ARRAY_A
		);

		wp_cache_set( $cache_key, $row, 'aegisseo', 60 );

		return $row ? $row : null;
	}


}
