<?php
namespace AegisSEO\SEO;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class GSC_Snapshots {
    private $options;
    private $gsc;
    private $events;

    public function __construct(Options $options, GSC $gsc, Events $events) {
        $this->options = $options;
        $this->gsc = $gsc;
        $this->events = $events;

        add_action('aegisseo_gsc_snapshot_runner', array($this, 'run_daily'));
    }

    public function run_daily() {
        $opts = $this->options->get_all();
        if (empty($opts['gsc_refresh_token']) || empty($opts['gsc_property'])) { return; }

        $property = (string) $opts['gsc_property'];
        $end = gmdate('Y-m-d');
        $start = gmdate('Y-m-d', strtotime('-3 days'));

        $posts = get_posts(array(
            'post_type' => 'any',
            'post_status' => 'publish',
            'posts_per_page' => 50,
            'orderby' => 'modified',
            'order' => 'DESC',
            'fields' => 'ids',
        ));
        if (empty($posts)) { return; }

        foreach ($posts as $post_id) {
            $url = get_permalink($post_id);
            if (!$url) continue;

            $m = $this->gsc->fetch_page_summary($property, $url, $start, $end);
            if (is_wp_error($m)) {
                if ($m->get_error_code() === 'aegisseo_gsc_rate_limited') { break; }
                continue;
            }
            $this->upsert_snapshot($end, (int)$post_id, $url, $m);
        }
    }

    public function upsert_snapshot($snap_date, $post_id, $url, $metrics) {
        global $wpdb;

        $table = $wpdb->prefix . 'aegisseo_gsc_snapshots';
        // Sanitize identifier (table name) - wpdb::prepare() cannot parameterize identifiers.
        $table = preg_replace('/[^A-Za-z0-9_]/', '', (string) $table);

        $snap_date = preg_replace('/[^0-9\-]/', '', (string)$snap_date);
        if ($snap_date === '') {
            return false;
        }

        $cache_key = 'aegisseo_gsc_snap_id_' . $snap_date . '_' . (int) $post_id;
        $row = wp_cache_get($cache_key, 'aegisseo');
        if ($row === false) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- read-only query; values prepared; identifier concatenated (prepare cannot parameterize identifiers).
            $row = $wpdb->get_row(
                $wpdb->prepare(
                    'SELECT id FROM %i WHERE snap_date=%s AND object_id=%d',
                    $table,
                    $snap_date,
                    (int) $post_id
                ),
                ARRAY_A
            );
            wp_cache_set($cache_key, $row, 'aegisseo', 300);
        }

        $data = array(
            'snap_date' => $snap_date,
            'object_id' => (int)$post_id,
            'page_url' => (string)$url,
            'clicks' => isset($metrics['clicks']) ? (float)$metrics['clicks'] : 0,
            'impressions' => isset($metrics['impressions']) ? (float)$metrics['impressions'] : 0,
            'ctr' => isset($metrics['ctr']) ? (float)$metrics['ctr'] : 0,
            'position' => isset($metrics['position']) ? (float)$metrics['position'] : 0,
            'created_at' => gmdate('Y-m-d H:i:s'),
        );

        if (!empty($row['id'])) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- wpdb helper; table identifier sanitized above.
            $wpdb->update($table, $data, array('id' => (int) $row['id']), array('%s','%d','%s','%f','%f','%f','%f','%s'), array('%d'));
        } else {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- wpdb helper; table identifier sanitized above.
            $wpdb->insert($table, $data, array('%s','%d','%s','%f','%f','%f','%f','%s'));
        }

        // Invalidate caches related to this object/date.
        wp_cache_delete('aegisseo_gsc_snap_id_' . $snap_date . '_' . (int) $post_id, 'aegisseo');

        return true;
    }

    public function get_series($post_id, $days = 28) {
        global $wpdb;

        $table = $wpdb->prefix . 'aegisseo_gsc_snapshots';
        // Sanitize identifier (table name) - wpdb::prepare() cannot parameterize identifiers.
        $table = preg_replace('/[^A-Za-z0-9_]/', '', (string) $table);

        $days = max(1, min(120, (int) $days));
        $start = gmdate('Y-m-d', strtotime('-' . ($days - 1) . ' days'));

        $cache_key = 'aegisseo_gsc_series_' . (int) $post_id . '_' . $start;
        $cached = wp_cache_get($cache_key, 'aegisseo');
        if ($cached !== false) {
            return $cached;
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- read-only query; values prepared; identifier concatenated (prepare cannot parameterize identifiers).
        $rows = $wpdb->get_results(
            $wpdb->prepare(
                'SELECT * FROM %i WHERE object_id=%d AND snap_date >= %s ORDER BY snap_date ASC',
                $table,
                (int) $post_id,
                $start
            ),
            ARRAY_A
        );
        wp_cache_set($cache_key, $rows, 'aegisseo', 300);

        return $rows;
    }
}