<?php
namespace AegisSEO\SEO;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Htaccess_Helper {

    private $options;

    const BEGIN_MARKER = '# BEGIN AegisSEO';
    const END_MARKER   = '# END AegisSEO';

    public function __construct(Options $options) {
        $this->options = $options;
    }

    public function is_supported() {
        $file = $this->get_file();
        return file_exists($file);
    }

    public function get_file() {
        return trailingslashit(ABSPATH) . '.htaccess';
    }

    public function generate_rules() {
        $force_https = (int) $this->options->get('htaccess_force_https', 0) === 1;
        $force_www   = (int) $this->options->get('htaccess_force_www', 0);
        $trail       = (int) $this->options->get('htaccess_trailing_slash', 0) === 1;

        $lines = array();
        $lines[] = self::BEGIN_MARKER;
        $lines[] = '<IfModule mod_rewrite.c>';
        $lines[] = 'RewriteEngine On';

        if ($force_https) {
            $lines[] = 'RewriteCond %{HTTPS} !=on';
            $lines[] = 'RewriteRule ^ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]';
        }

        if ($force_www === 1) {
            $lines[] = 'RewriteCond %{HTTP_HOST} !^www\. [NC]';
            $lines[] = 'RewriteRule ^ https://www.%{HTTP_HOST}%{REQUEST_URI} [L,R=301]';
        } elseif ($force_www === -1) {
            $lines[] = 'RewriteCond %{HTTP_HOST} ^www\.(.+)$ [NC]';
            $lines[] = 'RewriteRule ^ https://%1%{REQUEST_URI} [L,R=301]';
        }

        if ($trail) {
            $lines[] = 'RewriteCond %{REQUEST_FILENAME} !-f';
            $lines[] = 'RewriteCond %{REQUEST_FILENAME} !-d';
            $lines[] = 'RewriteCond %{REQUEST_URI} !(\.[a-zA-Z0-9]{1,6})$';
            $lines[] = 'RewriteRule ^(.+[^/])$ $1/ [L,R=301]';
        }

        $lines[] = '</IfModule>';
        $lines[] = self::END_MARKER;

        return implode("\n", $lines) . "\n";
    }

    public function apply() {
        $file = $this->get_file();
        if (!file_exists($file)) {
            return new \WP_Error('missing_htaccess', '.htaccess not found.');
        }
        if (!wp_is_writable($file)) {
            return new \WP_Error('htaccess_not_writable', '.htaccess is not writable.');
        }

        $contents = file_get_contents($file);
        if ($contents === false) {
            return new \WP_Error('read_failed', 'Unable to read .htaccess.');
        }

        $backup = $file . '.aegisseo-bak-' . gmdate('Ymd-His');
        @file_put_contents($backup, $contents);

        $new = $this->replace_block($contents, $this->generate_rules());
        $ok = file_put_contents($file, $new);

        if ($ok === false) {
            return new \WP_Error('write_failed', 'Failed to write .htaccess.');
        }

        return true;
    }

    public function restore_latest_backup() {
        $file = $this->get_file();
        $dir = dirname($file);
        $pattern = $dir . '/.htaccess.aegisseo-bak-*';
        $files = glob($pattern);
        if (empty($files)) {
            return new \WP_Error('no_backup', 'No AegisSEO backups found.');
        }
        rsort($files);
        $latest = $files[0];
        $contents = file_get_contents($latest);
        if ($contents === false) {
            return new \WP_Error('read_backup_failed', 'Failed to read backup.');
        }
        if (!wp_is_writable($file)) {
            return new \WP_Error('htaccess_not_writable', '.htaccess is not writable.');
        }
        $ok = file_put_contents($file, $contents);
        if ($ok === false) {
            return new \WP_Error('write_failed', 'Failed to restore .htaccess.');
        }
        return true;
    }

    private function replace_block($contents, $block) {
        $start = strpos($contents, self::BEGIN_MARKER);
        $end   = strpos($contents, self::END_MARKER);

        if ($start !== false && $end !== false && $end > $start) {
            $end = $end + strlen(self::END_MARKER);
            $before = substr($contents, 0, $start);
            $after  = substr($contents, $end);
            return rtrim($before) . "\n\n" . rtrim($block) . "\n\n" . ltrim($after);
        }

        return rtrim($contents) . "\n\n" . rtrim($block) . "\n";
    }
}
