<?php
namespace AegisSEO\SEO;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Redirects {

    private $options;

    public function __construct(Options $options) {
        $this->options = $options;

        add_action('template_redirect', array($this, 'maybe_redirect'), 0);
    }

    private function is_enabled() {
        return ((int)$this->options->get('redirects_enabled', 1) === 1);
    }

    private function request_path_query() {
        $uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        $uri = is_string($uri) ? trim($uri) : '';
        if ($uri === '') { return '/'; }

        if ($uri[0] !== '/') { $uri = '/' . $uri; }
        return $uri;
    }

    public function maybe_redirect() {
        if (!$this->is_enabled()) { return; }
        if (is_admin() || is_feed() || (defined('DOING_AJAX') && DOING_AJAX)) { return; }

        global $wpdb;
        $tred = $wpdb->prefix . 'aegisseo_redirects';
        $tred = preg_replace('/[^a-zA-Z0-9_]/', '', (string) $tred);

        $req = $this->request_path_query();

        $hash = md5('exact:' . $req);

        $cache_key = 'aegisseo_red_exact_' . $hash;
        $row = wp_cache_get($cache_key, 'aegisseo');
        if (false === $row) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name is plugin-controlled and sanitized; result is cached.
            $row = $wpdb->get_row(
                $wpdb->prepare(
                    "SELECT * FROM %i WHERE enabled=%d AND match_type=%s AND source_hash=%s LIMIT 1",
                    $tred,
                    1,
                    'exact',
                    $hash
                ),
                ARRAY_A
            );
            wp_cache_set($cache_key, $row, 'aegisseo');
        }if ($row) {
            $this->do_redirect($row, $req);
            return;
        }

        $rules = wp_cache_get('aegisseo_red_rules_regex', 'aegisseo');
        if (false === $rules) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,PluginCheck.Security.DirectDB.UnescapedDBParameter,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is plugin-controlled and sanitized; result is cached.
            $rules = $wpdb->get_results("SELECT * FROM {$tred} WHERE enabled=1 AND match_type='regex' ORDER BY id DESC LIMIT 200", ARRAY_A);
            wp_cache_set('aegisseo_red_rules_regex', $rules, 'aegisseo');
        }if (!$rules) { return; }

        foreach ($rules as $r) {
            $pattern = $r['source'];
            if (!$this->is_valid_regex($pattern)) { continue; }
            if (@preg_match($pattern, $req)) {
                $this->do_redirect($r, $req);
                return;
            }
        }
    }

    private function is_valid_regex($pattern) {
        if (!is_string($pattern) || trim($pattern) === '') { return false; }
        $delim = substr($pattern, 0, 1);
        if (!in_array($delim, array('#','~','/'), true)) { return false; }
        $ok = @preg_match($pattern, '/test') !== false;
        return $ok;
    }

    private function do_redirect($row, $req) {
        $status = (int) $row['status_code'];
        if (!in_array($status, array(301,302,307,410,451), true)) { $status = 301; }

        $target = (string) $row['target'];

        if ($row['match_type'] === 'regex' && $this->is_valid_regex($row['source'])) {
            $replaced = @preg_replace($row['source'], $target, $req);
            if (is_string($replaced) && $replaced !== '') {
                $target = $replaced;
            }
        }

        if ($status === 410 || $status === 451) {
            status_header($status);
            if (!empty($target)) {
                header('Location: ' . esc_url_raw($this->normalize_target($target)), true, $status);
            }
            exit;
        }

        if (empty($target)) { return; }
        wp_safe_redirect($this->normalize_target($target), $status);
        exit;
    }

    private function normalize_target($target) {
        $target = trim((string)$target);
        if ($target === '') { return home_url('/'); }

        if (preg_match('#^https?://#i', $target)) {
            return esc_url_raw($target);
        }

        if ($target[0] !== '/') { $target = '/' . $target; }
        return home_url($target);
    }

    public function upsert($data) {
        global $wpdb;
        $tred = $wpdb->prefix . 'aegisseo_redirects';
        $tred = preg_replace('/[^a-zA-Z0-9_]/', '', (string) $tred);

        $id = isset($data['id']) ? (int)$data['id'] : 0;
        $source = isset($data['source']) ? (string)$data['source'] : '';
        $target = isset($data['target']) ? (string)$data['target'] : '';
        $match_type = isset($data['match_type']) ? (string)$data['match_type'] : 'exact';
        $status = isset($data['status_code']) ? (int)$data['status_code'] : 301;
        $enabled = isset($data['enabled']) ? (int)$data['enabled'] : 1;
        $is_suggestion = isset($data['is_suggestion']) ? (int)$data['is_suggestion'] : 0;
        $suggestion_status = isset($data['suggestion_status']) ? (string)$data['suggestion_status'] : 'approved';
        $source_post_id = isset($data['source_post_id']) ? (int)$data['source_post_id'] : 0;
        $reason = isset($data['reason']) ? (string)$data['reason'] : '';

        $source = trim($source);
        $target = trim($target);

        if ($match_type !== 'regex') { $match_type = 'exact'; }

        if ($match_type === 'exact') {
            if ($source === '' || $source[0] !== '/') { $source = '/' . ltrim($source, '/'); }
        } else {
            if (!$this->is_valid_regex($source)) { return new \WP_Error('invalid_regex', 'Invalid regex pattern.'); }
        }

        if (!in_array($status, array(301,302,307,410,451), true)) { $status = 301; }

        $now = current_time('mysql');
        $hash = md5($match_type . ':' . $source);

        if ($id > 0) {
            $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $tred,
                array(
                    'source' => $source,
                    'source_hash' => $hash,
                    'target' => $target,
                    'match_type' => $match_type,
                    'status_code' => $status,
                    'enabled' => $enabled ? 1 : 0,
                    'updated_at' => $now,
                'is_suggestion' => $is_suggestion,
                'suggestion_status' => $suggestion_status,
                'source_post_id' => $source_post_id,
                'reason' => $reason,

                    'is_suggestion' => $is_suggestion,
                    'suggestion_status' => $suggestion_status,
                    'source_post_id' => $source_post_id,
                    'reason' => $reason,

                ),
                array('id' => $id),
                array('%s','%s','%s','%s','%d','%d','%s','%d','%s','%d','%s'),
                array('%d')
            );
            return $id;
        }

        $wpdb->insert( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $tred,
            array(
                'source' => $source,
                'source_hash' => $hash,
                'target' => $target,
                'match_type' => $match_type,
                'status_code' => $status,
                'enabled' => $enabled ? 1 : 0,
                'created_at' => $now,
                'updated_at' => $now,
                'is_suggestion' => $is_suggestion,
                'suggestion_status' => $suggestion_status,
                'source_post_id' => $source_post_id,
                'reason' => $reason,

                    'is_suggestion' => $is_suggestion,
                    'suggestion_status' => $suggestion_status,
                    'source_post_id' => $source_post_id,
                    'reason' => $reason,

            ),
            array('%s','%s','%s','%s','%d','%d','%s','%s','%d','%s','%d','%s')
        );

        return (int) $wpdb->insert_id;
    }

    public function delete($id) {
        global $wpdb;
        $tred = $wpdb->prefix . 'aegisseo_redirects';
        $tred = preg_replace('/[^a-zA-Z0-9_]/', '', (string) $tred);
        return $wpdb->delete($tred, array('id' => (int)$id), array('%d')); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    }

    public function get($id) {
        global $wpdb;
        $tred = $wpdb->prefix . 'aegisseo_redirects';
        $tred = preg_replace('/[^a-zA-Z0-9_]/', '', (string) $tred);
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name is plugin-controlled and sanitized; value is prepared.
        return $wpdb->get_row($wpdb->prepare("SELECT * FROM %i WHERE id=%d", $tred, (int) $id), ARRAY_A);
    }

    public function list($args = array()) {
        global $wpdb;
        $tred = $wpdb->prefix . 'aegisseo_redirects';
        $tred = preg_replace('/[^a-zA-Z0-9_]/', '', (string) $tred);

        // Build a safe WHERE clause using only fixed, known column names and placeholders.
        $enabled_is_set = (isset($args['enabled']) && $args['enabled'] !== '');
        $enabled_val    = $enabled_is_set ? (int) $args['enabled'] : 0;

        $match_is_set = (!empty($args['match_type']));
        $mt           = $match_is_set ? (($args['match_type'] === 'regex') ? 'regex' : 'exact') : '';

        $limit  = isset($args['limit']) ? max(1, min(500, (int) $args['limit'])) : 50;
        $offset = isset($args['offset']) ? max(0, (int) $args['offset']) : 0;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Query is prepared inline; read-only admin listing.
        if ($enabled_is_set && $match_is_set) {
            return $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    'SELECT * FROM %i WHERE enabled=%d AND match_type=%s ORDER BY id DESC LIMIT %d OFFSET %d',
                    $tred,
                    $enabled_val,
                    $mt,
                    $limit,
                    $offset
                ),
                ARRAY_A
            );
        }

        if ($enabled_is_set) {
            return $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    'SELECT * FROM %i WHERE enabled=%d ORDER BY id DESC LIMIT %d OFFSET %d',
                    $tred,
                    $enabled_val,
                    $limit,
                    $offset
                ),
                ARRAY_A
            );
        }

        if ($match_is_set) {
            return $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    'SELECT * FROM %i WHERE match_type=%s ORDER BY id DESC LIMIT %d OFFSET %d',
                    $tred,
                    $mt,
                    $limit,
                    $offset
                ),
                ARRAY_A
            );
        }

        return $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->prepare(
                'SELECT * FROM %i ORDER BY id DESC LIMIT %d OFFSET %d',
                $tred,
                $limit,
                $offset
            ),
            ARRAY_A
        );
    }
}
