<?php
namespace AegisSEO\SEO;

use AegisSEO\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Security_SEO {

    private $options;

    private $events;

    public function __construct(Options $options, Events $events) {
        $this->options = $options;
        $this->events  = $events;

        add_action('init', array($this, 'inspect_request'), 1);
    }

    public function inspect_request() {
        if (!(int) $this->options->get('security_seo_enabled', 1)) { return; }

        if (is_admin() || (defined('WP_CLI') && WP_CLI)) { return; }

        $uri = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
        $uri = is_string( $uri ) ? trim( $uri ) : '';
        if ( $uri === '' ) { return; }

        $ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
        $ua = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';

        $path = wp_parse_url($uri, PHP_URL_PATH);
        $query = wp_parse_url($uri, PHP_URL_QUERY);

        if ($query) {
            $q_lc = strtolower(rawurldecode((string)$query));
            $keywords = (array) $this->options->get('security_seo_keywords', array());
            foreach ($keywords as $kw) {
                $kw = strtolower((string)$kw);
                if ($kw !== '' && strpos($q_lc, $kw) !== false) {
                    $this->log('security_spammy_query', array(
                        'uri' => $uri,
                        'ip'  => $ip,
                        'ua'  => $ua,
                        'hit' => $kw,
                    ));
                    break;
                }
            }

            if (preg_match('/(?:redirect|url|next|target)=https?:\/\//i', $query)) {
                $this->log('security_redirect_abuse', array(
                    'uri' => $uri,
                    'ip'  => $ip,
                    'ua'  => $ua,
                ));
            }
        }

        if ($path && preg_match('/sitemap[^\/]*\.xml$/i', $path)) {
            $p_lc = strtolower((string)$path);

            if (strpos($p_lc, '..') !== false || strpos($p_lc, '%2e') !== false) {
                $this->log('security_sitemap_poison', array(
                    'uri' => $uri,
                    'ip'  => $ip,
                    'ua'  => $ua,
                    'reason' => 'traversal_pattern',
                ));
            }

            if ((int)$this->options->get('security_seo_sitemap_strict', 1) === 1) {
                $known = array(
                    '/sitemap_index.xml',
                );
                $is_known = in_array($p_lc, $known, true)
                    || preg_match('#/sitemap-pt-[^/]+(?:-[0-9]+)?\.xml$#i', $p_lc)
                    || preg_match('#/sitemap-tax-[^/]+(?:-[0-9]+)?\.xml$#i', $p_lc)
                    || preg_match('#/sitemap-video\.xml$#i', $p_lc)
                    || preg_match('#/sitemap-news\.xml$#i', $p_lc);

                if (!$is_known) {
                    $this->log('security_sitemap_poison', array(
                        'uri' => $uri,
                        'ip'  => $ip,
                        'ua'  => $ua,
                        'reason' => 'unknown_sitemap_endpoint',
                    ));
                }
            }
        }
    }

    private function log($type, array $payload) {
        if (!(int) $this->options->get('security_seo_log_events', 1)) { return; }

        $key = 'aegisseo_secseo_' . md5($type . '|' . (string)($payload['uri'] ?? ''));
        if (get_transient($key)) { return; }
        set_transient($key, 1, 10 * MINUTE_IN_SECONDS);

        $this->events->log_event($type, 'request', 0, 'security', null, wp_json_encode($payload));
    }
}
